#include <assert.h>
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <poll.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <map>
#include <string>
#include <utility>
#include <vector>

#include "httpinput.h"
#include "log.h"
#include "metacube2.h"
#include "mutexlock.h"
#include "parse.h"
#include "serverpool.h"
#include "state.pb.h"
#include "stream.h"
#include "timespec.h"
#include "util.h"
#include "version.h"

using namespace std;

namespace {

string host_header(const string &host, const string &port)
{
	if (port == "http" || atoi(port.c_str()) == 80) {
		return host;
	} else {
		return host + ":" + port;
	}
}

}  // namespace

extern ServerPool *servers;

HTTPInput::HTTPInput(const string &url, Input::Encoding encoding)
	: state(NOT_CONNECTED),
	  url(url),
	  encoding(encoding),
	  has_metacube_header(false),
	  sock(-1),
	  num_connection_attempts(0),
	  suppress_logging(false)
{
	pthread_mutex_init(&stats_mutex, NULL);
	stats.url = url;
	stats.bytes_received = 0;
	stats.data_bytes_received = 0;
	stats.connect_time = -1;

	last_verbose_connection.tv_sec = -3600;
	last_verbose_connection.tv_nsec = 0;
}

HTTPInput::HTTPInput(const InputProto &serialized)
	: state(State(serialized.state())),
	  url(serialized.url()),
	  encoding(serialized.is_metacube_encoded() ?
	           Input::INPUT_ENCODING_METACUBE :
	           Input::INPUT_ENCODING_RAW),
	  request(serialized.request()),
	  request_bytes_sent(serialized.request_bytes_sent()),
	  response(serialized.response()),
	  http_header(serialized.http_header()),
	  stream_header(serialized.stream_header()),
	  has_metacube_header(serialized.has_metacube_header()),
	  sock(serialized.sock()),
	  num_connection_attempts(0),
	  suppress_logging(false)
{
	pending_data.resize(serialized.pending_data().size());
	memcpy(&pending_data[0], serialized.pending_data().data(), serialized.pending_data().size());

	string protocol, user;
	parse_url(url, &protocol, &user, &host, &port, &path);  // Don't care if it fails.

	pthread_mutex_init(&stats_mutex, NULL);
	stats.url = url;
	stats.bytes_received = serialized.bytes_received();
	stats.data_bytes_received = serialized.data_bytes_received();
	if (serialized.has_connect_time()) {
		stats.connect_time = serialized.connect_time();
	} else {
		stats.connect_time = time(NULL);
	}

	last_verbose_connection.tv_sec = -3600;
	last_verbose_connection.tv_nsec = 0;
}

void HTTPInput::close_socket()
{
	if (sock != -1) {
		safe_close(sock);
		sock = -1;
	}

	MutexLock lock(&stats_mutex);
	stats.connect_time = -1;
}

InputProto HTTPInput::serialize() const
{
	InputProto serialized;
	serialized.set_state(state);
	serialized.set_url(url);
	serialized.set_request(request);
	serialized.set_request_bytes_sent(request_bytes_sent);
	serialized.set_response(response);
	serialized.set_http_header(http_header);
	serialized.set_stream_header(stream_header);
	serialized.set_pending_data(string(pending_data.begin(), pending_data.end()));
	serialized.set_has_metacube_header(has_metacube_header);
	serialized.set_sock(sock);
	serialized.set_bytes_received(stats.bytes_received);
	serialized.set_data_bytes_received(stats.data_bytes_received);
	serialized.set_connect_time(stats.connect_time);
	if (encoding == Input::INPUT_ENCODING_METACUBE) {
		serialized.set_is_metacube_encoded(true);
	} else {
		assert(encoding == Input::INPUT_ENCODING_RAW);
		serialized.set_is_metacube_encoded(false);
	}
	return serialized;
}

int HTTPInput::lookup_and_connect(const string &host, const string &port)
{
	addrinfo *ai;
	int err = getaddrinfo(host.c_str(), port.c_str(), NULL, &ai);
	if (err != 0) {
		if (!suppress_logging) {
			log(WARNING, "[%s] Lookup of '%s' failed (%s).",
				url.c_str(), host.c_str(), gai_strerror(err));
		}
		return -1;
	}

	addrinfo *base_ai = ai;

	// Connect to everything in turn until we have a socket.
	for ( ; ai && !should_stop(); ai = ai->ai_next) {
		int sock = socket(ai->ai_family, SOCK_STREAM, IPPROTO_TCP);
		if (sock == -1) {
			// Could be e.g. EPROTONOSUPPORT. The show must go on.
			continue;
		}

		// Now do a non-blocking connect. This is important because we want to be able to be
		// woken up, even though it's rather cumbersome.

		// Set the socket as nonblocking.
		int one = 1;
		if (ioctl(sock, FIONBIO, &one) == -1) {
			log_perror("ioctl(FIONBIO)");
			safe_close(sock);
			return -1;			
		}

		// Do a non-blocking connect.
		do {
			err = connect(sock, ai->ai_addr, ai->ai_addrlen);
		} while (err == -1 && errno == EINTR);

		if (err == -1 && errno != EINPROGRESS) {
			log_perror("connect");
			safe_close(sock);
			continue;
		}

		// Wait for the connect to complete, or an error to happen.
		for ( ;; ) {
			bool complete = wait_for_activity(sock, POLLIN | POLLOUT, NULL);
			if (should_stop()) {
				safe_close(sock);
				return -1;
			}
			if (complete) {
				break;
			}
		}

		// Check whether it ended in an error or not.
		socklen_t err_size = sizeof(err);
		if (getsockopt(sock, SOL_SOCKET, SO_ERROR, &err, &err_size) == -1) {
			log_perror("getsockopt");
			safe_close(sock);
			continue;
		}

		errno = err;

		if (err == 0) {
			// Successful connect.
			freeaddrinfo(base_ai);
			return sock;
		}

		safe_close(sock);
	}

	// Give the last one as error.
	if (!suppress_logging) {
		log(WARNING, "[%s] Connect to '%s' failed (%s)",
			url.c_str(), host.c_str(), strerror(errno));
	}
	freeaddrinfo(base_ai);
	return -1;
}
	
bool HTTPInput::parse_response(const string &request)
{
	vector<string> lines = split_lines(response);
	if (lines.empty()) {
		if (!suppress_logging) {
			log(WARNING, "[%s] Empty HTTP response from input.", url.c_str());
		}
		return false;
	}

	vector<string> first_line_tokens = split_tokens(lines[0]);
	if (first_line_tokens.size() < 2) {
		if (!suppress_logging) {
			log(WARNING, "[%s] Malformed response line '%s' from input.",
				url.c_str(), lines[0].c_str());
		}
		return false;
	}

	int response = atoi(first_line_tokens[1].c_str());
	if (response != 200) {
		if (!suppress_logging) {
			log(WARNING, "[%s] Non-200 response '%s' from input.",
				url.c_str(), lines[0].c_str());
		}
		return false;
	}

	multimap<string, string> parameters = extract_headers(lines, url);

	// Remove “Content-encoding: metacube”.
	// TODO: Make case-insensitive.
	multimap<string, string>::iterator encoding_it =
		parameters.find("Content-encoding");
	if (encoding_it != parameters.end() && encoding_it->second == "metacube") {
		parameters.erase(encoding_it);
	}

	// Change “Server: foo” to “Server: metacube/0.1 (reflecting: foo)”
	// TODO: Make case-insensitive.
	// XXX: Use a Via: instead?
	if (parameters.count("Server") == 0) {
		parameters.insert(make_pair("Server", SERVER_IDENTIFICATION));
	} else {
		for (multimap<string, string>::iterator it = parameters.begin();
		     it != parameters.end();
		     ++it) {
			if (it->first != "Server") {
				continue;
			}
			it->second = SERVER_IDENTIFICATION " (reflecting: " + it->second + ")";
		}
	}

	// Set “Connection: close”.
	// TODO: Make case-insensitive.
	parameters.erase("Connection");
	parameters.insert(make_pair("Connection", "close"));

	// Construct the new HTTP header.
	http_header = "HTTP/1.0 200 OK\r\n";
	for (multimap<string, string>::iterator it = parameters.begin();
	     it != parameters.end();
	     ++it) {
		http_header.append(it->first + ": " + it->second + "\r\n");
	}

	for (size_t i = 0; i < stream_indices.size(); ++i) {
		servers->set_header(stream_indices[i], http_header, stream_header);
	}

	return true;
}

void HTTPInput::do_work()
{
	timespec last_activity;

	// TODO: Make the timeout persist across restarts.
	if (state == SENDING_REQUEST || state == RECEIVING_HEADER || state == RECEIVING_DATA) {
		int err = clock_gettime(CLOCK_MONOTONIC_COARSE, &last_activity);
		assert(err != -1);
	}

	while (!should_stop()) {
		if (state == SENDING_REQUEST || state == RECEIVING_HEADER || state == RECEIVING_DATA) {
			// Give the socket 30 seconds since last activity before we time out.
			static const int timeout_secs = 30;

			timespec now;
			int err = clock_gettime(CLOCK_MONOTONIC_COARSE, &now);
			assert(err != -1);

			timespec elapsed = clock_diff(last_activity, now);
			if (elapsed.tv_sec >= timeout_secs) {
				// Timeout!
				if (!suppress_logging) {
					log(ERROR, "[%s] Timeout after %d seconds, closing.", url.c_str(), elapsed.tv_sec);
				}
				state = CLOSING_SOCKET;
				continue;
			}

			// Basically calculate (30 - (now - last_activity)) = (30 + (last_activity - now)).
			// Add a second of slack to account for differences between clocks.
			timespec timeout = clock_diff(now, last_activity);
			timeout.tv_sec += timeout_secs + 1;
			assert(timeout.tv_sec > 0 || (timeout.tv_sec >= 0 && timeout.tv_nsec > 0));

			bool activity = wait_for_activity(sock, (state == SENDING_REQUEST) ? POLLOUT : POLLIN, &timeout);
			if (activity) {
				err = clock_gettime(CLOCK_MONOTONIC_COARSE, &last_activity);
				assert(err != -1);
			} else {
				// OK. Most likely, should_stop was set, or we have timed out.
				continue;
			}
		}

		switch (state) {
		case NOT_CONNECTED:
			request.clear();
			request_bytes_sent = 0;
			response.clear();
			pending_data.clear();
			has_metacube_header = false;
			for (size_t i = 0; i < stream_indices.size(); ++i) {
				servers->set_header(stream_indices[i], "", "");
			}

			{
				string protocol, user;  // Thrown away.
				if (!parse_url(url, &protocol, &user, &host, &port, &path)) {
					if (!suppress_logging) {
						log(WARNING, "[%s] Failed to parse URL '%s'", url.c_str(), url.c_str());
					}
					break;
				}

				// Remove the brackets around IPv6 address literals.
				// TODO: See if we can join this with the code in parse_ip_address(),
				// or maybe even more it into parse_url().
				if (!host.empty() && host[0] == '[' && host[host.size() - 1] == ']') {
					host = host.substr(1, host.size() - 2);
				}
			}

			if (suppress_logging) {
				// See if there's more than one minute since last time we made a connection
				// with logging enabled. If so, turn it on again.
				timespec now;
				int err = clock_gettime(CLOCK_MONOTONIC_COARSE, &now);
				assert(err != -1);

				double elapsed = now.tv_sec - last_verbose_connection.tv_sec +
					1e-9 * (now.tv_nsec - last_verbose_connection.tv_nsec);
				if (elapsed > 60.0) {
					suppress_logging = false;
				}
			}
			if (!suppress_logging) {
				int err = clock_gettime(CLOCK_MONOTONIC_COARSE, &last_verbose_connection);
				assert(err != -1);
			}
			++num_connection_attempts;
			sock = lookup_and_connect(host, port);
			if (sock != -1) {
				// Yay, successful connect. Try to set it as nonblocking.
				int one = 1;
				if (ioctl(sock, FIONBIO, &one) == -1) {
					log_perror("ioctl(FIONBIO)");
					state = CLOSING_SOCKET;
				} else {
					state = SENDING_REQUEST;
					request = "GET " + path + " HTTP/1.0\r\nHost: " + host_header(host, port) + "\r\nUser-Agent: cubemap\r\n\r\n";
					request_bytes_sent = 0;
				}

				MutexLock lock(&stats_mutex);
				stats.connect_time = time(NULL);
				clock_gettime(CLOCK_MONOTONIC_COARSE, &last_activity);
			}
			break;
		case SENDING_REQUEST: {
			size_t to_send = request.size() - request_bytes_sent;
			int ret;

			do {
				ret = write(sock, request.data() + request_bytes_sent, to_send);
			} while (ret == -1 && errno == EINTR);

			if (ret == -1) {
				log_perror("write");
				state = CLOSING_SOCKET;
				continue;
			}

			assert(ret >= 0);
			request_bytes_sent += ret;

			if (request_bytes_sent == request.size()) {
				state = RECEIVING_HEADER;
			}
			break;
		}
		case RECEIVING_HEADER: {
			char buf[4096];
			int ret;

			do {
				ret = read(sock, buf, sizeof(buf));
			} while (ret == -1 && errno == EINTR);

			if (ret == -1) {
				log_perror("read");
				state = CLOSING_SOCKET;
				continue;
			}

			if (ret == 0) {
				// This really shouldn't happen...
				if (!suppress_logging) {
					log(ERROR, "[%s] Socket unexpectedly closed while reading header",
						   url.c_str());
				}
				state = CLOSING_SOCKET;
				continue;
			}
			
			RequestParseStatus status = wait_for_double_newline(&response, buf, ret);
			
			if (status == RP_OUT_OF_SPACE) {
				if (!suppress_logging) {
					log(WARNING, "[%s] Server sent overlong HTTP response!", url.c_str());
				}
				state = CLOSING_SOCKET;
				continue;
			} else if (status == RP_NOT_FINISHED_YET) {
				continue;
			}
	
			// OK, so we're fine, but there might be some of the actual data after the response.
			// We'll need to deal with that separately.
			string extra_data;
			if (status == RP_EXTRA_DATA) {
				char *ptr = static_cast<char *>(
					memmem(response.data(), response.size(), "\r\n\r\n", 4));
				assert(ptr != NULL);
				extra_data = string(ptr + 4, &response[0] + response.size());
				response.resize(ptr - response.data());
			}

			if (!parse_response(response)) {
				state = CLOSING_SOCKET;
				continue;
			}

			if (!extra_data.empty()) {
				process_data(&extra_data[0], extra_data.size());
			}

			if (!suppress_logging) {
				if (encoding == Input::INPUT_ENCODING_RAW) {
					log(INFO, "[%s] Connected to '%s', receiving raw data.",
						   url.c_str(), url.c_str());
				} else {
					assert(encoding == Input::INPUT_ENCODING_METACUBE);
					log(INFO, "[%s] Connected to '%s', receiving data.",
						   url.c_str(), url.c_str());
				}
			}
			state = RECEIVING_DATA;
			break;
		}
		case RECEIVING_DATA: {
			char buf[4096];
			int ret;

			do {
				ret = read(sock, buf, sizeof(buf));
			} while (ret == -1 && errno == EINTR);

			if (ret == -1) {
				log_perror("read");
				state = CLOSING_SOCKET;
				continue;
			}

			if (ret == 0) {
				// This really shouldn't happen...
				if (!suppress_logging) {
					log(ERROR, "[%s] Socket unexpectedly closed while reading data",
						   url.c_str());
				}
				state = CLOSING_SOCKET;
				continue;
			}

			num_connection_attempts = 0;  // Reset, since we have a successful read.
			if (suppress_logging) {
				// This was suppressed earlier, so print it out now.
				if (encoding == Input::INPUT_ENCODING_RAW) {
					log(INFO, "[%s] Connected to '%s', receiving raw data.",
						   url.c_str(), url.c_str());
				} else {
					assert(encoding == Input::INPUT_ENCODING_METACUBE);
					log(INFO, "[%s] Connected to '%s', receiving data.",
						   url.c_str(), url.c_str());
				}
				suppress_logging = false;
			}

			process_data(buf, ret);
			break;
		}
		case CLOSING_SOCKET: {
			close_socket();
			state = NOT_CONNECTED;
			break;
		}
		default:
			assert(false);
		}

		// If we are still in NOT_CONNECTED, either something went wrong,
		// or the connection just got closed.
		// The earlier steps have already given the error message, if any.
		if (state == NOT_CONNECTED && !should_stop()) {
			if (!suppress_logging) {
				log(INFO, "[%s] Waiting 0.2 seconds and restarting...", url.c_str());
			}

			if (num_connection_attempts >= 3 && !suppress_logging) {
				log(INFO, "[%s] %d failed connection attempts, suppressing logging for one minute.",
					url.c_str(), num_connection_attempts);
				suppress_logging = true;
			}
			timespec timeout_ts;
			timeout_ts.tv_sec = 0;
			timeout_ts.tv_nsec = 200000000;
			wait_for_wakeup(&timeout_ts);
		}
	}
}

void HTTPInput::process_data(char *ptr, size_t bytes)
{
	{
		MutexLock mutex(&stats_mutex);
		stats.bytes_received += bytes;
	}

	if (encoding == Input::INPUT_ENCODING_RAW) {
		for (size_t i = 0; i < stream_indices.size(); ++i) {
			servers->add_data(stream_indices[i], ptr, bytes, /*metacube_flags=*/0);
		}
		return;
	}

	assert(encoding == Input::INPUT_ENCODING_METACUBE);
	pending_data.insert(pending_data.end(), ptr, ptr + bytes);

	for ( ;; ) {
		// If we don't have enough data (yet) for even the Metacube header, just return.
		if (pending_data.size() < sizeof(metacube2_block_header)) {
			return;
		}

		// Make sure we have the Metacube sync header at the start.
		// We may need to skip over junk data (it _should_ not happen, though).
		if (!has_metacube_header) {
			char *ptr = static_cast<char *>(
				memmem(pending_data.data(), pending_data.size(),
				       METACUBE2_SYNC, strlen(METACUBE2_SYNC)));
			if (ptr == NULL) {
				// OK, so we didn't find the sync marker. We know then that
				// we do not have the _full_ marker in the buffer, but we
				// could have N-1 bytes. Drop everything before that,
				// and then give up.
				drop_pending_data(pending_data.size() - (strlen(METACUBE2_SYNC) - 1));
				return;
			} else {
				// Yay, we found the header. Drop everything (if anything) before it.
				drop_pending_data(ptr - pending_data.data());
				has_metacube_header = true;

				// Re-check that we have the entire header; we could have dropped data.
				if (pending_data.size() < sizeof(metacube2_block_header)) {
					return;
				}
			}
		}

		// Now it's safe to read the header.
		metacube2_block_header hdr;
		memcpy(&hdr, pending_data.data(), sizeof(hdr));
		assert(memcmp(hdr.sync, METACUBE2_SYNC, sizeof(hdr.sync)) == 0);
		uint32_t size = ntohl(hdr.size);
		uint16_t flags = ntohs(hdr.flags);
		uint16_t expected_csum = metacube2_compute_crc(&hdr);

		if (expected_csum != ntohs(hdr.csum)) {
			log(WARNING, "[%s] Metacube checksum failed (expected 0x%x, got 0x%x), "
				"not reading block claiming to be %d bytes (flags=%x).",
				url.c_str(), expected_csum, ntohs(hdr.csum),
				size, flags);

			// Drop only the first byte, and let the rest of the code handle resync.
			pending_data.erase(pending_data.begin(), pending_data.begin() + 1);
			has_metacube_header = false;
			continue;
		}
		if (size > 10485760) {
			log(WARNING, "[%s] Metacube block of %d bytes (flags=%x); corrupted header??",
				url.c_str(), size, flags);
		}

		// See if we have the entire block. If not, wait for more data.
		if (pending_data.size() < sizeof(metacube2_block_header) + size) {
			return;
		}

		// Send this block on to the servers.
		{
			MutexLock lock(&stats_mutex);
			stats.data_bytes_received += size;
		}
		char *inner_data = pending_data.data() + sizeof(metacube2_block_header);
		if (flags & METACUBE_FLAGS_HEADER) {
			stream_header = string(inner_data, inner_data + size);
			for (size_t i = 0; i < stream_indices.size(); ++i) {
				servers->set_header(stream_indices[i], http_header, stream_header);
			}
		}
		for (size_t i = 0; i < stream_indices.size(); ++i) {
			servers->add_data(stream_indices[i], inner_data, size, flags);
		}

		// Consume the block. This isn't the most efficient way of dealing with things
		// should we have many blocks, but these routines don't need to be too efficient
		// anyway.
		pending_data.erase(pending_data.begin(), pending_data.begin() + sizeof(metacube2_block_header) + size);
		has_metacube_header = false;
	}
}

void HTTPInput::drop_pending_data(size_t num_bytes)
{
	if (num_bytes == 0) {
		return;
	}
	log(WARNING, "[%s] Dropping %lld junk bytes; not a Metacube2 stream, or data was dropped from the middle of the stream.",
		url.c_str(), (long long)num_bytes);
	assert(pending_data.size() >= num_bytes);
	pending_data.erase(pending_data.begin(), pending_data.begin() + num_bytes);
}

void HTTPInput::add_destination(int stream_index)
{
	stream_indices.push_back(stream_index);
	servers->set_header(stream_index, http_header, stream_header);
}

InputStats HTTPInput::get_stats() const
{
	MutexLock lock(&stats_mutex);
	return stats;
}
