# elements.rb: all drawable objects
# copyright (c) 2009 by Vincent Fourmond
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details (in the COPYING file).


require 'ctioga2/graphics/types'
require 'ctioga2/graphics/styles'
require 'ctioga2/graphics/elements/element'
require 'ctioga2/graphics/elements/plot-elements'
require 'ctioga2/graphics/elements/containers'
require 'ctioga2/graphics/elements/redirecting-container'
require 'ctioga2/graphics/elements/subplot'
require 'ctioga2/graphics/elements/region'
require 'ctioga2/graphics/elements/gradient-region'
require 'ctioga2/graphics/elements/curve2d'
require 'ctioga2/graphics/elements/histogram'
require 'ctioga2/graphics/elements/parametric2d'
require 'ctioga2/graphics/elements/xyz-map'
require 'ctioga2/graphics/elements/xyz-contour'
require 'ctioga2/graphics/elements/primitive'

require 'ctioga2/graphics/elements/tangent'
require 'ctioga2/graphics/elements/style-lists'
require 'ctioga2/graphics/elements/contour'

module CTioga2

  module Graphics

    # Now, various commands pertaining to various drawables

    PlotCoordinatesGroup = CmdGroup.new('coordinates',
                                        "Plot coordinates",
                                        "Plot coordinates", 2)
    PlotMarginCommand = 
      Cmd.new("margin",nil,"--margin", 
              [ CmdArg.new('float') ]) do |plotmaker, margin|
      plotmaker.root_object.current_plot.style.plot_margin = margin
    end

    PlotMarginCommand.describe("Leaves a margin around data points",
                               <<EOH, PlotCoordinatesGroup)
Leaves a margin around the data points. Expressed in relative size of the
whole plot.
EOH


    # Various coordinate-related commands:
    # 
    # @todo All these commands should be axis-dependent, and not
    # plot-dependent.
    CoordinateRelatedCommands = []
    [:x, :y].each do |x|
      cmd = 
        Cmd.new("#{x}range",nil,"--#{x}range", 
                [ CmdArg.new('partial-float-range') ]) do |plotmaker, range|
        plotmaker.root_object.current_plot.
          set_user_boundaries(x.to_s,range)
      end
      cmd.describe("Sets the #{x.to_s.upcase} range",
                           <<EOH, PlotCoordinatesGroup)
Sets the range of the #{x.to_s.upcase} coordinates.

*Important note:* when the axis is in log range (using
{command: #{x.to_s.upcase}log}), the numbers you give are not the or
{command: ylog} values, but their @log10@, so that to
display #{x.to_s.upcase} values from @1e-2@ to @1e3@, use:

# #{x.to_s}yrange -2:3
EOH
      CoordinateRelatedCommands << cmd

      cmd = 
        Cmd.new("#{x}offset",nil,"--#{x}offset", 
                [ CmdArg.new('float') ]) do |plotmaker, val|
        plotmaker.root_object.current_plot.
          style.transforms.send("#{x}_offset=", val)
      end
      cmd.describe("Offset #{x.to_s.upcase} data",
                           <<EOH, PlotCoordinatesGroup)
Adds the given offset to all #{x.to_s.upcase} coordinates.
EOH
      CoordinateRelatedCommands << cmd

      cmd = 
        Cmd.new("#{x}scale",nil,"--#{x}scale", 
                [ CmdArg.new('float') ]) do |plotmaker, val|
        plotmaker.root_object.current_plot.
          style.transforms.send("#{x}_scale=", val)
      end
      cmd.describe("Scale #{x.to_s.upcase} data",
                           <<EOH, PlotCoordinatesGroup)
Multiplies the #{x.to_s.upcase} coordinates by this factor.
EOH
      CoordinateRelatedCommands << cmd

      # Alias
      cmd = 
        Cmd.new("#{x}fact",nil,"--#{x}fact", 
                [ CmdArg.new('float') ]) do |plotmaker, val|
        plotmaker.root_object.current_plot.
          style.transforms.send("#{x}_scale=", val)
      end
      cmd.describe("Alias for #{x}scale",
                           <<EOH, PlotCoordinatesGroup)
Alias for {command: #{x}scale}.
EOH
      CoordinateRelatedCommands << cmd

      cmd = 
        Cmd.new("#{x}log",nil,"--#{x}log", 
                [ CmdArg.new('boolean') ]) do |plotmaker, val|
        plotmaker.root_object.current_plot.
          style.send("set_log_scale", x, val)
      end
      cmd.describe("Use log scale for #{x.to_s.upcase}",
                           <<EOH, PlotCoordinatesGroup)
Uses a logarithmic scale for the #{x.to_s.upcase} axis.
EOH
      CoordinateRelatedCommands << cmd
    end
  end
end
