# cache.rb : A cache for various objects
# Copyright (C) 2007 Vincent Fourmond

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

module SciYAG

  module Backends

    # The Cache class aims at providing a small and easy-to-use cache
    # with validation checking, usage statistics and flushing (automatically
    # removes some entries when it grows too big).
    class Cache

      # An element of the cache
      class CacheEntry
        # The name under which it is cached
        attr_accessor :name

        # The data cached
        attr_accessor :data
        
        # The meta-data that should be matched against when checking
        # the cache's relevance
        attr_accessor :meta_data
        
        def initialize(name, data, meta_data)
          @name = name
          @data = data
          # Maybe we should even .dup the contents ?
          @meta_data = meta_data.dup # (just to make sure)
        end

        # Checks that the meta_data corresponds to this cache entry
        def relevant?(meta_data)
          return @meta_data == meta_data
        end
      end

      # Records all accessed items and the number of cache misses/succes
      # and so on. 
      attr_accessor :statistics

      # Creates a Cache
      def initialize
        # The cache itself
        @cache = {}

        @statistics = {}
      end

      # Look inside the cache for a cached element. If it is found and
      # up-to-date, it is returned. If not, #get_cache runs _code_,
      # stores its return value as a new cache for (_name_, _meta_data_)
      # and returns it.
      def get_cache(name, meta_data, &code)
        if (@cache.key?(name)) and
            ((cached = @cache[name]).relevant?(meta_data))
          stats(name)[:accesses] += 1
          return cached.data
        elsif @cache.key?(name)
          stats(name)[:updates] += 1
        end
        stats(name)[:misses] += 1
        
        # Now, we run the code to update the cache entry:
        data = code.call
        @cache[name] = CacheEntry.new(name, data, meta_data)
        return data
      end

      # Returns the statistics for the given item, or create them
      # on the fly if necessary:
      def stats(name)
        if @statistics[name]
          return @statistics[name]
        else
          @statistics[name] = {
            :accesses => 0,
            :misses => 0,
            :updates => 0
          }
          return @statistics[name]
        end
      end

    end

  end
end
