/* this file is part of criawips, a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2005 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "cria-canvas-text-priv.h"

#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkimmulticontext.h>

#define CDEBUG_TYPE cria_canvas_text_get_type
#include <cdebug/cdebug.h>

G_DEFINE_TYPE(CriaCanvasText, cria_canvas_text, GNOME_TYPE_CANVAS_TEXT)

enum {
	PROP_0,
	PROP_CURSOR_POS,
	PROP_CURSOR_TRAIL
};

static void
cct_move_cursor(CriaCanvasText* self, gint direction) {
	PangoLayout* layout = GNOME_CANVAS_TEXT(self)->layout;
	gint         new_index, old_index;
	gint         new_trailing, old_trailing;
	old_index = self->cursor;
	old_trailing = self->cursor_trail;
	pango_layout_move_cursor_visually(layout, TRUE,
					  self->cursor, self->cursor_trail,
					  direction,
					  &new_index, &new_trailing);

	if(new_index == G_MAXINT) {
		return;
	}

	if((new_index != self->cursor && new_index >= 0) ||
	   (new_trailing != self->cursor_trail)) {
		if(new_index != self->cursor && new_index >= 0) {
			self->cursor = new_index;
		}
	
		if(new_trailing != self->cursor_trail) {
			self->cursor_trail = new_trailing;
		}

		cdebug("moveCursor()", "moved from {%d+%d} into %d to {%d+%d}", old_index, old_trailing, direction, new_index, new_trailing);
		g_object_notify(G_OBJECT(self), "cursor");
	}
}

static void
cct_insert(CriaCanvasText* self, gchar const* text, gsize position) {
	GString* str = g_string_new(pango_layout_get_text(GNOME_CANVAS_TEXT(self)->layout));
	g_string_insert(str, position, text);
	g_object_set(self, "text", str->str, NULL);
	g_string_free(str, TRUE);
}

static gsize
g_utf8_length_next(gchar const* utf8) {
	return g_utf8_find_next_char(utf8, NULL) - utf8;
}

static void
cct_insert_and_move_cursor(CriaCanvasText* self, char* text) {
	if(CRIA_CANVAS_TEXT_GET_CLASS(self)->insert) {
		gsize i = self->cursor;

		if(self->cursor_trail) {
			gchar const* old  = pango_layout_get_text(GNOME_CANVAS_TEXT(self)->layout);
			gchar const* next = g_utf8_find_next_char(old + self->cursor, NULL);
			i += (next - (old + self->cursor));
		}
		
		CRIA_CANVAS_TEXT_GET_CLASS(self)->insert(self, text, i);
		
		/* move the cursor AFTER inserting text */
		for(i = text ? g_utf8_strlen(text, -1) : 0; i > 0; i--) {
			cct_move_cursor(self, 1);
		}
	}
}

static void
cria_canvas_text_init(CriaCanvasText* self) {
	self->im_context = gtk_im_multicontext_new();
#warning "FIXME: set the gdk window for the im context (eventually try to just use one context for the canvas)"
	//gtk_im_context_set_client_window(self->im_context, gdk_window_from_the_canvas);

	g_signal_connect_swapped(self->im_context, "commit",
				 G_CALLBACK(cct_insert_and_move_cursor), self);
}

static void
cct_finalize(GObject* object) {
	CriaCanvasText* self = CRIA_CANVAS_TEXT(object);

	g_signal_handlers_disconnect_by_func(self->im_context, cct_insert_and_move_cursor, self);
	g_object_unref(self->im_context);
}

static void
cct_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
	CriaCanvasText* self = CRIA_CANVAS_TEXT(object);

	switch(prop_id) {
	case PROP_CURSOR_POS:
		g_value_set_int(value, self->cursor);
		break;
	case PROP_CURSOR_TRAIL:
		g_value_set_int(value, self->cursor_trail);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static void
cct_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	switch(prop_id) {
	case PROP_CURSOR_POS:
	case PROP_CURSOR_TRAIL:
		/* read only properties fall through */
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static gboolean
cct_event(GnomeCanvasItem* item, GdkEvent* ev) {
	gboolean        retval = FALSE;
	CriaCanvasText* self = CRIA_CANVAS_TEXT(item);

	switch(ev->type) {
	case GDK_KEY_PRESS:
		if(gtk_im_context_filter_keypress(self->im_context, &(ev->key))) {
			retval = TRUE;
		} else {
			guint offset = 0;
			switch(ev->key.keyval) {
			case GDK_Left:
				cct_move_cursor(self, -1);
				retval = TRUE;
				break;
			case GDK_Right:
				cct_move_cursor(self, 1);
				retval = TRUE;
				break;
			case GDK_BackSpace:
				offset = 1;
				/* fall through */
			case GDK_Delete:
				if(CRIA_CANVAS_TEXT_GET_CLASS(self)->delete) {
					gchar const* text     = pango_layout_get_text(GNOME_CANVAS_TEXT(self)->layout);
					gsize        cursor   = self->cursor +
								self->cursor_trail * g_utf8_length_next(text + self->cursor);
					gchar const* cursor_p = cursor + text;
					gsize length_of_deleted_char = 0;
					
					if(!offset) {
						length_of_deleted_char = (gsize)g_utf8_find_next_char(cursor_p, NULL);
					} else {
						length_of_deleted_char = (gsize)g_utf8_find_prev_char(text, cursor_p);
					}

					length_of_deleted_char = labs(length_of_deleted_char - (gsize)cursor_p);
					
					/* offset is 0 or 1, so we don't do evil stuff by multiplying */
					offset *= length_of_deleted_char;

					if(cursor >= offset && (cursor + 1 - offset) <= strlen(text)) {
						/* don't delete characters before or after the string */

						/* the cursor MUST be moved before the new text is set because
						 * the old cursor position might be invalid in the new text */
						if(offset) {
							cct_move_cursor(self, -offset);
						}
						
						cdebug("event()", "calling delete(%p, %li, %li)",
						       self,
						       cursor - offset,
						       length_of_deleted_char);
						CRIA_CANVAS_TEXT_GET_CLASS(self)->delete(self, cursor - offset, length_of_deleted_char);
					}
				}
				retval = TRUE;
				break;
			case GDK_Return:
			case GDK_KP_Enter:
				{
					cct_insert_and_move_cursor(self, "\n");
					retval = TRUE;
				}
				break;
			case GDK_Home:
			case GDK_KP_Home:
				{
					gchar const* text = pango_layout_get_text(GNOME_CANVAS_TEXT(self)->layout);
					
					while(self->cursor > 0 && text[self->cursor - 1] != '\n') {
						cct_move_cursor(self, -1);
					}

					retval = TRUE;
				}
				break;
			case GDK_End:
			case GDK_KP_End:
				{
					const gchar* text = pango_layout_get_text(GNOME_CANVAS_TEXT(self)->layout);
					guint        len  = text ? strlen(text) : 0;
					
					while((guint)(self->cursor + self->cursor_trail * g_utf8_length_next(text + self->cursor)) < len &&
					  text[(self->cursor + self->cursor_trail * g_utf8_length_next(text + self->cursor))] != '\n') {
						cct_move_cursor(self, 1);
					}
					retval = TRUE;
				}
				break;
			default:
				break;
			}
		}
		break;
	default:
		break;
	}

	if(!retval && GNOME_CANVAS_ITEM_CLASS(cria_canvas_text_parent_class)->event) {
		retval = GNOME_CANVAS_ITEM_CLASS(cria_canvas_text_parent_class)->event(item, ev);
	}
	
	return retval;
}

static void
cct_delete(CriaCanvasText* self, gsize offset, gsize num_deleted) {
	GString* str = g_string_new(pango_layout_get_text(GNOME_CANVAS_TEXT(self)->layout));
	g_string_erase(str, offset, num_deleted);
	g_object_set(self, "text", str->str, NULL);
	g_string_free(str, TRUE);
}

static void
cria_canvas_text_class_init(CriaCanvasTextClass* self_class) {
	GObjectClass        * go_class;
	GnomeCanvasItemClass* gci_class;
	
	/* GObjectClass */
	go_class = G_OBJECT_CLASS(self_class);
	go_class->finalize     = cct_finalize;
	go_class->get_property = cct_get_property;
	go_class->set_property = cct_set_property;

	g_object_class_install_property(go_class,
					PROP_CURSOR_POS,
					g_param_spec_int("cursor",
							 "cursor",
							 "The position of the cursor",
							 0, G_MAXINT,
							 0,
							 G_PARAM_READABLE));
	g_object_class_install_property(go_class,
					PROP_CURSOR_TRAIL,
					g_param_spec_int("cursor-trail",
							 "cursor-trail",
							 "BLURB",
							 0, G_MAXINT,
							 0,
							 G_PARAM_READABLE));

	/* GnomeCanvasItemClass */
	gci_class = GNOME_CANVAS_ITEM_CLASS(self_class);
	gci_class->event = cct_event;

	/* CriaCanvasTextClass */
	self_class->delete = cct_delete;
	self_class->insert = cct_insert;
}

