/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004,2005 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "canvas.h"
#include "canvas-priv.h"

#include <gdk/gdkenumtypes.h>

#define CDEBUG_TYPE cria_canvas_get_type
#include <cdebug/cdebug.h>
#include <helpers/gnome-canvas-helpers.h>

#include "cc-marshallers.h"

enum {
	CANVAS_PROP_0,
	CANVAS_PROP_PADDING,
	CANVAS_PROP_ZOOM
};

enum {
	CANVAS_FOCUS_CHANGED,
	CANVAS_N_SIGNALS
};

enum {
	ITEM_PROP_0,
	ITEM_PROP_INTERACTIVE,
};

enum {
	ITEM_BUTTON_PRESS_EVENT,
	ITEM_FOCUS,
	ITEM_FOCUS_IN_EVENT,
	ITEM_FOCUS_OUT_EVENT,
	ITEM_GRAB_FOCUS,
	ITEM_KEY_PRESS_EVENT,
	ITEM_SET_CANVAS,
	ITEM_N_SIGNALS
};

G_DEFINE_TYPE(CriaCanvas, cria_canvas, GNOME_TYPE_CANVAS);
G_DEFINE_TYPE(CriaItem,   cria_item,   GNOME_TYPE_CANVAS_GROUP);

static void cria_canvas_get_property           (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void cria_canvas_init		       (CriaCanvas	* self);
static void cria_canvas_set_property           (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
static void cria_canvas_update_zoom	       (CriaCanvas     * self);
static void cria_canvas_update_extents	       (CriaCanvas     * self);
static void cria_item_get_property	       (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void cria_item_set_property	       (GObject		* object,
						guint		  prop_id,
						const GValue	* value,
						GParamSpec	* param_spec);

/* enable these to add support for signals */
static	guint	cria_canvas_signals[CANVAS_N_SIGNALS] = { 0 };
static	guint	cria_item_signals[ITEM_N_SIGNALS] = { 0 };

static void
find_first_focusable_item(gpointer item, CriaItem** retval) {
	if(retval && !*retval && CRIA_IS_ITEM(item) && CRIA_ITEM_CAN_FOCUS(item)) {
		*retval = CRIA_ITEM(item);
	}
}

static gboolean
cc_focus(GtkWidget* widget, GtkDirectionType dir) {
	gboolean retval = FALSE;

	if(GTK_WIDGET_CAN_FOCUS(widget)) {
		CriaItem* item = NULL;
		g_list_foreach(gnome_canvas_root(GNOME_CANVAS(widget))->item_list, (GFunc)find_first_focusable_item, &item);
		
		if(item) {
			if(!GTK_WIDGET_HAS_FOCUS(widget)) {
				gtk_widget_grab_focus(widget);

				if(GNOME_CANVAS(widget)->focused_item) {
					retval = TRUE;
				}
			}
			
			if(!retval && !GNOME_CANVAS(widget)->focused_item) {
				g_signal_emit(item, cria_item_signals[ITEM_FOCUS], 0, dir, &retval);
			}
			
			if(!retval) {
#warning "cc_focus(): FIXME: try to find another child to take the focus"
			}
		}
	}
	
	return retval;
}

static gboolean
cc_focus_in_event(GtkWidget* widget, GdkEventFocus* ev) {
	gboolean retval = FALSE;
	if(GNOME_CANVAS(widget)->focused_item) {
		g_signal_emit(GNOME_CANVAS(widget)->focused_item, cria_item_signals[ITEM_FOCUS_IN_EVENT], 0, ev, &retval);
	}
	return retval;
}

static gboolean
cc_focus_out_event(GtkWidget* widget, GdkEventFocus* ev) {
	gboolean retval = FALSE;
	if(GNOME_CANVAS(widget)->focused_item) {
		g_signal_emit(GNOME_CANVAS(widget)->focused_item, cria_item_signals[ITEM_FOCUS_OUT_EVENT], 0, ev, &retval);
	}
	return retval;
}

static gboolean
cc_button_press_event(GtkWidget* widget, GdkEventButton* ev) {
	/* try to find a focusable renderer to focus */
	gdouble          x, y;
	GnomeCanvasItem* it;
	gboolean         retval = FALSE;
	
	gnome_canvas_window_to_world(GNOME_CANVAS(widget), ev->x, ev->y, &x, &y);
	it = gnome_canvas_get_item_at(GNOME_CANVAS(widget), x, y);

	if(it) {
		g_signal_emit(it, cria_item_signals[ITEM_BUTTON_PRESS_EVENT], 0, ev, &retval);
	}
	
	return retval;
}

static gboolean
cc_focus_changed(CriaCanvas* self, CriaItem* item) {
	return FALSE;
}

static void
cria_canvas_class_init (CriaCanvasClass	* cria_canvas_class) {
	GObjectClass	* g_object_class;
	GtkWidgetClass  * widget_class;

	/* setting up GObjectClass */
	g_object_class = G_OBJECT_CLASS(cria_canvas_class);
	g_object_class->set_property = cria_canvas_set_property;
	g_object_class->get_property = cria_canvas_get_property;

	cria_canvas_signals[CANVAS_FOCUS_CHANGED] = g_signal_new("focus-changed",
					CRIA_TYPE_CANVAS,
					G_SIGNAL_RUN_LAST,
					G_STRUCT_OFFSET(CriaCanvasClass, focus_changed),
#warning "canvasClassInit(): FIXME: add accumulator"
					NULL, NULL,
					cc_marshal_BOOLEAN__OBJECT,
					G_TYPE_BOOLEAN,
					1,
					CRIA_TYPE_ITEM);
	g_object_class_install_property(g_object_class,
					CANVAS_PROP_PADDING,
					g_param_spec_uint64("padding",
							    "Padding",
							    "The padding between the slide and the widget edges.",
							    0,
							    G_MAXUINT64,
							    0,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					CANVAS_PROP_ZOOM,
					g_param_spec_double("zoom",
							    "Zoom",
							    "The zoom factor of this display: 1.0 means \"presentation size\"",
							    0.01,
							    G_MAXDOUBLE,
							    0.65,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

	/* setting up the GtkWidget class */
	widget_class = GTK_WIDGET_CLASS(cria_canvas_class);
	widget_class->button_press_event = cc_button_press_event;
	widget_class->focus              = cc_focus;
	widget_class->focus_in_event     = cc_focus_in_event;
	widget_class->focus_out_event    = cc_focus_out_event;

	/* setting up CriaCanvasClass */
	cria_canvas_class->focus_changed = cc_focus_changed;
}

/**
 * cria_canvas_get_focused:
 * @self: a #CriaCanvas
 *
 * Get the currently selected renderer.
 *
 * Returns the renderer that's currently selected; NULL if there is none
 */
GnomeCanvasItem*
cria_canvas_get_focused(CriaCanvas* self) {
	g_return_val_if_fail(CRIA_IS_CANVAS(self), NULL);

	return GNOME_CANVAS(self)->focused_item;
}

/**
 * cria_canvas_set_padding:
 * @self: a #CriaCanvas
 *
 * Get the padding of the canvas.
 * 
 * Returns the padding of the canvas.
 */
guint64
cria_canvas_get_padding(CriaCanvas* self) {
	g_return_val_if_fail(CRIA_IS_CANVAS(self), 0);

	return self->padding;
}

static void
cria_canvas_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaCanvas	* self;

	self = CRIA_CANVAS(object);

	switch (prop_id) {
	case CANVAS_PROP_PADDING:
		g_value_set_uint64(value, cria_canvas_get_padding(self));
		break;
	case CANVAS_PROP_ZOOM:
		g_value_set_double(value, cria_canvas_get_zoom(self));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

/**
 * cria_canvas_get_zoom:
 * @self: a #CriaCanvas
 *
 * Get the current zoom value of this canvas
 *
 * Returns the zoom level of this canvas.
 */
gdouble
cria_canvas_get_zoom(CriaCanvas* self) {
	g_return_val_if_fail(CRIA_IS_CANVAS(self), 1.0);
	
	return self->zoom;
}

static void
cria_canvas_init(CriaCanvas* self) {
	self->extents = g_new0(CcRectangle,1);

	/* initialize the value with this value, so setting 1.0 in construct
	 * time will cause a redraw */
	self->zoom = 0.01;

	gtk_widget_add_events(GTK_WIDGET(self), GDK_KEY_PRESS_MASK);
}

/**
 * cria_canvas_new:
 *
 * Creates a new #CriaCanvas.
 *
 * Returns the newly created and canvas.
 */
CriaCanvas*
cria_canvas_new(void) {
	CriaCanvas * self = g_object_new(CRIA_TYPE_CANVAS, "aa", TRUE, NULL);
	return self;
}

/**
 * cria_canvas_set_padding:
 * @self: a #CriaCanvas
 * @padding: the new padding
 *
 * Set the padding of the canvas.
 */
void
cria_canvas_set_padding(CriaCanvas* self, guint64 padding) {
	g_return_if_fail(CRIA_IS_CANVAS(self));

	if(padding != self->padding) {
		self->padding = padding;

		cria_canvas_update_extents(self);
		
		g_object_notify(G_OBJECT(self), "padding");
	}
}

static void
cria_canvas_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaCanvas	* self;
	
	self = CRIA_CANVAS (object);
	
	switch(prop_id) {
	case CANVAS_PROP_PADDING:
		cria_canvas_set_padding(self, g_value_get_uint64(value));
		break;
	case CANVAS_PROP_ZOOM:
		cria_canvas_set_zoom(self, g_value_get_double(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

/**
 * cria_canvas_set_extents:
 * @self: a #CriaCanvas
 * @extents: a #CcRectangle (using canvas coordinates)
 *
 * Specify the visible area of this canvas.
 */
void
cria_canvas_set_extents(CriaCanvas* self, const CcRectangle* extents) {
	g_return_if_fail(CRIA_IS_CANVAS(self));
	g_return_if_fail(self->extents);
	g_return_if_fail(extents);

	/* copy the whole structure at once */
	*(self->extents) = *extents;

	cria_canvas_update_extents(self);
	cria_canvas_update_zoom(self);
}

void
cria_canvas_set_zoom(CriaCanvas* self, gdouble zoom) {
	g_return_if_fail(CRIA_IS_CANVAS(self));

	cdebug("setZoom()", "to %f", zoom);

	if(zoom != self->zoom) {
		self->zoom = zoom;

		cria_canvas_update_zoom(self);

		g_object_notify(G_OBJECT(self), "zoom");
	}
}

static void
cria_canvas_update_extents(CriaCanvas* self) {
	gnome_canvas_set_scroll_region(GNOME_CANVAS(self),
#warning "updateExtents(): FIXME: make self->padding a gdouble"
				       self->extents->x - self->padding,
				       self->extents->y - self->padding,
				       self->extents->x + self->extents->w + self->padding,
				       self->extents->y + self->extents->h + self->padding);
	gnome_canvas_set_center_scroll_region(GNOME_CANVAS(self), TRUE);
}

static void
cria_canvas_update_zoom(CriaCanvas* self) {
	g_return_if_fail(CRIA_IS_CANVAS(self));
	g_return_if_fail(GNOME_IS_CANVAS(self));

	if(self->zoom > 0.0 && self->canvas.layout.hadjustment && self->canvas.layout.vadjustment) {
		gnome_canvas_set_zoom(GNOME_CANVAS(self), self->zoom);
	}
}

#if defined(G_HAVE_ISO_VARARGS) || defined(G_HAVE_GNUC_VARARGS)
# undef cdebug
# undef CDEBUG_TYPE
# define CDEBUG_TYPE cria_item_get_type
# ifdef G_HAVE_ISO_VARARGS
#  define cdebug(nspace, message, ...) cdebugt(CDEBUG_TYPE(), nspace, __VA_ARGS__)
# else
#  define cdebug(nspace, message...) cdebugt(CDEBUG_TYPE(), nspace, message)
# endif
#else
#endif

static gdouble
ci_point(GnomeCanvasItem* item, double x, double y, int cx, int cy, GnomeCanvasItem** actual_item) {
	GList          * it;
	GnomeCanvasItem* child;
	gdouble          best = 0.0;
	g_return_val_if_fail(actual_item, 0.0);
	*actual_item = NULL;
	gdouble comp_x = 0.0 + cx,
		comp_y = 0.0 + cy;
	
	for(it = GNOME_CANVAS_GROUP(item)->item_list; it; it = it->next) {
		child = GNOME_CANVAS_ITEM(it->data);
		if(!CRIA_IS_ITEM(child)) {
			continue;
		}

		if((child->x1 > comp_x) || (child->x2 < comp_x) || (child->y1 > comp_y) || (child->y2 < comp_y)) {
			continue;
		}
		
		*actual_item = child;
		best = 0.0;
	}
	
	return best;
}

static gboolean
ci_focus(CriaItem* self, GtkDirectionType dir) {
	gboolean retval = FALSE;
#if 0
	if(CRIA_ITEM(GNOME_CANVAS_ITEM(self)->parent)->focused_child != self) {
		/* we are not focused yet, focus ourselves if we can */
		if(CRIA_ITEM_CAN_FOCUS(self)) {
			retval = TRUE;
			cria_item_grab_focus(self, GTK_DIR_TAB_FORWARD);
		}
	}
#endif
	if(!GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(GNOME_CANVAS_ITEM(self)->canvas))) {
		retval = TRUE;
	}
	
	if(!retval && GNOME_CANVAS_ITEM(self)->canvas->focused_item != GNOME_CANVAS_ITEM(self)) {
		CriaItem* current_focus = self->focused_child;
		
		/* try to focus a child, then focus ourselves */
		if(!current_focus) {
			/* focus the first child */
			CriaItem* item = NULL;
			g_list_foreach(GNOME_CANVAS_GROUP(self)->item_list, (GFunc)find_first_focusable_item, &item);

			if(item) {
				g_signal_emit(item, cria_item_signals[ITEM_FOCUS], 0, dir, &retval);
			}
		}
		
		if(!retval && current_focus) {
			/* emit a focus event on the focused item, if this handler returns TRUE
			 * the sub-item seems to have another sub-item to focus */
			g_signal_emit(self->focused_child, cria_item_signals[ITEM_FOCUS], 0, dir, &retval);
		}
		
		if(!retval) {
			GList* focus = g_list_find(GNOME_CANVAS_GROUP(self)->item_list, current_focus);

			if(focus && focus->next) {
				do {
					/* just try to find the next item to focus */
					focus = focus->next;
				} while(focus != NULL && !(CRIA_IS_ITEM(focus->data) && CRIA_ITEM_CAN_FOCUS(focus->data)));
			}

			if(focus) {
				g_signal_emit(focus->data, cria_item_signals[ITEM_FOCUS], 0, dir, &retval);
			}
		}
			
		if(!retval && CRIA_ITEM_CAN_FOCUS(self)) {
			/* no next renderer found, focus ourselves */
			retval = TRUE;
			cria_item_grab_focus(self);
		}
	}
	
	return retval;
}

static gboolean
ci_focus_in_event(CriaItem* self, GdkEventFocus* ev) {
	CRIA_ITEM(GNOME_CANVAS_ITEM(self)->parent)->focused_child = self;
	return FALSE;
}

static gboolean
ci_focus_out_event(CriaItem* self, GdkEventFocus* ev) {
	g_return_val_if_fail(GNOME_CANVAS_ITEM(self)->canvas->focused_item == GNOME_CANVAS_ITEM(self), TRUE);
	g_return_val_if_fail(CRIA_ITEM(GNOME_CANVAS_ITEM(self)->parent)->focused_child == CRIA_ITEM(self), TRUE);
	CRIA_ITEM(GNOME_CANVAS_ITEM(self)->parent)->focused_child = NULL;
	return FALSE;
}

static void
ci_grab_focus(CriaItem* self) {
	GnomeCanvasItem* it;
	gboolean         retval = FALSE;
	GdkEventFocus    ev;

	ev.type = GDK_FOCUS_CHANGE;
	ev.window = GTK_WIDGET(GNOME_CANVAS_ITEM(self)->canvas)->window;
	ev.send_event = FALSE;
	ev.in = FALSE;

	if(CRIA_IS_ITEM(GNOME_CANVAS_ITEM(self)->canvas->focused_item)) {
		g_signal_emit(GNOME_CANVAS_ITEM(self)->canvas->focused_item, cria_item_signals[ITEM_FOCUS_OUT_EVENT], 0, &ev, &retval);

		for(it = GNOME_CANVAS_ITEM(GNOME_CANVAS_ITEM(self)->parent); CRIA_IS_ITEM(it); it = GNOME_CANVAS_ITEM(it->parent)) {
			CRIA_ITEM(it)->focused_child = NULL;
		}
	}
	gnome_canvas_item_grab_focus(GNOME_CANVAS_ITEM(self));

	/* emit a focus-in-event on the current item */
	ev.in = TRUE;
	g_signal_emit(self, cria_item_signals[ITEM_FOCUS_IN_EVENT], 0, &ev, &retval);
	
	for(it = GNOME_CANVAS_ITEM(self); it && CRIA_IS_ITEM(it->parent); it = GNOME_CANVAS_ITEM(it->parent)) {
		CRIA_ITEM(it->parent)->focused_child = CRIA_ITEM(it);
	}

	g_signal_emit(GNOME_CANVAS_ITEM(self)->canvas, cria_canvas_signals[CANVAS_FOCUS_CHANGED], 0, self, &retval);
}

static gboolean
ci_event(GnomeCanvasItem* item, GdkEvent* ev) {
	gboolean retval = FALSE;

	switch(ev->type) {
	case GDK_KEY_PRESS:
		g_signal_emit(item, cria_item_signals[ITEM_KEY_PRESS_EVENT], 0, ev, &retval);
		break;
	case GDK_BUTTON_PRESS:
		g_signal_emit(item, cria_item_signals[ITEM_BUTTON_PRESS_EVENT], 0, ev, &retval);
		break;
/*	case GDK_FOCUS_CHANGE:
		g_signal_emit(item, cria_item_signals[ITEM_FOCUS], 0, ev, &retval);
		break;
#warning "CriaItem::event(): FIXME: fix focus handling"
	case GDK_FOCUS_IN_EVENT:
		g_signal_emit(item, cria_item_signals[ITEM_FOCUS_IN_EVENT], 0, ev, &retval);
		break;
	case GDK_FOCUS_OUT_EVENT:
		g_signal_emit(item, cria_item_signals[ITEM_FOCUS_OUT_EVENT], 0, ev, &retval);
		break;*/
	default:
		{
			GTypeClass* c = g_type_class_ref(GDK_TYPE_EVENT_TYPE);
			cdebug("event()", "unhandled event of type %s", g_enum_get_value(G_ENUM_CLASS(c), ev->type)->value_name);
			g_type_class_unref(c);
		}
		break;
	}
	
	if(!retval && GNOME_CANVAS_ITEM_CLASS(cria_item_parent_class)->event) {
		retval = GNOME_CANVAS_ITEM_CLASS(cria_item_parent_class)->event(item, ev);
	}
	
	return retval;
}

static void
cria_item_class_init(CriaItemClass* cria_item_class) {
	GObjectClass	    * g_object_class;
	GnomeCanvasItemClass* item_class;

	/* setting up the object class */
	g_object_class = G_OBJECT_CLASS(cria_item_class);
	g_object_class->set_property = cria_item_set_property;
	g_object_class->get_property = cria_item_get_property;

	g_object_class_install_property(g_object_class,
					ITEM_PROP_INTERACTIVE,
					g_param_spec_boolean("interactive",
							     "Interactive",
							     "Specifies whether this item should be able to "
							     "receive user events",
							     FALSE,
							     G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

	/* setting up the gnome canvas item class */
	item_class = GNOME_CANVAS_ITEM_CLASS(cria_item_class);
	item_class->event = ci_event;
	item_class->point = ci_point;
	
	/* setting up the cria canvas item class */
	cria_item_class->focus           = ci_focus;
	cria_item_class->focus_in_event  = ci_focus_in_event;
	cria_item_class->focus_out_event = ci_focus_out_event;
	cria_item_class->grab_focus      = ci_grab_focus;

	cria_item_signals[ITEM_BUTTON_PRESS_EVENT] = g_signal_new("button-press-event",
							CRIA_TYPE_ITEM,
							G_SIGNAL_RUN_LAST,
							G_STRUCT_OFFSET(CriaItemClass, button_press_event),
#warning "init(): FIXME: add an accumulator"
							NULL, NULL,
							cc_marshal_BOOLEAN__ENUM,
							G_TYPE_BOOLEAN,
							1,
							GDK_TYPE_EVENT);

	/**
	 * CriaItem::focus:
	 * @dir: a #GtkDirectionType
	 *
	 * The ::focus signal is emitted when focus events happen.
	 *
	 * Returns %TRUE to stop other handlers from being invoked for this event.
	 *   %FALSE to porpagate the event further.
	 */
	cria_item_signals[ITEM_FOCUS] = g_signal_new("focus",
							CRIA_TYPE_ITEM,
							G_SIGNAL_RUN_LAST,
							G_STRUCT_OFFSET(CriaItemClass, focus),
#warning "init(): FIXME: add an accumulator"
							NULL, NULL,
							cc_marshal_BOOLEAN__ENUM,
							G_TYPE_BOOLEAN,
							1,
							GTK_TYPE_DIRECTION_TYPE);
	cria_item_signals[ITEM_FOCUS_IN_EVENT] = g_signal_new("focus-in-event",
							CRIA_TYPE_ITEM,
							G_SIGNAL_RUN_LAST,
							G_STRUCT_OFFSET(CriaItemClass, focus_in_event),
#warning "init(): FIXME: add an accumulator"
							NULL, NULL,
							cc_marshal_BOOLEAN__BOXED,
							G_TYPE_BOOLEAN,
							1,
							GDK_TYPE_EVENT);
	cria_item_signals[ITEM_FOCUS_OUT_EVENT] = g_signal_new("focus-out-event",
							CRIA_TYPE_ITEM,
							G_SIGNAL_RUN_LAST,
							G_STRUCT_OFFSET(CriaItemClass, focus_out_event),
#warning "init(): FIXME: add an accumulator"
							NULL, NULL,
							cc_marshal_BOOLEAN__BOXED,
							G_TYPE_BOOLEAN,
							1,
							GDK_TYPE_EVENT);
	cria_item_signals[ITEM_GRAB_FOCUS] = g_signal_new("grab-focus",
							CRIA_TYPE_ITEM,
							G_SIGNAL_RUN_LAST,
							G_STRUCT_OFFSET(CriaItemClass, grab_focus),
#warning "init(): FIXME: add an accumulator"
							NULL, NULL,
							g_cclosure_marshal_VOID__VOID,
							G_TYPE_NONE,
							0);
	cria_item_signals[ITEM_KEY_PRESS_EVENT] = g_signal_new("key-press-event",
							CRIA_TYPE_ITEM,
							G_SIGNAL_RUN_LAST,
							G_STRUCT_OFFSET(CriaItemClass, key_press_event),
#warning "init(): FIXME: add an accumulator"
							NULL, NULL,
							cc_marshal_BOOLEAN__BOXED,
							G_TYPE_BOOLEAN,
							1,
							GDK_TYPE_EVENT);
}

/**
 * cria_item_is_interactive:
 * @self: a #CriaItem
 *
 * Find out whether an item is interactive or not. Interactive items may
 * respond or react to events like user input etc.
 *
 * Returns TRUE if the item is interactive, FALSE if not
 */
gboolean
cria_item_is_interactive(CriaItem* self) {
	g_return_val_if_fail(CRIA_IS_ITEM(self), FALSE);

	return CRIA_ITEM_IS_INTERACTIVE(self);
}

static void
cria_item_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaItem* self;

	self = CRIA_ITEM(object);

	switch(prop_id) {
	case ITEM_PROP_INTERACTIVE:
		g_value_set_boolean(value, cria_item_is_interactive(self));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

/**
 * cria_item_grab_focus:
 * @self: a #CriaItem
 *
 * Grabs the keyboard focus to this renderer. This function should only be
 * called on renderers which can focus.
 */
void
cria_item_grab_focus(CriaItem* self) {
	g_return_if_fail(CRIA_IS_ITEM(self));

	if(!CRIA_ITEM_CAN_FOCUS(self)) {
		return;
	}

	if(CRIA_ITEM(GNOME_CANVAS_ITEM(self)->canvas->focused_item) == self) {
		/* just make sure the widget is focused */
		gtk_widget_grab_focus(GTK_WIDGET(GNOME_CANVAS_ITEM(self)->canvas));
	}

	/* emit a focus on the current item, if the result is non-FALSE,
	 * grab the focus, return if not */
	g_signal_emit(self, cria_item_signals[ITEM_GRAB_FOCUS], 0);
}

/**
 * cria_item_set_interactive:
 * @self: a #CriaItem
 * @interactive: a #gboolean
 *
 * Allow this item to get user events (TRUE) or not.
 */
void
cria_item_set_interactive(CriaItem* self, gboolean interactive) {
	g_return_if_fail(CRIA_IS_ITEM(self));

	if(interactive == CRIA_ITEM_IS_INTERACTIVE(self)) {
		return;
	} 

	if(interactive) {
		CRIA_ITEM_SET_FLAG(self, CRIA_INTERACTIVE);
	} else {
		CRIA_ITEM_UNSET_FLAG(self, CRIA_INTERACTIVE);
	}
	
	cdebug("setInteractive()", "set interactive to %s", interactive ? "true" : "false");
	g_object_notify(G_OBJECT(self), "interactive");
}

static void
cria_item_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaItem* self;
	
	self = CRIA_ITEM(object);
	
	switch(prop_id) {
	case ITEM_PROP_INTERACTIVE:
		cria_item_set_interactive(self, g_value_get_boolean(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

static void
cria_item_init(CriaItem* self) {}

