"
" Filename: cream-slide-generator.vim
"
" Cream -- An easy-to-use configuration of the famous Vim text editor
" [ http://cream.sourceforge.net ] Copyright (C) 2001-2006  Steve Hall
"
" License:
" This program is free software; you can redistribute it and/or modify
" it under the terms of the GNU General Public License as published by
" the Free Software Foundation; either version 2 of the License, or
" (at your option) any later version.
" [ http://www.gnu.org/licenses/gpl.html ]
"
" This program is distributed in the hope that it will be useful, but
" WITHOUT ANY WARRANTY; without even the implied warranty of
" MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
" General Public License for more details.
"
" You should have received a copy of the GNU General Public License
" along with this program; if not, write to the Free Software
" Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
" 02111-1307, USA.
"

" Description: 
"
" A auto-slideshow generator:
" o Maintains all the original images.
" o Creates thumbnails of all the images in the directory.
" o Creates an HTML document that displays all the thumbnails, where
"   each links to another HMTL page displaying the fullsize image.
" o Indicates each filename and size on the slide page.
"
" TODO:
" o Link subdirectories.
"

" register as a Cream add-on
if exists("$CREAM")

	call Cream_addon_register(
	\ 'Slide Generator',
	\ "Generate an HTML thumbnail page from a directory's images.",
	\ "Generate an HTML thumbnail page displaying all the images in a directory.",
	\ 'Slide Generator',
	\ 'call Cream_slide_generator()',
	\ '<Nil>'
	\ )
endif

function! Cream_slide_generator()

	" verify dependencies
	" on PATH
	if exists("g:CREAM_SLIDE_GENERATOR_CONVERTPATH")
		if !executable(g:CREAM_SLIDE_GENERATOR_CONVERTPATH)
			call confirm(
				\ "Remembered ImageMagick convert utility not found.\n" .
				\ "\n", "&Ok", 1, "Info")
			unlet g:CREAM_SLIDE_GENERATOR_CONVERTPATH
		endif
	endif

	if exists("g:CREAM_SLIDE_GENERATOR_CONVERTPATH")
		let myconvert = g:CREAM_SLIDE_GENERATOR_CONVERTPATH
	else
		if !executable("convert")
			let msg = "ImageMagick's convert utility not found on PATH."
		" Windows convert.exe test
		elseif stridx(system('convert /?'), "Converts FAT volumes to NTFS") >= 0
			let msg = "The convert.exe found on PATH is the Windows system utility, not the ImageMagick application."
		endif
		" option to find manually
		if exists("msg")
			let n = confirm(
				\ msg . "\n" .
				\ "Would you like to find the ImageMagick convert application?\n" .
				\ "\n", "&Yes\n&No", 1, "Info")
			if n != 1
				return
			endif
			let myconvert = browse(0, "ImageMagick convert utility", getcwd(), "")
			if myconvert == ""
				return
			endif
			let myconvert = fnamemodify(myconvert, ":p:8")
			let myconvert = Cream_path_fullsystem(myconvert)
			let g:CREAM_SLIDE_GENERATOR_CONVERTPATH = myconvert
		" default: on PATH
		else
			let myconvert = "convert"
		endif
	endif

	" get the directory
	if exists("g:CREAM_SLIDE_GENERATOR")
		let slidepath = g:CREAM_SLIDE_GENERATOR
	elseif exists("b:cream_pathfilename")
		let slidepath = b:cream_pathfilename
	else
		let slidepath = getcwd()
	endif
	let slidepath = browsedir("Confirm location to slideshow:", slidepath)
	if slidepath == ""
		" cancelled...
		return
	endif

	" reduce spaces
	if Cream_has("ms")
		let slidepath = fnamemodify(slidepath, ":p:8")
		let slidepath = Cream_path_fullsystem(slidepath)
	endif
	" ensure has trailing slash
	let slidepath = Cream_path_addtrailingslash(slidepath)

	" remember
	let g:CREAM_SLIDE_GENERATOR = slidepath
	
	" get all files in directory
	let myfiles = Cream_getfilelist(slidepath . '*')
	" start new slideshow
	if myfiles != ""
		silent! call Cream_file_new()
		execute 'silent! write! ' . slidepath . 'cream-slideshow.html'
		let mybuf = bufnr("%")
	endif

	function! s:Html_header()
	" dump an HTML skeleton in the current file
		" HTML header (from template
		execute 'normal i' . g:cream_template_html_html
		" go to find char
		?{+}
		" delete line
		normal dd
		normal k
		" open up some space
		normal 5o
		" go back to top of space
		normal 4k
	endfunction
	call s:Html_header()

	" for each image
	let i = 0
	let max = MvNumberOfElements(myfiles, "\n")
	while i < max
		let mypathfile = MvElementAt(myfiles, "\n", i) 
		let myfileext = fnamemodify(mypathfile, ":e")
		" only web-compatible graphic files
		if myfileext !~? '\(jpg\|gif\|png\)'
			let i = i + 1
			continue
		endif

		" use current path to avoid pathing issues
		let mycd = getcwd()
		execute "cd " . slidepath

		let myfilename = fnamemodify(mypathfile, ":t:r")
		let myfilenameext = myfilename . '.' . myfileext
		"let myfilethumb = myfilename . '-thumb.' . myfileext
		let myfilethumb = myfilename . '-thumb.' . myfileext
		let myfilehtml = myfilename . '.html'

""*** DEBUG:
"let n = confirm(
"    \ "DEBUG:\n" .
"    \ "  slidepath  = \"" . slidepath . "\"\n" .
"    \ "  myfileext  = \"" . myfileext . "\"\n" .
"    \ "  myfilename  = \"" . myfilename . "\"\n" .
"    \ "  myfilenameext  = \"" . myfilenameext . "\"\n" .
"    \ "  myfilethumb  = \"" . myfilethumb . "\"\n" .
"    \ "  myfilehtml  = \"" . myfilehtml . "\"\n" .
"    \ "\n", "&Ok\n&Cancel", 1, "Info")
"if n != 1
"    return
"endif
""***
		" progress indication
		let mycmdheight = &cmdheight
		set cmdheight=2
		echo " Progress: " . i . " of " . max . " (" . ((i*100) / max) . "%)"
		let &cmdheight = mycmdheight

		" generate thumbnail
		if Cream_has("ms")
			let quote = '"'
		else
			let quote = ''
		endif
		set noshellslash
		" command line
		execute 'silent! !' . myconvert . ' ' . myfilenameext . " -thumbnail 120x120 -bordercolor white -border 50 -gravity center -crop 120x120+0+0 +repage " . myfilethumb
		set shellslash

		let @x = ""
		let @x = @x . '<a href="' . myfilehtml . '"><img class="slide" src="' . myfilethumb . '"></a>' . "\n"
		put x

		" HTML document displaying each image
		call Cream_file_new()
""*** DEBUG:
"let n = confirm(
"    \ "DEBUG:\n" .
"    \ "  slidepath . myfilehtml  = \"" . slidepath . myfilehtml . "\"\n" .
"    \ "\n", "&Ok\n&Cancel", 1, "Info")
"if n != 1
"    return
"endif
""***
		call s:Html_header()
		" add reference to original image
		let @x = "\n"
		let @x = @x . '<h4>' . myfilename . '.' . myfileext . '<h4>' . "\n"
		let @x = @x . '<a href="cream-slideshow.html"><img src="' . myfilename . '.' . myfileext . '" alt="' . myfilename . '.' . myfileext .'"></a>' . "\n"
		let @x = @x . "\n"
		put x
		" save and close
		execute 'silent! write! ' . slidepath . myfilehtml
		silent! call Cream_close()

		" make sure we return to correct file
		execute ":buf " . mybuf

		let i = i + 1
	endwhile
	" final save and close
	silent! write!
	"" make sure we return to correct file
	"execute ":buf " . mybuf

	" generate simple .css
	silent! call Cream_file_new()
	" paste in reasonable CSS
	let @x = "\n"
	let @x = @x . 'BODY {' . "\n"
	let @x = @x . '    font-family: sans-serif;' . "\n"
	let @x = @x . '    background-color: #999;' . "\n"
	let @x = @x . '    color: #fff;' . "\n"
	let @x = @x . '}' . "\n"
	let @x = @x . 'IMG.slide {' . "\n"
	let @x = @x . '    padding: 10px;' . "\n"
	let @x = @x . '    background-color: #fff;' . "\n"
	let @x = @x . '}' . "\n"
	let @x = @x . 'H4 {' . "\n"
	let @x = @x . '    color: #fff;' . "\n"
	let @x = @x . '}' . "\n"
	put x

	execute 'silent! write! ' . slidepath . 'main.css'
	silent! call Cream_close()

	" make sure we return to cream-slide.html
	execute ":buf " . mybuf

	" open file in browser
	call Cream_file_open_defaultapp()
	" fix highlighting
	filetype detect

	let @x = ''

endfunction

