/* $Id: cpl_imagelist_io-test.c,v 1.51 2011/08/17 14:44:45 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2011/08/17 14:44:45 $
 * $Revision: 1.51 $
 * $Name: cpl-6_1_1 $
 */

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "cpl_imagelist_io.h"
#include "cpl_image_gen.h"
#include "cpl_image_io.h"
#include "cpl_vector.h"
#include "cpl_test.h"
#include "cpl_memory.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

#define     IMAGE_SZ    127
#define     NIMAGES     10

/*-----------------------------------------------------------------------------
                                  Main
 -----------------------------------------------------------------------------*/
int main(void)
{
    const char    * file = "cpl_imagelist_test.fits";
    cpl_imagelist * imlist;
    cpl_imagelist * imlist2;
    cpl_imagelist * imlista;
    cpl_imagelist * nulllist;
    cpl_image     * image;
    cpl_image     * nullimg;
    double          flags[NIMAGES];
    cpl_vector    * eraser;
    cpl_size        i;
    FILE          * stream;
    cpl_size        boxsize = IMAGE_SZ > 127 ? 5 :
    ( IMAGE_SZ < 25 ? 1 : (IMAGE_SZ / 25) );
    cpl_error_code  error;


    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    stream = cpl_msg_get_level() > CPL_MSG_INFO
        ? fopen("/dev/null", "a") : stdout;

    /* Insert tests below */

    cpl_test_nonnull(stream);

    eraser = cpl_vector_wrap(NIMAGES, flags);

    cpl_msg_info("", "Checking various failures");

    image = cpl_image_fill_test_create(IMAGE_SZ, IMAGE_SZ);
    cpl_test_nonnull( image );

    /* Test cpl_imagelist_new() */
    imlist = cpl_imagelist_new();
    cpl_test_nonnull( imlist );

    cpl_test_zero( cpl_imagelist_get_size(imlist) );
    cpl_test_eq( cpl_imagelist_is_uniform(imlist), 1);

    /* Failure tests involving NULL end empty lists */
    i = cpl_imagelist_get_size(NULL);
    cpl_test_error( CPL_ERROR_NULL_INPUT );
    cpl_test_eq( i, -1 );

    i = cpl_imagelist_is_uniform(NULL);
    cpl_test_error( CPL_ERROR_NULL_INPUT );
    cpl_test( i < 0);

    error = cpl_imagelist_set(NULL, image, 0);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT );

    error = cpl_imagelist_set(imlist, NULL, 0);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT );

    error = cpl_imagelist_set(imlist, image, -1);
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT );

    error = cpl_imagelist_set(imlist, image, 1);
    cpl_test_eq_error( error, CPL_ERROR_ACCESS_OUT_OF_RANGE );

    nullimg = cpl_imagelist_get(NULL, 0);
    cpl_test_error( CPL_ERROR_NULL_INPUT );
    cpl_test_null( nullimg );

    nullimg = cpl_imagelist_get(imlist, -1);
    cpl_test_error( CPL_ERROR_ILLEGAL_INPUT );
    cpl_test_null( nullimg );

    nullimg = cpl_imagelist_get(imlist, 0);
    cpl_test_error( CPL_ERROR_ACCESS_OUT_OF_RANGE );
    cpl_test_null( nullimg );

    error = cpl_imagelist_erase(imlist, NULL);
    cpl_test_eq_error( error, CPL_ERROR_NULL_INPUT );

    /* The elements of eraser are not initialized at this point,
       but they are also not supposed to be accessed */
    error = cpl_imagelist_erase(NULL, eraser);
    cpl_test_eq_error( error, CPL_ERROR_NULL_INPUT );

    error = cpl_imagelist_erase(imlist, eraser);
    cpl_test_eq_error( error, CPL_ERROR_INCOMPATIBLE_INPUT );

    nulllist = cpl_imagelist_duplicate(NULL);
    cpl_test_error( CPL_ERROR_NULL_INPUT );
    cpl_test_null( nulllist );

    error = cpl_imagelist_save(imlist, NULL,  CPL_TYPE_FLOAT, NULL,
                               CPL_IO_CREATE);
    cpl_test_eq_error( error, CPL_ERROR_NULL_INPUT );

    error = cpl_imagelist_save(NULL,  file, CPL_TYPE_FLOAT, NULL,
                               CPL_IO_CREATE);
    cpl_test_eq_error( error, CPL_ERROR_NULL_INPUT );

    error = cpl_imagelist_save(imlist, file,  CPL_TYPE_FLOAT, NULL,
                               CPL_IO_CREATE);
    cpl_test_eq_error( error, CPL_ERROR_ILLEGAL_INPUT );

    nulllist = cpl_imagelist_load(NULL, CPL_TYPE_INT, 0);
    cpl_test_error( CPL_ERROR_NULL_INPUT );
    cpl_test_null( nulllist );

    nulllist = cpl_imagelist_load(".", CPL_TYPE_INT, 0);
    cpl_test_error( CPL_ERROR_FILE_IO );
    cpl_test_null( nulllist );


    nulllist = cpl_imagelist_load_window(NULL, CPL_TYPE_INT, 0, 1, 1, 2, 2);
    cpl_test_error( CPL_ERROR_NULL_INPUT );
    cpl_test_null( nulllist );

    nulllist = cpl_imagelist_load_window(file, CPL_TYPE_INT, -1, 1, 1, 2, 2);
    cpl_test_error( CPL_ERROR_ILLEGAL_INPUT );
    cpl_test_null( nulllist );

    nulllist = cpl_imagelist_load_window(".", CPL_TYPE_INT, 0, 1, 1, 2, 2);
    cpl_test_error( CPL_ERROR_FILE_IO );
    cpl_test_null( nulllist );


    /* Test cpl_imagelist_duplicate() of empty list */
    imlist2 = cpl_imagelist_duplicate(imlist);
    cpl_test_nonnull( imlist2 );
    
    cpl_test_zero( cpl_imagelist_get_size(imlist2) );
    cpl_test_eq( cpl_imagelist_is_uniform(imlist2), 1);

    cpl_imagelist_empty(imlist2); /* Test cpl_imagelist_unset() */
    cpl_imagelist_delete(imlist2);
    
    cpl_msg_info("", "Create an image list of %d images", NIMAGES);

    /* Test cpl_imagelist_set() */
    for (i=0; i < NIMAGES; i++) {
        cpl_image * copy = cpl_image_duplicate(image);

        flags[i] = i % 2 ? 1.0 : -1.0;

        error = cpl_imagelist_set(imlist, copy, i);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

        cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
        cpl_test_eq( cpl_imagelist_get_size(imlist), i+1 );
        cpl_test_eq_ptr( cpl_imagelist_get(imlist, i), copy );

        /* Insert image twice, last image will remain twice in list */
        error = cpl_imagelist_set(imlist, copy, i + 1);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

    }
    cpl_image_delete(image);
    (void)cpl_imagelist_unset(imlist, NIMAGES);

    /* No real testing, but at least make use of 
       cpl_imagelist_dump_{structure, window} */
    cpl_imagelist_dump_structure(imlist, stream);

    cpl_imagelist_dump_window(imlist, 
                              (IMAGE_SZ / 2) - boxsize,
                              (IMAGE_SZ / 2) - boxsize,
                              (IMAGE_SZ / 2) + boxsize,
                              (IMAGE_SZ / 2) + boxsize,
                              stream);

    cpl_msg_info("", "Duplicate the image list");

    imlist2 = cpl_imagelist_duplicate(imlist);
    cpl_test_nonnull( imlist2 );

    cpl_test_zero( cpl_imagelist_is_uniform(imlist2));
    cpl_test_eq( cpl_imagelist_get_size(imlist2), NIMAGES );

    /* Failure tests involving non-empty, valid list */
    error = cpl_imagelist_save(imlist2, ".",  CPL_TYPE_FLOAT, NULL,
                               CPL_IO_CREATE);
    cpl_test_eq_error( error, CPL_ERROR_FILE_IO );

    remove(file);
    error = cpl_imagelist_save(imlist2, file,  CPL_TYPE_FLOAT, NULL,
                               CPL_IO_APPEND);
    cpl_test_eq_error( error, CPL_ERROR_FILE_IO );

    error = cpl_imagelist_save(imlist2, file,  CPL_TYPE_FLOAT, NULL,
                               CPL_IO_CREATE | CPL_IO_EXTEND);
    cpl_test_eq_error( error, CPL_ERROR_ILLEGAL_INPUT );

    error = cpl_imagelist_save(imlist2, file,  CPL_TYPE_FLOAT, NULL,
                               CPL_IO_CREATE | CPL_IO_EXTEND | CPL_IO_APPEND);
    cpl_test_eq_error( error, CPL_ERROR_ILLEGAL_INPUT );

    image = cpl_image_new(IMAGE_SZ, IMAGE_SZ, CPL_TYPE_INT);
    cpl_test_nonnull( image );

    error = cpl_imagelist_set(imlist2, image, 0);
    cpl_test_eq_error( error, CPL_ERROR_TYPE_MISMATCH );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist2) );
    cpl_test_eq( cpl_imagelist_get_size(imlist2), NIMAGES );

    cpl_image_delete(image);

    image = cpl_image_new(IMAGE_SZ/2, IMAGE_SZ, CPL_TYPE_INT);
    cpl_test_nonnull( image );

    error = cpl_imagelist_set(imlist2, image, 0);
    cpl_test_eq_error( error, CPL_ERROR_INCOMPATIBLE_INPUT );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist2) );
    cpl_test_eq( cpl_imagelist_get_size(imlist2), NIMAGES );

    nullimg = cpl_imagelist_get(imlist2, NIMAGES+1);
    cpl_test_error( CPL_ERROR_ACCESS_OUT_OF_RANGE );
    cpl_test_null( nullimg );

    cpl_test_zero( cpl_imagelist_is_uniform(imlist2) );
    cpl_test_eq( cpl_imagelist_get_size(imlist2), NIMAGES );

    cpl_imagelist_empty(imlist2); /* Test cpl_imagelist_unset() */

    /* Imagelist with 1 image */
    error = cpl_imagelist_set(imlist2, image, 0);
    cpl_test_eq_error( error, CPL_ERROR_NONE );

    /* Must be allowed to replace it w. image of different size/type  */
    image = cpl_image_new(IMAGE_SZ, IMAGE_SZ/2, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( image );

    error = cpl_imagelist_set(imlist2, image, 0);
    cpl_test_eq_error( error, CPL_ERROR_NONE );

    cpl_imagelist_delete(imlist2);

    /* Normal conditions */

    /* Create double-length list, with images inserted twice */
    imlista = cpl_imagelist_duplicate(imlist);
    for (i = 0; i < NIMAGES; i++) {
        image = cpl_imagelist_get(imlista, i);

        error = cpl_imagelist_set(imlista, image, i + NIMAGES);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

        cpl_test_zero( cpl_imagelist_is_uniform(imlista) );
        cpl_test_eq( cpl_imagelist_get_size(imlista), i+1 + NIMAGES );
        cpl_test_eq_ptr( cpl_imagelist_get_const(imlista, i + NIMAGES),
                         cpl_imagelist_get_const(imlista, i)  );    
    }

    /* Test cpl_imagelist_save() */
    cpl_msg_info("", "Save the image list to %s", file);
    cpl_test_zero( cpl_imagelist_save(imlist, file,  CPL_TYPE_FLOAT, NULL,
                                          CPL_IO_CREATE) );
    cpl_test_zero( cpl_imagelist_save(imlist, file,  CPL_TYPE_FLOAT, NULL,
                                          CPL_IO_EXTEND) );

    /* cpl_image_load(): Test loading of NAXIS=3 data (DFS09929) */
    image = cpl_image_load(file, CPL_TYPE_UNSPECIFIED, 0, 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_image_abs(image, cpl_imagelist_get_const(imlist, 0),
                       15.0 * FLT_EPSILON);
    cpl_image_delete(image);

    /* Test cpl_imagelist_load() handling of negative extension */

    nulllist = cpl_imagelist_load(file, CPL_TYPE_INT, -1);
    cpl_test_error( CPL_ERROR_ILLEGAL_INPUT );
    cpl_test_null( nulllist );

    /* Append error test */
    error = cpl_imagelist_save(imlist, file, CPL_TYPE_DOUBLE, NULL,
                               CPL_IO_APPEND);
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);

    image = cpl_image_new(IMAGE_SZ+1,IMAGE_SZ+1, CPL_TYPE_FLOAT);
    imlist2 = cpl_imagelist_new();
    cpl_imagelist_set(imlist2, image, 0);
    error = cpl_imagelist_save(imlist2, file, CPL_TYPE_FLOAT, NULL,
                               CPL_IO_APPEND);
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);
    cpl_imagelist_delete(imlist2);

    /* Tests cpl_imagelist_load() for no-casting type */
    imlist2 = cpl_imagelist_load(file, CPL_TYPE_UNSPECIFIED, 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(imlist2);

    cpl_test_imagelist_abs(imlist, imlist2, 10.0 * FLT_EPSILON);
    cpl_imagelist_delete(imlist2);

    cpl_test_zero( cpl_imagelist_save(imlist, file,  CPL_TYPE_FLOAT, NULL,
                                      CPL_IO_APPEND) );

    imlist2 = cpl_imagelist_load(file, CPL_TYPE_UNSPECIFIED, 1);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(imlist2);

    cpl_test_imagelist_abs(imlista, imlist2, 10.0 * FLT_EPSILON);
    cpl_imagelist_delete(imlista);
    
    cpl_test_zero( cpl_imagelist_erase(imlist, eraser) );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( 2*cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_imagelist_empty(imlist); /* Test cpl_imagelist_unset() */
    cpl_imagelist_delete(imlist);

    /* Tests cpl_imagelist_load() for no-casting type */
    imlist = cpl_imagelist_load(file, CPL_TYPE_UNSPECIFIED, 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(imlist);

    image = cpl_imagelist_get(imlist, 0);
    cpl_test_eq(cpl_image_get_type(image), CPL_TYPE_FLOAT);

    cpl_imagelist_delete(imlist);

    imlist = cpl_imagelist_load_window(file, CPL_TYPE_UNSPECIFIED, 0,
                       IMAGE_SZ/4, IMAGE_SZ/4,
                       3*IMAGE_SZ/4, 3*IMAGE_SZ/4);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(imlist);

    image = cpl_imagelist_get(imlist, 0);
    cpl_test_eq(cpl_image_get_type(image), CPL_TYPE_FLOAT);

    cpl_imagelist_delete(imlist);

    /* Test cpl_imagelist_load() */
    cpl_msg_info("", "Load image list as type DOUBLE");
    imlist = cpl_imagelist_load(file, CPL_TYPE_DOUBLE, 0);

    cpl_test_nonnull( imlist );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_test_zero( cpl_imagelist_erase(imlist, eraser) );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( 2*cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_imagelist_empty(imlist); /* Test cpl_imagelist_unset() */
    cpl_imagelist_delete(imlist);
    
    /* Test cpl_imagelist_load() */
    cpl_msg_info("", "Load image list as type FLOAT");
    imlist = cpl_imagelist_load(file, CPL_TYPE_FLOAT, 0);

    cpl_test_nonnull( imlist );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_test_zero( cpl_imagelist_erase(imlist, eraser) );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( 2*cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_imagelist_empty(imlist); /* Test cpl_imagelist_unset() */
    cpl_imagelist_delete(imlist);

    /* Test cpl_imagelist_load() */
    cpl_msg_info("", "Load image list as type INTEGER");
    imlist = cpl_imagelist_load(file, CPL_TYPE_INT, 0);

    cpl_test_nonnull( imlist );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_test_zero( cpl_imagelist_erase(imlist, eraser) );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( 2*cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_imagelist_empty(imlist); /* Test cpl_imagelist_unset() */
    cpl_imagelist_delete(imlist);

    /* Test cpl_imagelist_load_window() */
    cpl_msg_info("", "Load image list as type DOUBLE from a window");
    imlist = cpl_imagelist_load_window(file, CPL_TYPE_DOUBLE, 0, IMAGE_SZ/4,
                                       IMAGE_SZ/4, 3*IMAGE_SZ/4, 3*IMAGE_SZ/4);

    cpl_test_nonnull( imlist );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_test_zero( cpl_imagelist_erase(imlist, eraser) );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( 2*cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_imagelist_empty(imlist); /* Test cpl_imagelist_unset() */
    cpl_imagelist_delete(imlist);
    

    /* Test cpl_imagelist_load_window() */
    cpl_msg_info("", "Load image list as type FLOAT from a window");
    imlist = cpl_imagelist_load_window(file, CPL_TYPE_FLOAT, 0, IMAGE_SZ/4,
                                       IMAGE_SZ/4, 3*IMAGE_SZ/4, 3*IMAGE_SZ/4);

    cpl_test_nonnull( imlist );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_test_zero( cpl_imagelist_erase(imlist, eraser) );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( 2*cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_imagelist_empty(imlist); /* Test cpl_imagelist_unset() */
    cpl_imagelist_delete(imlist);
    

    /* Test cpl_imagelist_load_window() */
    cpl_msg_info("", "Load image list as type INTEGER from a window");
    imlist = cpl_imagelist_load_window(file, CPL_TYPE_INT, 0, IMAGE_SZ/4,
                                       IMAGE_SZ/4, 3*IMAGE_SZ/4, 3*IMAGE_SZ/4);

    cpl_test_nonnull( imlist );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( cpl_imagelist_get_size(imlist), NIMAGES );

    cpl_test_zero( cpl_imagelist_erase(imlist, eraser) );
    cpl_test_zero( cpl_imagelist_is_uniform(imlist) );
    cpl_test_eq( 2*cpl_imagelist_get_size(imlist), NIMAGES );

    /* Clean up  */

    cpl_imagelist_empty(imlist); /* Test cpl_imagelist_unset() */
    cpl_imagelist_delete(imlist);

    cpl_imagelist_empty(imlist2); /* Test cpl_imagelist_unset() */
    cpl_imagelist_delete(imlist2);
    cpl_vector_unwrap(eraser);
    remove(file);

    if (stream != stdout) cpl_test_zero( fclose(stream) );

    /* End of tests */
    return cpl_test_end(0);
}
