/* $Id: cpl_fft-test.c,v 1.19 2012/03/12 13:28:43 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2012/03/12 13:28:43 $
 * $Revision: 1.19 $
 * $Name: cpl-6_0 $
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "cpl_fft.h"

#include "cpl_test.h"

#include "cpl_image_io_impl.h"

/*-----------------------------------------------------------------------------
                                Defines
 -----------------------------------------------------------------------------*/

#ifndef IMAGESZ
#define IMAGESZ         10
#endif

#ifndef CONSTANT
#define CONSTANT        200
#endif

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cpl_fft_test Unit tests of the CPL FFT functions
 */
/*----------------------------------------------------------------------------*/



/*-----------------------------------------------------------------------------
                            Private Function prototypes
 -----------------------------------------------------------------------------*/
static void cpl_fft_image_test(void);
#if defined CPL_FFTWF_INSTALLED || defined CPL_FFTW_INSTALLED
static void cpl_fft_image_test_one(cpl_size, cpl_size, cpl_type);
#endif

/*----------------------------------------------------------------------------*/
/**
   @brief   Unit tests of cpl_fft module
**/
/*----------------------------------------------------------------------------*/

int main(void)
{

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Insert tests below */
#ifdef CPL_FFTWF_INSTALLED
    cpl_fft_image_test_one(16, 16, CPL_TYPE_FLOAT);
    cpl_fft_image_test_one( 4, 32, CPL_TYPE_FLOAT);
    cpl_fft_image_test_one( 4, 4, CPL_TYPE_FLOAT);
    cpl_fft_image_test_one( 2, 128, CPL_TYPE_FLOAT);
    cpl_fft_image_test_one( 128, 2, CPL_TYPE_FLOAT);
#endif

#ifdef CPL_FFTW_INSTALLED
    cpl_fft_image_test_one(16, 16, CPL_TYPE_DOUBLE);
    cpl_fft_image_test_one(32,  4, CPL_TYPE_DOUBLE);
    cpl_fft_image_test_one( 4, 4, CPL_TYPE_DOUBLE);
    cpl_fft_image_test_one( 2, 128, CPL_TYPE_DOUBLE);
    cpl_fft_image_test_one( 128, 2, CPL_TYPE_DOUBLE);
#endif

    cpl_fft_image_test();

    /* End of tests */
    return cpl_test_end(0);
}


/*----------------------------------------------------------------------------*/
/**
   @internal
   @brief  Unit tests of the function
   @see cpl_fft_image()
**/
/*----------------------------------------------------------------------------*/

static void cpl_fft_image_test(void)
{
    const cpl_type imtypes[] = {CPL_TYPE_DOUBLE, CPL_TYPE_FLOAT,
                                CPL_TYPE_INT, CPL_TYPE_DOUBLE_COMPLEX,
                                CPL_TYPE_FLOAT_COMPLEX};
    int            ityp;
    int            nok = 0; /* Number of successful calls */

    /* Insert tests below */

    /* Iterate through all pixel types */
    for (ityp = 0; ityp < (int)(sizeof(imtypes)/sizeof(imtypes[0])); ityp++) {
        const cpl_type imtype = imtypes[ityp];

        int ityp2;

        cpl_image * img1 = cpl_image_new(IMAGESZ, IMAGESZ, imtype);
        cpl_image * img3 = cpl_image_new(IMAGESZ, IMAGESZ, imtype);
        cpl_error_code error;

        /* Various error checks */
        error = cpl_fft_image(img3, NULL, CPL_FFT_FORWARD);
        cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

        error = cpl_fft_image(NULL, img3, CPL_FFT_FORWARD);
        cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

        error = cpl_fft_image(img3, img3, CPL_FFT_FORWARD | CPL_FFT_BACKWARD);
        if (imtype & CPL_TYPE_COMPLEX) {
            if (imtype & CPL_TYPE_DOUBLE) {
#ifdef CPL_FFTW_INSTALLED
                cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);
#else
                cpl_test_eq_error(error, CPL_ERROR_UNSUPPORTED_MODE);
#endif
            } else if (imtype & CPL_TYPE_FLOAT) {
#ifdef CPL_FFTWF_INSTALLED
                cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);
#else
                cpl_test_eq_error(error, CPL_ERROR_UNSUPPORTED_MODE);
#endif
            } else {
                cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);
            }
        } else if (imtype == CPL_TYPE_DOUBLE) {
#ifdef CPL_FFTW_INSTALLED
            cpl_test_eq_error(error, CPL_ERROR_TYPE_MISMATCH);
#else
            cpl_test_eq_error(error, CPL_ERROR_UNSUPPORTED_MODE);
#endif
        } else if (imtype == CPL_TYPE_FLOAT) {
#ifdef CPL_FFTWF_INSTALLED
            cpl_test_eq_error(error, CPL_ERROR_TYPE_MISMATCH);
#else
            cpl_test_eq_error(error, CPL_ERROR_UNSUPPORTED_MODE);
#endif
        } else {
            cpl_test_eq_error(error, CPL_ERROR_TYPE_MISMATCH);
        }

        if (!(imtype & CPL_TYPE_COMPLEX)) {
            error = cpl_image_fill_noise_uniform(img1, 0, CONSTANT);
            cpl_test_eq_error(error, CPL_ERROR_NONE);
        }

        for (ityp2 = 0; ityp2 < (int)(sizeof(imtypes)/sizeof(imtypes[0]));
             ityp2++) {
            const cpl_type imtype2 = imtypes[ityp2];
            cpl_image * img2 = cpl_image_new(IMAGESZ, IMAGESZ, imtype2);
            const cpl_image * imgin = img3;
            cpl_image * imgout = img2;
            int idir;
            /* No scaling on the forward transform has no effect */
            unsigned mode = CPL_FFT_FORWARD | CPL_FFT_NOSCALE;


            error = cpl_image_copy(img3, img1, 1, 1);
            cpl_test_eq_error(error, CPL_ERROR_NONE);

            /* Transform first forward, then backward */
            /* Those two iterations will succeed iff the input image
               and output image have matching non-integer precision */

            for (idir = 0; idir < 2; idir++, mode = CPL_FFT_BACKWARD,
                     imgin = img2, imgout = img3) {

                error = cpl_fft_image(imgout, imgin, mode);

                if (cpl_image_get_type(img3) == CPL_TYPE_FLOAT &&
                           cpl_image_get_type(img2) ==
                           (CPL_TYPE_FLOAT | CPL_TYPE_COMPLEX)) {
#ifdef CPL_FFTWF_INSTALLED
                    cpl_test_eq_error(CPL_ERROR_NONE, error);
                    nok++;

                    if (mode == CPL_FFT_BACKWARD) {
                        /* Transformed forward and backwards, so the result
                           should equal the original input */
                        cpl_test_image_abs(img1, img3,
                                           3.0 * FLT_EPSILON * CONSTANT);
                    }
#else
                    cpl_test_eq_error(CPL_ERROR_UNSUPPORTED_MODE, error);
#endif
                } else if (cpl_image_get_type(img3) == CPL_TYPE_DOUBLE &&
                           cpl_image_get_type(img2) ==
                           (CPL_TYPE_DOUBLE | CPL_TYPE_COMPLEX)) {
#ifdef CPL_FFTW_INSTALLED
                    cpl_test_eq_error(CPL_ERROR_NONE, error);
                    nok++;

                    if (mode == CPL_FFT_BACKWARD) {
                        /* Transformed forward and backwards, so the result
                           should equal the original input */
                        cpl_test_image_abs(img1, img3,
                                           5.0 * DBL_EPSILON * CONSTANT);
                    }
#else
                    cpl_test_eq_error(CPL_ERROR_UNSUPPORTED_MODE, error);
#endif

                } else if (cpl_image_get_type(img3) ==
                           (CPL_TYPE_DOUBLE | CPL_TYPE_COMPLEX) &&
                           cpl_image_get_type(img2) ==
                           (CPL_TYPE_DOUBLE | CPL_TYPE_COMPLEX)) {
#ifdef CPL_FFTW_INSTALLED
                    cpl_test_eq_error(CPL_ERROR_NONE, error);
#else
                    cpl_test_eq_error(CPL_ERROR_UNSUPPORTED_MODE, error);
#endif
                } else if (cpl_image_get_type(img3) ==
                           (CPL_TYPE_FLOAT | CPL_TYPE_COMPLEX) &&
                           cpl_image_get_type(img2) ==
                           (CPL_TYPE_FLOAT | CPL_TYPE_COMPLEX)) {
#ifdef CPL_FFTWF_INSTALLED
                    cpl_test_eq_error(CPL_ERROR_NONE, error);
#else
                    cpl_test_eq_error(CPL_ERROR_UNSUPPORTED_MODE, error);
#endif
                } else if ((imtype & CPL_TYPE_INT) ||
                           (imtype2 & CPL_TYPE_INT)) {
                    cpl_test_eq_error(CPL_ERROR_TYPE_MISMATCH, error);
                } else if ((imtype  & (CPL_TYPE_FLOAT | CPL_TYPE_DOUBLE | CPL_TYPE_INT)) !=
                           (imtype2 & (CPL_TYPE_FLOAT | CPL_TYPE_DOUBLE | CPL_TYPE_INT))) {
                    cpl_test_eq_error(CPL_ERROR_TYPE_MISMATCH, error);
                } else if (!((imtype  & CPL_TYPE_COMPLEX) ^
                             (imtype2 & CPL_TYPE_COMPLEX))) {
                    /* None or both are complex */
                    if (imtype == CPL_TYPE_DOUBLE) {
#ifdef CPL_FFTW_INSTALLED
                        cpl_test_eq_error(CPL_ERROR_TYPE_MISMATCH, error);
#else
                        cpl_test_eq_error(CPL_ERROR_UNSUPPORTED_MODE, error);
#endif
                    } else if (imtype == CPL_TYPE_FLOAT) {
#ifdef CPL_FFTWF_INSTALLED
                        cpl_test_eq_error(CPL_ERROR_TYPE_MISMATCH, error);
#else
                        cpl_test_eq_error(CPL_ERROR_UNSUPPORTED_MODE, error);
#endif
                    } else {
                        cpl_test_eq_error(CPL_ERROR_TYPE_MISMATCH, error);
                    }
                } else if (imtype & CPL_TYPE_DOUBLE) {
#ifdef CPL_FFTW_INSTALLED
                        cpl_test_eq_error(CPL_ERROR_TYPE_MISMATCH, error);
#else
                        cpl_test_eq_error(CPL_ERROR_UNSUPPORTED_MODE, error);
#endif
                } else if (imtype & CPL_TYPE_FLOAT) {
#ifdef CPL_FFTWF_INSTALLED
                        cpl_test_eq_error(CPL_ERROR_TYPE_MISMATCH, error);
#else
                        cpl_test_eq_error(CPL_ERROR_UNSUPPORTED_MODE, error);
#endif
                } else {
                    cpl_test_eq_error(CPL_ERROR_TYPE_MISMATCH, error);
                }
            }
            cpl_image_delete(img2);
        }
        cpl_image_delete(img1);
        cpl_image_delete(img3);
    }
#if defined CPL_FFTWF_INSTALLED && defined CPL_FFTW_INSTALLED
    cpl_test_eq(nok, 4); /* Forward and backward of float and double */
#elif defined CPL_FFTWF_INSTALLED
    cpl_msg_warning(cpl_func, "Double precision FFT not available for "
                    "unit testing");
    cpl_test_eq(nok, 2); /* Forward and backward of type float */
#elif defined CPL_FFTW_INSTALLED
    cpl_msg_warning(cpl_func, "Single precision FFT not available for "
                    "unit testing");
    cpl_test_eq(nok, 2); /* Forward and backward of type double */
#else
    cpl_msg_warning(cpl_func, "FFT not available for unit testing");
    cpl_test_zero(nok);
#endif

}

#if defined CPL_FFTWF_INSTALLED || defined CPL_FFTW_INSTALLED
/*----------------------------------------------------------------------------*/
/**
   @internal
   @brief  Unit tests of the function
   @param  nx  Size in x (the number of columns)
   @param  ny  Size in y (the number of rows)
   @param type One of CPL_TYPE_DOUBLE or CPL_TYPE_FLOAT
   @see cpl_fft_image()
**/
/*----------------------------------------------------------------------------*/

static void cpl_fft_image_test_one(cpl_size nx, cpl_size ny, cpl_type type)
{

    cpl_image    * image1r = cpl_image_new(nx, ny, type);
    cpl_image    * image1c;
    cpl_image    * image2  = cpl_image_new(nx, ny, type);
    cpl_image    * image3r = cpl_image_new(nx, ny, type | CPL_TYPE_COMPLEX);
    cpl_image    * image3c = cpl_image_new(nx, ny, type | CPL_TYPE_COMPLEX);
    cpl_image    * image3h = cpl_image_new(nx/2+1, ny, type | CPL_TYPE_COMPLEX);
    cpl_image    * image4;
    cpl_image    * image4r;
    cpl_image    * image4c;
    cpl_image    * image5  = cpl_image_new(nx, ny, type);
    cpl_error_code error;

    error = cpl_image_fill_noise_uniform(image1r, 0.0, 1.0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    image1c = cpl_image_cast(image1r, type | CPL_TYPE_COMPLEX);

    /* Real-to-complex, both full size */
    error = cpl_fft_image(image3r, image1r, CPL_FFT_FORWARD);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    /* Extract half of r2c transform */
    image4 = cpl_image_extract(image3r, 1, 1, nx/2 + 1, ny);

    /* Real-to-complex, complex is half size */
    error = cpl_fft_image(image3h, image1r, CPL_FFT_FORWARD);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    /* That half has to match the transform onto the half-sized image */
    cpl_test_image_abs(image3h, image4, type == CPL_TYPE_DOUBLE ?
                       DBL_EPSILON : FLT_EPSILON);

    /* Complex-to-complex of same real values */
    error = cpl_fft_image(image3c, image1c, CPL_FFT_FORWARD);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    /* Extract half of c2c transform */
    cpl_image_delete(image4);
    image4 = cpl_image_extract(image3c, 1, 1, nx/2 + 1, ny);

    cpl_test_image_abs(image3h, image4, 10.0 * nx *
                       (type == CPL_TYPE_DOUBLE ? DBL_EPSILON : FLT_EPSILON));

    /* Complex-to-real, both full size */
    error = cpl_fft_image(image2, image3r, CPL_FFT_BACKWARD);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    /* The back-transformed must match the original image */
    cpl_test_image_abs(image1r, image2, 6.0 * (type == CPL_TYPE_DOUBLE ?
                                               DBL_EPSILON : FLT_EPSILON));

    /* Complex-to-real, complex is half size */
    error = cpl_fft_image(image2, image3h, CPL_FFT_BACKWARD);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    /* The back-transformed must match the original image */
    cpl_test_image_abs(image1r, image2, 6.0 * (type == CPL_TYPE_DOUBLE ?
                                               DBL_EPSILON : FLT_EPSILON));

    /* Complex-to-complex of same real values */
    error = cpl_fft_image(image3r, image3c, CPL_FFT_BACKWARD);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    /* The back-transformed must match the original image - on the real part */
    image4r = cpl_image_extract_real(image3r);

    cpl_test_image_abs(image1r, image4r, 6.0 * (type == CPL_TYPE_DOUBLE ?
                                                DBL_EPSILON : FLT_EPSILON));

    /* The back-transformed must have a zero-valued imaginary part */
    image4c = cpl_image_extract_imag(image3r);
    cpl_image_delete(image4);
    image4 = cpl_image_new(nx, ny, type);

    cpl_test_image_abs(image4c, image4, 2.0 * (type == CPL_TYPE_DOUBLE ?
                                               DBL_EPSILON : FLT_EPSILON));

    cpl_image_delete(image1r);
    cpl_image_delete(image1c);
    cpl_image_delete(image2);
    cpl_image_delete(image3r);
    cpl_image_delete(image3c);
    cpl_image_delete(image3h);
    cpl_image_delete(image4);
    cpl_image_delete(image4r);
    cpl_image_delete(image4c);
    cpl_image_delete(image5);

}
#endif
