/* $Id: cpl_fft.c,v 1.21 2012/03/12 09:48:27 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


/*----------------------------------------------------------------------------
                                   Includes
 ----------------------------------------------------------------------------*/

/* Must be included first to ensure declaration of complex image accessors */
#include <complex.h>

#include "cpl_fft.h"

#include "cpl_error_impl.h"

#if defined CPL_FFTWF_INSTALLED || defined CPL_FFTW_INSTALLED
/* If FFTW is installed */
#include <fftw3.h>
#endif

#include <math.h>
#include <string.h>

/*---------------------------------------------------------------------------*/
/**
 * @defgroup cpl_fft FFTW wrappers
 *
 * This module provides FFTW wrappers
 *
 * @par Synopsis:
 * @code
 *   #include "cpl_fft.h"
 * @endcode
 */
/*---------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                                   Private functions
 -----------------------------------------------------------------------------*/

#ifdef CPL_FFTW_INSTALLED
#  define CPL_FFTW fftw
/* Cannot concatenate the reserved macro complex :-( */
#  define CPL_FFTW_TYPE fftw_complex
#  define CPL_TYPE double
/* Cannot concatenate the reserved macro complex :-( */
#  define CPL_TYPE_C double_complex
#  include "cpl_fft_body.h"
#  undef CPL_FFTW
#  undef CPL_TYPE
#  undef CPL_TYPE_T
#  undef CPL_TYPE_C
#  undef CPL_FFTW_TYPE
#endif

#ifdef CPL_FFTWF_INSTALLED
#  define CPL_FFTW fftwf
/* Cannot concatenate the reserved macro complex :-( */
#  define CPL_FFTW_TYPE fftwf_complex
#  define CPL_TYPE float
/* Cannot concatenate the reserved macro complex :-( */
#  define CPL_TYPE_C float_complex
#  include "cpl_fft_body.h"
#  undef CPL_FFTW
#  undef CPL_TYPE
#  undef CPL_TYPE_T
#  undef CPL_TYPE_C
#  undef CPL_FFTW_TYPE
#endif

/*---------------------------------------------------------------------------*/
/**
  @brief    Perform a FFT operation on an image
  @param  self  Pre-allocated output image to transform to
  @param  other Input image to transform from
  @param  mode  CPL_FFT_FORWARD or CPL_FFT_BACKWARD, optionally CPL_FFT_NOSCALE
  @return CPL_ERROR_NONE or the corresponding #_cpl_error_code_ on error
  
  This function performs an FFT on an image, using FFTW. CPL may be configured
  without this library, in this case an otherwise valid call will set and return
  the error CPL_ERROR_UNSUPPORTED_MODE.

  The input and output images must match in precision level. Integer images are
  not supported.

  In a forward transform the input image may be non-complex. In this case a
  real-to-complex transform is performed. This will only compute the first
  ny/2 + 1 columns of the transform. In this transform it is allowed to pass
  an output image with ny/2 + 1 columns.

  Similarly, in a backward transform the output image may be non-complex. In
  this case a complex-to-real transform is performed. This will only transform
  the first ny/2 + 1 columns of the input. In this transform it is allowed to
  pass an input image with ny/2 + 1 columns.

  Per default the backward transform scales (divides) the result with the
  number of elements transformed (i.e. the number of pixels in the result
  image). This scaling can be turned off with CPL_FFT_NOSCALE.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an image is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the mode is illegal
  - CPL_ERROR_INCOMPATIBLE_INPUT if the image sizes do not match
  - CPL_ERROR_TYPE_MISMATCH if the image types are incompatible with each other
                            or with the transform
  - CPL_ERROR_UNSUPPORTED_MODE if FFTW has not been installed
 */
/*---------------------------------------------------------------------------*/
cpl_error_code cpl_fft_image(cpl_image * self, const cpl_image * other,
                             cpl_fft_mode mode)
{
    const cpl_type typin  = cpl_image_get_type(other);
    const cpl_type typout = cpl_image_get_type(self);

    const cpl_size lnxin  = cpl_image_get_size_x(other);
    const cpl_size lnyin  = cpl_image_get_size_y(other);
    const cpl_size lnxout = cpl_image_get_size_x(self);
    const cpl_size lnyout = cpl_image_get_size_y(self);

    const int      nxin   = (int)lnxin;
    const int      nyin   = (int)lnyin;
    const int      nxout  = (int)lnxout;
    const int      nyout  = (int)lnyout;
    const int      nxh    = ((mode & CPL_FFT_FORWARD) ? nxin : nxout) / 2 + 1;

    cpl_error_code error;


    cpl_ensure_code(self  != NULL,             CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(other    != NULL,             CPL_ERROR_NULL_INPUT);

    /* FFTW (only) supports dimensions of type int */
    cpl_ensure_code((cpl_size)nxin  == lnxin,  CPL_ERROR_UNSUPPORTED_MODE);
    cpl_ensure_code((cpl_size)nyin  == lnyin,  CPL_ERROR_UNSUPPORTED_MODE);
    cpl_ensure_code((cpl_size)nxout == lnxout, CPL_ERROR_UNSUPPORTED_MODE);
    cpl_ensure_code((cpl_size)nyout == lnyout, CPL_ERROR_UNSUPPORTED_MODE);

    if (mode & CPL_FFT_FORWARD && !(typin & CPL_TYPE_COMPLEX)) {
        cpl_ensure_code(nxin == nxout || nxout == nxh,
                        CPL_ERROR_INCOMPATIBLE_INPUT);
    } else if (mode & CPL_FFT_BACKWARD && !(typout & CPL_TYPE_COMPLEX)) {
        cpl_ensure_code(nxin == nxout || nxin == nxh,
                        CPL_ERROR_INCOMPATIBLE_INPUT);
    } else {
        cpl_ensure_code(nxin == nxout,         CPL_ERROR_INCOMPATIBLE_INPUT);
    }
    cpl_ensure_code(lnyin == lnyout,           CPL_ERROR_INCOMPATIBLE_INPUT);

   if (typin & CPL_TYPE_FLOAT) {

       cpl_ensure_code(typout & CPL_TYPE_FLOAT, CPL_ERROR_TYPE_MISMATCH);

#ifdef CPL_FFTWF_INSTALLED
       error = cpl_fft_image_float(self, other, mode);
#else
       error = CPL_ERROR_UNSUPPORTED_MODE;
#endif
   } else if (typin & CPL_TYPE_DOUBLE) {

       cpl_ensure_code(typout & CPL_TYPE_DOUBLE, CPL_ERROR_TYPE_MISMATCH);

#ifdef CPL_FFTW_INSTALLED
       error = cpl_fft_image_double(self, other, mode);
#else
       error = CPL_ERROR_UNSUPPORTED_MODE;
#endif
   } else {
       error = CPL_ERROR_TYPE_MISMATCH;
   }

   /* Set or propagate error, if any */
   return cpl_error_set_message_(error, "mode=%d. %d X %d (%s) => "
                                 "%d X %d (%s)", (int)mode,
                                 nxin, nyin, cpl_type_get_name(typin),
                                 nxout, nyout, cpl_type_get_name(typout));
}

/*----------------------------------------------------------------------------*/
/**
  @brief  Perform a FFT operation on the images in an imagelist
  @param  self  Pre-allocated output imagelist to transform to
  @param  other Input imagelist to transform from
  @param  mode  CPL_FFT_FORWARD or CPL_FFT_BACKWARD, optionally CPL_FFT_NOSCALE
  @return CPL_ERROR_NONE or the corresponding #_cpl_error_code_ on error
  @see cpl_fft_image()

 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_fft_imagelist(cpl_imagelist       * self,
                                 const cpl_imagelist * other,
                                 cpl_fft_mode          mode)
{
    const cpl_size sizein = cpl_imagelist_get_size(other);
    cpl_size i;

    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(other  != NULL, CPL_ERROR_NULL_INPUT);
        
    cpl_ensure_code(cpl_imagelist_get_size(self) == sizein,
                    CPL_ERROR_INCOMPATIBLE_INPUT);

    for (i = 0; i < sizein; i++) {
        cpl_image       * imgout = cpl_imagelist_get(self, i);
        const cpl_image * imgin  = cpl_imagelist_get_const(other, i);

        if (cpl_fft_image(imgout, imgin, mode)) break;
    }

    return i == sizein ? CPL_ERROR_NONE : cpl_error_set_where_();
}

/**@}*/
