/* $Id: cpl_io_fits.c,v 1.31 2012/02/03 10:53:32 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2012/02/03 10:53:32 $
 * $Revision: 1.31 $
 * $Name: cpl-6_0 $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "cpl_io_fits.h"

#include <cpl_memory.h>
#include <cpl_error_impl.h>

#include <string.h>
#include <assert.h>

/* Needed for patch of fits_flush_file(). */
#include <fitsio2.h>

/* Try to use stat() */

#undef CPL_HAVE_STAT

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#if defined HAVE_STAT && defined HAVE_DECL_STAT
#define CPL_HAVE_STAT
#endif

/*----------------------------------------------------------------------------*/
/**
 * defgroup cpl_io_fits   Optimize open and close of FITS files
 *
 * The CPL API for FITS I/O passes only the FITS file name, and per default
 * opens and closes each file for each I/O operation. Since the FITS standard
 * does not allow random access to a given extension, the open/close approach
 * causes the writing of a file with N extensions to have complexity O(N^2).
 * The same is true for reading all N extensions.
 *
 * The complexity of those operations can be reduced to the expected O(N) by
 * keeping the FITS files open between operations. This is done with 
 * static (thread-shared) storage of the relevant data.
 *
 * @par Synopsis:
 * @code
 *   #include "cpl_io_fits.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                               Private types
 -----------------------------------------------------------------------------*/

typedef struct cpl_fitsfiles_t {
    /* The two pointers are either both NULL or both non-NULL */
    char      * name;
    fitsfile  * fptr;
    int         iomode; /* CFITSIO currently defines: READONLY, READWRITE */
    cpl_boolean has_stat; /* Set to true iff stat() can & was called OK. */
                          /* When false, the below members are undefined */
#ifdef CPL_HAVE_STAT
    dev_t      st_dev; /* ID of device containing file */
    ino_t      st_ino; /* inode number */
#endif
} cpl_fitsfiles_t;

/*-----------------------------------------------------------------------------
                        Private variables
 -----------------------------------------------------------------------------*/

#ifdef CPL_IO_FITS_MAX_OPEN
#if CPL_IO_FITS_MAX_OPEN > 0
static cpl_fitsfiles_t * cpl_fitsfiles = NULL;
static cpl_size cpl_nfitsfiles = 0;
#endif
static cpl_size cpl_maxfitsfiles = 0;
#endif
/*-----------------------------------------------------------------------------
                                   Private functions
 -----------------------------------------------------------------------------*/

static fitsfile * cpl_io_fits_remove_fptr(const char *, int *);
static const char * cpl_io_fits_find_name(fitsfile *, int *);
static void cpl_io_fits_set(fitsfile *, const char *, int) CPL_ATTR_NONNULL;
#if defined CPL_IO_FITS_MAX_OPEN && CPL_IO_FITS_MAX_OPEN > 0
static void cpl_io_fits_resize(void);
#endif

static int cpl_fits_flush_file(fitsfile *, int *);

/*-----------------------------------------------------------------------------
                              Function definitions
 -----------------------------------------------------------------------------*/

/**
 * @internal
 * @brief Initialize the internal storage
 * @param  enable Iff CPL_TRUE enable the optimized FITS I/O module
 * @return void
 * @see cpl_io_fits_end()
 */
void cpl_io_fits_init(cpl_boolean enable)
{
    if (enable) {
#ifndef CPL_IO_FITS_NDEBUG
        cpl_msg_debug(cpl_func, "Starting, max file pointers: "
                      CPL_STRINGIFY(CPL_IO_FITS_MAX_OPEN));
#endif

#if defined CPL_IO_FITS_MAX_OPEN && CPL_IO_FITS_MAX_OPEN > 0
#ifdef _OPENMP
#pragma omp critical(cpl_io_fits)
#endif
        {
            if (cpl_fitsfiles == NULL) {
                cpl_maxfitsfiles = CPL_IO_FITS_MAX_OPEN;
                cpl_fitsfiles = (cpl_fitsfiles_t*)
                    cpl_calloc(cpl_maxfitsfiles, sizeof(cpl_fitsfiles_t));
            }
        }
#endif
    }
}


#if defined CPL_IO_FITS_MAX_OPEN && CPL_IO_FITS_MAX_OPEN > 0
/**
 * @internal
 * @brief  Double the size of the I/O structure
 * @return void
 * @see cpl_io_fits_init()
 * @note Call (inside the critical section) if the current structure is full
 * 
 */
static void cpl_io_fits_resize(void)
{

#ifndef CPL_IO_FITS_NDEBUG
    cpl_msg_debug(cpl_func, "Resizing: %d", (int)cpl_maxfitsfiles);
#endif

    /* Resize the array to double the size */
    cpl_fitsfiles = (cpl_fitsfiles_t*)
        cpl_realloc(cpl_fitsfiles, 2 * cpl_maxfitsfiles *
                    sizeof(cpl_fitsfiles_t));
    /* Initialize the second half */
    (void)memset(cpl_fitsfiles + cpl_maxfitsfiles, 0,
                 cpl_maxfitsfiles * sizeof(cpl_fitsfiles_t));
    /* Double the limit */
    cpl_maxfitsfiles *= 2;

}
#endif

/**
 * @internal
 * @brief  Close all open FITS files
 * @return CPL_ERROR_NONE or the relevant #_cpl_error_code_ on error
 * @see cpl_io_fits_init()
 * @note Must be called before program termination, after it is called
 *       no other functions from this module may be called
 * 
 */
cpl_error_code cpl_io_fits_end(void)
{
    int status = 0;

#ifndef CPL_IO_FITS_NDEBUG
    cpl_msg_debug(cpl_func, "Finishing: %d", (int)cpl_maxfitsfiles);
#endif

    (void)cpl_io_fits_close_all(&status);

#if defined CPL_IO_FITS_MAX_OPEN && CPL_IO_FITS_MAX_OPEN > 0
#ifdef _OPENMP
#pragma omp critical(cpl_io_fits)
#endif
    {
        cpl_free((void*)cpl_fitsfiles);
        cpl_fitsfiles = NULL;
        cpl_maxfitsfiles = 0;
    }
#endif
    return status
        ? cpl_error_set_fits(CPL_ERROR_BAD_FILE_FORMAT, status, fits_close_file,
                             "Closing all open CFITSIO-files")
        : CPL_ERROR_NONE;
}

/**
 * @internal
 * @brief  Return true iff the I/O FITS optimized mode is enabled
 * @return CPL_TRUE iff the I/O FITS optimized mode is enabled
 * @see cpl_io_fits_init()
 * 
 */
cpl_boolean cpl_io_fits_is_enabled(void)
{

    return cpl_maxfitsfiles > 0 ? CPL_TRUE : CPL_FALSE;

}


/**
 * @internal
 * @brief Open a fits file and destroy any preexisting file
 * @param pfptr    CFITSIO file pointer pointer to file
 * @param filename Name of FITS file to open
 * @param status   Pointer to CFITSIO error status
 * @return         The CFITSIO error status
 * @see fits_create_file()
 * @note Since the underlying CFITSIO call supports meta-characters _all_
 * currently open files are closed prior to opening this one.
 *
 */
int cpl_io_fits_create_file(fitsfile **pfptr, const char *filename, int *status)
{
    if (*status == 0) { /* Duplicate CFITSIO behaviour */

        /* Look for the CFITSIO pointer - if it is there, the file is
           closed regardless of mode - and then reopened */

        *pfptr = cpl_io_fits_remove_fptr(filename, NULL);

        if ((*pfptr == NULL || !fits_close_file(*pfptr, status)) &&
            !fits_create_file(pfptr, filename, status)) {

#ifndef CPL_IO_FITS_NDEBUG
            cpl_msg_debug(cpl_func, "Reopened file for writing: %s", filename);
#endif

            /* FIXME: Assume READWRITE */
            cpl_io_fits_set(*pfptr, filename, READWRITE);
        }
    }

    return *status;
}

/**
 * @internal
 * @brief Try to reuse an already existing CFITSIO file pointer or reopen
 * @param pfptr    CFITSIO file pointer pointer to file
 * @param filename Name of FITS file to open
 * @param iomode   The CFITSIO iomode
 * @param status   Pointer to CFITSIO error status
 * @return         The CFITSIO error status
 * @see fits_open_diskfile()
 * @note Since this call may not actually open the file, any subsequent call to
 * fits_movrel_hdu() should be reimplemented with a call to fits_movabs_hdu().
 *
 */
int cpl_io_fits_open_diskfile(fitsfile **pfptr, const char * filename,
                              int iomode, int *status)
{

    if (*status == 0) { /* Duplicate CFITSIO behaviour */
        int myiomode = iomode;

        /* Determine if an already open file can be reused */
        *pfptr = cpl_io_fits_remove_fptr(filename, &myiomode);

        if (*pfptr != NULL) {

            if (myiomode == iomode) {
#ifndef CPL_IO_FITS_NDEBUG
                cpl_msg_debug(cpl_func, "Reusing handle (%p) for: %s (I/O-mode"
                              ": %d)", (const void*)*pfptr, filename, iomode);
#endif
                return *status;
            }

#ifndef CPL_IO_FITS_NDEBUG
            cpl_msg_debug(cpl_func, "Cannot reuse handle for: %s (I/O-mode: "
                          "%d <=> %d)", filename, iomode, myiomode);
#endif
            /* The file was opened, but with a different mode, so it has been
               unset from the I/O structure and can be closed for reopening */
            if (fits_close_file(*pfptr, status)) return *status;

        }

#ifndef CPL_IO_FITS_NDEBUG
        cpl_msg_debug(cpl_func, "Opening file: %s (I/O-mode: %d)",
                      filename, iomode);
#endif
        if (!fits_open_diskfile(pfptr, filename, iomode, status)) {
            cpl_io_fits_set(*pfptr, filename, iomode);
        }
    }

    return *status;

}

/**
 * @internal
 * @brief Instead of closing the file, just flush any written data
 * @param fptr    CFITSIO file pointer to file
 * @param status  Pointer to CFITSIO error status
 * @return        The CFITSIO error status
 * @see fits_flush_file()

  From the 3.280 source code of fits_flush_file():
  Flush all the data in the current FITS file to disk. This ensures that if
  the program subsequently dies, the disk FITS file will be closed correctly.

 */
int cpl_io_fits_close_file(fitsfile *fptr, int *status)
{

    if (*status == 0 && fptr != NULL) { /* Duplicate CFITSIO behaviour */
        int          iomode;
        const char * name = cpl_io_fits_find_name(fptr, &iomode);

        if (name != NULL) {

#ifndef CPL_IO_FITS_NDEBUG
            cpl_msg_debug(cpl_func, "Flushing handle for: %s (%d)",
                          name, iomode);
#endif

#ifdef CPL_IO_FITS_READONLY_FLUSH
            if (iomode != READONLY)
#endif
                (void)cpl_fits_flush_file(fptr, status);
        } else {
            /* It is actually a bug to enter here */
            (void)fits_close_file(fptr, status);
        }
    }

    return *status;
}


/*----------------------------------------------------------------------------*/
/**
  @internal
  @param status The CFITSIO status
  @return Zero on success or else the CFITSIO status
*/
/*----------------------------------------------------------------------------*/
int cpl_io_fits_close_all(int * status)
{
    if (*status == 0) { /* Duplicate CFITSIO behaviour */
        fitsfile * fptr;

        while ((fptr = cpl_io_fits_remove_fptr(NULL, NULL)) != NULL &&
               !fits_close_file(fptr, status)) {
#ifndef CPL_IO_FITS_NDEBUG
            cpl_msg_debug(cpl_func, "Closed CFITSIO-file: %p",
                          (const void*)fptr);
#endif
        }
    }

    return *status;
}

/**@}*/

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief Insert a CFITSIO triplet into the CPL I/O structure
  @param  fptr    The CFITSIO pointer to insert
  @param  name    The filename to insert
  @param  iomode  The I/O mode to insert
  @return True iff the CPL I/O struct is full, so a file must be closed
  @note Since this call is only done after a succesful opening of the named file
        name (and fptr) can safely be assumed to be non-NULL.
*/
/*----------------------------------------------------------------------------*/
static void cpl_io_fits_set(fitsfile * fptr, const char * name, int iomode)
{
    if (cpl_maxfitsfiles > 0) {
        char * filename;
        int    stat_err = 1; /* Set to non-zero in case stat() is unavailable */
#ifdef CPL_HAVE_STAT
        struct stat statbuf;
#endif

#if defined CPL_IO_FITS_MAX_OPEN && CPL_IO_FITS_MAX_OPEN > 0

        filename = cpl_strdup(*name == '!' ? name+1 : name);

#ifdef CPL_HAVE_STAT
        stat_err = stat(filename, &statbuf);
#endif


#ifdef _OPENMP
#pragma omp critical(cpl_io_fits)
#endif
        {
            cpl_size i;

            for (i = 0; i < cpl_nfitsfiles; i++) {
                if (cpl_fitsfiles[i].fptr == NULL) break;
            }

            if (i == cpl_nfitsfiles) {
                /* Not found */
                if (i == cpl_maxfitsfiles) {
                    cpl_io_fits_resize();
                }
                cpl_nfitsfiles++;
            }

            cpl_free(cpl_fitsfiles[i].name);
            cpl_fitsfiles[i].fptr   = fptr;
            cpl_fitsfiles[i].name   = filename;
            cpl_fitsfiles[i].iomode = iomode;

            if (stat_err) {
                cpl_fitsfiles[i].has_stat = CPL_FALSE;
            } else {
                cpl_fitsfiles[i].has_stat = CPL_TRUE;
#ifdef CPL_HAVE_STAT
                cpl_fitsfiles[i].st_dev = statbuf.st_dev;
                cpl_fitsfiles[i].st_ino = statbuf.st_ino; 
#endif
            }
        }
#endif
    }
}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief Search by name for an already opened FITS file with optional unset
  @param  name    The filename to look for, @em NULL will unset and return one
  @param  piomode Iff non-@em NULL and found, the I/O mode
  @return When found, the CFITSIO pointer structure otherwise NULL
  @note If the file is found, and if piomode is NULL or if it is non-NULL
        and different from the I/O mode of the already open file, then the entry
        is unset.
*/
/*----------------------------------------------------------------------------*/
static fitsfile * cpl_io_fits_remove_fptr(const char * name, int * piomode)
{

    fitsfile * fptr = NULL;

    if (cpl_maxfitsfiles > 0) {

#if defined CPL_IO_FITS_MAX_OPEN && CPL_IO_FITS_MAX_OPEN > 0

        const char * filename = name != NULL && *name == '!' ? name+1 : name;

#ifdef _OPENMP
#pragma omp critical(cpl_io_fits)
#endif
        {
            cpl_size i;

            for (i = 0; i < cpl_nfitsfiles; i++) {
                if (cpl_fitsfiles[i].name != NULL &&
                    (filename == NULL || !strcmp(cpl_fitsfiles[i].name,
                                                 filename))) break;
            }

#ifdef CPL_HAVE_STAT
            if (i == cpl_nfitsfiles && filename != NULL) {
                /* We did not find the file - with exactly that name -
                   so look for it under a different name */

                struct stat statbuf;

                if (!stat(filename, &statbuf)) {
                    for (i = 0; i < cpl_nfitsfiles; i++) {
                        if (cpl_fitsfiles[i].has_stat &&
                            cpl_fitsfiles[i].st_dev == statbuf.st_dev &&
                            cpl_fitsfiles[i].st_ino == statbuf.st_ino) break;
                    }
                }
            }
#endif

            if (i < cpl_nfitsfiles) {
                /* Found it */
                const int iomode = cpl_fitsfiles[i].iomode;

                fptr = cpl_fitsfiles[i].fptr;
                if (piomode == NULL || *piomode != iomode) {
                    cpl_free(cpl_fitsfiles[i].name);
                    cpl_fitsfiles[i].fptr = NULL;
                    cpl_fitsfiles[i].name = NULL;
                    cpl_fitsfiles[i].iomode = 0;
                    cpl_fitsfiles[i].has_stat = CPL_FALSE;
                    if (i == cpl_nfitsfiles-1) {
                        do {
                            cpl_nfitsfiles--;
                        } while (cpl_nfitsfiles > 0 &&
                                 cpl_fitsfiles[cpl_nfitsfiles-1].name == NULL);
                    }
                }
                if (piomode != NULL) *piomode = iomode;
            }
        }
#endif

    }

    return fptr;
}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief Try to find by CFITSIO pointer an already opened FITS file
  @param  fptr    The CFITSIO pointer to look for
  @param  piomode When found, the I/O mode
  @return When found, the name otherwise NULL
*/
/*----------------------------------------------------------------------------*/
static const char * cpl_io_fits_find_name(fitsfile * fptr, int * piomode)
{

    const char * name = NULL;

#if defined CPL_IO_FITS_MAX_OPEN && CPL_IO_FITS_MAX_OPEN > 0

    if (fptr != NULL && piomode != NULL)
#ifdef _OPENMP
#pragma omp critical(cpl_io_fits)
#endif
        {
            cpl_size i;

            for (i = 0; i < cpl_nfitsfiles; i++) {
                if (fptr == cpl_fitsfiles[i].fptr) break;
            }

            if (i < cpl_nfitsfiles) {
                /* Found it */
                name     = cpl_fitsfiles[i].name;
                *piomode = cpl_fitsfiles[i].iomode;
            }
        }
#endif

    return name;

}


/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief Perform proper flushing of CFITSIO files
  @param  fptr    The CFITSIO pointer to flush
  @param  status  Pointer to the CFITSIO error status
  @return 0 iff successful
  @see fits_flush_file() v. 3.28
  @note The call to ffflsh() is done with clearbuf set to TRUE instead of FALSE

*/
/*----------------------------------------------------------------------------*/
static int cpl_fits_flush_file(fitsfile *fptr,   /* I - FITS file pointer   */
           int *status)      /* IO - error status                           */
/*
  Flush all the data in the current FITS file to disk. This ensures that if
  the program subsequently dies, the disk FITS file will be closed correctly.
*/
{
    int hdunum, hdutype;

    if (*status > 0)
        return(*status);

    ffghdn(fptr, &hdunum);     /* get the current HDU number */

    if (ffchdu(fptr,status) > 0)   /* close out the current HDU */
        ffpmsg("ffflus could not close the current HDU.");

    ffflsh(fptr, TRUE, status);  /* flush any modified IO buffers to disk */

    if (ffgext(fptr, hdunum - 1, &hdutype, status) > 0) /* reopen HDU */
        ffpmsg("ffflus could not reopen the current HDU.");

    return(*status);
}

