/* $Id: cpl_fit-test.c,v 1.31 2011/02/14 15:47:18 cizzo Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cizzo $
 * $Date: 2011/02/14 15:47:18 $
 * $Revision: 1.31 $
 * $Name: cpl-5_3_0-BRANCH $
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <float.h>

#include "cpl_fit.h"

#include "cpl_memory.h"
#include "cpl_stats.h"
#include "cpl_tools.h"
#include "cpl_test.h"
#include "cpl_math_const.h"

/*-----------------------------------------------------------------------------
                                Defines
 -----------------------------------------------------------------------------*/

#ifndef IMAGESZ
#define IMAGESZ         10
#endif

#ifndef IMAGESZFIT
#define IMAGESZFIT      256
#endif

#ifndef WINSTART
#define WINSTART      3
#endif


/*----------------------------------------------------------------------------*/
/**
 * @defgroup cpl_fit_test Testing of the CPL utilities
 */
/*----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                            Defines
 -----------------------------------------------------------------------------*/

#define cpl_fit_imagelist_is_zero(A, B)              \
    cpl_fit_imagelist_is_zero_macro(A, B, __LINE__)
#define cpl_fit_image_is_zero(A, B)                  \
    cpl_fit_image_is_zero_macro(A, B, __LINE__)

/*-----------------------------------------------------------------------------
                            Private Function prototypes
 -----------------------------------------------------------------------------*/

static void cpl_fit_imagelist_polynomial_tests(void);
static void cpl_fit_image_gaussian_tests(void);
static void cpl_fit_imagelist_is_zero_macro(const cpl_imagelist *, double,
                                               int);
static void cpl_fit_image_is_zero_macro(const cpl_image *, double, int);
static void eval_gauss(const double x[], const double a[], double *);
static int  check_gauss_success(const char **, const double *, const double *,
                                const char **, const double *, const double *,
                                cpl_array   *, cpl_array    *, 
                                double, double, double, cpl_matrix   *);
static void cpl_fit_image_gaussian_test_one(void);


/*----------------------------------------------------------------------------*/
/**
   @brief   Unit tests of fit module
**/
/*----------------------------------------------------------------------------*/

int main(void)
{

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Insert tests below */

    cpl_fit_image_gaussian_tests();
    cpl_fit_imagelist_polynomial_tests();
    cpl_fit_image_gaussian_test_one();

    /* End of tests */
    return cpl_test_end(0);
}


static void cpl_fit_imagelist_polynomial_tests(void)
{

    const double     ditval[]    = {0.0, 1.0, 2.0, 3.0, 4.0, 
                    5.0, 6.0, 7.0, 8.0};
    const double     neqditval[] = {1.0,   2.0,  4.0,  6.0, 10.0,
                    14.0, 16.0, 18.0, 19.0};
    cpl_imagelist  * fit;
    cpl_imagelist  * fit_eq;
    cpl_imagelist  * input;
    cpl_image      * dfiterror
        = cpl_image_new(IMAGESZFIT, IMAGESZFIT, CPL_TYPE_DOUBLE);
    cpl_image      * ffiterror
        = cpl_image_new(IMAGESZFIT, IMAGESZFIT, CPL_TYPE_FLOAT);
    cpl_image      * ifiterror
        = cpl_image_new(IMAGESZFIT, IMAGESZFIT, CPL_TYPE_INT);
    cpl_image      * dfiterrorwin;
    const int        ndits = (int)(sizeof(ditval)/sizeof(double));
    cpl_vector     * vdit  = cpl_vector_wrap(ndits, (double*)ditval);
    cpl_vector     * nvdit = cpl_vector_wrap(ndits, (double*)neqditval);
    cpl_polynomial * vfiller = cpl_polynomial_new(1);
    const double     sqsum = 204.0; /* Sum of squares of ditvals */
    const double     mytol = 5.52 * FLT_EPSILON;
    int              i;
    const cpl_type   pixel_type[]
        = {CPL_TYPE_DOUBLE, CPL_TYPE_FLOAT, CPL_TYPE_INT};
    const cpl_type   pixeltype    = CPL_TYPE_FLOAT;
    int ntest;
    cpl_flops   flopsum = 0;
    cpl_flops   flops;
    double      secs = 0.0;
    size_t      bytes = 0;
    double      cputime;
    double      demax;


    cpl_msg_info(cpl_func, "Testing with %d %d X %d images",
                 ndits, IMAGESZFIT, IMAGESZFIT);

    fit = cpl_fit_imagelist_polynomial(NULL, NULL, 0, 0, CPL_FALSE,
                       CPL_TYPE_FLOAT, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null( fit );
    cpl_imagelist_delete(fit);

    fit = cpl_fit_imagelist_polynomial_window(NULL, NULL, 1, 1,
                          IMAGESZFIT, IMAGESZFIT,
                          0, 0, CPL_FALSE,
                          CPL_TYPE_FLOAT, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null( fit );
    cpl_imagelist_delete(fit);

    input = cpl_imagelist_new();
    fit = cpl_fit_imagelist_polynomial(vdit, input, 0, 0, CPL_FALSE,
                                          CPL_TYPE_FLOAT, NULL);
    cpl_test_error(CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_test_null( fit );
    cpl_imagelist_delete(fit);

    fit = cpl_fit_imagelist_polynomial_window(vdit, input, 1, 1,
                          IMAGESZFIT, IMAGESZFIT,
                          0, 0, CPL_FALSE,
                          CPL_TYPE_FLOAT, NULL);
    cpl_test_error(CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_test_null( fit );
    cpl_imagelist_delete(fit);

    fit = cpl_fit_imagelist_polynomial(vdit, input, 1, 1, CPL_FALSE,
                                          CPL_TYPE_FLOAT, NULL);
    cpl_test_error(CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_test_null( fit );
    cpl_imagelist_delete(fit);

    fit = cpl_fit_imagelist_polynomial_window(vdit, input, 1, 1,
                          IMAGESZFIT, IMAGESZFIT,
                          0, 0, CPL_FALSE,
                          CPL_TYPE_FLOAT, NULL);
    cpl_test_error(CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_test_null( fit );
    cpl_imagelist_delete(fit);


    /* Test with all types of pixels */
    for (ntest = 0; ntest < 3; ntest++) {

        const cpl_type test_type = pixel_type[ntest];

        cpl_image * image = cpl_image_new(IMAGESZFIT, IMAGESZ, test_type);

        
        cpl_msg_info(cpl_func, "Fitting with pixel type %u",
                     (unsigned)test_type);

        cpl_test_zero(cpl_imagelist_set(input, image, 0));

        image = cpl_image_duplicate(image);

        cpl_test_zero(cpl_image_fill_noise_uniform(image, 1.0, 20.0));

        cpl_test_zero(cpl_image_multiply_scalar(image, ditval[1]));

        cpl_test_zero(cpl_imagelist_set(input, image, 1));

        /* A perfectly linear set */
        for (i = 2; i < ndits; i++) {

            image
                = cpl_image_multiply_scalar_create(cpl_imagelist_get(input, 1),
                                                   ditval[i]);

            cpl_test_zero(cpl_imagelist_set(input, image, i));

        }

        flops = cpl_tools_get_flops();

        cputime = cpl_test_get_cputime();

        fit = cpl_fit_imagelist_polynomial(vdit, input, 1, ndits-1, CPL_FALSE,
                                              test_type, NULL);
        secs += cpl_test_get_cputime() - cputime;
        flopsum += cpl_tools_get_flops() - flops;
        bytes += cpl_test_get_bytes_imagelist(input);

        cpl_test_error(CPL_ERROR_NONE );
        cpl_test_eq( cpl_imagelist_get_size(fit), ndits - 1 );

        /* The linarity must be equal to the values in image 1
           - normalize */
        cpl_test_zero(cpl_image_divide(cpl_imagelist_get(fit, 0),
                                            cpl_imagelist_get(input, 1)));

        /* Subtract the expected value in the 1st image */
        cpl_test_zero(cpl_image_subtract_scalar(cpl_imagelist_get(fit, 0), 1.0));

        cpl_fit_imagelist_is_zero(fit, IMAGESZFIT * mytol);

        cpl_imagelist_delete(fit);

        flops = cpl_tools_get_flops();

        cputime = cpl_test_get_cputime();

        fit = cpl_fit_imagelist_polynomial_window(vdit, input,
                          WINSTART, WINSTART,
                          IMAGESZFIT - WINSTART,
                          IMAGESZ - WINSTART,
                          1, ndits-1, CPL_FALSE,
                          test_type, NULL);
        secs += cpl_test_get_cputime() - cputime;
        flopsum += cpl_tools_get_flops() - flops;
        bytes += cpl_test_get_bytes_imagelist(input);


        cpl_test_error(CPL_ERROR_NONE );
        cpl_test_eq( cpl_imagelist_get_size(fit), ndits - 1 );

        cpl_imagelist_delete(fit);

    /* 
     * For the _window() version of the fitting function, we don't test
     * the correctness of the products any further here,
     * as it would add too much complexity and the objective inside 
     * this loop is mainly to test on different pixel tests. We will do
     * do it in later steps.
     */

        cpl_imagelist_delete(input);
        input = cpl_imagelist_new();

    /*
     * Repeat with non-equidistant but symmetric x-axis values
     * and check validity of the eq_dist in this case
     */

        image = cpl_image_new(IMAGESZFIT, IMAGESZ, test_type);

        cpl_test_zero(cpl_image_fill_noise_uniform(image, 1.0, 20.0));

        cpl_test_zero(cpl_image_multiply_scalar(image, neqditval[0]));

        cpl_test_zero(cpl_imagelist_set(input, image, 0));

        /* A perfectly linear set */
        for (i = 1; i < ndits; i++) {

            image
                = cpl_image_multiply_scalar_create(cpl_imagelist_get(input, 0),
                                                   neqditval[i]);

            cpl_test_zero(cpl_imagelist_set(input, image, i));

        }

    /* Fit without is_eqdist option */
        flops = cpl_tools_get_flops();

        cputime = cpl_test_get_cputime();

        fit = cpl_fit_imagelist_polynomial(nvdit, input, 1, ndits-1,
                       CPL_FALSE, test_type, NULL);
        secs += cpl_test_get_cputime() - cputime;
        flopsum += cpl_tools_get_flops() - flops;
        bytes += cpl_test_get_bytes_imagelist(input);

        cpl_test_error(CPL_ERROR_NONE );
        cpl_test_eq( cpl_imagelist_get_size(fit), ndits - 1 );

    /* Repeat with is_eqdist */
        flops = cpl_tools_get_flops();

        cputime = cpl_test_get_cputime();

        fit_eq = cpl_fit_imagelist_polynomial(nvdit, input, 1, ndits-1,
                          CPL_TRUE, test_type, NULL);
        secs += cpl_test_get_cputime() - cputime;
        flopsum += cpl_tools_get_flops() - flops;
        bytes += cpl_test_get_bytes_imagelist(input);

        cpl_test_zero(cpl_imagelist_subtract(fit, fit_eq));

        cpl_fit_imagelist_is_zero(fit, IMAGESZFIT * mytol);

        cpl_imagelist_delete(fit);
        cpl_imagelist_delete(fit_eq);
        cpl_imagelist_delete(input);
        input = cpl_imagelist_new();

    }



    /* 2nd order function check with and without eq_dist */
    for (i=0; i < ndits; i++) {
        cpl_image * image = cpl_image_new(IMAGESZFIT, IMAGESZFIT,
                                          CPL_TYPE_DOUBLE);

        cpl_test_zero(cpl_image_add_scalar(image, neqditval[i]*neqditval[i]));

        cpl_test_zero(cpl_imagelist_set(input, image, i));

        cpl_msg_debug(cpl_func, "Dit and mean of input image no. %d: %g %g",
                      i, ditval[i], cpl_image_get_mean(image));
    }

    fit = cpl_fit_imagelist_polynomial(nvdit, input, 1, ndits, CPL_FALSE,
                       pixeltype, NULL);
    if (cpl_error_get_code() != CPL_ERROR_NONE ) {
        /* Fails on 32-bit intel, but not on others */
        cpl_test_error(CPL_ERROR_SINGULAR_MATRIX );
        cpl_test_null( fit );
    }

    fit_eq = cpl_fit_imagelist_polynomial(nvdit, input, 1, ndits, CPL_TRUE,
                      pixeltype, NULL);
    if (cpl_error_get_code() != CPL_ERROR_NONE ) {
        /* Fails on 32-bit intel, but not on others */
        cpl_test_error(CPL_ERROR_SINGULAR_MATRIX );
        cpl_test_null( fit );
    }

    cpl_test_zero(cpl_imagelist_subtract(fit, fit_eq));

    cpl_fit_imagelist_is_zero(fit, IMAGESZFIT * mytol);

    cpl_imagelist_delete(fit);
    cpl_imagelist_delete(fit_eq);
    cpl_imagelist_delete(input);
    input = cpl_imagelist_new();

    cpl_vector_unwrap(nvdit);

    /* Create a list of images with a 2nd order function */
    for (i=0; i < ndits; i++) {
        cpl_image * image = cpl_image_new(IMAGESZFIT, IMAGESZFIT,
                                          CPL_TYPE_DOUBLE);

        cpl_test_zero(cpl_image_add_scalar(image, ditval[i]*ditval[i]));

        cpl_test_zero(cpl_imagelist_set(input, image, i));

        cpl_msg_debug(cpl_func, "Dit and mean of input image no. %d: %g %g",
                      i, ditval[i], cpl_image_get_mean(image));
    }

    fit = cpl_fit_imagelist_polynomial(vdit, input, 1, ndits, CPL_FALSE,
                                          pixeltype,
                                          NULL);
    if (cpl_error_get_code() != CPL_ERROR_NONE ) {
        /* Fails on 32-bit intel, but not on others */
        cpl_test_error(CPL_ERROR_SINGULAR_MATRIX );
        cpl_test_null( fit );
    }
    cpl_imagelist_delete(fit);

    /* Illegal max-degree */
    fit = cpl_fit_imagelist_polynomial(vdit, input, 1, 0, CPL_FALSE,
                                          pixeltype, NULL);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_null( fit );
    cpl_imagelist_delete(fit);

    /* Illegal min-degree */
    fit = cpl_fit_imagelist_polynomial(vdit, input, -1, 0, CPL_FALSE,
                                          pixeltype, NULL);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_null( fit );
    cpl_imagelist_delete(fit);

    /* Illegal pixeltype */
    fit = cpl_fit_imagelist_polynomial(vdit, input, 0, 0, CPL_FALSE,
                                          CPL_TYPE_INVALID, NULL);
    cpl_test_error(CPL_ERROR_UNSUPPORTED_MODE);
    cpl_test_null( fit );
    cpl_imagelist_delete(fit);


    flops = cpl_tools_get_flops();
    cputime = cpl_test_get_cputime();

    /* Fit with zero-order term */
    /* Also, try to use an integer-type image for fitting error */
    fit = cpl_fit_imagelist_polynomial(vdit, input, 0, 2, CPL_TRUE,
                                          pixeltype,
                                          ifiterror);
    secs += cpl_test_get_cputime() - cputime;
    flopsum += cpl_tools_get_flops() - flops;
    bytes += cpl_test_get_bytes_imagelist(input);

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_eq( cpl_imagelist_get_size(fit), 3 );
    cpl_fit_image_is_zero(ifiterror, mytol); 

    cpl_test_zero(cpl_image_subtract_scalar(cpl_imagelist_get(fit, 2), 1.0));

    cpl_fit_imagelist_is_zero(fit, mytol);

    cpl_imagelist_delete(fit);


    flops = cpl_tools_get_flops();
    cputime = cpl_test_get_cputime();

    /* Fit with zero-order term */
    /* Also, try to use an integer-type image for fitting error */
    fit = cpl_fit_imagelist_polynomial(vdit, input, 0, ndits-1, CPL_TRUE,
                                          pixeltype,
                                          ifiterror);
    secs += cpl_test_get_cputime() - cputime;
    flopsum += cpl_tools_get_flops() - flops;
    bytes += cpl_test_get_bytes_imagelist(input);

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_eq( cpl_imagelist_get_size(fit), ndits );
    cpl_fit_image_is_zero(ifiterror, mytol);

    cpl_test_zero(cpl_image_subtract_scalar(cpl_imagelist_get(fit, 2), 1.0));

    cpl_fit_imagelist_is_zero(fit, mytol);

    cpl_imagelist_delete(fit);

    flops = cpl_tools_get_flops();
    cputime = cpl_test_get_cputime();

    /* Fit without zero-order term */
    fit = cpl_fit_imagelist_polynomial(vdit, input, 1, ndits-1, CPL_FALSE,
                                          pixeltype,
                                          dfiterror);
    secs += cpl_test_get_cputime() - cputime;
    flopsum += cpl_tools_get_flops() - flops;
    bytes += cpl_test_get_bytes_imagelist(input);

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_eq( cpl_imagelist_get_size(fit), ndits-1 );

    cpl_test_zero(cpl_image_subtract_scalar(cpl_imagelist_get(fit, 1), 1.0));

    cpl_fit_imagelist_is_zero(fit, mytol);
    cpl_fit_image_is_zero(dfiterror, mytol);

    cpl_imagelist_delete(fit);

    flops = cpl_tools_get_flops();
    cputime = cpl_test_get_cputime();

    /* Repeat previous test on _window() function */
    dfiterrorwin = cpl_image_new(IMAGESZFIT - 2 * WINSTART + 1,
                 IMAGESZ - 2 * WINSTART + 1,
                 CPL_TYPE_DOUBLE);

    fit = cpl_fit_imagelist_polynomial_window(vdit, input,
                          WINSTART, WINSTART,
                          IMAGESZFIT - WINSTART,
                          IMAGESZ - WINSTART,
                          1, ndits-1, CPL_FALSE,
                          pixeltype, dfiterrorwin);
    secs += cpl_test_get_cputime() - cputime;
    flopsum += cpl_tools_get_flops() - flops;
    bytes += cpl_test_get_bytes_imagelist(input); /* FIXME: Too large... */

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_eq( cpl_imagelist_get_size(fit), ndits-1 );

    cpl_test_eq(cpl_image_get_size_x(cpl_imagelist_get(fit, 0)), 
         IMAGESZFIT - 2 * WINSTART + 1 );
    cpl_test_eq(cpl_image_get_size_y(cpl_imagelist_get(fit, 0)), 
         IMAGESZ - 2 * WINSTART + 1 );

    cpl_test_zero(cpl_image_subtract_scalar(cpl_imagelist_get(fit, 1), 1.0));

    cpl_fit_imagelist_is_zero(fit, mytol);
    cpl_fit_image_is_zero(dfiterrorwin, mytol);

    cpl_imagelist_delete(fit);
    cpl_image_delete(dfiterrorwin);

    flops = cpl_tools_get_flops();
    cputime = cpl_test_get_cputime();

    /* Fit with no zero- and 1st-order terms */
    fit = cpl_fit_imagelist_polynomial(vdit, input, 2, ndits, CPL_TRUE,
                                          pixeltype,
                                          ffiterror);
    secs += cpl_test_get_cputime() - cputime;
    flopsum += cpl_tools_get_flops() - flops;
    bytes += cpl_test_get_bytes_imagelist(input);

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_eq( cpl_imagelist_get_size(fit), ndits-1 );

    cpl_test_zero(cpl_image_subtract_scalar(cpl_imagelist_get(fit, 0), 1.0));

    cpl_fit_imagelist_is_zero(fit, mytol);
    cpl_fit_image_is_zero(ffiterror, mytol);

    cpl_imagelist_delete(fit);

    flops = cpl_tools_get_flops();
    cputime = cpl_test_get_cputime();

    /* Fit with one zero-term */
    fit = cpl_fit_imagelist_polynomial(vdit, input, 0, 0, CPL_TRUE,
                                          pixeltype,
                                          dfiterror);
    secs += cpl_test_get_cputime() - cputime;
    flopsum += cpl_tools_get_flops() - flops;
    bytes += cpl_test_get_bytes_imagelist(input);

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_eq( cpl_imagelist_get_size(fit), 1 );

    cpl_test_zero(cpl_image_subtract_scalar(cpl_imagelist_get(fit, 0),
                                           sqsum/(double)ndits));

    cpl_fit_imagelist_is_zero(fit, mytol);

    cpl_imagelist_delete(fit);

    cpl_imagelist_delete(input);

    (void)cpl_vector_unwrap(vdit);

    /* Try to fit as many coefficients are there are data points */
    /* Also, use floats this time */

    /* Use a polynomial to filler vdit */
    i = 0;
    cpl_polynomial_set_coeff(vfiller, &i, 0.0);
    i = 1;
    cpl_polynomial_set_coeff(vfiller, &i, 1.0);


    input = cpl_imagelist_new();

    vdit = cpl_vector_new(1);

    for (ntest = 1; ntest <= ndits; ntest++) {
        const double gain = 4.0; /* Some random number */

        cpl_msg_info(cpl_func, "Fitting %d coefficients to as many points",
                     ntest);

        cpl_vector_set_size(vdit, ntest);

        cpl_vector_fill_polynomial(vdit, vfiller, 0.0, 1.0);

        /* Create a list of images with a 2nd order function */
        for (i = ntest - 1; i < ntest; i++) {
            cpl_image * image = cpl_image_new(IMAGESZFIT, IMAGESZFIT,
                                              pixeltype);

            cpl_test_zero(cpl_image_add_scalar(image, gain * ditval[i]*ditval[i]));

            cpl_test_zero(cpl_imagelist_set(input, image, i));

            cpl_msg_debug(cpl_func, "Dit and mean of input image no. %d: %g %g",
                          i, ditval[i], cpl_image_get_mean(image));
        }

        /* Ready for fitting */

        flops = cpl_tools_get_flops();
        cputime = cpl_test_get_cputime();

        /* Fit with zero-order term */
        fit = cpl_fit_imagelist_polynomial(vdit, input, 0, ntest-1, CPL_TRUE,
                                           pixeltype, ffiterror);
        secs += cpl_test_get_cputime() - cputime;
        flopsum += cpl_tools_get_flops() - flops;
        bytes += cpl_test_get_bytes_imagelist(input);

        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            cpl_test_null( fit );

            cpl_msg_warning(cpl_func, "Could not fit %d coefficients to as many "
                            "points", ntest);

            cpl_test_error(CPL_ERROR_SINGULAR_MATRIX );

            break;
        }

        cpl_test_eq( cpl_imagelist_get_size(fit), ntest );

        if (ntest == 2) {
            cpl_test_zero(cpl_image_subtract_scalar(cpl_imagelist_get(fit, 1),
                                                   gain));
        } else if (ntest > 2) {
            cpl_test_zero(cpl_image_subtract_scalar(cpl_imagelist_get(fit, 2),
                                                   gain));
        }

        cpl_fit_imagelist_is_zero(fit, ntest * mytol);

        cpl_fit_image_is_zero(ffiterror, ntest * mytol);

        cpl_imagelist_delete(fit);
    }

    cpl_imagelist_delete(input);
    cpl_vector_delete(vdit);

    /* Multiple samples at three different (equidistant) points */

    input = cpl_imagelist_new();

    vdit = cpl_vector_new(9);
    cpl_vector_set(vdit, 0, 5.0);
    cpl_vector_set(vdit, 1, 5.0);
    cpl_vector_set(vdit, 2, 5.0);
    cpl_vector_set(vdit, 3, 3.0);
    cpl_vector_set(vdit, 4, 3.0);
    cpl_vector_set(vdit, 5, 3.0);
    cpl_vector_set(vdit, 6, 7.0);
    cpl_vector_set(vdit, 7, 7.0);
    cpl_vector_set(vdit, 8, 7.0);

    cpl_imagelist_set(input, cpl_image_fill_test_create(IMAGESZFIT, IMAGESZFIT), 0);
    cpl_imagelist_set(input, cpl_image_fill_test_create(IMAGESZFIT, IMAGESZFIT), 1);
    cpl_imagelist_set(input, cpl_image_fill_test_create(IMAGESZFIT, IMAGESZFIT), 2);
    cpl_imagelist_set(input, cpl_image_fill_test_create(IMAGESZFIT, IMAGESZFIT), 3);
    cpl_imagelist_set(input, cpl_image_fill_test_create(IMAGESZFIT, IMAGESZFIT), 4);
    cpl_imagelist_set(input, cpl_image_fill_test_create(IMAGESZFIT, IMAGESZFIT), 5);
    cpl_imagelist_set(input, cpl_image_fill_test_create(IMAGESZFIT, IMAGESZFIT), 6);
    cpl_imagelist_set(input, cpl_image_fill_test_create(IMAGESZFIT, IMAGESZFIT), 7);
    cpl_imagelist_set(input, cpl_image_fill_test_create(IMAGESZFIT, IMAGESZFIT), 8);

    /* Fit a non-over-determined set without constant and linear terms */
    flops = cpl_tools_get_flops();
    cputime = cpl_test_get_cputime();

    fit = cpl_fit_imagelist_polynomial(vdit, input, 2, 4, CPL_FALSE,
                                       CPL_TYPE_DOUBLE, dfiterror);

    secs += cpl_test_get_cputime() - cputime;
    flopsum += cpl_tools_get_flops() - flops;
    bytes += cpl_test_get_bytes_imagelist(input);

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_eq( cpl_imagelist_get_size(fit), 3 );

    cpl_fit_image_is_zero(dfiterror, DBL_MAX);

    cpl_imagelist_delete(fit);

    /* Fit a non-over-determined set with a constant term */
    flops = cpl_tools_get_flops();
    cputime = cpl_test_get_cputime();

    fit = cpl_fit_imagelist_polynomial(vdit, input, 0, 2, CPL_FALSE,
                                       CPL_TYPE_DOUBLE, dfiterror);

    secs += cpl_test_get_cputime() - cputime;
    flopsum += cpl_tools_get_flops() - flops;
    bytes += cpl_test_get_bytes_imagelist(input);

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_eq( cpl_imagelist_get_size(fit), 3 );

    cpl_fit_image_is_zero(dfiterror, DBL_MAX);

    demax = cpl_image_get_sqflux(dfiterror);

    cpl_imagelist_delete(fit);


    /* Repeat the previous test - this time exploiting that
       the sampling points are equidistant */
    flops = cpl_tools_get_flops();
    cputime = cpl_test_get_cputime();

    fit = cpl_fit_imagelist_polynomial(vdit, input, 0, 2, CPL_TRUE,
                                       CPL_TYPE_DOUBLE, dfiterror);

    secs += cpl_test_get_cputime() - cputime;
    flopsum += cpl_tools_get_flops() - flops;
    bytes += cpl_test_get_bytes_imagelist(input);

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_eq( cpl_imagelist_get_size(fit), 3 );

    cpl_fit_image_is_zero(dfiterror, DBL_MAX);

    cpl_test( cpl_image_get_sqflux(dfiterror) <= demax );

    cpl_imagelist_delete(fit);

    /* Repeat the previous test - after sorting the sampling points */
    cpl_vector_sort(vdit, 1);
    flops = cpl_tools_get_flops();
    cputime = cpl_test_get_cputime();

    fit = cpl_fit_imagelist_polynomial(vdit, input, 0, 2, CPL_TRUE,
                                       CPL_TYPE_DOUBLE, dfiterror);

    secs += cpl_test_get_cputime() - cputime;
    flopsum += cpl_tools_get_flops() - flops;
    bytes += cpl_test_get_bytes_imagelist(input);

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_eq( cpl_imagelist_get_size(fit), 3 );

    cpl_fit_image_is_zero(dfiterror, DBL_MAX);

    cpl_imagelist_delete(fit);

    /* Fit an under-determined set without a constant term */
    fit = cpl_fit_imagelist_polynomial(vdit, input, 1, 4, CPL_FALSE,
                                       CPL_TYPE_DOUBLE, dfiterror);

    cpl_test_error(CPL_ERROR_SINGULAR_MATRIX );
    cpl_test_null( fit );

    cpl_imagelist_delete(input);
    cpl_vector_delete(vdit);


    cpl_msg_info("","Speed while fitting with image size %d in "
                 "%g secs [Mflop/s]: %g", IMAGESZFIT, secs, flopsum/secs/1e6);
    if (secs > 0.0) {
        cpl_msg_info(cpl_func,"Processing rate [MB/s]: %g",
                     1e-6 * (double)bytes / secs);
    }

    /* Done testing */
    cpl_image_delete(dfiterror);
    cpl_image_delete(ffiterror);
    cpl_image_delete(ifiterror);
    cpl_polynomial_delete(vfiller);

}



/*----------------------------------------------------------------------------*/
/**
  @brief  Verify that all elements in an imagelist are zero (within a tolerance)
  @param  self    The list of images to check
  @param  tol     The non-negative tolerance
  param   line    The line number of the caller
  @return void

 */
/*----------------------------------------------------------------------------*/
static void cpl_fit_imagelist_is_zero_macro(const cpl_imagelist * self,
                                               double tol, int line)
{

    const int n = cpl_imagelist_get_size(self);
    int i;

    for (i = 0; i < n; i++) {

        /* FIXME: Need traceback of failure */

        cpl_fit_image_is_zero_macro(cpl_imagelist_get_const(self, i), tol,
                                    line);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief  Verify that all elements in an image are zero (within a tolerance)
  @param  self    The image to check
  @param  tol     The non-negative tolerance
  param   line    The line number of the caller
  @return void

 */
/*----------------------------------------------------------------------------*/
static void cpl_fit_image_is_zero_macro(const cpl_image * self, double tol,
                                           int line)
{

    cpl_stats * stats = cpl_stats_new_from_image(self,
                                                 CPL_STATS_MIN | CPL_STATS_MAX
                                                 | CPL_STATS_MEAN);

    const double mymin = cpl_stats_get_min(stats);
    const double mymax = cpl_stats_get_max(stats);

    /* FIXME: Need traceback of failure */
    cpl_test(line > 0);

    cpl_test_leq( fabs(mymin), tol );
    cpl_test_leq( fabs(mymax), tol );

    cpl_stats_delete(stats);

}

static void cpl_fit_image_gaussian_tests(void)
{
    cpl_image  *box        = NULL;
    cpl_image  *image      = NULL;
    cpl_image  *eimage     = NULL;
    cpl_matrix *covariance = NULL;
    cpl_matrix *phys_cov   = NULL;
    cpl_array  *parameters = NULL;
    cpl_array  *err_params = NULL;
    cpl_array  *fit_params = NULL;
    double      major, minor, angle, rms, chisq, value;
    const int   nx         = 30;
    const int   ny         = 30;
    const int   lnx        = 2438;
    const int   lny        = 2438;
    int         i, j;

    double x[2];
    const char  *p[7] = { /* Parameter names */
                            "Background       ",
                            "Normalisation    ",
                            "Correlation      ",
                            "Center position x",
                            "Center position y",
                            "Sigma x          ",
                            "Sigma y          "};

    double u[7] = { /* Parameter values of simulated gaussian */
                            2.130659, 
                       104274.700696, 
                            0.779320, 
                          796.851741, 
                         1976.324361, 
                            4.506552, 
                            3.248286 };

    const double e[7] = { /* Errors on parameters of simulated gaussian */
                            2.029266,
                          955.703656, 
                            0.004452, 
                            0.035972, 
                            0.025949, 
                            0.037186, 
                            0.026913 };

    const char  *fp[3] = { /* Physical parameter names */
                            "Angle            ",
                            "Major semi-axis  ",
                            "Minor semi-axis  "};
                            
    const double fu[3] = { /* Physical parameter values of simulated gaussian */
                            33.422720,
                             5.276152,
                             1.738563};

    const double fe[3] = { /* Errors on physical parameters of gaussian */
                             0.238359,
                             0.283202,
                             0.052671};


    /*
     * Fill image with model gaussian.
     */

    box = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);

    u[3] -= 782;
    u[4] -= 1961;
    for (j = 1; j <= ny; j++) {
        x[1] = j;
        for (i = 1; i <= nx; i++) {
             x[0] = i;
             eval_gauss(x, u, &value);
             cpl_image_set(box, i, j, value);
        }
    }
    u[3] += 782;
    u[4] += 1961;


    /*
     * Insert it into a bigger image.
     */

    image = cpl_image_new(lnx, lny, CPL_TYPE_FLOAT);
    cpl_image_copy(image, box, 783, 1962);
    cpl_image_delete(box); box = NULL;


    /*
     * Add some noise, and create corresponding error image
     * (not really physical, but this is just for testing).
     */

    eimage = cpl_image_new(lnx, lny, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(eimage, -50, 50);
    cpl_image_add(image, eimage);
    cpl_image_delete(eimage);
    eimage = cpl_image_new(lnx, lny, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(eimage, 50, 60);


    /*
     * Allocate the parameters array, with the error array, and leave
     * them empty (for the moment).
     */

    parameters = cpl_array_new(7, CPL_TYPE_DOUBLE);

    err_params = cpl_array_new(7, CPL_TYPE_DOUBLE);


    /*
     * Now see if we can find a gaussian without any first guess:
     */

    cpl_test_zero(cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 30,
                                         parameters, err_params, fit_params, 
                                         &rms, &chisq, &covariance, &major, 
                                         &minor, &angle, &phys_cov));
    cpl_test(check_gauss_success(p, u, e, fp, fu, fe, parameters, err_params, 
                                 angle, major, minor, phys_cov));

    cpl_matrix_delete(covariance);
    cpl_matrix_delete(phys_cov);

    /*
     * Now retry with a first-guess, all parameters free.
     */

    fit_params = cpl_array_new(7, CPL_TYPE_INT);

    for (i = 0; i < 7; i++)
        cpl_array_set(fit_params, i, 1);

    for (i = 0; i < 7; i++)
        cpl_array_set(parameters, i, u[i]);

    for (i = 0; i < 7; i++)
        cpl_array_set(err_params, i, e[i]);

    cpl_test_zero(cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 30,
                                         parameters, err_params, fit_params, 
                                         &rms, &chisq, &covariance, &major, 
                                         &minor, &angle, &phys_cov));

    cpl_test(check_gauss_success(p, u, e, fp, fu, fe, parameters, err_params, 
                                 angle, major, minor, phys_cov));

    cpl_matrix_delete(covariance);
    cpl_matrix_delete(phys_cov);

    /*
     * Now retry with a first-guess, two parameters are frozen.
     */

    for (i = 0; i < 7; i++)
        cpl_array_set(parameters, i, u[i]);

    for (i = 0; i < 7; i++)
        cpl_array_set(err_params, i, e[i]);

    cpl_array_set(fit_params, 0, 0);
    cpl_array_set(fit_params, 4, 0);


    cpl_test_zero(cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 30,
                                         parameters, err_params, fit_params,
                                         &rms, &chisq, &covariance, &major,
                                         &minor, &angle, &phys_cov));

    cpl_test(check_gauss_success(p, u, e, fp, fu, fe, parameters, err_params,
                                 angle, major, minor, phys_cov));

    cpl_matrix_delete(covariance);
    cpl_matrix_delete(phys_cov);


    /*
     * Now try without eny error propagation
     */

    cpl_test_zero(cpl_fit_image_gaussian(image, NULL, 797, 1976, 30, 30,
                                         parameters, NULL, fit_params,
                                         &rms, NULL, NULL, &major,
                                         &minor, &angle, NULL));

    cpl_test(check_gauss_success(p, u, e, fp, fu, fe, parameters, err_params,
                                 angle, major, minor, NULL));

    /*
     * Try different error situations:
     */

    cpl_fit_image_gaussian(NULL, eimage, 797, 1976, 30, 30,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 30,
                           NULL, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_fit_image_gaussian(image, eimage, -1, 1976, 30, 30,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);

    cpl_fit_image_gaussian(image, eimage, 797, 197600, 30, 30,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 300000, 30,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 2, 30,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);
    
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 300000,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 2,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    box = cpl_image_new(46, 72, CPL_TYPE_FLOAT);
    cpl_fit_image_gaussian(image, box, 797, 1976, 30, 30,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);
    cpl_image_delete(box);
    cpl_test_error(CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 30,
                           parameters, fit_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_INVALID_TYPE);

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 30,
                           fit_params, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_INVALID_TYPE);

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 30,
                           parameters, err_params, err_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_INVALID_TYPE);

    cpl_fit_image_gaussian(image, NULL, 797, 1976, 30, 30,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);

    for (i = 0; i < 7; i++)
        cpl_array_set(fit_params, i, 0); /* All frozen */

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 30,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    cpl_image_delete(image);
    image = cpl_image_new(lnx, lny, CPL_TYPE_FLOAT_COMPLEX);

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 30,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_INVALID_TYPE);

    cpl_image_delete(image);
    image = cpl_image_new(lnx, lny, CPL_TYPE_DOUBLE);

    cpl_fit_image_gaussian(image, eimage, 797, 1976, 30, 30,
                           parameters, err_params, fit_params,
                           &rms, &chisq, &covariance, &major,
                           &minor, &angle, &phys_cov);

    cpl_test_error(CPL_ERROR_TYPE_MISMATCH);

    cpl_image_delete(image);
    cpl_image_delete(eimage);
    cpl_array_delete(parameters);
    cpl_array_delete(err_params);
    cpl_array_delete(fit_params);

}


static void eval_gauss(const double x[], const double a[], double *result)
{
    if (a[5] == 0.0) {
        if (x[0] == a[3]) {
            *result = DBL_MAX;
        }
        else {
            *result = 0.0;
        }
    }
    else if (a[6] == 0.0) {
        if (x[1] == a[4]) {
            *result = DBL_MAX;
        }
        else {
            *result = 0.0;
        }
    }
    else {
        double b1 = -0.5 / (1 - a[2] * a[2]);
        double b2 = (x[0] - a[3]) / a[5];
        double b3 = (x[1] - a[4]) / a[6];

        *result = a[0]
                + a[1] / (CPL_MATH_2PI * a[5] * a[6] * sqrt(1 - a[2] * a[2]))
                * exp(b1 * (b2 * b2 - 2 * a[2] * b2 * b3 + b3 * b3));
    }

    return;
}


static int check_gauss_success(const char  **p, 
                               const double *u, 
                               const double *e,
                               const char  **fp, 
                               const double *fu, 
                               const double *fe,
                               cpl_array    *parameters,
                               cpl_array    *err_params,
                               double        angle,
                               double        major,
                               double        minor,
                               cpl_matrix   *phys_cov)
{
    cpl_matrix *matrix  = NULL;
    int         success = 0;
    int         i;


    if (phys_cov == NULL) {
        matrix = cpl_matrix_new(3, 3);
        cpl_matrix_fill(matrix, 0.0);
        phys_cov = matrix;
    } else {
        /* The covariance matrix must be 3 X 3, symmetric, positive definite */
        cpl_error_code error;

        cpl_test_eq(cpl_matrix_get_nrow(phys_cov), 3);
        cpl_test_eq(cpl_matrix_get_ncol(phys_cov), 3);

        matrix = cpl_matrix_transpose_create(phys_cov);
        cpl_test_matrix_abs(phys_cov, matrix, DBL_EPSILON);

        error = cpl_matrix_decomp_chol(matrix);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

    }

    for (i = 0; i < 7; i++) {

        /*
         * The info message is only printed if 
         * cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
         * or if running with
         * export CPL_MSG_LEVEL=info
         */

        cpl_msg_info(cpl_func, "%s = %f +- %f", p[i],
                                cpl_array_get_double(parameters, i, NULL),
                                cpl_array_get_double(err_params, i, NULL));

        /*
         * The result must not differ from expectations more than 3 sigmas.
         * The reason of this loose test is that cpl_image_fill_noise_uniform()
         * gives different results depending on how many times it is called,
         * and other tests may be added in future.
         * It may therefore happen that a test "fails" because of an offset 
         * passing the 3-sigma limit (1% probability). A solution to this 
         * problem would be to store simulated images in the CPL repository.
         */

        success = fabs(cpl_array_get_double(parameters, i, NULL) - u[i])
                < 3 * e[i]; 

        if (!success) {
            cpl_msg_error(cpl_func, "Expected value for %s = %f +- %f", p[i],
                          u[i], e[i]);
            cpl_msg_error(cpl_func, "Obtained value for %s = %f +- %f", p[i],
                          cpl_array_get_double(parameters, i, NULL),
                          cpl_array_get_double(err_params, i, NULL));
            return success;
        }
    }
    
    cpl_msg_info(cpl_func, "%s = %f +- %f (degrees)", fp[0],
                 angle * CPL_MATH_DEG_RAD, 
                 sqrt(cpl_matrix_get(phys_cov, 0, 0)) * CPL_MATH_DEG_RAD);

    success = fabs(angle * CPL_MATH_DEG_RAD - fu[0])
            < 3 * fe[0] * CPL_MATH_DEG_RAD ;

    if (!success) {
        cpl_msg_error(cpl_func, "Expected value for %s = %f +- %f", fp[0],
                      fu[0], fe[0]);
        cpl_msg_error(cpl_func, "Obtained value for %s = %f +- %f", fp[0],
                      angle * CPL_MATH_DEG_RAD,
                      sqrt(cpl_matrix_get(phys_cov, 0, 0)) * CPL_MATH_DEG_RAD);
        return success;
    }

    cpl_msg_info(cpl_func, "%s = %f +- %f", fp[1],
                 major, sqrt(cpl_matrix_get(phys_cov, 1, 1)));

    success = fabs(major - fu[1]) < 3 * fe[1];

    if (!success) {
        cpl_msg_error(cpl_func, "Expected value for %s = %f +- %f", fp[1],
                      fu[1], fe[1]);
        cpl_msg_error(cpl_func, "Obtained value for %s = %f +- %f", fp[1],
                      major,  sqrt(cpl_matrix_get(phys_cov, 1, 1)));
        return success;
    }

    cpl_msg_info(cpl_func, "%s = %f +- %f\n", fp[2],
                 minor, sqrt(cpl_matrix_get(phys_cov, 2, 2)));

    success = fabs(minor - fu[2]) < 3 * fe[2];

    if (!success) {
        cpl_msg_error(cpl_func, "Expected value for %s = %f +- %f", fp[2],
                      fu[2], fe[2]);
        cpl_msg_error(cpl_func, "Obtained value for %s = %f +- %f", fp[2],
                      major,  sqrt(cpl_matrix_get(phys_cov, 2, 2)));
    }

    cpl_matrix_delete(matrix);

    return success;
}

/*----------------------------------------------------------------------------*/
/**
  @brief   Test cpl_fit_image_gaussian() with no returned covariance matrix
  @return  void
  @see DFS10000

 */
/*----------------------------------------------------------------------------*/
static void cpl_fit_image_gaussian_test_one(void)
{
    double         rms = 0.0;
    double         redchi = 0.0;
    cpl_image    * image;
    cpl_image    * imerr;
    cpl_array    * params;
    cpl_array    * paramserr;
    cpl_error_code error;

    params = cpl_array_new(7, CPL_TYPE_DOUBLE);
    paramserr = cpl_array_new(7, CPL_TYPE_DOUBLE);

    image = cpl_image_new(IMAGESZFIT, IMAGESZFIT, CPL_TYPE_DOUBLE);
    error = cpl_image_fill_gaussian(image, IMAGESZFIT/2, IMAGESZFIT/2,
                                    20.0, 20.0, 20.0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    imerr = cpl_image_new(IMAGESZFIT, IMAGESZFIT, CPL_TYPE_DOUBLE);

    error = cpl_image_fill_noise_uniform(imerr,1.0,1.00001);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    error = cpl_fit_image_gaussian(image, imerr, IMAGESZFIT/2, IMAGESZFIT/2,
                                   IMAGESZFIT/5, IMAGESZFIT/5,
                                   params, paramserr, NULL, &rms, NULL,
                                   NULL, NULL, NULL, NULL, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    error = cpl_fit_image_gaussian(image, imerr, IMAGESZFIT/2, IMAGESZFIT/2,
                                   IMAGESZFIT/5, IMAGESZFIT/5,
                                   params, paramserr, NULL, &rms, &redchi,
                                   NULL, NULL, NULL, NULL, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    cpl_image_delete(image);
    cpl_image_delete(imerr);
    cpl_array_delete(params);
    cpl_array_delete(paramserr);

}

