/* $Id: cpl_column.h,v 1.48 2010/12/23 13:22:32 cizzo Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cizzo $
 * $Date: 2010/12/23 13:22:32 $
 * $Revision: 1.48 $
 * $Name: cpl-5_3_0-BRANCH $
 */

#ifndef CPL_COLUMN_H
#define CPL_COLUMN_H

#include <complex.h>
#include <cpl_macros.h>
#include <cpl_type.h>
#include <cpl_error.h>
#include <cpl_array.h>

CPL_BEGIN_DECLS


/*
 *  This is the type used for any flag used in cpl_column.
 */

typedef int cpl_column_flag;

typedef struct _cpl_column_ cpl_column;

void cpl_column_dump_structure(cpl_column *column);
void cpl_column_dump(cpl_column *column, int start, int count);

/*
 * Constructors and destructors:
 */

cpl_column *cpl_column_new_int(int length);
cpl_column *cpl_column_new_float(int length);
cpl_column *cpl_column_new_double(int length);
cpl_column *cpl_column_new_float_complex(int length);
cpl_column *cpl_column_new_double_complex(int length);
cpl_column *cpl_column_new_string(int length);
cpl_column *cpl_column_new_array(cpl_type, int, int);

cpl_error_code cpl_column_set_save_type(cpl_column *, cpl_type);
cpl_type cpl_column_get_save_type(const cpl_column *);

cpl_column *cpl_column_wrap_int(int *, int length);
cpl_column *cpl_column_wrap_float(float *, int length);
cpl_column *cpl_column_wrap_double(double *, int length);
cpl_column *cpl_column_wrap_float_complex(float complex *, int length);
cpl_column *cpl_column_wrap_double_complex(double complex *, int length);
cpl_column *cpl_column_wrap_string(char **, int length);

void *cpl_column_unwrap(cpl_column *);
void cpl_column_delete_but_strings(cpl_column *);
void cpl_column_delete_but_arrays(cpl_column *);

cpl_error_code cpl_column_copy_data(cpl_column *, const double *);
cpl_error_code cpl_column_copy_data_int(cpl_column *, const int *);
cpl_error_code cpl_column_copy_data_float(cpl_column *, const float *);
cpl_error_code cpl_column_copy_data_double(cpl_column *, const double *);
cpl_error_code cpl_column_copy_data_complex(cpl_column *,
                                            const double complex *);
cpl_error_code cpl_column_copy_data_float_complex(cpl_column *,
                                                  const float complex *);
cpl_error_code cpl_column_copy_data_double_complex(cpl_column *,
                                                   const double complex *);
cpl_error_code cpl_column_copy_data_string(cpl_column *, const char **);

void cpl_column_delete(cpl_column *);


/*
 * Methods:
 */

cpl_error_code cpl_column_set_name(cpl_column *, const char *);
const char *cpl_column_get_name(const cpl_column *);

cpl_error_code cpl_column_set_unit(cpl_column *, const char *);
const char *cpl_column_get_unit(const cpl_column *);

cpl_error_code cpl_column_set_format(cpl_column *, const char *);
const char *cpl_column_get_format(const cpl_column *);

int cpl_column_get_size(const cpl_column *);
int cpl_column_get_depth(const cpl_column *);
int cpl_column_get_dimensions(const cpl_column *);
cpl_error_code cpl_column_set_dimensions(cpl_column *, const cpl_array *);
int cpl_column_get_dimension(const cpl_column *, int);

cpl_type cpl_column_get_type(const cpl_column *);

int *cpl_column_get_data_int(cpl_column *);
const int *cpl_column_get_data_int_const(const cpl_column *);
float *cpl_column_get_data_float(cpl_column *);
const float *cpl_column_get_data_float_const(const cpl_column *);
double *cpl_column_get_data_double(cpl_column *);
const double *cpl_column_get_data_double_const(const cpl_column *);
float complex *cpl_column_get_data_float_complex(cpl_column *);
const float complex *
cpl_column_get_data_float_complex_const(const cpl_column *);
double complex *cpl_column_get_data_double_complex(cpl_column *);
const double complex *
cpl_column_get_data_double_complex_const(const cpl_column *);
char **cpl_column_get_data_string(cpl_column *);
const char **cpl_column_get_data_string_const(const cpl_column *);
cpl_array **cpl_column_get_data_array(cpl_column *);
const cpl_array **cpl_column_get_data_array_const(const cpl_column *);
cpl_column_flag *cpl_column_get_data_invalid(cpl_column *);
const cpl_column_flag *cpl_column_get_data_invalid_const(const cpl_column *);
cpl_error_code cpl_column_set_data_invalid(cpl_column *, 
                                           cpl_column_flag *, int);

cpl_error_code cpl_column_set_size(cpl_column *, int);
cpl_error_code cpl_column_set_depth(cpl_column *, int);

double cpl_column_get(cpl_column *, int, int *null); 
double complex cpl_column_get_complex(cpl_column *, int , int *null);
int cpl_column_get_int(cpl_column *, int, int *null); 
float cpl_column_get_float(cpl_column *, int, int *null); 
double cpl_column_get_double(cpl_column *, int, int *null); 
float complex cpl_column_get_float_complex(cpl_column *, int, int *null);
double complex cpl_column_get_double_complex(cpl_column *, int, int *null);
char *cpl_column_get_string(cpl_column *, int);
const char *cpl_column_get_string_const(const cpl_column *, int);
cpl_array *cpl_column_get_array(cpl_column *, int);
const cpl_array *cpl_column_get_array_const(const cpl_column *, int);

cpl_error_code cpl_column_set(cpl_column *, int, double); 
cpl_error_code cpl_column_set_int(cpl_column *, int, int); 
cpl_error_code cpl_column_set_float(cpl_column *, int, float); 
cpl_error_code cpl_column_set_double(cpl_column *, int, double); 
cpl_error_code cpl_column_set_complex(cpl_column *, int, double complex);
cpl_error_code cpl_column_set_float_complex(cpl_column *, int, float complex);
cpl_error_code cpl_column_set_double_complex(cpl_column *, int, double complex);
cpl_error_code cpl_column_set_string(cpl_column *, int, const char *);
cpl_error_code cpl_column_set_array(cpl_column *, int, const cpl_array *);

cpl_error_code cpl_column_fill(cpl_column *, int, int, double); 
cpl_error_code cpl_column_fill_int(cpl_column *, int, int, int); 
cpl_error_code cpl_column_fill_float(cpl_column *, int, int, float); 
cpl_error_code cpl_column_fill_double(cpl_column *, int, int, double);
cpl_error_code cpl_column_fill_complex(cpl_column *, int, int, double complex);
cpl_error_code cpl_column_fill_float_complex(cpl_column *, 
                                             int, int, float complex);
cpl_error_code cpl_column_fill_double_complex(cpl_column *, 
                                              int, int, double complex);
cpl_error_code cpl_column_fill_string(cpl_column *, int, int, const char *);
cpl_error_code cpl_column_fill_array(cpl_column *, int, int, const cpl_array *);

cpl_error_code cpl_column_copy_segment(cpl_column *, int, int, double *);
cpl_error_code cpl_column_copy_segment_int(cpl_column *, int, int, int *);
cpl_error_code cpl_column_copy_segment_float(cpl_column *, int, int, float *);
cpl_error_code cpl_column_copy_segment_double(cpl_column *, int, int, double *);
cpl_error_code cpl_column_copy_segment_complex(cpl_column *, 
                                               int, int, double complex *);
cpl_error_code cpl_column_copy_segment_float_complex(cpl_column *, 
                                               int, int, float complex *);
cpl_error_code cpl_column_copy_segment_double_complex(cpl_column *, 
                                               int, int, double complex *);
cpl_error_code cpl_column_copy_segment_string(cpl_column *, int, int, char **);
cpl_error_code cpl_column_copy_segment_array(cpl_column *, int, int, 
                                             cpl_array **);

cpl_error_code cpl_column_shift(cpl_column *, int);

cpl_error_code cpl_column_set_invalid(cpl_column *, int);
cpl_error_code cpl_column_fill_invalid(cpl_column *, int, int);
int cpl_column_is_invalid(cpl_column *, int);
int cpl_column_has_invalid(cpl_column *);
int cpl_column_has_valid(cpl_column *);
int cpl_column_count_invalid(cpl_column *);

cpl_error_code cpl_column_fill_invalid_int(cpl_column *, int);
cpl_error_code cpl_column_fill_invalid_float(cpl_column *, float);
cpl_error_code cpl_column_fill_invalid_double(cpl_column *, double);
cpl_error_code cpl_column_fill_invalid_float_complex(cpl_column *, 
                                                     float complex);
cpl_error_code cpl_column_fill_invalid_double_complex(cpl_column *, 
                                                     double complex);

cpl_error_code cpl_column_erase_segment(cpl_column *, int, int);
cpl_error_code cpl_column_erase_pattern(cpl_column *, int *);
cpl_error_code cpl_column_insert_segment(cpl_column *, int, int);

cpl_column *cpl_column_duplicate(cpl_column *);

cpl_column *cpl_column_extract(cpl_column *, int, int);

cpl_column *cpl_column_cast_to_int(cpl_column *);
cpl_column *cpl_column_cast_to_float(cpl_column *);
cpl_column *cpl_column_cast_to_double(cpl_column *);
cpl_column *cpl_column_cast_to_float_complex(cpl_column *);
cpl_column *cpl_column_cast_to_double_complex(cpl_column *);
cpl_column *cpl_column_cast_to_int_array(cpl_column *);
cpl_column *cpl_column_cast_to_float_array(cpl_column *);
cpl_column *cpl_column_cast_to_double_array(cpl_column *);
cpl_column *cpl_column_cast_to_float_complex_array(cpl_column *);
cpl_column *cpl_column_cast_to_double_complex_array(cpl_column *);
cpl_column *cpl_column_cast_to_int_flat(cpl_column *);
cpl_column *cpl_column_cast_to_float_flat(cpl_column *);
cpl_column *cpl_column_cast_to_double_flat(cpl_column *);
cpl_column *cpl_column_cast_to_float_complex_flat(cpl_column *);
cpl_column *cpl_column_cast_to_double_complex_flat(cpl_column *);

cpl_error_code cpl_column_merge(cpl_column *, cpl_column *, int);

cpl_error_code cpl_column_add(cpl_column *, cpl_column *);
cpl_error_code cpl_column_subtract(cpl_column *, cpl_column *);
cpl_error_code cpl_column_multiply(cpl_column *, cpl_column *);
cpl_error_code cpl_column_divide(cpl_column *, cpl_column *);

cpl_error_code cpl_column_add_scalar(cpl_column *, double);
cpl_error_code cpl_column_add_scalar_complex(cpl_column *, double complex);
cpl_error_code cpl_column_subtract_scalar(cpl_column *, double);
cpl_error_code cpl_column_subtract_scalar_complex(cpl_column *, double complex);
cpl_error_code cpl_column_multiply_scalar(cpl_column *, double);
cpl_error_code cpl_column_multiply_scalar_complex(cpl_column *, double complex);
cpl_error_code cpl_column_divide_scalar(cpl_column *, double);
cpl_error_code cpl_column_divide_scalar_complex(cpl_column *, double complex);
cpl_error_code cpl_column_absolute(cpl_column *);
cpl_column *cpl_column_absolute_complex(cpl_column *);
cpl_column *cpl_column_phase_complex(cpl_column *);
cpl_column *cpl_column_extract_real(cpl_column *);
cpl_column *cpl_column_extract_imag(cpl_column *);
cpl_error_code cpl_column_logarithm(cpl_column *, double);
cpl_error_code cpl_column_power(cpl_column *, double);
cpl_error_code cpl_column_exponential(cpl_column *, double);
cpl_error_code cpl_column_conjugate(cpl_column *);

double cpl_column_get_mean(cpl_column *);
double complex cpl_column_get_mean_complex(cpl_column *);
double cpl_column_get_median(cpl_column *);
double cpl_column_get_stdev(cpl_column *);

double cpl_column_get_max(cpl_column *);
double cpl_column_get_min(cpl_column *);
cpl_error_code cpl_column_get_maxpos(cpl_column *, int *);
cpl_error_code cpl_column_get_minpos(cpl_column *, int *);

CPL_END_DECLS

#endif
/* end of cpl_column.h */
