/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set sw=2 sts=2 et cin: */
/*
 * This file is part of the MUSE Instrument Pipeline
 * Copyright (C) 2007-2012 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <muse.h>

/* structure for the expected arc line detections */
typedef struct {
  int no;
  float pos;
  float flux;
} expected;

void
test_line_locations(cpl_table *aLines, expected aPositions[], int aNLines)
{
  cpl_test_nonnull(aLines);
  if (aLines) {
    int nlines = cpl_table_get_nrow(aLines);
    cpl_test(nlines == aNLines);
    if (nlines != aNLines) {
      cpl_msg_error(__func__, "nlines=%d != aNLines=%d", nlines, aNLines);
      cpl_table_dump(aLines, 0, nlines, stdout);
      fflush(stdout);
    }

    /* test for expected positions of all lines with a tolerance of +/-0.5 pix */
    int i;
    for (i = 0; aPositions[i].pos > 0. && aPositions[i].no < nlines; i++) {
      double detpos = cpl_table_get(aLines, "center", aPositions[i].no, NULL);
      int doesmatch = (detpos > aPositions[i].pos - 0.5)
                   && (detpos < aPositions[i].pos + 0.5);
      cpl_test(doesmatch);
      if (!doesmatch) {
        cpl_msg_error(__func__, "position[%d][] = %d %f %f does not match %f",
                      i, aPositions[i].no, aPositions[i].pos, aPositions[i].flux,
                      detpos);
        cpl_table_dump(aLines, 0, nlines, stdout);
        fflush(stdout);
        break;
      }
#if 0
      else {
        cpl_msg_debug(__func__, "position[%d][] = %d %f %f does match %f",
                      i, aPositions[i].no, aPositions[i].pos, aPositions[i].flux,
                      detpos);
      }
#endif
    }
  }
  cpl_table_delete(aLines);
} /* test_line_locations() */

/*----------------------------------------------------------------------------*/
/**
  @brief    test program to check that muse_wave_lines_search() does what it
            should when called on a column taken from a real and an
            INM-simulated arc exposure
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char **argv)
{
  UNUSED_ARGUMENTS(argc, argv);
  cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

  cpl_error_reset();

  cpl_msg_info(__func__, "===== Data from real exposure (not flat-fielded) =====");
  expected positions1[] = { /* list of {no, pos, flux} for each expected arc line */
    /* {  108.8068  87.8679 }. not detected, too faint */
    { 0,  120.2017, 3518.22 },
    { 1,  214.8117, 11558.4 },
    { 2,  305.4193, 208.807 },
    { 3,  438.1213, 24984.0 },
    { 4,  492.0264, 129.439 },
    { 5,  732.8964, 99274.6 },
    /* {  902.3192  39.8464 }, not detected, too faint */
    { 6,  976.9777, 18461.0 },
    { 7,  993.6136, 18575.8 },
    /* {  1345.762, 50.1498 }, not detected, too faint */
    /* {  1417.484, 37.8722 }, not detected, too faint */
    { 8,  1508.089, 10287.0 },
    /* {  1729.771, 44.8923 }, not detected, too faint */
    { 9,  1882.141, 481.420 },
    { 10, 2021.659, 124.865 },
    /* {  2029.547, 51.6413 }, not detected, too faint */
    { 11, 2232.977, 104.027 },
    /* {  2261.895, 69.4551 }, not detected, too faint */
    /* {  2272.911  25.9585 }, not detected, too faint */
    /* {  2464.896, 39.1081 }, not detected, too faint */
    { 12,  2540.088, 86.9954 },
    { 13,  2851.337, 81.1261 },
    /* {  2919.494  80.5648 }, not detected, too faint */
    /* {  3087.383, 46.9465 }, not detected, too faint */
    { 0,  -1.,     -1.      } /* end "tag" with negative position */
  };
  muse_image *arc = muse_image_load(BASEFILENAME"_real.fits");
  if (!arc || !arc->data) {
    cpl_msg_error(__func__, "Could not load test image: %s", cpl_error_get_message());
    muse_image_delete(arc);
    return cpl_test_end(1);
  }
  test_line_locations(muse_wave_lines_search(arc, 1.0, 24, 10), positions1, 14);
  muse_image_delete(arc);

  cpl_msg_info(__func__, "===== Data from real exposure (flat-fielded) =====");
  expected positions2[] = { /* list of {no, pos, flux} for each expected arc line */
    /* {  108.3552, 541.482 }, not detected, too faint */
    { 0,  120.1869, 42553.2 },
    { 1,  214.8108, 88305.5 },
    { 2,  305.4236, 1190.07 },
    { 3,  438.1479, 94265.4 },
    { 4,  492.0470, 417.331 },
    { 5,  732.9244, 212000. },
    { 6,  976.9833, 27728.1 },
    { 7,  993.6230, 27242.3 },
    /* {  1345.801, 58.0849 }, not detected, too faint */
    { 8,  1508.106, 9084.03 },
    /* {  1729.733, 47.2774 }, not detected, too faint */
    { 9,  1882.136, 330.631 },
    { 10, 2021.659, 74.5212 },
    /* {  2029.588, 29.0155 }, not detected, too faint */
    { 11, 2232.979, 57.3139 },
    /* {  2261.933, 35.5726 }, not detected, too faint */
    /* {  2272.913, 14.0534 }, not detected, too faint */
    /* {  2464.890, 29.5937 }, not detected, too faint */
    /* {  2540.115, 50.3002 }, not detected, too faint */
    /* {  2851.316, 48.4706 }, not detected, too faint */
    { 0,  -1.,      -1.     } /* end "tag" with negative position */
  };
  arc = muse_image_load(BASEFILENAME"_realflat.fits");
  if (!arc || !arc->data) {
    cpl_msg_error(__func__, "Could not load test image: %s", cpl_error_get_message());
    muse_image_delete(arc);
    return cpl_test_end(1);
  }
  test_line_locations(muse_wave_lines_search(arc, 1.0, 24, 10), positions2, 12);
  muse_image_delete(arc);

  cpl_msg_info(__func__, "===== Data from INM exposure (not flat-fielded) =====");
  expected positions3[] = { /* list of {no, pos, flux} for each expected arc line */
    /* {  109.639, 28.1381 }, not detected, too faint */
    { 0,  121.964, 4504.78 },
    { 1,  216.983, 31689.7 },
    { 2,  308.003, 312.151 },
    { 3,  440.911, 60906.6 },
    { 4,  494.925, 293.064 },
    { 5,  734.845, 111166. },
    { 6,  978.879, 23999.8 },
    { 7,  994.857, 24438.3 },
    { 8,  1345.56, 78.8044 },
    { 9,  1417.82, 144.573 },
    { 10, 1507.95, 29936.9 },
    { 11, 1559.85, 75.9608 },
    { 12, 1573.98, 501.176 },
    { 13, 1702.09, 177.416 },
    { 14, 1729.02, 89.1105 },
    { 15, 1800.95, 805.578 },
    { 16, 1880.99, 741.682 },
    { 17, 1901.63, 2730.02 },
    { 18, 2020.84, 205.277 },
    { 19, 2029.20, 102.573 },
    { 20, 2087.97, 342.069 },
    { 21, 2132.20, 5578.56 },
    { 22, 2194.98, 6051.73 },
    { 23, 2202.02, 2255.08 }, /* sometimes not detected, too close to brighter line?! */
    { 24, 2215.97, 1409.06 },
    { 25, 2232.02, 314.620 },
    { 26, 2263.85, 138.113 },
    { 27, 2830.89, 13079.6 },
    { 28, 2881.89, 868.914 },
    { 29, 3303.91, 63.3318 },
    { 30, 3316.95, 211.574 },
    { 31, 3334.95, 12801.4 },
    { 32, 3852.90, 1404.62 },
    { 33, 4051.76, 2010.92 },
    { 0,  -1.,     -1.     } /* end "tag" with negative position */
  };
  arc = muse_image_load(BASEFILENAME"_inm.fits");
  if (!arc || !arc->data) {
    cpl_msg_error(__func__, "Could not load test image: %s", cpl_error_get_message());
    muse_image_delete(arc);
    return cpl_test_end(3);
  }
  test_line_locations(muse_wave_lines_search(arc, 1.0, 24, 10), positions3, 34);
  muse_image_delete(arc);

  return cpl_test_end(0);
}
