/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set sw=2 sts=2 et cin: */
/*
 * This file is part of the MUSE Instrument Pipeline
 * Copyright (C) 2005-2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef MUSE_SKY_H
#define MUSE_SKY_H

/*----------------------------------------------------------------------------*
 *                              Includes                                      *
 *----------------------------------------------------------------------------*/
#include <cpl.h>

#include "muse_utils.h"
#include "muse_cplwrappers.h"
#include "muse_pixtable.h"
#include "muse_resampling.h"
#include "muse_lsf.h"

/*----------------------------------------------------------------------------*
 *                              Defines                                       *
 *----------------------------------------------------------------------------*/
extern const muse_cpltable_def muse_fluxspectrum_def[];
extern const muse_cpltable_def muse_relativespectrum_def[];
extern const muse_cpltable_def muse_extinction_def[];
extern const muse_cpltable_def muse_pixtable_def[];

/*----------------------------------------------------------------------------*
 *                          Special variable types                            *
 *----------------------------------------------------------------------------*/
/** @addtogroup muse_skysub */
/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Structure to hold the MASTER SKY result.

  This structure contains the resulting MASTER SKY parameters.
 */
/*----------------------------------------------------------------------------*/
typedef struct {
  /** @brief Table of @ref muse_sky_lines_lines_def
      "Atmospheric emission lines and their intensities".
      @copydetails muse_sky_lines_lines_def */
  cpl_table *lines;
  /** @brief @ref muse_fluxspectrum_def "Continuum flux" table */
  cpl_table *continuum;
  /** @brief LSF parameter for the resampled spectrum */
  muse_lsf_params **lsf; /*<< the LSF parameters */
} muse_sky_master;

/*----------------------------------------------------------------------------*/
/**
   @brief Structure to define the major settable sky parameters
*/
/*----------------------------------------------------------------------------*/
typedef struct {
  double fraction;
  double sampling;
  double csampling;
} muse_sky_params;

/*----------------------------------------------------------------------------*/
/**
   @brief Structure to define which slice parameters are fit.
*/
/*----------------------------------------------------------------------------*/
typedef struct {
  cpl_size n_param;
  cpl_size offset;
  cpl_size refraction;
  cpl_size sensitivity;
  cpl_size slit_width;
  cpl_size bin_width;
  cpl_size lsf_width;
  cpl_size hermit[MAX_HERMIT_ORDER];
} muse_sky_fit_params;

/**@}*/

/*----------------------------------------------------------------------------*
 *                           Function prototypes                              *
 *----------------------------------------------------------------------------*/
cpl_table *muse_sky_lines_load(muse_processing *);
cpl_table *muse_sky_continuum_load(muse_processing *);
cpl_error_code muse_sky_lines_set_range(cpl_table *, double, double);
muse_sky_master *muse_sky_master_new(void);
void muse_sky_master_delete(muse_sky_master *);

muse_mask *muse_sky_create_skymask(muse_image *, double, const char *);
cpl_table *muse_sky_spectrum_from_cube(muse_datacube *, const cpl_mask *);
void muse_sky_mark_cosmic(cpl_table *aSpectrum, muse_pixtable *aPixtable);
cpl_error_code muse_sky_save_continuum(muse_processing *, const cpl_table *, cpl_propertylist *);

cpl_error_code muse_sky_lines_save(muse_processing *, const cpl_table *, cpl_propertylist *);
muse_sky_fit_params *muse_sky_fit_params_new(cpl_boolean, cpl_boolean, cpl_size, cpl_size, cpl_size, cpl_size, cpl_size, cpl_size, cpl_size, cpl_size);

void muse_sky_fit_params_delete(muse_sky_fit_params *);

muse_sky_master *muse_sky_master_fit(const cpl_array *, const cpl_array *, const cpl_array *, const cpl_table *);

muse_sky_master *muse_sky_master_load(muse_processing *);

muse_lsf_params *muse_sky_lsf_params_fit(const cpl_array *, const cpl_array *, const cpl_array *, const cpl_table *, const cpl_array *, const cpl_array *, muse_lsf_params *, int, const muse_sky_fit_params *);
cpl_error_code muse_sky_subtract_slice(muse_pixtable *, muse_sky_master *, muse_lsf_params *);
cpl_error_code muse_sky_subtract_pixtable(muse_pixtable *a, muse_sky_master *, muse_lsf_params **);

cpl_error_code muse_sky_lines_apply_strength(cpl_table *, const cpl_array *);
cpl_error_code muse_sky_lines_cut(cpl_table *, double);
cpl_table *muse_sky_lines_create(const cpl_table *, const cpl_table *, double);
cpl_array *muse_sky_apply_lsf(const cpl_array *, const cpl_table *, const muse_lsf_params *);

muse_lsf_params *muse_lsf_params_fit(const cpl_array *, cpl_array *, const cpl_array *, const cpl_table *, muse_lsf_params *, int, const muse_sky_fit_params *);

/* functios about QC parameters (muse_sky_qc.c) */
void muse_sky_qc_lines(cpl_propertylist *, cpl_table *, const char *);
void muse_sky_qc_continuum(cpl_propertylist *, cpl_table *, const char *);

/* functions about row-by-row sky subtraction (muse_sky_rowbyrow.c) */
cpl_error_code muse_sky_subtract_rowbyrow_mask(muse_image *, cpl_table *);
cpl_error_code muse_sky_subtract_rowbyrow(muse_image *, cpl_table *, float, unsigned int);

#endif /* MUSE_SKY_H */
