!--------------------------------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations                              !
!   Copyright 2000-2023 CP2K developers group <https://cp2k.org>                                   !
!                                                                                                  !
!   SPDX-License-Identifier: GPL-2.0-or-later                                                      !
!--------------------------------------------------------------------------------------------------!

! **************************************************************************************************
!> \brief Filters for interpolating scaling functions .
!> \author Florian Schiffmann (09.2007,fschiff)
! **************************************************************************************************
MODULE lazy

   USE kinds,                           ONLY: dp
#include "../base/base_uses.f90"

   IMPLICIT NONE

   PRIVATE

   CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'lazy'

! *** Public data types ***

   PUBLIC :: lazy_arrays

CONTAINS

! **************************************************************************************************
!> \brief ...
!> \param itype ...
!> \param m ...
!> \param ch ...
!> \param cg ...
!> \param cgt ...
!> \param cht ...
!> \par History
!>      09.2007 created [Florian Schiffmann]
!> \author fschiff
! **************************************************************************************************
   SUBROUTINE lazy_arrays(itype, m, ch, cg, cgt, cht)
      INTEGER, INTENT(in)                                :: itype, m
      REAL(KIND=dp), DIMENSION(:), POINTER               :: ch, cg, cgt, cht

      INTEGER                                            :: i

      ALLOCATE (ch(-m:m))
      ALLOCATE (cg(-m:m))
      ALLOCATE (cgt(-m:m))
      ALLOCATE (cht(-m:m))

      SELECT CASE (itype)
      CASE (8)

         DO i = -m, m
            ch(i) = 0._dp
            cht(i) = 0._dp
            cg(i) = 0._dp
            cgt(i) = 0._dp
         END DO

         ! The normalization is chosen such that a constant function remains the same constant
         ! on each level of the transform

         ch(-7) = -5._dp/2048._dp
         ch(-6) = 0._dp
         ch(-5) = 49._dp/2048._dp
         ch(-4) = 0._dp
         ch(-3) = -245._dp/2048._dp
         ch(-2) = 0._dp
         ch(-1) = 1225._dp/2048._dp
         ch(0) = 1._dp
         ch(1) = 1225._dp/2048._dp
         ch(2) = 0._dp
         ch(3) = -245._dp/2048._dp
         ch(4) = 0._dp
         ch(5) = 49._dp/2048._dp
         ch(6) = 0._dp
         ch(7) = -5._dp/2048._dp
         !
         cht(0) = 1._dp

         ! g coefficients from h coefficients
         DO i = -m, m - 1
            cg(i + 1) = cht(-i)*(-1)**(i + 1)
            cgt(i + 1) = ch(-i)*(-1)**(i + 1)
         END DO

      CASE (14)
         ch = (/ &
              0._dp, 0._dp, 0._dp, 0.0000275373458862304687_dp, 0._dp, -0.000423073768615722656_dp, 0._dp, &
              0.00310254096984863281_dp, 0._dp, -0.0146262645721435547_dp, 0._dp, &
              0.0511919260025024414_dp, 0._dp, -0.153575778007507324_dp, 0._dp, 0.614303112030029297_dp, &
              1._dp, 0.614303112030029297_dp, 0._dp, -0.153575778007507324_dp, 0._dp, &
              0.0511919260025024414_dp, 0._dp, -0.0146262645721435547_dp, 0._dp, &
              0.00310254096984863281_dp, 0._dp, -0.000423073768615722656_dp, 0._dp, &
              0.0000275373458862304687_dp, 0._dp, 0._dp, 0._dp &
              /)

         DO i = -m, m
            cht(i) = 0._dp
            cg(i) = 0._dp
            cgt(i) = 0._dp
         END DO

         ! the normalization is chosen such that a constant function remains the same constant
         ! on each level of the transform

         cht(0) = 1._dp

         ! g coefficients from h coefficients
         DO i = -m, m - 1
            cg(i + 1) = cht(-i)*(-1)**(i + 1)
            cgt(i + 1) = ch(-i)*(-1)**(i + 1)
         END DO

      CASE (16)
         ch = (/ &
              0._dp, 0._dp, 0._dp, -6.39259815216064453e-6_dp, 0._dp, 0.000110641121864318848_dp, 0._dp, &
              -0.000915303826332092285_dp, 0._dp, 0.00484772026538848877_dp, 0._dp, &
              -0.0186983495950698853_dp, 0._dp, 0.0575909167528152466_dp, 0._dp, &
              -0.159974768757820129_dp, 0._dp, 0.617045536637306213_dp, 1._dp, 0.617045536637306213_dp, &
              0._dp, -0.159974768757820129_dp, 0._dp, 0.0575909167528152466_dp, 0._dp, &
              -0.0186983495950698853_dp, 0._dp, 0.00484772026538848877_dp, 0._dp, &
              -0.000915303826332092285_dp, 0._dp, 0.000110641121864318848_dp, 0._dp, &
              -6.39259815216064453e-6_dp, 0._dp, 0._dp, 0._dp &
              /)

         DO i = -m, m
            cht(i) = 0._dp
            cg(i) = 0._dp
            cgt(i) = 0._dp
         END DO

         ! the normalization is chosen such that a constant function remains the same constant
         ! on each level of the transform

         cht(0) = 1._dp

         ! g coefficients from h coefficients
         DO i = -m, m - 1
            cg(i + 1) = cht(-i)*(-1)**(i + 1)
            cgt(i + 1) = ch(-i)*(-1)**(i + 1)
         END DO

      CASE (20)
         ch = (/ &
              0._dp, 0._dp, 0._dp, &
              -3.53757059201598167e-7_dp, 0._dp, 7.51213519833981991e-6_dp, 0._dp, &
              -0.0000766237790230661631_dp, 0._dp, 0.000501001632073894143_dp, 0._dp, &
              -0.00236837135162204504_dp, 0._dp, 0.00868402828928083181_dp, 0._dp, &
              -0.0260520848678424954_dp, 0._dp, 0.0677354206563904881_dp, 0._dp, &
              -0.16933855164097622_dp, 0._dp, 0.620908022683579475_dp, 1._dp, 0.620908022683579475_dp, &
              0._dp, -0.16933855164097622_dp, 0._dp, 0.0677354206563904881_dp, 0._dp, &
              -0.0260520848678424954_dp, 0._dp, 0.00868402828928083181_dp, 0._dp, &
              -0.00236837135162204504_dp, 0._dp, 0.000501001632073894143_dp, 0._dp, &
              -0.0000766237790230661631_dp, 0._dp, 7.51213519833981991e-6_dp, 0._dp, &
              -3.53757059201598167e-7_dp, &
              0._dp, 0._dp, 0._dp/)

         !******** coefficients for wavelet transform *********************
         DO i = -m, m
            cht(i) = 0._dp
            cg(i) = 0._dp
            cgt(i) = 0._dp
         END DO

         ! the normalization is chosen such that a constant function remains the same constant
         ! on each level of the transform

         cht(0) = 1._dp

         ! g coefficients from h coefficients
         DO i = -m, m - 1
            cg(i + 1) = cht(-i)*(-1)**(i + 1)
            cgt(i + 1) = ch(-i)*(-1)**(i + 1)
         END DO

      CASE (24)
         ch = (/ &
              0._dp, 0._dp, 0._dp, &
              -2.00495833269087598e-8_dp, 0._dp, 5.0505855142546352e-7_dp, 0._dp, &
              -6.14044870417274069e-6_dp, 0._dp, 0.0000480399810385279125_dp, 0._dp, &
              -0.000272226559218324837_dp, 0._dp, 0.00119360875964957813_dp, 0._dp, &
              -0.00423188560239395883_dp, 0._dp, 0.0125613112325027032_dp, 0._dp, &
              -0.032300514597864094_dp, 0._dp, 0.0753678673950162192_dp, 0._dp, &
              -0.175858357255037845_dp, 0._dp, 0.623497812086043268_dp, 1._dp, 0.623497812086043268_dp, &
              0._dp, -0.175858357255037845_dp, 0._dp, 0.0753678673950162192_dp, 0._dp, &
              -0.032300514597864094_dp, 0._dp, 0.0125613112325027032_dp, 0._dp, &
              -0.00423188560239395883_dp, 0._dp, 0.00119360875964957813_dp, 0._dp, &
              -0.000272226559218324837_dp, 0._dp, 0.0000480399810385279125_dp, 0._dp, &
              -6.14044870417274069e-6_dp, 0._dp, 5.0505855142546352e-7_dp, 0._dp, &
              -2.00495833269087598e-8_dp, &
              0._dp, 0._dp, 0._dp/)

         !******** coefficients for wavelet transform *********************
         DO i = -m, m
            cht(i) = 0._dp
            cg(i) = 0._dp
            cgt(i) = 0._dp
         END DO

         ! the normalization is chosen such that a constant function remains the same constant
         ! on each level of the transform

         cht(0) = 1._dp

         ! g coefficients from h coefficients
         DO i = -m, m - 1
            cg(i + 1) = cht(-i)*(-1)**(i + 1)
            cgt(i + 1) = ch(-i)*(-1)**(i + 1)
         END DO

      CASE (30)
         ch = (/ &
              0._dp, 0._dp, 0._dp, &
              2.78364831185484718e-10_dp, 0._dp, -8.6705490009997277e-9_dp, 0._dp, &
              1.31098700895115883e-7_dp, 0._dp, -1.28248729136526407e-6_dp, 0._dp, &
              9.13008809805271326e-6_dp, 0._dp, -0.000050455750015554468_dp, 0._dp, &
              0.000225566882422478798_dp, 0._dp, -0.00083996810502084962_dp, 0._dp, &
              0.0026652834101623113_dp, 0._dp, -0.00734972091893243418_dp, 0._dp, &
              0.0179659844685015058_dp, 0._dp, -0.039898484988490357_dp, 0._dp, &
              0.0837868184758297496_dp, 0._dp, -0.18261229667809048_dp, 0._dp, 0.626099302896310217_dp, &
              1._dp, 0.626099302896310217_dp, 0._dp, -0.18261229667809048_dp, 0._dp, &
              0.0837868184758297496_dp, 0._dp, -0.039898484988490357_dp, 0._dp, &
              0.0179659844685015058_dp, 0._dp, -0.00734972091893243418_dp, 0._dp, &
              0.0026652834101623113_dp, 0._dp, -0.00083996810502084962_dp, 0._dp, &
              0.000225566882422478798_dp, 0._dp, -0.000050455750015554468_dp, 0._dp, &
              9.13008809805271326e-6_dp, 0._dp, -1.28248729136526407e-6_dp, 0._dp, &
              1.31098700895115883e-7_dp, 0._dp, -8.6705490009997277e-9_dp, 0._dp, &
              2.78364831185484718e-10_dp, &
              0._dp, 0._dp, 0._dp/)

         !******** coefficients for wavelet transform *********************
         DO i = -m, m
            cht(i) = 0._dp
            cg(i) = 0._dp
            cgt(i) = 0._dp
         END DO

         ! the normalization is chosen such that a constant function remains the same constant
         ! on each level of the transform

         cht(0) = 1._dp

         ! g coefficients from h coefficients
         DO i = -m, m - 1
            cg(i + 1) = cht(-i)*(-1)**(i + 1)
            cgt(i + 1) = ch(-i)*(-1)**(i + 1)
         END DO

      CASE (40)
         ch = (/ &
              0._dp, 0._dp, 0._dp, &
              -2.33895335687458837e-13_dp, 0._dp, 9.61499474542229436e-12_dp, 0._dp, &
              -1.9312403731519637e-10_dp, 0._dp, 2.52621846791090201e-9_dp, 0._dp, &
              -2.42028027409528354e-8_dp, 0._dp, 1.81103730854716044e-7_dp, 0._dp, &
              -1.10227332483179024e-6_dp, 0._dp, 5.61214589957214346e-6_dp, 0._dp, &
              -0.0000244006343459658411_dp, 0._dp, 0.0000920510703104425647_dp, 0._dp, &
              -0.000305221969976730609_dp, 0._dp, 0.000899343879450152758_dp, 0._dp, &
              -0.00237826492565707063_dp, 0._dp, 0.00569939227746221067_dp, 0._dp, &
              -0.0125090557778066702_dp, 0._dp, 0.0254814099177543281_dp, 0._dp, &
              -0.0491427191270976329_dp, 0._dp, 0.0930820915230908105_dp, 0._dp, &
              -0.189611667917407207_dp, 0._dp, 0.628712372568244948_dp, 1._dp, 0.628712372568244948_dp, &
              0._dp, -0.189611667917407207_dp, 0._dp, 0.0930820915230908105_dp, 0._dp, &
              -0.0491427191270976329_dp, 0._dp, 0.0254814099177543281_dp, 0._dp, &
              -0.0125090557778066702_dp, 0._dp, 0.00569939227746221067_dp, 0._dp, &
              -0.00237826492565707063_dp, 0._dp, 0.000899343879450152758_dp, 0._dp, &
              -0.000305221969976730609_dp, 0._dp, 0.0000920510703104425647_dp, 0._dp, &
              -0.0000244006343459658411_dp, 0._dp, 5.61214589957214346e-6_dp, 0._dp, &
              -1.10227332483179024e-6_dp, 0._dp, 1.81103730854716044e-7_dp, 0._dp, &
              -2.42028027409528354e-8_dp, 0._dp, 2.52621846791090201e-9_dp, 0._dp, &
              -1.9312403731519637e-10_dp, 0._dp, 9.61499474542229436e-12_dp, 0._dp, &
              -2.33895335687458837e-13_dp, &
              0._dp, 0._dp, 0._dp/)

         !******** coefficients for wavelet transform *********************
         DO i = -m, m
            cht(i) = 0._dp
            cg(i) = 0._dp
            cgt(i) = 0._dp
         END DO

         ! the normalization is chosen such that a constant function remains the same constant
         ! on each level of the transform

         cht(0) = 1._dp

         ! g coefficients from h coefficients
         DO i = -m, m - 1
            cg(i + 1) = cht(-i)*(-1)**(i + 1)
            cgt(i + 1) = ch(-i)*(-1)**(i + 1)
         END DO

      CASE (50)
         !******** coefficients for wavelet transform *********************

         ch(-m:0) = (/0._dp, 0._dp, 0._dp, &
                      2.03510990661269603e-16_dp, 0._dp, -1.03963806080363471e-14_dp, 0._dp, &
                      2.60602607241444435e-13_dp, 0._dp, -4.27267065360972853e-12_dp, 0._dp, &
                      5.15325765417075794e-11_dp, 0._dp, -4.87577454971540944e-10_dp, 0._dp, &
                      3.76884194923947865e-9_dp, 0._dp, -2.44743981275102471e-8_dp, 0._dp, &
                      1.36277898664545694e-7_dp, 0._dp, -6.60874540835592558e-7_dp, 0._dp, &
                      2.82580838150391301e-6_dp, 0._dp, -0.0000107609066649189415_dp, 0._dp, &
                      0.0000368023007940227798_dp, 0._dp, -0.000113853271687194218_dp, 0._dp, &
                      0.000320647989649649023_dp, 0._dp, -0.00082693428909646327_dp, 0._dp, &
                      0.00196396893660410027_dp, 0._dp, -0.00432073166052902059_dp, 0._dp, &
                      0.008863039303649273_dp, 0._dp, -0.0170899753079935742_dp, 0._dp, &
                      0.0313316213979882195_dp, 0._dp, -0.0556296135025505121_dp, 0._dp, &
                      0.0991218567863627306_dp, 0._dp, -0.193934067625492299_dp, 0._dp, &
                      0.630285719782849972_dp, 1._dp/)
         ch(1:m) = (/ &
                   0.630285719782849972_dp, 0._dp, -0.193934067625492299_dp, 0._dp, &
                   0.0991218567863627306_dp, 0._dp, -0.0556296135025505121_dp, 0._dp, &
                   0.0313316213979882195_dp, 0._dp, -0.0170899753079935742_dp, 0._dp, &
                   0.008863039303649273_dp, 0._dp, -0.00432073166052902059_dp, 0._dp, &
                   0.00196396893660410027_dp, 0._dp, -0.00082693428909646327_dp, 0._dp, &
                   0.000320647989649649023_dp, 0._dp, -0.000113853271687194218_dp, 0._dp, &
                   0.0000368023007940227798_dp, 0._dp, -0.0000107609066649189415_dp, 0._dp, &
                   2.82580838150391301e-6_dp, 0._dp, -6.60874540835592558e-7_dp, 0._dp, &
                   1.36277898664545694e-7_dp, 0._dp, -2.44743981275102471e-8_dp, 0._dp, &
                   3.76884194923947865e-9_dp, 0._dp, -4.87577454971540944e-10_dp, 0._dp, &
                   5.15325765417075794e-11_dp, 0._dp, -4.27267065360972853e-12_dp, 0._dp, &
                   2.60602607241444435e-13_dp, 0._dp, -1.03963806080363471e-14_dp, 0._dp, &
                   2.03510990661269603e-16_dp, &
                   0._dp, 0._dp, 0._dp/)
         DO i = -m, m
            cht(i) = 0._dp
            cg(i) = 0._dp
            cgt(i) = 0._dp
         END DO

         ! the normalization is chosen such that a constant function remains the same constant
         ! on each level of the transform

         cht(0) = 1._dp

         ! g coefficients from h coefficients
         DO i = -m, m - 1
            cg(i + 1) = cht(-i)*(-1)**(i + 1)
            cgt(i + 1) = ch(-i)*(-1)**(i + 1)
         END DO

      CASE (60)
         !******** coefficients for wavelet transform *********************
         ch(-m:0) = (/ &
                    0._dp, 0._dp, 0._dp, &
                    -1.80960777804178984e-19_dp, 0._dp, 1.10513064480060885e-17_dp, 0._dp, &
                    -3.32141991973710259e-16_dp, 0._dp, 6.54883738891560793e-15_dp, 0._dp, &
                    -9.5279163579909433e-14_dp, 0._dp, 1.09084919935365698e-12_dp, 0._dp, &
                    -1.02354148279779304e-11_dp, 0._dp, 8.09410105920413478e-11_dp, 0._dp, &
                    -5.50587106934234749e-10_dp, 0._dp, 3.27218841600752522e-9_dp, 0._dp, &
                    -1.71999647508087864e-8_dp, 0._dp, 8.07595396530604197e-8_dp, 0._dp, &
                    -3.41497481961512632e-7_dp, 0._dp, 1.30947169656570694e-6_dp, 0._dp, &
                    -4.58013372208926991e-6_dp, 0._dp, 0.0000146880150398035208_dp, 0._dp, &
                    -0.0000433840444231233622_dp, 0._dp, 0.000118515001353520526_dp, 0._dp, &
                    -0.000300581525171972349_dp, 0._dp, 0.000710396937937769235_dp, 0._dp, &
                    -0.00157035112596770041_dp, 0._dp, 0.00325946830347917649_dp, 0._dp, &
                    -0.00638065613347741822_dp, 0._dp, 0.0118436928230099234_dp, 0._dp, &
                    -0.0209956372771539552_dp, 0._dp, 0.0359258682297967678_dp, 0._dp, &
                    -0.0604028333973506096_dp, 0._dp, 0.103355959368799932_dp, 0._dp, &
                    -0.196868494035809394_dp, 0._dp, 0.631336894666561161_dp, 1._dp/)
         ch(1:m) = (/0.631336894666561161_dp, &
                     0._dp, -0.196868494035809394_dp, 0._dp, 0.103355959368799932_dp, 0._dp, &
                     -0.0604028333973506096_dp, 0._dp, 0.0359258682297967678_dp, 0._dp, &
                     -0.0209956372771539552_dp, 0._dp, 0.0118436928230099234_dp, 0._dp, &
                     -0.00638065613347741822_dp, 0._dp, 0.00325946830347917649_dp, 0._dp, &
                     -0.00157035112596770041_dp, 0._dp, 0.000710396937937769235_dp, 0._dp, &
                     -0.000300581525171972349_dp, 0._dp, 0.000118515001353520526_dp, 0._dp, &
                     -0.0000433840444231233622_dp, 0._dp, 0.0000146880150398035208_dp, 0._dp, &
                     -4.58013372208926991e-6_dp, 0._dp, 1.30947169656570694e-6_dp, 0._dp, &
                     -3.41497481961512632e-7_dp, 0._dp, 8.07595396530604197e-8_dp, 0._dp, &
                     -1.71999647508087864e-8_dp, 0._dp, 3.27218841600752522e-9_dp, 0._dp, &
                     -5.50587106934234749e-10_dp, 0._dp, 8.09410105920413478e-11_dp, 0._dp, &
                     -1.02354148279779304e-11_dp, 0._dp, 1.09084919935365698e-12_dp, 0._dp, &
                     -9.5279163579909433e-14_dp, 0._dp, 6.54883738891560793e-15_dp, 0._dp, &
                     -3.32141991973710259e-16_dp, 0._dp, 1.10513064480060885e-17_dp, 0._dp, &
                     -1.80960777804178984e-19_dp, &
                     0._dp, 0._dp, 0._dp/)

         DO i = -m, m
            cht(i) = 0._dp
            cg(i) = 0._dp
            cgt(i) = 0._dp
         END DO

         ! the normalization is chosen such that a constant function remains the same constant
         ! on each level of the transform

         cht(0) = 1._dp

         ! g coefficients from h coefficients
         DO i = -m, m - 1
            cg(i + 1) = cht(-i)*(-1)**(i + 1)
            cgt(i + 1) = ch(-i)*(-1)**(i + 1)
         END DO

      CASE (100)
         !******** coefficients for wavelet transform *********************
         ch(-m:-m + 22) = (/ &
                          0._dp, 0._dp, 0._dp, &
                          -1.2683805626484815e-31_dp, 0._dp, 1.28158741180595538e-29_dp, 0._dp, &
                          -6.41198417717232202e-28_dp, 0._dp, 2.11779334023809847e-26_dp, 0._dp, &
                          -5.19441179715542395e-25_dp, 0._dp, 1.00911662890805933e-23_dp, 0._dp, &
                          -1.61729304931893186e-22_dp, 0._dp, 2.19924673311587861e-21_dp, 0._dp, &
                          -2.59007672484430282e-20_dp, 0._dp, 2.68351845030850743e-19_dp, 0._dp/)
         ch(-m + 23:0) = (/ &
                         -2.47631006363911635e-18_dp, 0._dp, 2.05560047903738218e-17_dp, 0._dp, &
                         -1.54763876066192241e-16_dp, 0._dp, 1.06410357358472536e-15_dp, 0._dp, &
                         -6.72076683598079057e-15_dp, 0._dp, 3.91882394735691508e-14_dp, 0._dp, &
                         -2.1187969775075261e-13_dp, 0._dp, 1.06630135673478305e-12_dp, 0._dp, &
                         -5.01180443685748998e-12_dp, 0._dp, 2.20666419683207691e-11_dp, 0._dp, &
                         -9.1258654919835045e-11_dp, 0._dp, 3.55352205999457932e-10_dp, 0._dp, &
                         -1.30569909741619005e-9_dp, 0._dp, 4.53620646231137398e-9_dp, 0._dp, &
                         -1.49279735540769725e-8_dp, 0._dp, 4.66118357913015673e-8_dp, 0._dp, &
                         -1.38309751701521672e-7_dp, 0._dp, 3.90568525175243503e-7_dp, 0._dp, &
                         -1.05103157937523999e-6_dp, 0._dp, 2.69873794098621337e-6_dp, 0._dp, &
                         -6.61998110310293365e-6_dp, 0._dp, 0.0000155312721433740143_dp, 0._dp, &
                         -0.0000348898934935080536_dp, 0._dp, 0.0000751302114254144957_dp, 0._dp, &
                         -0.000155250095336387829_dp, 0._dp, 0.000308205854288494064_dp, 0._dp, &
                         -0.000588508297900663563_dp, 0._dp, 0.00108221904295030132_dp, 0._dp, &
                         -0.00191926718143360074_dp, 0._dp, 0.00328782888345707184_dp, 0._dp, &
                         -0.00545087420152093489_dp, 0._dp, 0.00876675750345045625_dp, 0._dp, &
                         -0.0137206712673049998_dp, 0._dp, 0.0209859998811194541_dp, 0._dp, &
                         -0.031565718829452402_dp, 0._dp, 0.0471537281279474153_dp, 0._dp, &
                         -0.0711699126403181485_dp, 0._dp, 0.112357606764076737_dp, 0._dp, &
                         -0.202867901101805219_dp, 0._dp, 0.633444670787269357_dp, 1._dp/)
         ch(1:80) = (/0.633444670787269357_dp, &
                      0._dp, -0.202867901101805219_dp, 0._dp, 0.112357606764076737_dp, 0._dp, &
                      -0.0711699126403181485_dp, 0._dp, 0.0471537281279474153_dp, 0._dp, &
                      -0.031565718829452402_dp, 0._dp, 0.0209859998811194541_dp, 0._dp, &
                      -0.0137206712673049998_dp, 0._dp, 0.00876675750345045625_dp, 0._dp, &
                      -0.00545087420152093489_dp, 0._dp, 0.00328782888345707184_dp, 0._dp, &
                      -0.00191926718143360074_dp, 0._dp, 0.00108221904295030132_dp, 0._dp, &
                      -0.000588508297900663563_dp, 0._dp, 0.000308205854288494064_dp, 0._dp, &
                      -0.000155250095336387829_dp, 0._dp, 0.0000751302114254144957_dp, 0._dp, &
                      -0.0000348898934935080536_dp, 0._dp, 0.0000155312721433740143_dp, 0._dp, &
                      -6.61998110310293365e-6_dp, 0._dp, 2.69873794098621337e-6_dp, 0._dp, &
                      -1.05103157937523999e-6_dp, 0._dp, 3.90568525175243503e-7_dp, 0._dp, &
                      -1.38309751701521672e-7_dp, 0._dp, 4.66118357913015673e-8_dp, 0._dp, &
                      -1.49279735540769725e-8_dp, 0._dp, 4.53620646231137398e-9_dp, 0._dp, &
                      -1.30569909741619005e-9_dp, 0._dp, 3.55352205999457932e-10_dp, 0._dp, &
                      -9.1258654919835045e-11_dp, 0._dp, 2.20666419683207691e-11_dp, 0._dp, &
                      -5.01180443685748998e-12_dp, 0._dp, 1.06630135673478305e-12_dp, 0._dp, &
                      -2.1187969775075261e-13_dp, 0._dp, 3.91882394735691508e-14_dp, 0._dp, &
                      -6.72076683598079057e-15_dp, 0._dp, 1.06410357358472536e-15_dp, 0._dp, &
                      -1.54763876066192241e-16_dp, 0._dp, 2.05560047903738218e-17_dp, 0._dp, &
                      -2.47631006363911635e-18_dp, 0._dp/)
         ch(81:m) = (/2.68351845030850743e-19_dp, 0._dp, &
                      -2.59007672484430282e-20_dp, 0._dp, 2.19924673311587861e-21_dp, 0._dp, &
                      -1.61729304931893186e-22_dp, 0._dp, 1.00911662890805933e-23_dp, 0._dp, &
                      -5.19441179715542395e-25_dp, 0._dp, 2.11779334023809847e-26_dp, 0._dp, &
                      -6.41198417717232202e-28_dp, 0._dp, 1.28158741180595538e-29_dp, 0._dp, &
                      -1.2683805626484815e-31_dp, &
                      0._dp, 0._dp, 0._dp/)

         DO i = -m, m
            cht(i) = 0._dp
            cg(i) = 0._dp
            cgt(i) = 0._dp
         END DO

         ! the normalization is chosen such that a constant function remains the same constant
         ! on each level of the transform

         cht(0) = 1._dp

         ! g coefficients from h coefficients
         DO i = -m, m - 1
            cg(i + 1) = cht(-i)*(-1)**(i + 1)
            cgt(i + 1) = ch(-i)*(-1)**(i + 1)
         END DO

      END SELECT

   END SUBROUTINE lazy_arrays
END MODULE lazy
