# autopkgtest check: helper routines for authdaemond tests
#
# Author: Markus Wanner <markus@bluegap.ch>

TEST_USERS="alice bob carol"

CONFIG_FILES="/etc/courier/authdaemonrc \
	    /etc/courier/authldaprc \
	    /etc/courier/authmysqlrc \
	    /etc/courier/authpgsqlrc \
	    /etc/courier/authsqliterc"

export PGOPTIONS='--client-min-messages=warning'

# exits with code 0 if the given user exists
user_exists() {
    id -u $1 > /dev/null 2>&1
}

# exits with code 0 if the postgresql client tools are installed
has_postgres_client() {
    which psql > /dev/null 2>&1
}

authenumerate_as_courier() {
    su -c "/usr/sbin/authenumerate" -s /bin/sh courier
}

authtest_as_courier() {
    user=$1
    password=$2
    su -c "/usr/sbin/authtest $user $password" -s /bin/sh courier
}

# emits a random (512bit, hex encoded) password on stdout
gen_random_password() {
    dd if=/dev/urandom bs=16 count=1 2> /dev/null | hexdump -e '"%x"'
}

# accepts SQL on stdin
postgres_superuser_exec() {
    su postgres -c "psql -X -q -v ON_ERROR_STOP=1 --pset pager=off"
}

create_test_users() {
    echo "create test users..."
    PASSWORD_ALICE=$(gen_random_password)
    PASSWORD_BOB=$(gen_random_password)
    PASSWORD_CAROL=$(gen_random_password)

    useradd --shell /bin/false --password $PASSWORD_ALICE alice
    useradd --shell /bin/false --password $PASSWORD_BOB   bob
    useradd --shell /bin/false --password $PASSWORD_CAROL carol

    ALICE_UID=$(id -u alice)
    ALICE_GID=$(id -g alice)
    BOB_UID=$(id -u bob)
    BOB_GID=$(id -g bob)
    CAROL_UID=$(id -u carol)
    CAROL_GID=$(id -g carol)
}

backup_config_files() {
    echo "backup config files..."
    for f in $CONFIG_FILES; do
	if [ -f $f ]; then
	    cp ${f} ${f}.autopkgtest.bak
	fi
    done
}

restore_config_files() {
    echo "restore config files..."
    for f in $CONFIG_FILES; do
	if [ -f ${f}.autopkgtest.bak ]; then
	    mv ${f}.autopkgtest.bak ${f}
	fi
    done
}

# cleanup after running tests
finish() {
    echo "finish..."
    # restore config files, then restart the authdaemon, so it
    # disconnects from the database. Otherwise authdaemon blocks the
    # database deletion.
    restore_config_files
    service courier-authdaemon restart

    # cleanup Postgres databases
    if has_postgres_client; then
        postgres_superuser_exec <<EOSQL
DROP DATABASE IF EXISTS test;
DROP ROLE IF EXISTS courier;
EOSQL
    fi

    # drop test users
    if user_exists alice; then
	echo "dropping user alice"
        userdel alice
    fi
    if user_exists bob; then
	echo "dropping user bob"
        userdel bob
    fi
    if user_exists carol; then
	echo "dropping user carol"
	userdel carol
    fi
}
trap finish EXIT

dump_file_if_exists() {
    if [ -f $1 ]; then
	echo "===== BEGIN $1 ====="
	cat $1
	echo "===== END $1 ====="
    fi
}

dump_config_files() {
    for f in $CONFIG_FILES; do
	if [ -f ${f}.autopkgtest.bak ]; then
	    dump_file_if_exists $f
	fi
    done
}


dump_environment() {
    echo "dump..."
    # dump and then restore the config files
    dump_config_files

    # cleanup as well, especially after an error
    finish
}
trap dump_environment INT QUIT ABRT PIPE TERM
