#!/usr/bin/env python
#
#   ConVirt   -  Copyright (c) 2008 Jd & Hap Hazard
#   ======
#
# ConVirt is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU Lesser General Public License (LGPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/lgpl.txt
#
# author : Jd <jd_jedi@users.sourceforge.net>
#

from utils import PyConfig
import re, os

from NodeProxy import Node
from constants import *



# Contains base classes representing
#        VM,
#        VMConfig,
#        VMStats,
#        VMOperationsException
# Also, has utility classes
#        DiskEntry
#        NetworkEntry

class VMOperationException(Exception):
   def __init__(self, message, errno = None):
      if errno:
         Exception.__init__(self, (errno, message))
      else:
         Exception.__init__(self,  message)
      self.message = message
      self.errno = errno

   def __repr__(self):
      if self.errno != None:
         return "[Error %s]: %s" % (str(self.errno), self.message)
      else:
         return self.message
      


class VM:
    """
    This represent Doms. It encapsulates information about
    running virtual machine : state as well as resource stats
    """
    RUNNING  = 0
    BLOCKED  = 1
    PAUSED   = 2
    SHUTDOWN = 3
    CRASHED  = 4
    NOT_STARTED    = -1
    UNKNOWN  = -2


    # interface methods



# Disk entry class : lets use xen format. need to abstract this out.
# and change correspoding provisioning and ui code as well.
class DiskEntry:
    """This class provides a structured representation of the disk
    string. The following attributes are available:

     type:     {file|lvm|part}
     filename: the path to the VBD or device
     device:   the device to be used in the guest system
     mode:     {r|w}
     """

    def __init__(self, input):
        if type(input) is str:
            m = re.match("^(phy|file|tap:aio|tap:qcow):([ \S]+),([a-z|0-9]+(:cdrom)?|ioemu:[a-z|0-9]+),(r|w)$", input)
            if m:
                self.filename = m.group(2)
                self.device = m.group(3)
                self.mode = m.group(5)

                if m.group(1) in ("file", "tap:aio", "tap:qcow"):
                    self.type = m.group(1)
                else:
                    device = os.path.realpath(m.group(2))
                    if device.startswith("/dev/mapper"):
                        self.type = "lvm"
                    else:
                        self.type = "part"
        elif type(input) in (list, tuple):
            self.type, self.filename, self.device, self.mode = input

    def __repr__(self):
        return "%s:%s,%s,%s" % (self.type, self.filename, self.device,
                                self.mode)


# Network class
#class NetworkEntry:



# DomConfig object, will use python config 
class VMConfig(PyConfig):
    """
    reprsents startup config object (information in the conf file)
    """
    
    # Module variable.
    imps = ["import sys",
            "import os",
            "import os.path",
            ]
    

    signature = "# Automtically generated by ConVirt\n"



    # DomConfig follows
    def __init__(self, node, filename = None):
        """
        read stuff from file and populate the config
        when filename is None, creates an empty config
        """
        self.node = node
        self.convirt_generated = False

        
        PyConfig.__init__(self,
                          node,
                          filename,
                          VMConfig.signature)

        # TBD : implement isfile in node proxy and use it.
        #if filename is None or not os.path.isfile(filename): return
        
        if filename is None : return

        if len(self.lines) > 0:
            if self.lines[0].find(self.signature) >= 0:
                    self.convirt_generated = True
            
        for k in ("xm_help", "xm_file"):
            if self.options.has_key(k):
                del self.options[k]
                
        if self["name"] is None:
            raise Exception("No dom name specified")

        #pick up the name from the file
        self.name = self["name"]

    
    # read_conf
    def read_conf(self, init_glob=None, init_locs=None):
        # Leverage the fact that conf files are python syntax.
        # save on parsing code
        
        # Create global and local dicts for the file.
        # Initialize locals to the vars.
        globs = {}
        locs = {}

        cmd = '\n'.join(self.imps)  

        # Use exec to do the standard imports and
        # define variables we are passing to the script.
        exec cmd in globs, locs
        return PyConfig.read_conf(self, globs, locs)
    
    def __setitem__(self, name, item):
        self.options[name] = item
        if name == "name":
            self.name = item

    # try to keep them in sync
    def set_name(self, name):
        self.name = name
        self["name"] = name
    
    
    # get the configured disks
    def getDisks(self):
        """
        This method returns a more structured version of the config file's
        disk entry
        """
        reslist = []
        for str in self["disk"]:
            reslist.append(DiskEntry(str))
        return reslist


    # Assumes that the file read contains single line values
    # now preserves the comments and order
    def write(self):
        """Writes the settings out to the filename specified during
        initialization"""
        self.name = self["name"]
        PyConfig.write(self)
        self.convirt_generated = True


    def is_convirt_generated(self):
        return self.convirt_generated

    # Get raw file content    
    def get_contents(self):
        f = self.node.node_proxy.open(filename)
        lines = f.readlines()
        f.close()
        contents = "".join(lines)
        return contents

    # Write raw contents
    def write_contents(self, contents):
        """Writes the settings out to the filename specified during
        initialization"""
        outfile = self.node.node_proxy.open(self.filename, 'w')
        outfile.write(contents)
        outfile.close()



    # get representation required for underlying create API
    # may be __repr__ ?
    def internal_rep(self):
        """ create config required by xen create_dom api"""
        # the problem is that this needs to be complete, otherwise some
        # existing conf files that are super set of convirt created files
        # would not work
        # May be we should just go the xm create route for now.


    # Validate the config. Actual details to be supplied by the
    # derived class
    
    def validate(self):
        """Attempts to validate that the settings are not going to lead to
        any errors when the dom is started, and returns a list of the
        errors as strings"""

        result = []
        return result




class VMStats:
    """
    represents statatistics/measurements for a dom. (CPU, I/O etc)
    This is abstracted out so we can cut over to some other source for
    runtime statastics/measurements
    """

    def __init__(self, dom):
        """
        constructor, dom for which the stats are to be obtained.
        """
        self.dom = dom
        self.stat = {}

    def get_snapshot(self):
        # to be implemented by derived class
        return self.stat
