/*
 * Copyright (C) 2003  Robert Collins  <robertc@squid-cache.org>
 * 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * DO NOT ALTER THE NEXT LINE
 * arch-tag: a0feccb0-784a-4c9e-8a61-41d248fed0c0
 * 
 */

#include "Config.h"
#include <getopt++/GetOption.h>
#include <getopt++/BoolOption.h>
#include <getopt++/StringOption.h>
#include <getopt++/StringCollector.h>
#include <cassert>
#include <sstream>
#include <fstream>
#include <errno.h>
#include "ConfigSource.h"
#include "ConfigArchSource.h"
#include "FileSystemVisitor.h"
#include "Directory.h"
#include "ConfigEntry.h"

/* TODO: generate separate OptionSets for the commands, and check parameter 2 for which option set to process - i.e. use the system. XXX This requires removing the use of getopt from libgetopt++, as we need more flexability that it offers. */

static BoolOption verboseOption(false, 'v', "verbose", "be verbose about operation.");
static StringOption outputOption(".", 'd', "dir", "location to output built tree in.");
static StringOption configSourceOption("",'s',"source", "The RCS source for the config itself");
static BoolOption pathsOption(false, 'p', "paths", "only list the paths from the config.");
static BoolOption releaseOption(false, 'r', "release-id", "Generate a ./=RELEASE-ID for this config.");
static BoolOption noSchemeOption(false, 'n', "no-url-schemes", "Strip url schemes from project locations.");

using namespace std;


void
usage()
{
    cout << "cm: config manager" << endl;
    cout << "usage:" << endl;
    cout << "cm [options] command config-file" << endl;
    cout << "commands: build cat update changes missing examine" << endl;
    cout << "valid options: " << endl;
    GetOption::GetInstance().ParameterUsage (cout);
}

istream &
getStream(std::vector<std::string> &unnamedparams, string &name)
{
    const char *filename;
    char tempfile[] = "/tmp/cm.XXXXXX";
    int filemade=0;

    if (unnamedparams.size() == 0)
        return cin;

    name = unnamedparams[0];

    unnamedparams.erase(unnamedparams.begin());

    filename = name.c_str();

    if (name.find("http://") == 0 || name.find("ftp://") == 0) {
        int handle = mkstemp(tempfile);
        close (handle);

        string wgetcmd = string("wget -O ") + tempfile + " " + filename;
        system(wgetcmd.c_str());

        filename = tempfile;
        filemade=1;
    }

    static ifstream input (filename, ifstream::in);

    if (!input.good())
        throw new runtime_error ("Could not open " + name + " for reading");

    if (filemade)
        unlink(tempfile);

    return input;

}

class Command
{

public:
    static Command *Factory(std::string command);
    virtual ~Command();
    std::string const & name() const;
    int execute(std::vector<std::string> const &params);
    virtual int execute() = 0;

protected:
    Command(std::string name);
    void processOptions(std::vector<std::string> const &params);
    virtual void makeConfig();
    void checkParameterCount(int const &minCount, int const &maxCount);
    virtual OptionSet &getOptionSet() const;
    std::vector<std::string> params;
    Config *config;

private:
    std::string const _name;
};

class CommandBuild : public Command
{

public:
    static std::string const Name;
    CommandBuild();
    virtual int execute();
};

class CommandCat : public Command
{

public:
    static std::string const Name;
    CommandCat();
    virtual int execute();
};

class CommandUpdate : public Command
{

public:
    static std::string const Name;
    CommandUpdate();
    virtual int execute();
};

class CommandChanges : public Command
{

public:
    static std::string const Name;
    CommandChanges();
    virtual int execute();
};

class CommandMissing : public Command
{

public:
    static std::string const Name;
    CommandMissing();
    virtual int execute();
};

class CommandExamine : public Command
{

public:
    static std::string const Name;
    CommandExamine();
    virtual int execute();
protected:
    virtual void makeConfig();
};

Command *
Command::Factory(std::string command)
{
    if (command == CommandBuild::Name)
        return new CommandBuild();

    if (command == CommandCat::Name)
        return new CommandCat();

    if (command == CommandUpdate::Name)
        return new CommandUpdate();

    if (command == CommandChanges::Name)
        return new CommandChanges();

    if (command == CommandMissing::Name)
        return new CommandMissing();

    if (command == CommandExamine::Name)
        return new CommandExamine();

    return NULL;
}

Command::Command(std::string aLabel) : config (NULL),_name(aLabel)
{}

Command::~Command()
{
    delete config;
}

std::string const &
Command::name() const
{
    return _name;
}

void
Command::makeConfig()
{
    checkParameterCount(0, 1);
    string streamName;
    config = new Config(getStream(params, streamName));

    if (streamName.size() > 0)
        config->name(streamName);
}

void
Command::checkParameterCount(int const &minCount, int const &maxCount)
{
    if (params.size() < minCount || params.size() > maxCount) {
	usage();
	throw new runtime_error("Incorrect non-option count to command" + name());
    }
}

void
Command::processOptions(std::vector<std::string> const &unprocessed)
{
    /* Find the config name. This should be pushed down to the individual commands */
    if (!getOptionSet().Process (unprocessed,NULL)) {
        usage();
	throw new runtime_error("Could not process command options for command " + name());
    }

    params = GetOption::GetInstance().nonOptions();
    makeConfig();
    ConfigSource::Verbose(verboseOption);
    config->verbose(verboseOption);

    if (((string)configSourceOption).size())
        config->setConfigSource(ConfigSource::Create(configSourceOption));
    else
        config->setConfigSource(new ConfigArchSource(""));
}

OptionSet &
Command::getOptionSet() const
{
    return GetOption::GetInstance();
}

std::string const CommandBuild::Name ("build");

CommandBuild::CommandBuild() : Command(Name)
{}

int
CommandBuild::execute()
{
    config->build((string)outputOption);
    return 0;
}

std::string const CommandCat::Name ("cat");

CommandCat::CommandCat() : Command(Name)
{}

int
CommandCat::execute()
{
    config->cat();
    return 0;
}

std::string const CommandUpdate::Name ("update");

CommandUpdate::CommandUpdate() : Command(Name)
{}

int
CommandUpdate::execute()
{
    config->update((string)outputOption);
    return 0;
}

std::string const CommandChanges::Name ("changes");

CommandChanges::CommandChanges() : Command(Name)
{}

int
CommandChanges::execute()
{
    return config->changes((string)outputOption);
}

std::string const CommandMissing::Name ("missing");

CommandMissing::CommandMissing() : Command(Name)
{}

int
CommandMissing::execute()
{
    return config->missing((string)outputOption);
}

std::string const CommandExamine::Name ("examine");

CommandExamine::CommandExamine() : Command(Name)
{}

int
CommandExamine::execute()
{
    config->examine((string)outputOption);
    return 0;
}

void
CommandExamine::makeConfig()
{
    checkParameterCount(0, 0);
    config = new Config(std::cin);
}

int
main (int argc, char ** argv)
{
    StringCollector theCommand(1, true);
    Command *command = NULL;

    /* Find the command, and any global options. */

    if (argc == 1 || !GetOption::GetInstance().Process (argc, argv, &theCommand)) {
        usage();
        return 1;
    }

    command = Command::Factory(theCommand(1));

    if (!command) {
        cout << "Unknown command \"" << theCommand(1) << "\"" << endl;
        usage();
        return 1;
    }

    if (verboseOption)
        cout << "Output files will be placed in : '" << (string)outputOption << "'" << endl;

    try {
        int result = command->execute(GetOption::GetInstance().remainingArgv());
        delete command;
        return result;
    } catch (exception *e) {
        cerr << "Fatal error: '" << e->what() << "'" << endl;
        return 1;
    }
}

int
Command::execute(std::vector<std::string> const &unprocessed)
{
    processOptions(unprocessed);
    int result = execute();

    if (releaseOption)
        config->makeReleaseId(outputOption);

    return result;
}

Config::Config (istream &aSource) : source (aSource), theSource(NULL), _verbose(false)
{}

void
Config::verbose(bool const &aBool)
{
    _verbose = aBool;
}

bool
Config::verbose() const
{
    return _verbose;
}

string
getLineFromComment(istream &source)
{
    char c;
    string result;

    while (source.good()) {
        source.get(c);

        if (c == '\n')
            return result ;

        result += c;
    }
}

struct BuildConfigEntry : public unary_function<void, ConfigEntry &>
{
    BuildConfigEntry(ostream &aStream, string const &aRelPath) : out (aStream), relPath(aRelPath)
    {}

    void operator() (ConfigEntry &anEntry)
    {
        while (anEntry.path().size() && nested.size() &&
                anEntry.path().find(nested.back()->path()) != 0)
            /* this is not a sub tree */
            nested.pop_back();

        out << "Building " << anEntry.source()->url(false) << " in ";

        out << relPath + "/" + anEntry.path();

        if (nested.size())
            out << " with parent " << nested.back()->path();

        out << endl;

        string parentPath = Path(relPath + "/" + anEntry.path()).dirName();

        if (!Directory::Exists(parentPath))
            Directory::Create(parentPath);

        anEntry.source()->get
        (relPath + "/" + anEntry.path());

        if (nested.size())
            nested.back()->source()->ignore(anEntry.source(), relPath + "/" + anEntry.path());

        nested.push_back(&anEntry);
    }

    ostream &out;
    string const &relPath;
    vector<ConfigEntry *> nested;
};

void
Config::build (string const &where) throw(exception *)
{
    parse(source);
    for_each(entries.begin(), entries.end(), BuildConfigEntry(cout, where));
}


/* bah uglification. tokenising and parsing mixed. bah. bah. */
string
Config::getToken(bool const &required) throw(exception *)
{
    while (source.good()) {
        string token;
        source >> token;

        if (token.size() != 0) {
            if (token[0] != '#')
                return token;
            else {
                string candidate = getLineFromComment(source);
                string::size_type position = candidate.find("config-name:");

                if (position != string::npos)
                    name(candidate.substr(position + 12));
            }

        }
    }

    if (!required)
        return string();

    throw new runtime_error("Could not get next token from config file: invalid config file");
}

string
Config::normalise(string const &aPath)
{
    string::size_type pos = aPath.find_first_not_of("./");

    if (pos != string::npos)
        return aPath.substr(pos);

    /* If solely composed of . and /, return an empty string */
    if (aPath.find_first_of("./") != string::npos)
        return string();

    return aPath;
}

void
Config::parse(istream &source) throw(exception *)
{
    while (source.good()) {
        string path = getToken(false);

        if (path.size() == 0)
            return ;

        string source = getToken(true);

        if (verbose())
            cout << "Found config line path:'" << path << "'" <<
            " source url:'" << source << "'" << endl;

        entries.push_back(ConfigEntry(normalise(path), ConfigSource::Create(source)));
    }
}

struct DumpConfigEntry : public unary_function<void, ConfigEntry &>
{
    DumpConfigEntry(ostream &aStream, bool const &aBool, bool const &anotherBool) : out (aStream), pathsOnly (aBool), stripSchemes(anotherBool)
    {}

    void operator() (ConfigEntry &anEntry)
    {
        /* insert ./ for root nodes */

        if (anEntry.path().size())
            out << anEntry.path();
        else
            out << ".";

        if (!pathsOnly)
            out << " " << anEntry.source()->url(stripSchemes);

        out << endl;
    }

    ostream &out;
    bool const pathsOnly;
    bool const stripSchemes;
};

void
Config::cat () throw (exception *)
{
    parse(source);
    for_each(entries.begin(), entries.end(), DumpConfigEntry(cout, pathsOption, noSchemeOption));
}

struct UpdateConfigEntry : public unary_function<void, ConfigEntry &>
{
    UpdateConfigEntry(ostream &aStream, string const &aRelPath) : out (aStream), relPath(aRelPath)
    {}

    void operator() (ConfigEntry &anEntry)
    {
        out << "Updating " << anEntry.source()->url(false) << " in ";
        out << relPath + "/" + anEntry.path() << endl;

	/* FIXME
	 * A quick hack for Debian bug #288887
	 * If the path does not exist, perform a simple (non-nested)
	 * get on it.  Otherwise an update will suffice
	 */
        if (!Directory::Exists(anEntry.path())) {
	  //BuildConfigEntry(out, relPath);
	  anEntry.source()->get    (relPath + "/" + anEntry.path());
	} else {
	  anEntry.source()->update (relPath + "/" + anEntry.path());
	}
    }

    ostream &out;
    string const &relPath;
};

void
Config::update (string const &where) throw (exception *)
{
    parse(source);
    for_each(entries.begin(), entries.end(), UpdateConfigEntry(cout, where));
}

struct ChangesConfigEntry : public unary_function<void, ConfigEntry &>
{
    ChangesConfigEntry(ostream &aStream, string const &aRelPath) : out (aStream), relPath(aRelPath), result(0)
    {}

    void operator() (ConfigEntry &anEntry)
    {
        if (verboseOption) {
            out << "Getting changes for " << anEntry.source()->url(false) << " in ";
            out << relPath + "/" + anEntry.path() << endl;
        }

        int holding = anEntry.source()->changes (relPath + "/" + anEntry.path());

        if (holding > result)
            result = holding;
    }

    ostream &out;
    string const &relPath;
    int result;
};

int
Config::changes (string const &where) throw (exception *)
{
    parse(source);
    ChangesConfigEntry result = for_each(entries.begin(), entries.end(), ChangesConfigEntry(cout, where));
    return result.result;
}

struct MissingConfigEntry : public unary_function<void, ConfigEntry &>
{
    MissingConfigEntry(ostream &aStream, string const &aRelPath) : out (aStream), relPath(aRelPath), result(0)
    {}

    void operator() (ConfigEntry &anEntry)
    {
        if (verboseOption) {
            out << "Getting missing patches for " << anEntry.source()->url(false) << " in ";
            out << relPath + "/" + anEntry.path() << endl;
        }

        int holding = anEntry.source()->missing (relPath + "/" + anEntry.path());

        if (holding > result)
            result = holding;
    }

    ostream &out;
    string const &relPath;
    int result;
};

int
Config::missing (string const &where) throw (exception *)
{
    parse(source);
    MissingConfigEntry result = for_each(entries.begin(), entries.end(), MissingConfigEntry(cout, where));
    return result.result;
}

class Examiner : public FileSystemVisitor
{

public:
    Examiner(Config &aConfig) : config(aConfig)
    {}

    virtual bool visitDirectory(Directory &aDir)
    {
        if (verboseOption)
            std::cout << aDir.path().fullName() << endl;

        bool newEntry (false);

        while (nested.size() && aDir.path().fullName().find(nested.back().path()) != 0)
            /* this is not a sub tree */
            nested.pop_back();

        if (!config.entries.size()) {
            config.entries.push_back(ConfigEntry(aDir.path().fullName(),
                                                 ConfigSource::Create(aDir.path(),NULL)));

            if (!config.entries.back().source())
                throw new runtime_error("Could not identify the RCS type for the supplied path (" + aDir.path().fullName() + ").");

            newEntry = true;
        } else {
            ConfigSource *newSource = ConfigSource::Create(aDir.path(),nested.back().source());

            if (newSource) {
                config.entries.push_back(ConfigEntry(aDir.path().fullName(), newSource));
                newEntry = true;
            }
        }

        if (newEntry)
            nested.push_back(config.entries.back());

        return true;
    }

    Config &config;
    vector<ConfigEntry> nested;
};

class NormaliseConfigEntry : public unary_function<void, ConfigEntry &>
{

public:
    NormaliseConfigEntry (string where)
    {
        path = Path(where).fullName();
    }

    void operator() (ConfigEntry &anEntry)
    {
        anEntry.path("." + anEntry.path().substr(path.size()));
    }

    string path;
};

void
Config::examine (string const &where) throw (exception *)
{
    /* visit the tree starting at where */
    Examiner examiner(*this);
    Directory(where).visit(examiner);
    for_each(entries.begin(), entries.end(), NormaliseConfigEntry(where));
    for_each(entries.begin(), entries.end(), DumpConfigEntry(cout, false, noSchemeOption));
}

void
Config::name(string const &aName) throw (exception *)
{
    if (theName.size() == 0)
        theName = aName;
}

void
Config::setConfigSource(ConfigSource *aSource) throw(exception *)
{
    if (!theSource)
        theSource = aSource;
}


struct GenReleaseLines : public unary_function<void, ConfigEntry &>
{
    GenReleaseLines(ostream &aStream, string const &aRelPath) : output(aStream), where(aRelPath)
    {}

    void operator() (ConfigEntry &entry)
    {
        output << entry.path() << "\t" << entry.source()->treeVersion(where + "/" + entry.path()) << endl;
    }

private:
    ostream &output;
    string const &where;

};

void
Config::makeReleaseId(string const &where) throw (exception *)
{
    if (unlink((where + "/=RELEASE-ID").c_str()) && errno != ENOENT)
        throw new runtime_error ("Could not unlink " + where + "/=RELEASE-ID.");

    ofstream output((where + "/=RELEASE-ID").c_str() , ios::trunc);

    output << "# automatically generated release id (by cm)" << endl;

    output << "#" << endl;

    string configRelease;

    if (theSource)
        output << theSource->treeVersion(where);

    output << "(" << theName << ")" << endl;

    output << endl;

    for_each(entries.begin(), entries.end(), GenReleaseLines(output, where));

    /*  generate config name (using '-' if stdin was used and no name detected)
     *   output tree canonical precise id's from configentries
     */
}
