/***************************************************************
 *
 * Copyright (C) 1990-2018, Condor Team, Computer Sciences Department,
 * University of Wisconsin-Madison, WI.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License.  You may
 * obtain a copy of the License at
 * 
 *	  http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***************************************************************/

#include "condor_common.h"
#include "condor_arglist.h"
#include "condor_attributes.h"
#include "condor_getcwd.h"
#include "condor_daemon_core.h"
#include "condor_version.h"
#include "dagman_utils.h"
#include "my_popen.h"
#include "read_multiple_logs.h"
#include "../condor_procapi/processid.h"
#include "../condor_procapi/procapi.h"
#include "tmp_dir.h"
#include "tokener.h"
#include "which.h"

namespace shallow = DagmanShallowOptions;
namespace deep = DagmanDeepOptions;

static void
AppendError(std::string &errMsg, const std::string &newError)
{
	if ( errMsg != "" ) errMsg += "; ";
	errMsg += newError;
}

static bool
ImportFilter( const std::string &var, const std::string &val ) {
	if ( (var.find(";") != std::string::npos) || (val.find(";") != std::string::npos) ) {
		return false;
	}
	return Env::IsSafeEnvV2Value( val.c_str() );
}

dag_tokener::dag_tokener(const char * line_in)
{
	tokener tkns(line_in);
	while(tkns.next()) {
		std::string token;
		tkns.copy_token(token);
		tokens.Append(&token);
	}
}


bool
DagmanUtils::writeSubmitFile( /* const */ SubmitDagDeepOptions &deepOpts,
			/* const */ SubmitDagShallowOptions &shallowOpts,
			/* const */ std::list<std::string> &dagFileAttrLines ) const
{
	FILE *pSubFile = safe_fopen_wrapper_follow(shallowOpts.strSubFile.c_str(), "w");
	if (!pSubFile)
	{
		fprintf( stderr, "ERROR: unable to create submit file %s\n",
				 shallowOpts.strSubFile.c_str() );
		return false;
	}

	const char *executable = NULL;
	std::string valgrindPath; // outside if so executable is valid!
	if ( shallowOpts[shallow::b::RunValgrind] ) {
		valgrindPath = which( valgrind_exe );
		if ( valgrindPath.empty()) {
			fprintf( stderr, "ERROR: can't find %s in PATH, aborting.\n",
						 valgrind_exe );
			fclose(pSubFile);
			return false;
		} else {
			executable = valgrindPath.c_str();
		}
	} else {
		executable = deepOpts[deep::str::DagmanPath].c_str();
	}

	/*	Set up DAGMan proper jobs getenv filter
	*	Update DAGMAN_MANAGER_JOB_APPEND_GETENV Macro documentation if base
	*	getEnv value changes. -Cole Bollig 2023-02-21
	*/
	std::string getEnv = "CONDOR_CONFIG,_CONDOR_*,PATH,PYTHONPATH,PERL*,PEGASUS_*,TZ,HOME,USER,LANG,LC_ALL";
	auto_free_ptr conf_getenvVars = param("DAGMAN_MANAGER_JOB_APPEND_GETENV");
	if (conf_getenvVars && strcasecmp(conf_getenvVars.ptr(),"true") == MATCH) {
		getEnv = "true";
	} else {
		//Scitoken related variables
		getEnv += ",BEARER_TOKEN,BEARER_TOKEN_FILE,XDG_RUNTIME_DIR";
		//Add user defined via flag vars to getenv
		if (!deepOpts[deep::str::GetFromEnv].empty()) { getEnv += ","; getEnv += deepOpts[deep::str::GetFromEnv]; }
		//Add config defined vars to add getenv
		if (conf_getenvVars) { getEnv += ","; getEnv += conf_getenvVars.ptr(); }
	}

	fprintf(pSubFile, "# Filename: %s\n", shallowOpts.strSubFile.c_str());

	fprintf(pSubFile, "# Generated by condor_submit_dag ");
	for (auto & dagFile : shallowOpts.dagFiles) {
		fprintf(pSubFile, "%s ", dagFile.c_str());
	}
	fprintf(pSubFile, "\n");

	fprintf(pSubFile, "universe\t= scheduler\n");
	fprintf(pSubFile, "executable\t= %s\n", executable);
	fprintf(pSubFile, "getenv\t\t= %s\n", getEnv.c_str());
	fprintf(pSubFile, "output\t\t= %s\n", shallowOpts.strLibOut.c_str());
	fprintf(pSubFile, "error\t\t= %s\n", shallowOpts.strLibErr.c_str());
	fprintf(pSubFile, "log\t\t= %s\n", shallowOpts.strSchedLog.c_str());
	if ( ! deepOpts.batchName.empty() ) {
		fprintf(pSubFile, "+%s\t= \"%s\"\n", ATTR_JOB_BATCH_NAME,
					deepOpts.batchName.c_str());
	}
	if ( ! deepOpts.batchId.empty() ) {
		fprintf(pSubFile, "+%s\t= \"%s\"\n", ATTR_JOB_BATCH_ID,
					deepOpts.batchId.c_str());
	}
#if !defined ( WIN32 )
	fprintf(pSubFile, "remove_kill_sig\t= SIGUSR1\n" );
#endif
	fprintf(pSubFile, "+%s\t= \"%s =?= $(cluster)\"\n",
				ATTR_OTHER_JOB_REMOVE_REQUIREMENTS, ATTR_DAGMAN_JOB_ID );

		// ensure DAGMan is automatically requeued by the schedd if it
		// exits abnormally or is killed (e.g., during a reboot)
	const char *defaultRemoveExpr = "( ExitSignal =?= 11 || "
				"(ExitCode =!= UNDEFINED && ExitCode >=0 && ExitCode <= 2))";
	std::string removeExpr;
	param(removeExpr, "DAGMAN_ON_EXIT_REMOVE", defaultRemoveExpr);
	fprintf(pSubFile, "# Note: default on_exit_remove expression:\n");
	fprintf(pSubFile, "# %s\n", defaultRemoveExpr);
	fprintf(pSubFile, "# attempts to ensure that DAGMan is automatically\n");
	fprintf(pSubFile, "# requeued by the schedd if it exits abnormally or\n");
	fprintf(pSubFile, "# is killed (e.g., during a reboot).\n");
	fprintf(pSubFile, "on_exit_remove\t= %s\n", removeExpr.c_str() );

	if (!usingPythonBindings) {
		fprintf(pSubFile, "copy_to_spool\t= %s\n", shallowOpts.copyToSpool ?
				"True" : "False" );
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Be sure to change MIN_SUBMIT_FILE_VERSION in dagman_main.cpp
	// if the arguments passed to condor_dagman change in an
	// incompatible way!!
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	ArgList args;

	if ( shallowOpts[shallow::b::RunValgrind] ) {
		args.AppendArg("--tool=memcheck");
		args.AppendArg("--leak-check=yes");
		args.AppendArg("--show-reachable=yes");
		args.AppendArg(deepOpts[deep::str::DagmanPath].c_str());
	}

		// -p 0 causes DAGMan to run w/o a command socket (see gittrac #4987).
	args.AppendArg("-p");
	args.AppendArg("0");
	args.AppendArg("-f");
	args.AppendArg("-l");
	args.AppendArg(".");
	if ( shallowOpts[shallow::i::DebugLevel] != DEBUG_UNSET ) {
		args.AppendArg("-Debug");
		args.AppendArg(std::to_string(shallowOpts[shallow::i::DebugLevel]));
	}
	args.AppendArg("-Lockfile");
	args.AppendArg(shallowOpts.strLockFile.c_str());
	args.AppendArg("-AutoRescue");
	args.AppendArg(std::to_string(deepOpts[deep::b::AutoRescue]));
	args.AppendArg("-DoRescueFrom");
	args.AppendArg(std::to_string(deepOpts.doRescueFrom));

	for (auto & dagFile : shallowOpts.dagFiles) {
		args.AppendArg("-Dag");
		args.AppendArg(dagFile.c_str());
	}

	if(shallowOpts[shallow::i::MaxIdle] != 0) 
	{
		args.AppendArg("-MaxIdle");
		args.AppendArg(std::to_string(shallowOpts[shallow::i::MaxIdle]));
	}

	if(shallowOpts[shallow::i::MaxJobs] != 0) 
	{
		args.AppendArg("-MaxJobs");
		args.AppendArg(std::to_string(shallowOpts[shallow::i::MaxJobs]));
	}

	if(shallowOpts[shallow::i::MaxPre] != 0) 
	{
		args.AppendArg("-MaxPre");
		args.AppendArg(std::to_string(shallowOpts[shallow::i::MaxPre]));
	}

	if(shallowOpts[shallow::i::MaxPost] != 0) 
	{
		args.AppendArg("-MaxPost");
		args.AppendArg(std::to_string(shallowOpts[shallow::i::MaxPost]));
	}

	if ( shallowOpts.bPostRunSet ) {
		if (shallowOpts[shallow::b::PostRun]) {
			args.AppendArg("-AlwaysRunPost");
		} else {
			args.AppendArg("-DontAlwaysRunPost");
		}
	}

	if(deepOpts[deep::b::UseDagDir])
	{
		args.AppendArg("-UseDagDir");
	}

	if(deepOpts[deep::b::SuppressNotification])
	{
		args.AppendArg("-Suppress_notification");
	}
	else
	{
		args.AppendArg("-Dont_Suppress_notification");
	}

	if ( shallowOpts.doRecovery ) {
		args.AppendArg( "-DoRecov" );
	}

	args.AppendArg("-CsdVersion");
	args.AppendArg(CondorVersion());

	if(deepOpts[deep::b::AllowVersionMismatch]) {
		args.AppendArg("-AllowVersionMismatch");
	}

	if(shallowOpts[shallow::b::DumpRescueDag]) {
		args.AppendArg("-DumpRescue");
	}

	if(deepOpts.bVerbose) {
		args.AppendArg("-Verbose");
	}

	if(deepOpts[deep::b::Force]) {
		args.AppendArg("-Force");
	}

	if(deepOpts.strNotification != "") {
		args.AppendArg("-Notification");
		args.AppendArg(deepOpts.strNotification);
	}

	if(!deepOpts[deep::str::DagmanPath].empty()) {
		args.AppendArg("-Dagman");
		args.AppendArg(deepOpts[deep::str::DagmanPath]);
	}

	if(deepOpts[deep::str::OutfileDir] != "") {
		args.AppendArg("-Outfile_dir");
		args.AppendArg(deepOpts[deep::str::OutfileDir]);
	}

	if(deepOpts[deep::b::UpdateSubmit]) {
		args.AppendArg("-Update_submit");
	}

	if(deepOpts[deep::b::ImportEnv]) {
		args.AppendArg("-Import_env");
	}

	if(!deepOpts[deep::str::GetFromEnv].empty()) {
		args.AppendArg("-Include_env");
		args.AppendArg(deepOpts[deep::str::GetFromEnv]);
	}

	for (auto &kv_pairs : deepOpts.addToEnv) {
		args.AppendArg( "-Insert_env" );
		args.AppendArg(kv_pairs);
	}

	if( shallowOpts[shallow::i::Priority] != 0 ) {
		args.AppendArg("-Priority");
		args.AppendArg(std::to_string(shallowOpts[shallow::i::Priority]));
	}

	if (!shallowOpts[shallow::str::SaveFile].empty()) {
		args.AppendArg("-load_save");
		args.AppendArg(shallowOpts[shallow::str::SaveFile]);
	}

	std::string arg_str,args_error;
	if(!args.GetArgsStringV1WackedOrV2Quoted(arg_str, args_error)) {
		fprintf(stderr,"Failed to insert arguments: %s",args_error.c_str());
		exit(1);
	}
	fprintf(pSubFile, "arguments\t= %s\n", arg_str.c_str());

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Be sure to change MIN_SUBMIT_FILE_VERSION in dagman_main.cpp
	// if the environment passed to condor_dagman changes in an
	// incompatible way!!
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	Env env;
	if ( deepOpts[deep::b::ImportEnv] ) {
		env.Import(ImportFilter);
	}

	for (auto const& kv_pairs : deepOpts.addToEnv) {
		std::string err_msg;
		env.MergeFromV1RawOrV2Quoted(kv_pairs.c_str(),err_msg);
		if (!err_msg.empty()) {
			fprintf(stderr,"Error: Failed to add %s to DAGMan manager jobs environment because %s\n",kv_pairs.c_str(),err_msg.c_str());
			exit(1);
		}
	}
	env.SetEnv("_CONDOR_DAGMAN_LOG", shallowOpts.strDebugLog.c_str());
	env.SetEnv("_CONDOR_MAX_DAGMAN_LOG=0");
	if ( shallowOpts[shallow::str::ScheddDaemonAdFile] != "" ) {
		env.SetEnv("_CONDOR_SCHEDD_DAEMON_AD_FILE",
				   shallowOpts[shallow::str::ScheddDaemonAdFile].c_str());
	}
	if ( shallowOpts[shallow::str::ScheddAddressFile] != "" ) {
		env.SetEnv("_CONDOR_SCHEDD_ADDRESS_FILE",
				   shallowOpts[shallow::str::ScheddAddressFile].c_str());
	}
	if ( shallowOpts[shallow::str::ConfigFile] != "" ) {
		if ( access( shallowOpts[shallow::str::ConfigFile].c_str(), F_OK ) != 0 ) {
			fprintf( stderr, "ERROR: unable to read config file %s "
						"(error %d, %s)\n",
						shallowOpts[shallow::str::ConfigFile].c_str(), errno, strerror(errno) );
			fclose(pSubFile);
			return false;
		}
		env.SetEnv("_CONDOR_DAGMAN_CONFIG_FILE", shallowOpts[shallow::str::ConfigFile].c_str());
	}

	std::string env_str;
	env.getDelimitedStringV2Quoted( env_str );
	fprintf(pSubFile, "environment\t= %s\n",env_str.c_str());

	if ( deepOpts.strNotification != "" ) {    
		fprintf( pSubFile, "notification\t= %s\n",
					deepOpts.strNotification.c_str() );
	}

		// Append user-specified stuff to submit file...

		// ...first, the insert file, if any...
	if ( shallowOpts.appendFile != "" ) {
		FILE *aFile = safe_fopen_wrapper_follow(
					shallowOpts.appendFile.c_str(), "r");
		if ( !aFile ) {
			fprintf( stderr, "ERROR: unable to read submit append file (%s)\n",
					 shallowOpts.appendFile.c_str() );
			return false;
		}

		char *line;
		int lineno = 0;
		while ( (line = getline_trim( aFile, lineno )) != NULL ) {
			fprintf(pSubFile, "%s\n", line);
		}

		fclose( aFile );
	}

		// ...now append lines specified in the DAG file...
	for (auto & dagFileAttrLine : dagFileAttrLines) {
			// Note:  prepending "+" here means that this only works
			// for setting ClassAd attributes.
		fprintf( pSubFile, "My.%s\n", dagFileAttrLine.c_str() );
	}

		// ...now things specified directly on the command line.
	for (auto & appendLine : shallowOpts.appendLines) {
		fprintf( pSubFile, "%s\n", appendLine.c_str() );
	}

	fprintf(pSubFile, "queue\n");

	fclose(pSubFile);

	return true;
}

/** Run condor_submit_dag on the given DAG file.
@param opts: the condor_submit_dag options
@param dagFile: the DAG file to process
@param directory: the directory from which the DAG file should
	be processed (ignored if NULL)
@param priority: the priority of this DAG
@param isRetry: whether this is a retry
@return 0 if successful, 1 if failed
*/
int
DagmanUtils::runSubmitDag( const SubmitDagDeepOptions &deepOpts,
			const char *dagFile, const char *directory, int priority,
			bool isRetry )
{
	int result = 0;

		// Change to the appropriate directory if necessary.
	TmpDir tmpDir;
	std::string errMsg;
	if ( directory ) {
		if ( !tmpDir.Cd2TmpDir( directory, errMsg ) ) {
			fprintf( stderr, "Error (%s) changing to node directory\n",
						errMsg.c_str() );
			result = 1;
			return result;
		}
	}

		// Build up the command line for the recursive run of
		// condor_submit_dag.  We need -no_submit so we don't
		// actually run the subdag now; we need -update_submit
		// so the lower-level .condor.sub file will get
		// updated, in case it came from an earlier version
		// of condor_submit_dag.
	ArgList args;
	args.AppendArg( "condor_submit_dag" );
	args.AppendArg( "-no_submit" );
	args.AppendArg( "-update_submit" );

		// Add in arguments we're passing along.
	if ( deepOpts.bVerbose ) {
		args.AppendArg( "-verbose" );
	}

	if ( deepOpts[deep::b::Force] && !isRetry ) {
		args.AppendArg( "-force" );
	}

	if (deepOpts.strNotification != "" ) {
		args.AppendArg( "-notification" );
		if(deepOpts[deep::b::SuppressNotification]) {
			args.AppendArg( "never" );
		} else { 
			args.AppendArg( deepOpts.strNotification.c_str() );
		}
	}

	if ( !deepOpts[deep::str::DagmanPath].empty() ) {
		args.AppendArg( "-dagman" );
		args.AppendArg( deepOpts[deep::str::DagmanPath].c_str() );
	}

	if ( deepOpts[deep::b::UseDagDir] ) {
		args.AppendArg( "-usedagdir" );
	}

	if ( deepOpts[deep::str::OutfileDir] != "" ) {
		args.AppendArg( "-outfile_dir" );
		args.AppendArg( deepOpts[deep::str::OutfileDir].c_str() );
	}

	args.AppendArg( "-autorescue" );
	args.AppendArg( std::to_string(deepOpts[deep::b::AutoRescue]) );

	if ( deepOpts.doRescueFrom != 0 ) {
		args.AppendArg( "-dorescuefrom" );
		args.AppendArg( std::to_string(deepOpts.doRescueFrom) );
	}

	if ( deepOpts[deep::b::AllowVersionMismatch] ) {
		args.AppendArg( "-allowver" );
	}

	if ( deepOpts[deep::b::ImportEnv] ) {
		args.AppendArg( "-import_env" );
	}

	if ( !deepOpts[deep::str::GetFromEnv].empty() ) {
		args.AppendArg( "-include_env" );
		args.AppendArg( deepOpts[deep::str::GetFromEnv] );
	}

	for (auto &kv_pairs : deepOpts.addToEnv) {
		args.AppendArg( "-insert_env" );
		args.AppendArg(kv_pairs.c_str());
	}

	if ( deepOpts[deep::b::Recurse] ) {
		args.AppendArg( "-do_recurse" );
	}

	if ( deepOpts[deep::b::UpdateSubmit] ) {
		args.AppendArg( "-update_submit" );
	}

	if( priority != 0) {
		args.AppendArg( "-Priority" );
		args.AppendArg( std::to_string(priority) );
	}

	if( deepOpts[deep::b::SuppressNotification] ) {
		args.AppendArg( "-suppress_notification" );
	} else {
		args.AppendArg( "-dont_suppress_notification" );
	}

	args.AppendArg( dagFile );

	std::string cmdLine;
	args.GetArgsStringForDisplay(cmdLine);
	dprintf( D_ALWAYS, "Recursive submit command: <%s>\n",
				cmdLine.c_str() );

		// Now actually run the command.
	int retval = my_system( args );
	if ( retval != 0 ) {
		dprintf( D_ALWAYS, "ERROR: condor_submit_dag -no_submit "
					"failed on DAG file %s.\n", dagFile );
		result = 1;
	}

		// Change back to the directory we started from.
	if ( !tmpDir.Cd2MainDir( errMsg ) ) {
		dprintf( D_ALWAYS, "Error (%s) changing back to original directory\n",
					errMsg.c_str() );
	}

	return result;
}

//---------------------------------------------------------------------------
/** Set up things in deep and shallow options that aren't directly specified
	on the command line.
	@param deepOpts: the condor_submit_dag deep options
	@param shallowOpts: the condor_submit_dag shallow options
	@return 0 if successful, 1 if failed
*/
int
DagmanUtils::setUpOptions( SubmitDagDeepOptions &deepOpts,
			SubmitDagShallowOptions &shallowOpts,
			std::list<std::string> &dagFileAttrLines )
{
	shallowOpts.strLibOut = shallowOpts.primaryDagFile + ".lib.out";
	shallowOpts.strLibErr = shallowOpts.primaryDagFile + ".lib.err";

	if ( deepOpts[deep::str::OutfileDir] != "" ) {
		shallowOpts.strDebugLog = deepOpts[deep::str::OutfileDir] + DIR_DELIM_STRING +
					condor_basename( shallowOpts.primaryDagFile.c_str() );
	} else {
		shallowOpts.strDebugLog = shallowOpts.primaryDagFile;
	}
	shallowOpts.strDebugLog += ".dagman.out";
	shallowOpts.strSchedLog = shallowOpts.primaryDagFile + ".dagman.log";
	shallowOpts.strSubFile = shallowOpts.primaryDagFile + DAG_SUBMIT_FILE_SUFFIX;

	std::string rescueDagBase;

		// If we're running each DAG in its own directory, write any rescue
		// DAG to the current directory, to avoid confusion (since the
		// rescue DAG must be run from the current directory).
	if ( deepOpts[deep::b::UseDagDir] ) {
		if ( !condor_getcwd( rescueDagBase ) ) {
			fprintf( stderr, "ERROR: unable to get cwd: %d, %s\n",
					errno, strerror(errno) );
			return 1;
		}
		rescueDagBase += DIR_DELIM_STRING;
		rescueDagBase += condor_basename(shallowOpts.primaryDagFile.c_str());
	} else {
		rescueDagBase = shallowOpts.primaryDagFile;
	}

		// If we're running multiple DAGs, put "_multi" in the rescue
		// DAG name to indicate that the rescue DAG is for *all* of
		// the DAGs we're running.
	if ( shallowOpts.dagFiles.size() > 1 ) {
		rescueDagBase += "_multi";
	}
	shallowOpts.strRescueFile = rescueDagBase + ".rescue";

	shallowOpts.strLockFile = shallowOpts.primaryDagFile + ".lock";

	if (deepOpts[deep::str::DagmanPath].empty()) {
		deepOpts[deep::str::DagmanPath] = which( dagman_exe );
	}

	if (deepOpts[deep::str::DagmanPath].empty())
	{
		fprintf( stderr, "ERROR: can't find %s in PATH, aborting.\n",
				 dagman_exe );
		return 1;
	}
	std::string msg;
	if ( !processDagCommands(deepOpts, shallowOpts, dagFileAttrLines, msg) ) {
		fprintf( stderr, "ERROR: %s\n", msg.c_str() );
		return 1;
	}

	return 0;
}

/** Read the DAG files for DAG commands that need to be parsed
	before the submission of the DAGMan job proper because the
	commands effect the produced .condor.sub file
	@param deepOpts: DAGMan deep options struct
	@param shallowOpts: DAGMan shallow options struct
	@param attrLines: list of strings of attributes to be added
	                  to the DAGMan job propers classad
	@param errMsg: error message
	@return true if the operation succeeded; otherwise false
*/
bool
DagmanUtils::processDagCommands( SubmitDagDeepOptions& deepOpts, SubmitDagShallowOptions& shallowOpts,
                                 std::list<std::string> &attrLines, std::string &errMsg )
{
	bool result = true;
	// Note: destructor will change back to original directory.
	TmpDir dagDir;

	for (auto & dagFile : shallowOpts.dagFiles) {
		std::string newDagFile;
		// Switch to DAG Dir if needed
		if (deepOpts[deep::b::UseDagDir]) {
			std::string	tmpErrMsg;
			if ( !dagDir.Cd2TmpDirFile( dagFile.c_str(), tmpErrMsg ) ) {
				errMsg = "Unable to change to DAG directory " + tmpErrMsg;
				return false;
			}
			newDagFile = condor_basename( dagFile.c_str() );
		} else {
			newDagFile = dagFile.c_str();
		}

		std::list<std::string> configFiles;
		// Note: destructor will close file.
		MultiLogFiles::FileReader reader;
		errMsg = reader.Open( newDagFile );
		if ( errMsg != "" ) {
			return false;
		}

		//Read DAG file
		std::string logicalLine;
		while ( reader.NextLogicalLine( logicalLine ) ) {
			if ( logicalLine != "" ) {
				trim(logicalLine);
				StringTokenIterator tokens(logicalLine);
				const char* cmd = tokens.first();
				// Parse CONFIG command
				if (strcasecmp(cmd, "CONFIG") == MATCH) {
					const char* newFile = tokens.remain();
					while (newFile && isspace(*newFile) && *newFile != '\0') { newFile++; }
					if (!newFile || *newFile == '\0') {
						AppendError(errMsg, "Improperly-formatted file: value missing after keyword CONFIG");
						result = false;
					} else {
						bool alreadyInList = false;
						for (auto & configFile : configFiles)
							if (strcmp(configFile.c_str(), newFile) == MATCH)
								alreadyInList = true;
						if (!alreadyInList)
							configFiles.emplace_back(newFile);
					}
				// Parse SET_JOB_ATTR command
				} else if (strcasecmp(cmd, "SET_JOB_ATTR") == MATCH) {
					const char* attr = tokens.remain();
					while (attr && isspace(*attr) && *attr != '\0') { attr++; }
					if (!attr || *attr == '\0') {
						AppendError(errMsg, "Improperly-formatted file: value missing after keyword SET_JOB_ATTR");
						result = false;
					} else {
						attrLines.emplace_back(attr);
					}
				// Parse ENV command
				} else if (strcasecmp(cmd, "ENV") == MATCH) {
					const char* type = tokens.next();
					// Parse GET option
					if (strcasecmp(type, "GET") == MATCH) {
						const char* remain = tokens.remain();
						while (remain && isspace(*remain) && *remain != '\0') { remain++; }
						if (!remain || *remain == '\0') {
							AppendError(errMsg, "Improperly-formatted file: environment variables missing after ENV GET");
							result = false;
						} else {
							StringTokenIterator vars(remain);
							for (auto& var : vars) {
								if (!deepOpts[deep::str::GetFromEnv].empty()) { deepOpts[deep::str::GetFromEnv] += ","; }
								deepOpts[deep::str::GetFromEnv] += var;
							}
						}
					// Parse SET option
					} else if (strcasecmp(type, "SET") == MATCH) {
						const char* info = tokens.remain();
						while (info && isspace(*info) && *info != '\0') { info++; }
						if (!info || *info == '\0') {
							AppendError(errMsg, "Improperly-formatted file: environment variables missing after ENV SET");
							result = false;
						} else { deepOpts.addToEnv.push_back(info); }
					// Else error
					} else {
						AppendError(errMsg, "Improperly-formatted file: sub-command (SET or GET) missing after keyword ENV");
						result = false;
					}
				}
			}
		}

		reader.Close();
		// Verify config files (only 1 file given)
		for (auto & configfile_it : configFiles) {
			std::string cfgFileMS = configfile_it.c_str();
			std::string tmpErrMsg;
			if ( MakePathAbsolute( cfgFileMS, tmpErrMsg ) ) {
				if (shallowOpts[shallow::str::ConfigFile].empty()) {
					shallowOpts[shallow::str::ConfigFile] = cfgFileMS;
				} else if ( shallowOpts[shallow::str::ConfigFile] != cfgFileMS ) {
					AppendError( errMsg, "Conflicting DAGMan config files specified: " +
					                     shallowOpts[shallow::str::ConfigFile] + " and " + cfgFileMS );
					result = false;
				}
			} else {
				AppendError( errMsg, tmpErrMsg );
				result = false;
			}
		}

		// Switch back to original directory
		std::string tmpErrMsg;
		if ( !dagDir.Cd2MainDir( tmpErrMsg ) ) {
			AppendError( errMsg, "Unable to change to original directory " + tmpErrMsg );
			result = false;
		}
	}

	return result;
}

/** Make the given path into an absolute path, if it is not already.
	@param filePath: the path to make absolute (filePath is changed)
	@param errMsg: a std::string to receive any error message.
	@return true if the operation succeeded; otherwise false
*/
bool
DagmanUtils::MakePathAbsolute(std::string &filePath, std::string &errMsg)
{
	bool result = true;

	if ( !fullpath( filePath.c_str() ) ) {
		std::string currentDir;
		if ( !condor_getcwd( currentDir ) ) {
			formatstr( errMsg, "condor_getcwd() failed with errno %d (%s) at %s:%d",
					   errno, strerror(errno), __FILE__, __LINE__ );
			result = false;
		}

		filePath = currentDir + DIR_DELIM_STRING + filePath;
	}

	return result;
}

/** Finds the number of the last existing rescue DAG file for the
	given "primary" DAG.
	@param primaryDagFile The primary DAG file name
	@param multiDags Whether we have multiple DAGs
	@param maxRescueDagNum the maximum legal rescue DAG number
	@return The number of the last existing rescue DAG (0 if there
		is none)
*/
int
DagmanUtils::FindLastRescueDagNum( const char *primaryDagFile, bool multiDags,
			int maxRescueDagNum )
{
	int lastRescue = 0;

	for ( int test = 1; test <= maxRescueDagNum; test++ ) {
		std::string testName = RescueDagName( primaryDagFile, multiDags,
					test );
		if ( access( testName.c_str(), F_OK ) == 0 ) {
			if ( test > lastRescue + 1 ) {
					// This should probably be a fatal error if
					// DAGMAN_USE_STRICT is set, but I'm avoiding
					// that for now because the fact that this code
					// is used in both condor_dagman and condor_submit_dag
					// makes that harder to implement. wenger 2011-01-28
				dprintf( D_ALWAYS, "Warning: found rescue DAG "
							"number %d, but not rescue DAG number %d\n",
							test, test - 1);
			}
			lastRescue = test;
		}
	}
	
	if ( lastRescue >= maxRescueDagNum ) {
		dprintf( D_ALWAYS,
					"Warning: FindLastRescueDagNum() hit maximum "
					"rescue DAG number: %d\n", maxRescueDagNum );
	}

	return lastRescue;
}

/** Creates a rescue DAG name, given a primary DAG name and rescue
	DAG number
	@param primaryDagFile The primary DAG file name
	@param multiDags Whether we have multiple DAGs
	@param rescueDagNum The rescue DAG number
	@return The full name of the rescue DAG
*/
std::string
DagmanUtils::RescueDagName(const char *primaryDagFile, bool multiDags,
			int rescueDagNum)
{
	ASSERT( rescueDagNum >= 1 );

	std::string fileName(primaryDagFile);
	if ( multiDags ) {
		fileName += "_multi";
	}
	fileName += ".rescue";
	formatstr_cat(fileName, "%.3d", rescueDagNum);

	return fileName;
}

/** Renames all rescue DAG files for this primary DAG after the
	given one (as long as the numbers are contiguous).	For example,
	if rescueDagNum is 3, we will rename .rescue4, .rescue5, etc.
	@param primaryDagFile The primary DAG file name
	@param multiDags Whether we have multiple DAGs
	@param rescueDagNum The rescue DAG number to rename *after*
	@param maxRescueDagNum the maximum legal rescue DAG number
*/
void
DagmanUtils::RenameRescueDagsAfter(const char *primaryDagFile, bool multiDags,
			int rescueDagNum, int maxRescueDagNum)
{
		// Need to allow 0 here so condor_submit_dag -f can rename all
		// rescue DAGs.
	ASSERT( rescueDagNum >= 0 );

	dprintf( D_ALWAYS, "Renaming rescue DAGs newer than number %d\n",
				rescueDagNum );

	int firstToDelete = rescueDagNum + 1;
	int lastToDelete = FindLastRescueDagNum( primaryDagFile, multiDags,
				maxRescueDagNum );

	for ( int rescueNum = firstToDelete; rescueNum <= lastToDelete;
				rescueNum++ ) {
		std::string rescueDagName = RescueDagName( primaryDagFile, multiDags,
					rescueNum );
		dprintf( D_ALWAYS, "Renaming %s\n", rescueDagName.c_str() );
		std::string newName = rescueDagName + ".old";
			// Unlink here to be safe on Windows.
		tolerant_unlink( newName.c_str() );
		if ( rename( rescueDagName.c_str(), newName.c_str() ) != 0 ) {
			EXCEPT( "Fatal error: unable to rename old rescue file "
						"%s: error %d (%s)\n", rescueDagName.c_str(),
						errno, strerror( errno ) );
		}
	}
}

/** Generates the halt file name based on the primary DAG name.
	@return The halt file name.
*/
std::string
DagmanUtils::HaltFileName( const std::string &primaryDagFile )
{
	std::string haltFile = primaryDagFile + ".halt";

	return haltFile;
}

/** Attempts to unlink the given file, and prints an appropriate error
	message if this fails (but doesn't return an error, so only call
	this if a failure of the unlink is okay).
	@param pathname The path of the file to unlink
*/
void
DagmanUtils::tolerant_unlink( const char *pathname )
{
	if ( unlink( pathname ) != 0 ) {
		if ( errno == ENOENT ) {
			dprintf( D_SYSCALLS,
						"Warning: failure (%d (%s)) attempting to unlink file %s\n",
						errno, strerror( errno ), pathname );
		} else {
			dprintf( D_ALWAYS,
						"Error (%d (%s)) attempting to unlink file %s\n",
						errno, strerror( errno ), pathname );

		}
	}
}

//---------------------------------------------------------------------------
bool 
DagmanUtils::fileExists(const std::string &strFile)
{
	int fd = safe_open_wrapper_follow(strFile.c_str(), O_RDONLY);
	if (fd == -1)
		return false;
	close(fd);
	return true;
}

//---------------------------------------------------------------------------
bool 
DagmanUtils::ensureOutputFilesExist(const SubmitDagDeepOptions &deepOpts,
			SubmitDagShallowOptions &shallowOpts)
{
	int maxRescueDagNum = param_integer("DAGMAN_MAX_RESCUE_NUM",
				MAX_RESCUE_DAG_DEFAULT, 0, ABS_MAX_RESCUE_DAG_NUM);

	if (deepOpts.doRescueFrom > 0)
	{
		std::string rescueDagName = RescueDagName(shallowOpts.primaryDagFile.c_str(),
				shallowOpts.dagFiles.size() > 1, deepOpts.doRescueFrom);
		if (!fileExists(rescueDagName))
		{
			fprintf( stderr, "-dorescuefrom %d specified, but rescue "
						"DAG file %s does not exist!\n", deepOpts.doRescueFrom,
						rescueDagName.c_str() );
			return false;
		}
	}

		// Get rid of the halt file (if one exists).
	tolerant_unlink( HaltFileName( shallowOpts.primaryDagFile ).c_str() );

	if (deepOpts[deep::b::Force])
	{
		tolerant_unlink(shallowOpts.strSubFile.c_str());
		tolerant_unlink(shallowOpts.strSchedLog.c_str());
		tolerant_unlink(shallowOpts.strLibOut.c_str());
		tolerant_unlink(shallowOpts.strLibErr.c_str());
		RenameRescueDagsAfter(shallowOpts.primaryDagFile.c_str(),
					shallowOpts.dagFiles.size() > 1, 0, maxRescueDagNum);
	}

		// Check whether we're automatically running a rescue DAG -- if
		// so, allow things to continue even if the files generated
		// by condor_submit_dag already exist.
	bool autoRunningRescue = false;
	if (deepOpts[deep::b::AutoRescue]) {
		int rescueDagNum = FindLastRescueDagNum(shallowOpts.primaryDagFile.c_str(),
					shallowOpts.dagFiles.size() > 1, maxRescueDagNum);
		if (rescueDagNum > 0) {
			printf("Running rescue DAG %d\n", rescueDagNum);
			autoRunningRescue = true;
		}
	}

	bool bHadError = false;
		// If not running a rescue DAG, check for existing files
		// generated by condor_submit_dag...
	if (!autoRunningRescue && deepOpts.doRescueFrom < 1 &&
				!deepOpts[deep::b::UpdateSubmit] && shallowOpts[shallow::str::SaveFile].empty()) {
		if (fileExists(shallowOpts.strSubFile))
		{
			fprintf( stderr, "ERROR: \"%s\" already exists.\n",
					 shallowOpts.strSubFile.c_str() );
			bHadError = true;
		}
		if (fileExists(shallowOpts.strLibOut))
		{
			fprintf( stderr, "ERROR: \"%s\" already exists.\n",
					 shallowOpts.strLibOut.c_str() );
			bHadError = true;
		}
		if (fileExists(shallowOpts.strLibErr))
		{
			fprintf( stderr, "ERROR: \"%s\" already exists.\n",
					 shallowOpts.strLibErr.c_str() );
			bHadError = true;
		}
		if (fileExists(shallowOpts.strSchedLog))
		{
			fprintf( stderr, "ERROR: \"%s\" already exists.\n",
					 shallowOpts.strSchedLog.c_str() );
			bHadError = true;
		}
	}

		// This is checking for the existance of an "old-style" rescue
		// DAG file.
	if (!deepOpts[deep::b::AutoRescue] && deepOpts.doRescueFrom < 1 &&
				fileExists(shallowOpts.strRescueFile))
	{
		fprintf( stderr, "ERROR: \"%s\" already exists.\n",
				 shallowOpts.strRescueFile.c_str() );
		fprintf( stderr, "	You may want to resubmit your DAG using that "
				 "file, instead of \"%s\"\n", shallowOpts.primaryDagFile.c_str());
		fprintf( stderr, "	Look at the HTCondor manual for details about DAG "
				 "rescue files.\n" );
		fprintf( stderr, "	Please investigate and either remove \"%s\",\n",
				 shallowOpts.strRescueFile.c_str() );
		fprintf( stderr, "	or use it as the input to condor_submit_dag.\n" );
		bHadError = true;
	}

	if (bHadError) 
	{
		fprintf( stderr, "\nSome file(s) needed by %s already exist.  ",
				 dagman_exe );
		if(!usingPythonBindings) {
			fprintf( stderr, "Either rename them,\nuse the \"-f\" option to "
				 "force them to be overwritten, or use\n"
				 "the \"-update_submit\" option to update the submit "
				 "file and continue.\n" );
		}
		else {
			fprintf( stderr, "Either rename them,\nor set the { \"force\" : True }"
				" option to force them to be overwritten.\n" );
		}
		return false;
	}

	return true;
}

//-----------------------------------------------------------------------------
int 
DagmanUtils::popen (ArgList &args) {
	std::string cmd; // for debug output
	args.GetArgsStringForDisplay(cmd);
    dprintf( D_ALWAYS, "Running: %s\n", cmd.c_str() );

	FILE *fp = my_popen( args, "r", MY_POPEN_OPT_WANT_STDERR );

    int r = 0;
    if (fp == NULL || (r = my_pclose(fp) & 0xff) != 0) {
		dprintf( D_ERROR, "Warning: failure: %s\n", cmd.c_str() );
		if( fp != NULL ) {
			dprintf ( D_ALWAYS,
						"\t(my_pclose() returned %d (errno %d, %s))\n",
						r, errno, strerror( errno ) );
		} else {
			dprintf ( D_ALWAYS,
						"\t(my_popen() returned NULL (errno %d, %s))\n",
						errno, strerror( errno ) );
			r = -1;
		}
    }
    return r;
}

//-----------------------------------------------------------------------------
int
DagmanUtils::create_lock_file(const char *lockFileName, bool abortDuplicates) {
	int result = 0;

	FILE *fp = safe_fopen_wrapper_follow( lockFileName, "w" );
	if ( fp == NULL ) {
		dprintf( D_ALWAYS,
					"ERROR: could not open lock file %s for writing.\n",
					lockFileName);
		result = -1;
	}

		//
		// Create the ProcessId object.
		//
	ProcessId *procId = NULL;
	if ( result == 0 && abortDuplicates ) {
		int status;
		int precision_range = 1;
		if ( ProcAPI::createProcessId( daemonCore->getpid(), procId,
					status, &precision_range ) != PROCAPI_SUCCESS ) {
			dprintf( D_ALWAYS, "ERROR: ProcAPI::createProcessId() "
						"failed; %d\n", status );
			result = -1;
		}
	}

		//
		// Write out the ProcessId object.
		//
	if ( result == 0 && abortDuplicates ) {
		if ( procId->write( fp ) != ProcessId::SUCCESS ) {
			dprintf( D_ALWAYS, "ERROR: ProcessId::write() failed\n");
			result = -1;
		}
	}

		//
		// Confirm the ProcessId object's uniqueness.
		//
	if ( result == 0 && abortDuplicates ) {
		int status;
		if ( ProcAPI::confirmProcessId( *procId, status ) !=
					PROCAPI_SUCCESS ) {
			dprintf( D_ERROR, "Warning: ProcAPI::"
						"confirmProcessId() failed; %d\n", status );
		} else {
			if ( !procId->isConfirmed() ) {
				dprintf( D_ERROR, "Warning: ProcessId not "
							"confirmed unique\n" );
			} else {

					//
					// Write out the confirmation.
					//
				if ( procId->writeConfirmationOnly( fp ) !=
							ProcessId::SUCCESS ) {
					dprintf( D_ERROR, "ERROR: ProcessId::"
								"writeConfirmationOnly() failed\n");
					result = -1;
				}
			}
		}
	}

	delete procId;

	if ( fp != NULL ) {
		if ( fclose( fp ) != 0 ) {
			dprintf( D_ALWAYS, "ERROR: closing lock "
						"file failed with errno %d (%s)\n", errno,
						strerror( errno ) );
		}
	}

	return result;
}

//-----------------------------------------------------------------------------
int
DagmanUtils::check_lock_file(const char *lockFileName) {
	int result = 0;

	FILE *fp = safe_fopen_wrapper_follow( lockFileName, "r" );
	if ( fp == NULL ) {
		dprintf( D_ALWAYS,
					"ERROR: could not open lock file %s for reading.\n",
					lockFileName );
		result = -1;
	}

	ProcessId *procId = NULL;
	if ( result != -1 ) {
		int status;
		procId = new ProcessId( fp, status );
		if ( status != ProcessId::SUCCESS ) {
			dprintf( D_ALWAYS, "ERROR: unable to create ProcessId "
						"object from lock file %s\n", lockFileName );
			result = -1;
		}
	}

	if ( result != -1 ) {
		int status;
		int aliveResult = ProcAPI::isAlive( *procId, status );
		if ( aliveResult != PROCAPI_SUCCESS ) {
			dprintf( D_ALWAYS, "ERROR: failed to determine "
						"whether DAGMan that wrote lock file is alive\n" );
			result = -1;
		} else {

			if ( status == PROCAPI_ALIVE ) {
				dprintf( D_ALWAYS,
						"Duplicate DAGMan PID %d is alive; this DAGMan "
						"should abort.\n", procId->getPid() );
				result = 1;

			} else if ( status == PROCAPI_DEAD ) {
				dprintf( D_ALWAYS,
						"Duplicate DAGMan PID %d is no longer alive; "
						"this DAGMan should continue.\n",
						procId->getPid() );
				result = 0;

			} else if ( status == PROCAPI_UNCERTAIN ) {
				dprintf( D_ALWAYS,
						"Duplicate DAGMan PID %d *may* be alive; this "
						"DAGMan is continuing, but this will cause "
						"problems if the duplicate DAGMan is alive.\n",
						procId->getPid() );
				result = 0;

			} else {
				EXCEPT( "Illegal ProcAPI::isAlive() status value: %d",
							status );
			}
		}
	}

	delete procId;

	if ( fp != NULL ) {
		if ( fclose( fp ) != 0 ) {
			dprintf( D_ALWAYS, "ERROR: closing lock "
						"file failed with errno %d (%s)\n", errno,
						strerror( errno ) );
		}
	}

	return result;
}


const std::string &
SubmitDagShallowOptions::operator[]( shallow::str opt ) const {
    return stringOpts[opt._to_integral()];
}

bool
SubmitDagShallowOptions::operator[]( shallow::b opt ) const {
    return boolOpts[opt._to_integral()];
}

int
SubmitDagShallowOptions::operator[]( shallow::i opt ) const {
    return intOpts[opt._to_integral()];
}

std::string &
SubmitDagShallowOptions::operator[]( shallow::str opt ) {
    return stringOpts[opt._to_integral()];
}

bool &
SubmitDagShallowOptions::operator[]( shallow::b opt ) {
    return boolOpts[opt._to_integral()];
}

int &
SubmitDagShallowOptions::operator[]( shallow::i opt ) {
    return intOpts[opt._to_integral()];
}

/*

//
// We don't need these, because the Python bindings have to do type-
// conversion for the values, but in case we ever do, here they are.
// You can do the same thing for the deep options.
//

bool
SubmitDagShallowOptions::set( const char * key, const std::string & value ) {
    auto maybe = shallow::str::_from_string_nocase_nothrow(key);
    if(! maybe) { return false; }
    stringOpts[*maybe] = value;
    return true;
}

bool
SubmitDagShallowOptions::set( const char * key, bool value ) {
    auto maybe = shallow::b::_from_string_nocase_nothrow(key);
    if(! maybe) { return false; }
    boolOpts[*maybe] = value;
    return true;
}

bool
SubmitDagShallowOptions::set( const char * key, int value ) {
    auto maybe = shallow::i::_from_string_nocase_nothrow(key);
    if(! maybe) { return false; }
    intOpts[*maybe] = value;
    return true;
}
*/

const std::string &
SubmitDagDeepOptions::operator[]( deep::str opt ) const {
    return stringOpts[opt._to_integral()];
}

bool
SubmitDagDeepOptions::operator[]( deep::b opt ) const {
    return boolOpts[opt._to_integral()];
}

std::string &
SubmitDagDeepOptions::operator[]( deep::str opt ) {
    return stringOpts[opt._to_integral()];
}

bool &
SubmitDagDeepOptions::operator[]( deep::b opt ) {
    return boolOpts[opt._to_integral()];
}
