/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.math.analysis;

import org.apache.commons.math.ConvergenceException;
import org.apache.commons.math.MathException;

import junit.framework.TestCase;

/**
 * @version $Revision: 566833 $ $Date: 2007-08-16 13:36:33 -0700 (Thu, 16 Aug 2007) $
 */
public class UnivariateRealSolverUtilsTest extends TestCase {
    
    protected UnivariateRealFunction sin = new SinFunction();
    
    public void testSolveNull() throws MathException {
        try {
            UnivariateRealSolverUtils.solve(null, 0.0, 4.0);
            fail();
        } catch(IllegalArgumentException ex){
            // success
        }
    }
    
    public void testSolveBadParameters() throws MathException {
        try { // bad endpoints
            UnivariateRealSolverUtils.solve(sin,0.0, 4.0, 4.0); 
        } catch (IllegalArgumentException ex) {
            // expected
        }    
        try { // bad accuracy
            UnivariateRealSolverUtils.solve(sin, 0.0, 4.0, 0.0); 
        } catch (IllegalArgumentException ex) {
            // expected
        }        
    }
    
    public void testSolveSin() throws MathException {     
        double x = UnivariateRealSolverUtils.solve(sin, 1.0, 4.0);
        assertEquals(Math.PI, x, 1.0e-4);
    }
    
    public void testSolveAccuracyNull()  throws MathException {
        try {
            double accuracy = 1.0e-6;
            UnivariateRealSolverUtils.solve(null, 0.0, 4.0, accuracy);
            fail();
        } catch(IllegalArgumentException ex){
            // success
        }
    }
    
    public void testSolveAccuracySin() throws MathException {
        double accuracy = 1.0e-6;
        double x = UnivariateRealSolverUtils.solve(sin, 1.0,
                4.0, accuracy);
        assertEquals(Math.PI, x, accuracy);
    }
    
    public void testSolveNoRoot() throws MathException {
        try {
            UnivariateRealSolverUtils.solve(sin, 1.0, 1.5);  
            fail("Expecting IllegalArgumentException ");  
        } catch (IllegalArgumentException ex) {
            // expected
        }
    }
    
    public void testBracketSin() throws MathException {
        double[] result = UnivariateRealSolverUtils.bracket(sin, 
                0.0, -2.0, 2.0);
        assertTrue(sin.value(result[0]) < 0);
        assertTrue(sin.value(result[1]) > 0);
    }
    
    public void testBracketCornerSolution() throws MathException {
        try {
            UnivariateRealSolverUtils.bracket(sin, 1.5, 0, 2.0); 
            fail("Expecting ConvergenceException");
        } catch (ConvergenceException ex) {
            // expected
        }
    }
    
    public void testBadParameters() throws MathException {
        try { // null function
            UnivariateRealSolverUtils.bracket(null, 1.5, 0, 2.0);
            fail("Expecting IllegalArgumentException");
        } catch (IllegalArgumentException ex) {
            // expected
        }
        try { // initial not between endpoints
            UnivariateRealSolverUtils.bracket(sin, 2.5, 0, 2.0);
            fail("Expecting IllegalArgumentException");
        } catch (IllegalArgumentException ex) {
            // expected
        }
        try { // endpoints not valid
            UnivariateRealSolverUtils.bracket(sin, 1.5, 2.0, 1.0);
            fail("Expecting IllegalArgumentException");
        } catch (IllegalArgumentException ex) {
            // expected
        }
        try { // bad maximum iterations
            UnivariateRealSolverUtils.bracket(sin, 1.5, 0, 2.0, 0);
            fail("Expecting IllegalArgumentException");
        } catch (IllegalArgumentException ex) {
            // expected
        }        
    }
    
}
