/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.math.estimation;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Simple implementation of the {@link EstimationProblem
 * EstimationProblem} interface for boilerplate data handling.
 * <p>This class <em>only</em> handles parameters and measurements
 * storage and unbound parameters filtering. It does not compute
 * anything by itself. It should either be used with measurements
 * implementation that are smart enough to know about the
 * various parameters in order to compute the partial derivatives
 * appropriately. Since the problem-specific logic is mainly related to
 * the various measurements models, the simplest way to use this class
 * is by extending it and using one internal class extending
 * {@link WeightedMeasurement WeightedMeasurement} for each measurement
 * type. The instances of the internal classes would have access to the
 * various parameters and their current estimate.</p>

 * @version $Revision: 627989 $ $Date: 2008-02-15 03:04:02 -0700 (Fri, 15 Feb 2008) $
 * @since 1.2

 */
public class SimpleEstimationProblem implements EstimationProblem {

    /**
     * Build an empty instance without parameters nor measurements.
     */
    public SimpleEstimationProblem() {
        parameters   = new ArrayList();
        measurements = new ArrayList();
    }

    /** 
     * Get all the parameters of the problem.
     * @return parameters
     */
    public EstimatedParameter[] getAllParameters() {
        return (EstimatedParameter[]) parameters.toArray(new EstimatedParameter[parameters.size()]);
    }

    /** 
     * Get the unbound parameters of the problem.
     * @return unbound parameters
     */
    public EstimatedParameter[] getUnboundParameters() {

        // filter the unbound parameters
        List unbound = new ArrayList(parameters.size());
        for (Iterator iterator = parameters.iterator(); iterator.hasNext();) {
            EstimatedParameter p = (EstimatedParameter) iterator.next();
            if (! p.isBound()) {
                unbound.add(p);
            }
        }

        // convert to an array
        return (EstimatedParameter[]) unbound.toArray(new EstimatedParameter[unbound.size()]);
        
    }

    /** 
     * Get the measurements of an estimation problem.
     * @return measurements
     */
    public WeightedMeasurement[] getMeasurements() {
        return (WeightedMeasurement[]) measurements.toArray(new WeightedMeasurement[measurements.size()]);
    }

    /** Add a parameter to the problem.
     * @param p parameter to add
     */
    protected void addParameter(EstimatedParameter p) {
        parameters.add(p);
    }

    /**
     * Add a new measurement to the set.
     * @param m measurement to add
     */
    protected void addMeasurement(WeightedMeasurement m) {
        measurements.add(m);
    }

    /** Estimated parameters. */
    private final List parameters;

    /** Measurements. */
    private final List measurements;

}
