import threading
import Queue
if 'Message' not in globals():
    from py.__.execnet.message import Message

class RemoteError(EOFError):
    """ Contains an Exceptions from the other side. """
    def __init__(self, formatted):
        self.formatted = formatted
        EOFError.__init__(self)

    def __str__(self):
        return self.formatted

    def __repr__(self):
        return "%s: %s" %(self.__class__.__name__, self.formatted)

class Channel(object):
    """Communication channel between two possibly remote threads of code. """
    RemoteError = RemoteError
    def __init__(self, gateway, id, receiver=None):
        assert isinstance(id, int)
        self.gateway = gateway
        self.id = id
        self._receiver = receiver 
        self._items = Queue.Queue()
        self._closeevent = threading.Event()

    def __repr__(self):
        flag = self.isclosed() and "closed" or "open"
        return "<Channel id=%d %s>" % (self.id, flag)

    #
    # internal methods, called from the receiver thread 
    #
    def _local_close(self, stickyerror=EOFError()):
        self.gateway._del_channelmapping(self.id)
        self._stickyerror = stickyerror
        self._items.put(ENDMARKER) 
        self._closeevent.set()

    def _local_receivechannel(self, newid):
        """ receive a remotely created new (sub)channel. """
        # executes in receiver thread 
        newchannel = Channel(self.gateway, newid)
        self.gateway.channelfactory[newid] = newchannel
        self._local_receivedata(newchannel) 

    def _local_receivedata(self, data): 
        # executes in receiver thread 
        if self._receiver is not None: 
            self._receiver(data) 
        else: 
            self._items.put(data) 

    def _local_schedulexec(self, sourcetask): 
        # executes in receiver thread 
        self.gateway._local_schedulexec(channel=self, sourcetask=sourcetask) 

    #
    # public API for channel objects 
    #
    def isclosed(self):
        """ return True if the channel is closed. A closed 
            channel may still hold items. 
        """ 
        return self._closeevent.isSet()

    def makefile(self, mode='w', proxyclose=False):
        """ return a file-like object.  Only supported mode right
            now is 'w' for binary writes.  If you want to have
            a subsequent file.close() mean to close the channel
            as well, then pass proxyclose=True. 
        """ 
        assert mode == 'w', "mode %r not availabe" %(mode,)
        return ChannelFile(channel=self, proxyclose=proxyclose)

    def close(self, error=None):
        """ close down this channel on both sides. """
        if self.id in self.gateway.channelfactory:
            put = self.gateway._outgoing.put
            if error is not None:
                put(Message.CHANNEL_CLOSE_ERROR(self.id, str(error)))
            else:
                put(Message.CHANNEL_CLOSE(self.id))
            self._local_close()

    def waitclose(self, timeout):
        """ wait until this channel is closed.  A closed
        channel may still hold receiveable items.  waitclose()
        reraises exceptions from executing code on the other 
        side as channel.RemoteErrors containing a a textual 
        representation of the remote traceback.
        """
        self._closeevent.wait(timeout=timeout)
        if not self._closeevent.isSet():
            raise IOError, "Timeout"
        if isinstance(self._stickyerror, self.RemoteError):
            raise self._stickyerror

    def send(self, item):
        """sends the given item to the other side of the channel,
        possibly blocking if the sender queue is full.
        Note that an item needs to be marshallable.
        """
        if self.isclosed(): 
            raise IOError, "cannot send to %r" %(self,) 
        if isinstance(item, Channel):
            data = Message.CHANNEL_NEW(self.id, item.id)
        else:
            data = Message.CHANNEL_DATA(self.id, item)
        self.gateway._outgoing.put(data)

    def receive(self):
        """receives an item that was sent from the other side,
        possibly blocking if there is none.
        Note that exceptions from the other side will be
        reraised as channel.RemoteError exceptions containing
        a textual representation of the remote traceback.
        """
        if self._receiver: 
            raise IOError("calling receive() on channel with receiver callback")
        x = self._items.get()
        if x is ENDMARKER: 
            self._items.put(x)  # for other receivers 
            raise self._stickyerror 
        else: 
            return x
    
    def __iter__(self):
        return self 

    def next(self): 
        try:
            return self.receive()
        except EOFError: 
            raise StopIteration 

#
# helpers
#

ENDMARKER = object() 

class ChannelFactory(object):
    def __init__(self, gateway, startcount=1):
        self._dict = dict()
        self._writelock = threading.Lock()
        self.gateway = gateway
        self.count = startcount

    def new(self, id=None, receiver=None):
        """ create a new Channel with 'id' (or create new id if None). """
        self._writelock.acquire()
        try:
            if id is None:
                id = self.count
                self.count += 2
            channel = Channel(self.gateway, id, receiver=receiver)
            self._dict[id] = channel
            return channel
        finally:
            self._writelock.release()

    def __contains__(self, key):
        return key in self._dict

    def values(self):
        self._writelock.acquire()
        try:
            return self._dict.values()
        finally:
            self._writelock.release()

    def __getitem__(self, key):
        return self._dict[key]

    def __setitem__(self, key, value):
        self._writelock.acquire()
        try:
            self._dict[key] = value
        finally:
            self._writelock.release()
    def __delitem__(self, key):
        self._writelock.acquire()
        try:
            del self._dict[key]
        finally:
            self._writelock.release()


class ChannelFile:
    def __init__(self, channel, proxyclose=True):
        self.channel = channel
        self._proxyclose = proxyclose 

    def write(self, out):
        self.channel.send(out)

    def flush(self):
        pass

    def close(self):
        if self._proxyclose: 
            self.channel.close()

    def __repr__(self):
        state = self.channel.isclosed() and 'closed' or 'open'
        return '<ChannelFile %d %s>' %(self.channel.id, state) 

