# -*- coding: utf-8 -*-

#-------------------------------------------------------------------------------

# This file is part of Code_Saturne, a general-purpose CFD tool.
#
# Copyright (C) 1998-2016 EDF S.A.
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
# Street, Fifth Floor, Boston, MA 02110-1301, USA.

#-------------------------------------------------------------------------------

"""
This module contains the following classes:
- BoundaryConditionsRoughWallView
"""

#-------------------------------------------------------------------------------
# Standard modules
#-------------------------------------------------------------------------------

import string, logging

#-------------------------------------------------------------------------------
# Third-party modules
#-------------------------------------------------------------------------------

from PyQt4.QtCore import *
from PyQt4.QtGui  import *

#-------------------------------------------------------------------------------
# Application modules import
#-------------------------------------------------------------------------------

from code_saturne.Base.Toolbox import GuiParam
from code_saturne.Base.QtPage import DoubleValidator, ComboModel, from_qvariant

from code_saturne.Pages.BoundaryConditionsRoughWallForm import Ui_BoundaryConditionsRoughWallForm

#-------------------------------------------------------------------------------
# log config
#-------------------------------------------------------------------------------

logging.basicConfig()
log = logging.getLogger("BoundaryConditionsRoughWallView")
log.setLevel(GuiParam.DEBUG)

#-------------------------------------------------------------------------------
# Main class
#-------------------------------------------------------------------------------

class BoundaryConditionsRoughWallView(QWidget, Ui_BoundaryConditionsRoughWallForm):
    """
    Boundary condition for smooth or rough wall.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        QWidget.__init__(self, parent)

        Ui_BoundaryConditionsRoughWallForm.__init__(self)
        self.setupUi(self)


    def setup(self, case):
        """
        Setup the widget
        """
        self.case = case
        self.__boundary = None

        self.case.undoStopGlobal()

        self.connect(self.radioButtonSmooth, SIGNAL("clicked()"), self.__slotRoughness)
        self.connect(self.radioButtonRough,  SIGNAL("clicked()"), self.__slotRoughness)

        self.connect(self.lineEditRoughCoef, SIGNAL("textChanged(const QString &)"), self.__slotRoughnessHeight)

        validatorRoughCoef = DoubleValidator(self.lineEditRoughCoef)
        self.lineEditRoughCoef.setValidator(validatorRoughCoef)

        self.case.undoStartGlobal()


    def showWidget(self, boundary):
        """
        Show the widget
        """
        self.show()
        self.__boundary = boundary

        if self.__boundary.getRoughnessChoice() == "on":
            self.radioButtonSmooth.setChecked(False)
            self.radioButtonRough.setChecked(True)
        else:
            self.radioButtonSmooth.setChecked(True)
            self.radioButtonRough.setChecked(False)

        self.__slotRoughness()


    def hideWidget(self):
        """
        Hide the widget
        """
        self.hide()


    @pyqtSignature("")
    def __slotRoughness(self):
        """
        Private slot.

        Selects if the wall is rought or smooth.
        """
        if self.radioButtonSmooth.isChecked():
            self.frameRoughness.hide()
            self.__boundary.setRoughnessChoice('off')

        elif self.radioButtonRough.isChecked():
            self.frameRoughness.show()
            self.__boundary.setRoughnessChoice('on')
            r = self.__boundary.getRoughness()
            self.lineEditRoughCoef.setText(str(r))


    @pyqtSignature("const QString&")
    def __slotRoughnessHeight(self, text):
        """
        Private slot.

        Input the roughness height for the selected wall.

        @type text: C{QString}
        @param text: roughness height.
        """
        if self.sender().validator().state == QValidator.Acceptable:
            r = from_qvariant(text, float)
            self.__boundary.setRoughness(r)


    def tr(self, text):
        """
        Translation
        """
        return text

#-------------------------------------------------------------------------------
# End
#-------------------------------------------------------------------------------
