/*
 * This file is part of Cockpit.
 *
 * Copyright (C) 2021 Red Hat, Inc.
 *
 * Cockpit is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * Cockpit is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Cockpit; If not, see <http://www.gnu.org/licenses/>.
 */

import React, { useState, useContext } from 'react';
import cockpit from 'cockpit';
import {
    FormGroup,
} from '@patternfly/react-core';

import { MacMenu, NetworkModal, dialogSave } from './dialogs-common.jsx';
import { ModelContext } from './model-context.jsx';

const _ = cockpit.gettext;

export const MacDialog = ({ connection, dev, setIsOpen, settings }) => {
    const idPrefix = "network-mac-settings";
    const model = useContext(ModelContext);

    const [mac, setMAC] = useState((settings.ethernet && settings.ethernet.assigned_mac_address) || "");
    const [dialogError, setDialogError] = useState(undefined);

    const onSubmit = (ev) => {
        const createSettingsObj = () => ({
            ...settings,
            ethernet: {
                assigned_mac_address: mac
            },
        });

        if (!mac) {
            setDialogError(_("Enter a valid MAC address"));
            return;
        }

        dialogSave({
            model,
            dev,
            connection,
            settings: createSettingsObj(),
            setDialogError,
            setIsOpen,
        });

        // Prevent dialog from closing because of <form> onsubmit event
        if (event)
            event.preventDefault();

        return false;
    };

    return (
        <NetworkModal dialogError={dialogError}
                      idPrefix={idPrefix}
                      onSubmit={onSubmit}
                      setIsOpen={setIsOpen}
                      title={_("Ethernet MAC")}
        >
            <FormGroup fieldId={idPrefix + "-mac-input"} label={_("MAC")}>
                <MacMenu idPrefix={idPrefix} model={model} mac={mac} setMAC={setMAC} />
            </FormGroup>
        </NetworkModal>
    );
};
