/*  Clzip - LZMA lossless data compressor
    Copyright (C) 2010, 2011, 2012, 2013, 2014 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

enum { max_num_trials = 1 << 13,
       price_shift_bits = 6,
       price_step_bits = 2,
       price_step = 1 << price_step_bits };

typedef uint8_t Dis_slots[1<<10];

extern Dis_slots dis_slots;

static inline void Dis_slots_init( void )
  {
  int i, size, slot;
  for( slot = 0; slot < 4; ++slot ) dis_slots[slot] = slot;
  for( i = 4, size = 2, slot = 4; slot < 20; slot += 2 )
    {
    memset( &dis_slots[i], slot, size );
    memset( &dis_slots[i+size], slot + 1, size );
    size <<= 1;
    i += size;
    }
  }

static inline uint8_t get_slot( const unsigned dis )
  {
  if( dis < (1 << 10) ) return dis_slots[dis];
  if( dis < (1 << 19) ) return dis_slots[dis>> 9] + 18;
  if( dis < (1 << 28) ) return dis_slots[dis>>18] + 36;
  return dis_slots[dis>>27] + 54;
  }


typedef short Prob_prices[bit_model_total >> price_step_bits];

extern Prob_prices prob_prices;

static inline void Prob_prices_init( void )
  {
  int i, j;
  for( i = 0; i < bit_model_total >> price_step_bits; ++i )
    {
    unsigned val = ( i * price_step ) + ( price_step / 2 );
    int bits = 0;				/* base 2 logarithm of val */
    for( j = 0; j < price_shift_bits; ++j )
      {
      val = val * val;
      bits <<= 1;
      while( val >= 1 << 16 ) { val >>= 1; ++bits; }
      }
    bits += 15;					/* remaining bits in val */
    prob_prices[i] = ( bit_model_total_bits << price_shift_bits ) - bits;
    }
  }

static inline int get_price( const int probability )
  { return prob_prices[probability >> price_step_bits]; }


static inline int price0( const Bit_model probability )
  { return get_price( probability ); }

static inline int price1( const Bit_model probability )
  { return get_price( bit_model_total - probability ); }

static inline int price_bit( const Bit_model bm, const int bit )
  { if( bit ) return price1( bm ); else return price0( bm ); }


static inline int price_symbol( const Bit_model bm[], int symbol,
                                const int num_bits )
  {
  int price = 0;
  symbol |= ( 1 << num_bits );
  while( symbol > 1 )
    {
    const int bit = symbol & 1;
    symbol >>= 1;
    price += price_bit( bm[symbol], bit );
    }
  return price;
  }


static inline int price_symbol_reversed( const Bit_model bm[], int symbol,
                                         const int num_bits )
  {
  int price = 0;
  int model = 1;
  int i;
  for( i = num_bits; i > 0; --i )
    {
    const int bit = symbol & 1;
    price += price_bit( bm[model], bit );
    model = ( model << 1 ) | bit;
    symbol >>= 1;
    }
  return price;
  }


static inline int price_matched( const Bit_model bm[], int symbol,
                                 int match_byte )
  {
  int price = 0;
  int mask = 0x100;
  symbol |= mask;

  do {
    int match_bit, bit;
    match_byte <<= 1;
    match_bit = match_byte & mask;
    symbol <<= 1;
    bit = symbol & 0x100;
    price += price_bit( bm[match_bit+(symbol>>9)+mask], bit );
    mask &= ~(match_byte ^ symbol);	/* if( match_bit != bit ) mask = 0; */
    }
  while( symbol < 0x10000 );
  return price;
  }


struct Pair			/* distance-length pair */
  {
  int dis;
  int len;
  };


enum { /* bytes to keep in buffer before dictionary */
       before_size = max_num_trials + 1,
       /* bytes to keep in buffer after pos */
       after_size = ( 2 * max_match_len ) + 1,
       num_prev_positions3 = 1 << 16,
       num_prev_positions2 = 1 << 10 };

struct Matchfinder
  {
  unsigned long long partial_data_pos;
  uint8_t * buffer;		/* input buffer */
  int32_t * prev_positions;	/* 1 + last seen position of key. else 0 */
  int32_t * prev_pos_tree;	/* previous positions of key */
  int match_len_limit;
  int buffer_size;
  int dictionary_size;		/* bytes to keep in buffer before pos */
  int pos;			/* current pos in buffer */
  int cyclic_pos;		/* cycles through [0, dictionary_size] */
  int stream_pos;		/* first byte not yet read from file */
  int pos_limit;		/* when reached, a new block must be read */
  int cycles;
  int key4_mask;
  int num_prev_positions;	/* size of prev_positions */
  int infd;			/* input file descriptor */
  bool at_stream_end;		/* stream_pos shows real end of file */
  };

bool Mf_read_block( struct Matchfinder * const mf );
void Mf_normalize_pos( struct Matchfinder * const mf );

bool Mf_init( struct Matchfinder * const mf, const int dict_size,
              const int match_len_limit, const int ifd );

static inline void Mf_free( struct Matchfinder * const mf )
  {
  free( mf->prev_positions );
  free( mf->buffer );
  }

static inline uint8_t Mf_peek( const struct Matchfinder * const mf,
                               const int distance )
  { return mf->buffer[mf->pos-distance]; }

static inline int Mf_available_bytes( const struct Matchfinder * const mf )
  { return mf->stream_pos - mf->pos; }

static inline unsigned long long
Mf_data_position( const struct Matchfinder * const mf )
  { return mf->partial_data_pos + mf->pos; }

static inline bool Mf_finished( const struct Matchfinder * const mf )
  { return mf->at_stream_end && mf->pos >= mf->stream_pos; }

static inline const uint8_t *
Mf_ptr_to_current_pos( const struct Matchfinder * const mf )
  { return mf->buffer + mf->pos; }

static inline bool Mf_dec_pos( struct Matchfinder * const mf,
                               const int ahead )
  {
  if( ahead < 0 || mf->pos < ahead ) return false;
  mf->pos -= ahead;
  mf->cyclic_pos -= ahead;
  if( mf->cyclic_pos < 0 ) mf->cyclic_pos += mf->dictionary_size + 1;
  return true;
  }

static inline int Mf_true_match_len( const struct Matchfinder * const mf,
                                     const int index, const int distance,
                                     int len_limit )
  {
  const uint8_t * const data = mf->buffer + mf->pos + index;
  int i = 0;

  if( index + len_limit > Mf_available_bytes( mf ) )
    len_limit = Mf_available_bytes( mf ) - index;
  while( i < len_limit && data[i-distance] == data[i] ) ++i;
  return i;
  }

static inline void Mf_move_pos( struct Matchfinder * const mf )
  {
  if( ++mf->cyclic_pos > mf->dictionary_size ) mf->cyclic_pos = 0;
  if( ++mf->pos >= mf->pos_limit ) Mf_normalize_pos( mf );
  }

void Mf_reset( struct Matchfinder * const mf );
int Mf_get_match_pairs( struct Matchfinder * const mf, struct Pair * pairs );


enum { re_buffer_size = 65536 };

struct Range_encoder
  {
  uint64_t low;
  unsigned long long partial_member_pos;
  uint8_t * buffer;		/* output buffer */
  int pos;			/* current pos in buffer */
  uint32_t range;
  unsigned ff_count;
  int outfd;			/* output file descriptor */
  uint8_t cache;
  };

void Re_flush_data( struct Range_encoder * const renc );

static inline void Re_put_byte( struct Range_encoder * const renc,
                                const uint8_t b )
  {
  renc->buffer[renc->pos] = b;
  if( ++renc->pos >= re_buffer_size ) Re_flush_data( renc );
  }

static inline void Re_shift_low( struct Range_encoder * const renc )
  {
  const bool carry = ( renc->low > 0xFFFFFFFFU );
  if( carry || renc->low < 0xFF000000U )
    {
    Re_put_byte( renc, renc->cache + carry );
    for( ; renc->ff_count > 0; --renc->ff_count )
      Re_put_byte( renc, 0xFF + carry );
    renc->cache = renc->low >> 24;
    }
  else ++renc->ff_count;
  renc->low = ( renc->low & 0x00FFFFFFU ) << 8;
  }

static inline bool Re_init( struct Range_encoder * const renc, const int ofd )
  {
  renc->low = 0;
  renc->partial_member_pos = 0;
  renc->buffer = (uint8_t *)malloc( re_buffer_size );
  if( !renc->buffer ) return false;
  renc->pos = 0;
  renc->range = 0xFFFFFFFFU;
  renc->ff_count = 0;
  renc->outfd = ofd;
  renc->cache = 0;
  return true;
  }

static inline void Re_free( struct Range_encoder * const renc )
  { free( renc->buffer ); }

static inline unsigned long long
Re_member_position( const struct Range_encoder * const renc )
  { return renc->partial_member_pos + renc->pos + renc->ff_count; }

static inline void Re_flush( struct Range_encoder * const renc )
  { int i; for( i = 0; i < 5; ++i ) Re_shift_low( renc ); }

static inline void Re_encode( struct Range_encoder * const renc,
                              const int symbol, const int num_bits )
  {
  int i;
  for( i = num_bits - 1; i >= 0; --i )
    {
    renc->range >>= 1;
    if( (symbol >> i) & 1 ) renc->low += renc->range;
    if( renc->range <= 0x00FFFFFFU )
      { renc->range <<= 8; Re_shift_low( renc ); }
    }
  }

static inline void Re_encode_bit( struct Range_encoder * const renc,
                                  Bit_model * const probability, const int bit )
  {
  const uint32_t bound = ( renc->range >> bit_model_total_bits ) * *probability;
  if( !bit )
    {
    renc->range = bound;
    *probability += (bit_model_total - *probability) >> bit_model_move_bits;
    }
  else
    {
    renc->low += bound;
    renc->range -= bound;
    *probability -= *probability >> bit_model_move_bits;
    }
  if( renc->range <= 0x00FFFFFFU )
    { renc->range <<= 8; Re_shift_low( renc ); }
  }

static inline void Re_encode_tree( struct Range_encoder * const renc,
                                   Bit_model bm[], const int symbol, const int num_bits )
  {
  int mask = ( 1 << ( num_bits - 1 ) );
  int model = 1;
  int i;
  for( i = num_bits; i > 0; --i, mask >>= 1 )
    {
    const int bit = ( symbol & mask );
    Re_encode_bit( renc, &bm[model], bit );
    model <<= 1;
    if( bit ) model |= 1;
    }
  }

static inline void Re_encode_tree_reversed( struct Range_encoder * const renc,
                                            Bit_model bm[], int symbol, const int num_bits )
  {
  int model = 1;
  int i;
  for( i = num_bits; i > 0; --i )
    {
    const int bit = symbol & 1;
    Re_encode_bit( renc, &bm[model], bit );
    model = ( model << 1 ) | bit;
    symbol >>= 1;
    }
  }

static inline void Re_encode_matched( struct Range_encoder * const renc,
                                      Bit_model bm[], int symbol,
                                      int match_byte )
  {
  int mask = 0x100;
  symbol |= mask;

  do {
    int match_bit, bit;
    match_byte <<= 1;
    match_bit = match_byte & mask;
    symbol <<= 1;
    bit = symbol & 0x100;
    Re_encode_bit( renc, &bm[match_bit+(symbol>>9)+mask], bit );
    mask &= ~(match_byte ^ symbol);	/* if( match_bit != bit ) mask = 0; */
    }
  while( symbol < 0x10000 );
  }

static inline void Re_encode_len( struct Range_encoder * const renc,
                                  struct Len_model * const lm,
                                  int symbol, const int pos_state )
  {
  bool bit = ( ( symbol -= min_match_len ) >= len_low_symbols );
  Re_encode_bit( renc, &lm->choice1, bit );
  if( !bit )
    Re_encode_tree( renc, lm->bm_low[pos_state], symbol, len_low_bits );
  else
    {
    bit = ( symbol >= len_low_symbols + len_mid_symbols );
    Re_encode_bit( renc, &lm->choice2, bit );
    if( !bit )
      Re_encode_tree( renc, lm->bm_mid[pos_state],
                      symbol - len_low_symbols, len_mid_bits );
    else
      Re_encode_tree( renc, lm->bm_high,
                      symbol - len_low_symbols - len_mid_symbols, len_high_bits );
    }
  }


struct Len_prices
  {
  const struct Len_model * lm;
  int len_symbols;
  int count;
  int prices[pos_states][max_len_symbols];
  int counters[pos_states];
  };

static inline void Lp_update_low_mid_prices( struct Len_prices * const lp,
                                             const int pos_state )
  {
  int * const pps = lp->prices[pos_state];
  int tmp = price0( lp->lm->choice1 );
  int len = 0;
  lp->counters[pos_state] = lp->count;
  for( ; len < len_low_symbols && len < lp->len_symbols; ++len )
    pps[len] = tmp + price_symbol( lp->lm->bm_low[pos_state], len, len_low_bits );
  if( len >= lp->len_symbols ) return;
  tmp = price1( lp->lm->choice1 ) + price0( lp->lm->choice2 );
  for( ; len < len_low_symbols + len_mid_symbols && len < lp->len_symbols; ++len )
    pps[len] = tmp +
               price_symbol( lp->lm->bm_mid[pos_state], len - len_low_symbols, len_mid_bits );
    }

static inline void Lp_update_high_prices( struct Len_prices * const lp )
  {
  const int tmp = price1( lp->lm->choice1 ) + price1( lp->lm->choice2 );
  int len;
  for( len = len_low_symbols + len_mid_symbols; len < lp->len_symbols; ++len )
    /* using 4 slots per value makes "Lp_price" faster */
    lp->prices[3][len] = lp->prices[2][len] =
    lp->prices[1][len] = lp->prices[0][len] = tmp +
      price_symbol( lp->lm->bm_high, len - len_low_symbols - len_mid_symbols, len_high_bits );
  }

static inline void Lp_init( struct Len_prices * const lp,
                            const struct Len_model * const lm,
                            const int match_len_limit )
  {
  int i;
  lp->lm = lm;
  lp->len_symbols = match_len_limit + 1 - min_match_len;
  lp->count = ( match_len_limit > 12 ) ? 1 : lp->len_symbols;
  for( i = 0; i < pos_states; ++i ) lp->counters[i] = 0;
  }

static inline void Lp_decrement_counter( struct Len_prices * const lp,
                                         const int pos_state )
  { --lp->counters[pos_state]; }

static inline void Lp_update_prices( struct Len_prices * const lp )
  {
  int pos_state;
  bool high_pending = false;
  for( pos_state = 0; pos_state < pos_states; ++pos_state )
    if( lp->counters[pos_state] <= 0 )
      { Lp_update_low_mid_prices( lp, pos_state ); high_pending = true; }
  if( high_pending && lp->len_symbols > len_low_symbols + len_mid_symbols )
    Lp_update_high_prices( lp );
  }

static inline int Lp_price( const struct Len_prices * const lp,
                            const int symbol, const int pos_state )
  { return lp->prices[pos_state][symbol - min_match_len]; }


enum { infinite_price = 0x0FFFFFFF,
       max_marker_size = 16,
       num_rep_distances = 4,			/* must be 4 */
       single_step_trial = -2,
       dual_step_trial = -1 };

struct Trial
  {
  State state;
  int price;		/* dual use var; cumulative price, match length */
  int dis;		/* rep index or match distance. (-1 for literal) */
  int prev_index;	/* index of prev trial in trials[] */
  int prev_index2;	/*   -2  trial is single step */
			/*   -1  literal + rep0 */
			/* >= 0  ( rep or match ) + literal + rep0 */
  int reps[num_rep_distances];
  };

static inline void Tr_update( struct Trial * const trial, const int pr,
                              const int distance, const int p_i )
  {
  if( pr < trial->price )
    {
    trial->price = pr; trial->dis = distance; trial->prev_index = p_i;
    trial->prev_index2 = single_step_trial;
    }
  }

static inline void Tr_update2( struct Trial * const trial, const int pr,
                               const int p_i )
  {
  if( pr < trial->price )
    {
    trial->price = pr; trial->dis = 0; trial->prev_index = p_i;
    trial->prev_index2 = dual_step_trial;
    }
  }

static inline void Tr_update3( struct Trial * const trial, const int pr,
                               const int distance, const int p_i,
                               const int p_i2 )
  {
  if( pr < trial->price )
    {
    trial->price = pr; trial->dis = distance; trial->prev_index = p_i;
    trial->prev_index2 = p_i2;
    }
  }


struct LZ_encoder
  {
  int pending_num_pairs;
  uint32_t crc;

  Bit_model bm_literal[1<<literal_context_bits][0x300];
  Bit_model bm_match[states][pos_states];
  Bit_model bm_rep[states];
  Bit_model bm_rep0[states];
  Bit_model bm_rep1[states];
  Bit_model bm_rep2[states];
  Bit_model bm_len[states][pos_states];
  Bit_model bm_dis_slot[len_states][1<<dis_slot_bits];
  Bit_model bm_dis[modeled_distances-end_dis_model];
  Bit_model bm_align[dis_align_size];

  struct Matchfinder * matchfinder;
  struct Range_encoder renc;
  struct Len_model match_len_model;
  struct Len_model rep_len_model;
  struct Len_prices match_len_prices;
  struct Len_prices rep_len_prices;

  struct Pair pairs[max_match_len+1];
  struct Trial trials[max_num_trials];

  int dis_slot_prices[len_states][2*max_dictionary_bits];
  int dis_prices[len_states][modeled_distances];
  int align_prices[dis_align_size];
  int num_dis_slots;
  };

bool LZe_init( struct LZ_encoder * const e, struct Matchfinder * const mf,
               const File_header header, const int outfd );

static inline void LZe_free( struct LZ_encoder * const e )
  { Re_free( &e->renc ); }

static inline unsigned LZe_crc( const struct LZ_encoder * const e )
  { return e->crc ^ 0xFFFFFFFFU; }

       /* move-to-front dis in/into reps if( dis > 0 ) */
static inline void mtf_reps( const int dis, int reps[num_rep_distances] )
  {
  int i;
  if( dis >= num_rep_distances )
    {
    for( i = num_rep_distances - 1; i > 0; --i ) reps[i] = reps[i-1];
    reps[0] = dis - num_rep_distances;
    }
  else if( dis > 0 )
    {
    const int distance = reps[dis];
    for( i = dis; i > 0; --i ) reps[i] = reps[i-1];
    reps[0] = distance;
    }
  }

static inline int LZe_price_shortrep( const struct LZ_encoder * const e,
                                      const State state, const int pos_state )
  {
  return price0( e->bm_rep0[state] ) + price0( e->bm_len[state][pos_state] );
  }

static inline int LZe_price_rep( const struct LZ_encoder * const e,
                                 const int rep,
                                 const State state, const int pos_state )
  {
  int price;
  if( rep == 0 ) return price0( e->bm_rep0[state] ) +
                        price1( e->bm_len[state][pos_state] );
  price = price1( e->bm_rep0[state] );
  if( rep == 1 )
    price += price0( e->bm_rep1[state] );
  else
    {
    price += price1( e->bm_rep1[state] );
    price += price_bit( e->bm_rep2[state], rep - 2 );
    }
  return price;
  }

static inline int LZe_price_rep0_len( const struct LZ_encoder * const e,
                                      const int len,
                                      const State state, const int pos_state )
  {
  return LZe_price_rep( e, 0, state, pos_state ) +
         Lp_price( &e->rep_len_prices, len, pos_state );
  }

static inline int LZe_price_pair( const struct LZ_encoder * const e,
                                  const int dis, const int len,
                                  const int pos_state )
  {
  const int price = Lp_price( &e->match_len_prices, len, pos_state );
  const int len_state = get_len_state( len );
  if( dis < modeled_distances )
    return price + e->dis_prices[len_state][dis];
  else
    return price + e->dis_slot_prices[len_state][get_slot( dis )] +
           e->align_prices[dis & (dis_align_size - 1)];
  }

static inline int LZe_price_literal( const struct LZ_encoder * const e,
                                     uint8_t prev_byte, uint8_t symbol )
  { return price_symbol( e->bm_literal[get_lit_state(prev_byte)], symbol, 8 ); }

static inline int LZe_price_matched( const struct LZ_encoder * const e,
                                     uint8_t prev_byte, uint8_t symbol,
                                     uint8_t match_byte )
  { return price_matched( e->bm_literal[get_lit_state(prev_byte)], symbol,
                          match_byte ); }

static inline void LZe_encode_literal( struct LZ_encoder * const e,
                                       uint8_t prev_byte, uint8_t symbol )
  { Re_encode_tree( &e->renc,
                    e->bm_literal[get_lit_state(prev_byte)], symbol, 8 ); }

static inline void LZe_encode_matched( struct LZ_encoder * const e,
                                       uint8_t prev_byte, uint8_t symbol,
                                       uint8_t match_byte )
  { Re_encode_matched( &e->renc, e->bm_literal[get_lit_state(prev_byte)],
                       symbol, match_byte ); }

static inline void LZe_encode_pair( struct LZ_encoder * const e,
                                    const unsigned dis, const int len,
                                    const int pos_state )
  {
  const int dis_slot = get_slot( dis );
  Re_encode_len( &e->renc, &e->match_len_model, len, pos_state );
  Re_encode_tree( &e->renc, e->bm_dis_slot[get_len_state(len)], dis_slot,
                  dis_slot_bits );

  if( dis_slot >= start_dis_model )
    {
    const int direct_bits = ( dis_slot >> 1 ) - 1;
    const unsigned base = ( 2 | ( dis_slot & 1 ) ) << direct_bits;
    const unsigned direct_dis = dis - base;

    if( dis_slot < end_dis_model )
      Re_encode_tree_reversed( &e->renc, e->bm_dis + base - dis_slot - 1,
                               direct_dis, direct_bits );
    else
      {
      Re_encode( &e->renc, direct_dis >> dis_align_bits,
                 direct_bits - dis_align_bits );
      Re_encode_tree_reversed( &e->renc, e->bm_align, direct_dis, dis_align_bits );
      }
    }
  }

static inline int LZe_read_match_distances( struct LZ_encoder * const e )
  {
  const int num_pairs = Mf_get_match_pairs( e->matchfinder, e->pairs );
  if( num_pairs > 0 )
    {
    int len = e->pairs[num_pairs-1].len;
    if( len == e->matchfinder->match_len_limit && len < max_match_len )
      {
      len += Mf_true_match_len( e->matchfinder, len,
                                e->pairs[num_pairs-1].dis + 1,
                                max_match_len - len );
      e->pairs[num_pairs-1].len = len;
      }
    }
  return num_pairs;
  }

static inline void LZe_move_pos( struct LZ_encoder * const e, int n )
  {
  while( true )
    {
    Mf_move_pos( e->matchfinder );
    if( --n <= 0 ) break;
    Mf_get_match_pairs( e->matchfinder, 0 );
    }
  }

static inline void LZe_backward( struct LZ_encoder * const e, int cur )
  {
  int * const dis = &e->trials[cur].dis;
  while( cur > 0 )
    {
    const int prev_index = e->trials[cur].prev_index;
    struct Trial * const prev_trial = &e->trials[prev_index];

    if( e->trials[cur].prev_index2 != single_step_trial )
      {
      prev_trial->dis = -1;
      prev_trial->prev_index = prev_index - 1;
      prev_trial->prev_index2 = single_step_trial;
      if( e->trials[cur].prev_index2 >= 0 )
        {
        struct Trial * const prev_trial2 = &e->trials[prev_index-1];
        prev_trial2->dis = *dis; *dis = 0;
        prev_trial2->prev_index = e->trials[cur].prev_index2;
        prev_trial2->prev_index2 = single_step_trial;
        }
      }
    prev_trial->price = cur - prev_index;			/* len */
    cur = *dis; *dis = prev_trial->dis; prev_trial->dis = cur;
    cur = prev_index;
    }
  }

bool LZe_encode_member( struct LZ_encoder * const e,
                        const unsigned long long member_size );
