/* 
 * Copyright (C) 2013 Simon Richter
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <CL/cl.h>
#include <CL/cl_ext.h>

#include <iostream>

int main(int, char **)
{
        try
        {
                cl_uint num_platforms;
                cl_int rc = clGetPlatformIDs(0, 0, &num_platforms);
                switch(rc)
                {
                case CL_SUCCESS:
                        break;

                case CL_PLATFORM_NOT_FOUND_KHR:
                        std::cerr << "I: ICD loader reports no usable platforms" << std::endl;
                        return 0;

                default:
                        throw rc;
                }

                cl_platform_id platforms[num_platforms];

                rc = clGetPlatformIDs(num_platforms, platforms, 0);
                if(rc != CL_SUCCESS)
                        throw rc;

                for(cl_platform_id const *i = platforms; i != platforms + num_platforms; ++i)
                {
                        struct platform_info_query
                        {
                                cl_platform_info info;
                                char const *tag;
                        };

                        static platform_info_query const platform_queries[] =
                        {
                                { CL_PLATFORM_NAME, "Platform" },
                                { CL_PLATFORM_VENDOR, "Vendor" },
                                { CL_PLATFORM_PROFILE, "Profile" },
                                { CL_PLATFORM_VERSION, "Version" },
                                { CL_PLATFORM_EXTENSIONS, "Extensions" }
                        };

                        char buffer[64];
                        size_t length;

                        for(platform_info_query const *j = platform_queries; j != platform_queries + 5; ++j)
                        {
                                rc = clGetPlatformInfo(*i, j->info, sizeof buffer, buffer, &length);
                                if(rc != CL_SUCCESS)
                                        throw rc;

                                std::cout << j->tag << ": " << buffer << std::endl;
                        }

                        std::cout << std::endl;

                        cl_uint num_devices;
                        rc = clGetDeviceIDs(*i, CL_DEVICE_TYPE_ALL, 0, 0, &num_devices);
                        if(rc != CL_SUCCESS)
                                throw rc;

                        cl_device_id devices[num_devices];
                        rc = clGetDeviceIDs(*i, CL_DEVICE_TYPE_ALL, num_devices, devices, 0);
                        if(rc != CL_SUCCESS)
                                throw rc;

                        for(cl_device_id const *j = devices; j != devices + num_devices; ++j)
                        {
                                enum data_type
                                {
                                        boolean,
                                        uint,
                                        string
                                };

                                struct device_info_query
                                {
                                        cl_device_info info;
                                        char const *tag;
                                        data_type type;
                                };

                                static device_info_query const device_queries[] =
                                {
                                        { CL_DEVICE_NAME, "Device", string },
                                        { CL_DEVICE_VENDOR, "Vendor", string },
                                        { CL_DEVICE_PROFILE, "Profile", string },
                                        { CL_DEVICE_VERSION, "Version", string },
                                        { CL_DEVICE_EXTENSIONS, "Extensions", string },
                                        { CL_DEVICE_VENDOR_ID, "VendorID", uint },
                                        { CL_DEVICE_MAX_COMPUTE_UNITS, "MaxComputeUnits", uint },
                                        { CL_DEVICE_MAX_WORK_ITEM_DIMENSIONS, "MaxWorkItemDimensions", uint },
                                        { CL_DEVICE_MAX_WORK_GROUP_SIZE, "MaxWorkGroupSize", uint },
                                        { CL_DEVICE_MAX_WORK_ITEM_SIZES, "MaxWorkItemSizes", uint },
                                        { CL_DEVICE_PREFERRED_VECTOR_WIDTH_CHAR, "PreferredVectorWidthChar", uint },
                                        { CL_DEVICE_NATIVE_VECTOR_WIDTH_CHAR, "NativeVectorWidthChar", uint },
                                        { CL_DEVICE_PREFERRED_VECTOR_WIDTH_SHORT, "PreferredVectorWidthShort", uint },
                                        { CL_DEVICE_NATIVE_VECTOR_WIDTH_SHORT, "NativeVectorWidthShort", uint },
                                        { CL_DEVICE_PREFERRED_VECTOR_WIDTH_INT, "PreferredVectorWidthInt", uint },
                                        { CL_DEVICE_NATIVE_VECTOR_WIDTH_INT, "NativeVectorWidthInt", uint },
                                        { CL_DEVICE_PREFERRED_VECTOR_WIDTH_LONG, "PreferredVectorWidthLong", uint },
                                        { CL_DEVICE_NATIVE_VECTOR_WIDTH_LONG, "NativeVectorWidthLong", uint },
                                        { CL_DEVICE_PREFERRED_VECTOR_WIDTH_FLOAT, "PreferredVectorWidthFloat", uint },
                                        { CL_DEVICE_NATIVE_VECTOR_WIDTH_FLOAT, "NativeVectorWidthFloat", uint },
                                        { CL_DEVICE_PREFERRED_VECTOR_WIDTH_DOUBLE, "PreferredVectorWidthDouble", uint },
                                        { CL_DEVICE_NATIVE_VECTOR_WIDTH_DOUBLE, "NativeVectorWidthDouble", uint },
                                        { CL_DEVICE_PREFERRED_VECTOR_WIDTH_HALF, "PreferredVectorWidthHalf", uint },
                                        { CL_DEVICE_NATIVE_VECTOR_WIDTH_HALF, "NativeVectorWidthHalf", uint },
                                        { CL_DEVICE_MAX_CLOCK_FREQUENCY, "MaxClockFrequency", uint },
                                        { CL_DEVICE_ADDRESS_BITS, "AddressBits", uint },
                                        { CL_DEVICE_MAX_READ_IMAGE_ARGS, "MaxReadImageArgs", uint },
                                        { CL_DEVICE_MAX_WRITE_IMAGE_ARGS, "MaxWriteImageArgs", uint },
                                        { CL_DEVICE_MAX_MEM_ALLOC_SIZE, "MaxMemAllocSize", uint },
                                        { CL_DEVICE_IMAGE2D_MAX_WIDTH, "Image2DMaxWidth", uint },
                                        { CL_DEVICE_IMAGE2D_MAX_HEIGHT, "Image2DMaxHeight", uint },
                                        { CL_DEVICE_IMAGE3D_MAX_WIDTH, "Image3DMaxWidth", uint },
                                        { CL_DEVICE_IMAGE3D_MAX_HEIGHT, "Image3DMaxHeight", uint },
                                        { CL_DEVICE_IMAGE3D_MAX_DEPTH, "Image3DMaxDepth", uint },
                                        { CL_DEVICE_IMAGE_SUPPORT, "ImageSupport", boolean },
                                        { CL_DEVICE_MAX_PARAMETER_SIZE, "MaxParameterSize", uint },
                                        { CL_DEVICE_MAX_SAMPLERS, "MaxSamplers", uint },
                                        { CL_DEVICE_MEM_BASE_ADDR_ALIGN, "MemBaseAddrAlign", uint },
                                        { CL_DEVICE_MIN_DATA_TYPE_ALIGN_SIZE, "MinDataTypeAlignSize", uint },
                                        { CL_DEVICE_SINGLE_FP_CONFIG, "SingleFpConfig", uint },
                                        { CL_DEVICE_GLOBAL_MEM_CACHE_TYPE, "GlobalMemCacheType", uint },
                                        { CL_DEVICE_GLOBAL_MEM_CACHELINE_SIZE, "GlobalMemCachelineSize", uint },
                                        { CL_DEVICE_GLOBAL_MEM_CACHE_SIZE, "GlobalMemCacheSize", uint },
                                        { CL_DEVICE_GLOBAL_MEM_SIZE, "GlobalMemSize", uint },
                                        { CL_DEVICE_MAX_CONSTANT_BUFFER_SIZE, "MaxConstantBufferSize", uint },
                                        { CL_DEVICE_MAX_CONSTANT_ARGS, "MaxConstantArgs", uint },
                                        { CL_DEVICE_LOCAL_MEM_TYPE, "LocalMemType", uint },
                                        { CL_DEVICE_LOCAL_MEM_SIZE, "LocalMemSize", uint },
                                        { CL_DEVICE_ERROR_CORRECTION_SUPPORT, "ErrorCorrectionSupport", boolean },
                                        { CL_DEVICE_PROFILING_TIMER_RESOLUTION, "ProfilingTimerResolution", uint },
                                        { CL_DEVICE_ENDIAN_LITTLE, "EndianLittle", boolean },
                                        { CL_DEVICE_AVAILABLE, "DeviceAvailable", boolean },
                                        { CL_DEVICE_COMPILER_AVAILABLE, "CompilerAvailable", boolean },
                                        { CL_DEVICE_EXECUTION_CAPABILITIES, "ExecutionCapabilities", uint },
                                        { CL_DEVICE_QUEUE_PROPERTIES, "QueueProperties", uint },
                                        //{ CL_DEVICE_PLATFORM, "Platform", uint },
                                        { CL_DEVICE_DOUBLE_FP_CONFIG, "DoubleFPConfig", uint },
                                        { CL_DEVICE_HOST_UNIFIED_MEMORY, "HostUnifiedMemory", uint },
                                        { CL_DEVICE_OPENCL_C_VERSION, "OpenCL-C-Version", string },
                                        { CL_DEVICE_LINKER_AVAILABLE, "LinkerAvailable", boolean },
                                        { CL_DEVICE_BUILT_IN_KERNELS, "BuiltInKernels", uint },
                                        { CL_DEVICE_IMAGE_MAX_BUFFER_SIZE, "ImageMaxBufferSize", uint },
                                        { CL_DEVICE_IMAGE_MAX_ARRAY_SIZE, "ImageMaxArraySize", uint },
                                        //{ CL_DEVICE_PARENT_DEVICE, "", uint },
                                        { CL_DEVICE_PARTITION_MAX_SUB_DEVICES, "PartitionMaxSubDevices", uint },
                                        { CL_DEVICE_PARTITION_PROPERTIES, "PartitionProperties", uint },
                                        { CL_DEVICE_PARTITION_AFFINITY_DOMAIN, "PartitionAffinityDomain", uint },
                                        { CL_DEVICE_PARTITION_TYPE, "PartitionType", uint },
                                        { CL_DEVICE_REFERENCE_COUNT, "ReferenceCount", uint },
                                        { CL_DEVICE_PREFERRED_INTEROP_USER_SYNC, "PreferredInteropUserSync", uint },
                                        { CL_DEVICE_PRINTF_BUFFER_SIZE, "PrintfBufferSize", uint },
                                };

                                for(device_info_query const *k = device_queries; k != device_queries + (sizeof device_queries / sizeof *device_queries); ++k)
                                {
                                        rc = clGetDeviceInfo(*j, k->info, sizeof buffer, buffer, &length);
                                        if(rc == CL_INVALID_VALUE)
                                                continue;

                                        if(rc != CL_SUCCESS)
                                                throw rc;

                                        std::cout << k->tag << ": ";
                                        switch(k->type)
                                        {
                                        case boolean:
                                                std::cout << ((*reinterpret_cast<cl_uint *>(buffer))?"true":"false");
                                                break;
                                        case uint:
                                                std::cout << *reinterpret_cast<cl_uint *>(buffer);
                                                break;
                                        case string:
                                                std::cout << buffer;
                                                break;
                                        }

                                        std::cout << std::endl;
                                }

                                std::cout << std::endl;
                        }
                }

                return 0;
        }
        catch(cl_int e)
        {
                std::cerr << "E: " << e << std::endl;
                return 1;
        }
}
