/*
 * Copyright 2013 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 3 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "clickmanager.h"
#include <QString>
#include <QStringList>
#include <QJsonDocument>
#include <QJsonObject>
#include <QJsonArray>
#include <QJsonValue>
#include <QDebug>
#include <QDebug>

ClickManager::ClickManager(QQuickItem *parent):
    QQuickItem(parent)
  , m_process(this)
{
    this->m_updatesAvailable = false;
    this->connect(&(this->m_process), SIGNAL(readyReadStandardOutput()),
                  this, SLOT(processOutput()));
    this->connect(&(this->network), SIGNAL(updatesFound()),
                  this, SLOT(processUpdates()));
    this->connect(&(this->network), SIGNAL(downloadUrlFound(QString,QString)),
                  this, SLOT(downloadUrlObtained(QString,QString)));
    this->connect(&(this->downloader), SIGNAL(downloadCreated(QString,QString)),
                  this, SLOT(downloadCreated(QString,QString)));
}

ClickManager::~ClickManager()
{
}

void ClickManager::checkUpdates()
{
    QStringList args("list");
    args << "--manifest";
    this->m_process.start("click", args);
}

void ClickManager::processOutput()
{
    QString output(this->m_process.readAllStandardOutput());

    QJsonDocument document = QJsonDocument::fromJson(output.toUtf8());

    QJsonArray array = document.array();

    int i;
    for(i = 0; i < array.size(); i++) {
        QJsonObject object = array.at(i).toObject();
        QString name = object.value("name").toString();
        QString title = object.value("title").toString();
        QString version = object.value("version").toString();
        Application* app = new Application();
        app->initializeApplication(name, title, version);
        this->m_apps[app->getPackageName()] = app;
    }

    this->network.checkForNewVersions(this->m_apps);
}

void ClickManager::processUpdates()
{
    foreach(QString id, this->m_apps.keys()) {
        Application* app = this->m_apps.value(id);
        if(app->updateRequired()) {
            this->m_model.append(QVariant::fromValue(app));
        }
    }

    emit this->modelChanged();
}

void ClickManager::startDownload(QString packagename)
{
    qDebug() << "Download Package:" << packagename;
    this->network.getResourceUrl(packagename);
}

void ClickManager::downloadUrlObtained(QString packagename, QString url)
{
    qDebug() << "Download Url:" << url;
    this->downloader.startDownload(packagename, url.append("?noauth=1"));
}

void ClickManager::downloadCreated(QString packagename, QString dbuspath)
{
    qDebug() << "Dbus Path:" << dbuspath;
    this->m_apps[packagename]->setDbusPath(dbuspath);
    this->m_apps[packagename]->setUpdateState(true);
}
