/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import Ubuntu.Components 0.1
import Ubuntu.Components.ListItems 0.1 as ListItem
import com.ubuntu.click 0.1

Page {
    id: root
    title: i18n.tr("Updates")

    property int updates: 0

    onUpdatesChanged: {
        if(root.updates > 1) {
            installAllButton.visible = true;
        } else {
            installAllButton.visible = false;
        }
    }

    ClickManager {
        id: clickManager
        objectName: "clickManager"

        Component.onCompleted: clickManager.checkUpdates()

        onModelChanged: {
            labelMessage.visible = false;
            root.updates = clickManager.model.length;
            loadingIndicator.visible = false;
        }

        onUpdatesNotFound: {
            loadingIndicator.visible = false;
            installAllButton.visible = false;
            labelMessage.text = i18n.tr("No Updates Found")
            labelMessage.visible = true;
        }

        onCredentialsNotFound: {
            loadingIndicator.visible = false;
            installAllButton.visible = false;
            labelMessage.text = i18n.tr("Please log into your Ubuntu One account in System Settings.")
            labelMessage.visible = true;
        }
    }

    Button {
        id: retryButton
        objectName: "retryButton"
        text: i18n.tr("Check again")
        anchors {
            left: parent.left
            right: parent.right
            top: parent.top
            topMargin: units.gu(2)
            leftMargin: units.gu(4)
            rightMargin: units.gu(4)
        }
        visible: root.updates == 0 ? true : false

        color: UbuntuColors.orange
        onClicked: {
            loadingIndicator.visible = true;
            clickManager.checkUpdates()
        }
    }

    Button {
        id: installAllButton
        objectName: "installAllButton"
        text: i18n.tr("Install %1 Updates").arg(root.updates)
        anchors {
            left: parent.left
            right: parent.right
            top: parent.top
            topMargin: units.gu(2)
            leftMargin: units.gu(4)
            rightMargin: units.gu(4)
        }
        visible: false

        color: UbuntuColors.orange
        onClicked: {
            for(var i=0; i < clickManager.model.length; i++){
                var item = clickManager.model[i];
                if(!item.selected) {
                    clickManager.startDownload(item.packageName);
                }
            }
        }
    }

    ListView {
        id: updateList
        objectName: "updateList"
        anchors {
            left: parent.left
            right: parent.right
            top: installAllButton.bottom
            bottom: parent.bottom
            margins: units.gu(2)
        }
        model: clickManager.model
        clip: true

        delegate: ListItem.Subtitled {
            id: listItem
            icon: Qt.resolvedUrl(modelData.iconUrl)
            enabled: !modelData.updateState
            height: units.gu(9)

            Rectangle {
                id: textArea
                objectName: "textArea"
                color: "transparent"
                anchors.fill: parent
                anchors.topMargin: units.gu(1)

                property string message: modelData.error

                onMessageChanged: {
                    if(message.length > 0) {
                        labelVersion.text = message;
                        buttonAppUpdate.text = i18n.tr("Retry");
                        modelData.updateState = false;
                    }
                }

                Column {
                    id: colRight
                    objectName: "colRight"
                    anchors.top: parent.top
                    anchors.right: parent.right
                    anchors.topMargin: units.gu(1)

                    Button {
                        id: buttonAppUpdate
                        text: i18n.tr("Update")
                        height: textArea.height / 2

                        onClicked: {
                            modelData.selected = true;
                            clickManager.startDownload(modelData.packageName);
                        }
                    }
                    Label {
                        text: parseFloat((modelData.binaryFilesize / 1024) / 1024).toFixed(2) + " Mb"
                        height: textArea.height / 2
                        anchors.right: parent.right
                    }
                }

                Column {
                    id: colLeft
                    objectName: "colLeft"
                    anchors {
                        left: parent.left
                        bottom: parent.bottom
                        top: parent.top
                        right: colRight.left
                        rightMargin: units.gu(1)
                        topMargin: units.gu(1)
                    }

                    Label {
                        anchors {
                            left: parent.left
                            right: parent.right
                        }
                        text: modelData.title
                        color: modelData.updateState ? "gray" : "white"
                        font.bold: true
                        elide: Text.ElideRight
                    }
                    Label {
                        id: labelVersion
                        anchors {
                            left: parent.left
                            right: parent.right
                        }
                        text: modelData.remoteVersion ? i18n.tr("Version: ") + modelData.remoteVersion : ""
                        color: "white"
                        visible: !modelData.updateState
                        elide: Text.ElideRight
                    }
                }
            }

            ProgressBar {
                id: progress
                objectName: "progress"
                width: textArea.width - colRight.width - units.gu(2)
                height: parent.height / 3
                anchors.left: textArea.left
                anchors.leftMargin: units.gu(1)
                anchors.bottom: textArea.bottom
                anchors.bottomMargin: units.dp(5)
                opacity: modelData.updateState ? 1 : 0
                value: 0
                minimumValue: 0
                maximumValue: 100

                DownloadTracker {
                    id: tracker
                    objectName: "tracker"
                    service: "com.canonical.applications.Downloader"
                    dbusPath: modelData.updateState ? modelData.dbusPath : ""

                    onProgress: {
                        var percentage = parseInt(received * 100 / total);
                        progress.value = percentage;
                    }

                    onFinished: {
                        progress.visible = false;
                        buttonAppUpdate.visible = false;
                        textArea.message = i18n.tr("Installed");
                        root.updates -= 1;
                    }

                    onError: {
                        textArea.message = error;
                    }
                }

                Behavior on opacity { PropertyAnimation { duration: UbuntuAnimation.SleepyDuration } }
            }
        }
    }

    Label {
        id: labelMessage
        objectName: "labelMessage"
        anchors {
            left: parent.left
            right: parent.right
            verticalCenter: parent.verticalCenter
            margins: units.gu(2)
        }
        color: "white"
        opacity: .9
        style: Text.Raised
        styleColor: "black"
        wrapMode: Text.WrapAtWordBoundaryOrAnywhere
        visible: false
        fontSize: "x-large"
    }

    Rectangle {
        id: loadingIndicator
        objectName: "loadingIndicator"
        anchors.fill: parent
        color: "black"
        opacity: 0.8
        visible: true

        ActivityIndicator {
            running: parent.visible
            anchors.centerIn: parent
        }
        MouseArea {
            anchors.fill: parent
        }
    }
}
