'''sr_lint.py: lint checks'''
#
# Copyright (C) 2013-2016 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function
from clickreviews.frameworks import Frameworks
from clickreviews.sr_common import (
    SnapReview,
)
from clickreviews.common import (
    find_external_symlinks,
)
import glob
import os
import re


class SnapReviewLint(SnapReview):
    '''This class represents snap lint reviews'''

    def __init__(self, fn, overrides=None):
        '''Set up the class.'''
        SnapReview.__init__(self, fn, "lint-snap-v2", overrides=overrides)
        if not self.is_snap2:
            return

        self.valid_compiled_architectures = ['armhf',
                                             'i386',
                                             'amd64',
                                             'arm64',
                                             ]
        self.valid_architectures = ['all'] + self.valid_compiled_architectures
        self.vcs_files = ['.bzr*',
                          # '.excludes',  # autogenerated by SDK
                          '.git*',
                          '.idea',
                          '.svn*',
                          '.hg',
                          '.project',
                          'CVS*',
                          'RCS*'
                          ]

        self._list_all_compiled_binaries()

        # Valid values for 'type' in packaging yaml
        # - app
        # - framework
        # - kernel
        # - gadget
        # - os
        self.valid_snap_types = ['app',
                                 'framework',
                                 'kernel',
                                 'gadget',
                                 'os',
                                 ]
        self.redflagged_snap_types = ['framework',
                                      'kernel',
                                      'gadget',
                                      'os',
                                      ]

    def check_architectures(self):
        '''Check architectures in snap.yaml is valid'''
        if not self.is_snap2:
            return

        t = 'info'
        n = self._get_check_name('architecture_valid')
        s = 'OK'

        key = 'architectures'
        if key not in self.snap_yaml:
            s = 'OK (%s not specified)' % key
            self._add_result(t, n, s)
            return

        if not isinstance(self.snap_yaml[key], list):
            t = 'error'
            s = "invalid %s entry: %s (not a list)" % (key,
                                                       self.snap_yaml[key])
        else:
            bad_archs = []
            for arch in self.snap_yaml[key]:
                if arch not in self.valid_architectures:
                    bad_archs.append(arch)
                if len(bad_archs) > 0:
                    t = 'error'
                    s = "invalid multi architecture: %s" % ",".join(bad_archs)
        self._add_result(t, n, s)

    def check_description(self):
        '''Check description'''
        if not self.is_snap2:
            return

        key = 'description'

        t = 'info'
        n = self._get_check_name('%s_present' % key)
        s = 'OK'
        if key not in self.snap_yaml:
            s = 'OK (optional %s field not specified)' % key
            self._add_result(t, n, s)
            return
        self._add_result(t, n, s)

        t = 'info'
        n = self._get_check_name(key)
        s = 'OK'
        if not isinstance(self.snap_yaml[key], str):
            t = 'error'
            s = "invalid %s entry: %s (not a str)" % (key, self.snap_yaml[key])
            self._add_result(t, n, s)
            return
        elif len(self.snap_yaml[key]) < 1:
            t = 'error'
            s = "invalid %s entry (empty)" % (key)
        elif len(self.snap_yaml[key]) < len(self.snap_yaml['name']):
            t = 'info'
            s = "%s is too short: '%s'" % (key, self.snap_yaml[key])
        self._add_result(t, n, s)

    def check_frameworks(self):
        '''Check framework'''
        if not self.is_snap2:
            return

        key = 'frameworks'

        t = 'info'
        n = self._get_check_name(key)
        l = "http://askubuntu.com/questions/460512/what-framework-should-i-use-in-my-manifest-file"

        if key not in self.snap_yaml:
            s = 'OK (%s not specified)' % key
            self._add_result(t, n, s)
            return

        if not isinstance(self.snap_yaml[key], list):
            t = 'error'
            s = "invalid %s entry: %s (not a list)" % (key,
                                                       self.snap_yaml[key])
            self._add_result(t, n, s)
            return
        elif len(self.snap_yaml[key]) < 1:
            t = 'error'
            s = "invalid %s entry (empty)" % (key)
            self._add_result(t, n, s)
            return

        t = 'info'
        n = self._get_check_name('framework_uses_%s' % key)
        s = "OK"
        if 'type' in self.snap_yaml and self.snap_yaml['type'] == 'framework':
            t = 'error'
            s = "framework may not specify '%s'" % key
        self._add_result(t, n, s)

        framework_overrides = self.overrides.get('framework', {})
        frameworks = Frameworks(overrides=framework_overrides)

        for framework in self.snap_yaml[key]:
            if framework in frameworks.AVAILABLE_FRAMEWORKS:
                t = 'info'
                s = 'OK'
                self._add_result(t, n, s)
                # If it's an available framework, we're done checking
                return
            elif framework in frameworks.DEPRECATED_FRAMEWORKS:
                t = 'warn'
                s = "'%s' is deprecated. Please use a newer framework" % \
                    framework
                self._add_result(t, n, s, l)
                return
            elif framework in frameworks.OBSOLETE_FRAMEWORKS:
                t = 'error'
                s = "'%s' is obsolete. Please use a newer framework" % \
                    framework
                self._add_result(t, n, s, l)
                return
            else:
                # None of the above checks triggered, this is an unknown
                # framework
                t = 'error'
                s = "'%s' is not a supported framework" % \
                    framework
                self._add_result(t, n, s, l)

    # TODO: verify this is a field
    def check_license_agreement(self):
        '''Check license-agreement'''
        if not self.is_snap2:
            return

        key = 'license-agreement'

        t = 'info'
        n = self._get_check_name('%s_present' % key)
        s = 'OK'
        if key not in self.snap_yaml:
            s = 'OK (optional %s field not specified)' % key
            self._add_result(t, n, s)
            return
        self._add_result(t, n, s)

        t = 'info'
        n = self._get_check_name(key)
        s = 'OK'
        if not isinstance(self.snap_yaml[key], str):
            t = 'error'
            s = "invalid %s entry: %s (not a str)" % (key, self.snap_yaml[key])
            self._add_result(t, n, s)
            return
        elif len(self.snap_yaml[key]) < 1:
            t = 'error'
            s = "invalid %s entry (empty)" % (key)
            self._add_result(t, n, s)
            return
        self._add_result(t, n, s)

    def check_license_version(self):
        '''license-version'''
        if not self.is_snap2:
            return

        key = 'license-version'
        t = 'info'
        n = self._get_check_name('%s_present' % key)
        s = 'OK'
        if key not in self.snap_yaml:
            s = 'OK (optional %s field not specified)' % key
            self._add_result(t, n, s)
            return
        self._add_result(t, n, s)

        t = 'info'
        n = self._get_check_name(key)
        s = 'OK'
        if not isinstance(self.snap_yaml[key], str):
            t = 'error'
            s = "invalid %s entry: %s (not a str)" % (key, self.snap_yaml[key])
            self._add_result(t, n, s)
            return
        elif len(self.snap_yaml[key]) < 1:
            t = 'error'
            s = "invalid %s entry (empty)" % (key)
            self._add_result(t, n, s)
            return
        self._add_result(t, n, s)

    def check_name(self):
        '''Check package name'''
        if not self.is_snap2:
            return

        t = 'info'
        n = self._get_check_name('name_valid')
        s = 'OK'
        if 'name' not in self.snap_yaml:
            t = 'error'
            s = "could not find 'name' in yaml"
        elif not isinstance(self.snap_yaml['name'], str):
            t = 'error'
            s = "malformed 'name': %s (not a str)" % (self.snap_yaml['name'])
        elif not self._verify_pkgname(self.snap_yaml['name']):
            t = 'error'
            s = "malformed 'name': '%s'" % self.snap_yaml['name']
        self._add_result(t, n, s)

    def check_summary(self):
        '''Check summary'''
        if not self.is_snap2:
            return

        key = 'summary'

        t = 'info'
        n = self._get_check_name('%s_present' % key)
        s = 'OK'
        if key not in self.snap_yaml:
            s = 'OK (optional %s field not specified)' % key
            self._add_result(t, n, s)
            return
        self._add_result(t, n, s)

        t = 'info'
        n = self._get_check_name(key)
        s = 'OK'
        if not isinstance(self.snap_yaml[key], str):
            t = 'error'
            s = "invalid %s entry: %s (not a str)" % (key, self.snap_yaml[key])
            self._add_result(t, n, s)
            return
        elif len(self.snap_yaml[key]) < 1:
            t = 'error'
            s = "invalid %s entry (empty)" % (key)
        elif len(self.snap_yaml[key]) < len(self.snap_yaml['name']):
            t = 'info'
            s = "%s is too short: '%s'" % (key, self.snap_yaml[key])
        self._add_result(t, n, s)

    def check_type(self):
        '''Check type'''
        if not self.is_snap2 or 'type' not in self.snap_yaml:
            return

        t = 'info'
        n = self._get_check_name('snap_type_valid')
        s = 'OK'
        if self.snap_yaml['type'] not in self.valid_snap_types:
            t = 'error'
            s = "unknown 'type': '%s'" % self.snap_yaml['type']
        self._add_result(t, n, s)

    def check_type_redflagged(self):
        '''Check if type is redflagged'''
        if not self.is_snap2 or 'type' not in self.snap_yaml:
            return

        t = 'info'
        n = self._get_check_name('snap_type_redflag')
        s = "OK"
        l = None
        manual_review = False
        if self.snap_yaml['type'] in self.redflagged_snap_types:
            t = 'error'
            s = "(NEEDS REVIEW) type '%s' not allowed" % self.snap_yaml['type']
            manual_review = True
            if self.snap_yaml['type'] == "framework":
                l = "https://developer.ubuntu.com/en/snappy/guides/frameworks/"
        self._add_result(t, n, s, link=l, manual_review=manual_review)

    def check_version(self):
        '''Check package version'''
        if not self.is_snap2:
            return

        t = 'info'
        n = self._get_check_name('version_valid')
        s = 'OK'
        if 'version' not in self.snap_yaml:
            t = 'error'
            s = "could not find 'version' in yaml"
        elif not self._verify_pkgversion(self.snap_yaml['version']):
            t = 'error'
            s = "malformed 'version': '%s'" % self.snap_yaml['version']
        self._add_result(t, n, s)

    def check_config(self):
        '''Check config'''
        if not self.is_snap2:
            return

        fn = os.path.join(self._get_unpack_dir(), 'meta/hooks/config')
        if fn not in self.pkg_files:
            return

        t = 'info'
        n = self._get_check_name('config_hook_executable')
        s = 'OK'
        if not self._check_innerpath_executable(fn):
            t = 'error'
            s = 'meta/hooks/config is not executable'
        self._add_result(t, n, s)

    def check_icon(self):
        '''Check icon'''
        # see docs/meta.md and docs/gadget.md
        if not self.is_snap2 or 'icon' not in self.snap_yaml:
            return

        t = 'info'
        n = self._get_check_name('icon_present')
        s = 'OK'
        if 'type' in self.snap_yaml and self.snap_yaml['type'] != "gadget":
            t = 'warn'
            s = 'icon only used with gadget snaps'
            self._add_result(t, n, s)
            return
        self._add_result(t, n, s)

        t = 'info'
        n = self._get_check_name('icon_empty')
        s = 'OK'
        if len(self.snap_yaml['icon']) == 0:
            t = 'error'
            s = "icon entry is empty"
            self._add_result(t, n, s)
            return
        self._add_result(t, n, s)

        t = 'info'
        n = self._get_check_name('icon_absolute_path')
        s = 'OK'
        if self.snap_yaml['icon'].startswith('/'):
            t = 'error'
            s = "icon entry '%s' should not specify absolute path" % \
                self.snap_yaml['icon']
        self._add_result(t, n, s)

        t = 'info'
        n = self._get_check_name('icon_exists')
        s = 'OK'
        fn = self._path_join(self._get_unpack_dir(), self.snap_yaml['icon'])
        if fn not in self.pkg_files:
            t = 'error'
            s = "icon entry '%s' does not exist" % self.snap_yaml['icon']
        self._add_result(t, n, s)

    def check_unknown_entries(self):
        '''Check for any unknown fields'''
        if not self.is_snap2:
            return

        t = 'info'
        n = self._get_check_name('unknown_field')
        s = 'OK'
        unknown = []
        for f in self.snap_yaml:
            if f not in self.snappy_required + self.snappy_optional:
                unknown.append(f)
        if len(unknown) > 0:
            t = 'warn'
            s = "unknown entries in snap.yaml: '%s'" % \
                (",".join(sorted(unknown)))
        self._add_result(t, n, s)

    def check_apps(self):
        '''Check apps'''
        if not self.is_snap2:
            return

        key = 'apps'

        t = 'info'
        n = self._get_check_name('%s_present' % key)
        s = 'OK'
        if key not in self.snap_yaml:
            s = 'OK (optional %s field not specified)' % key
            self._add_result(t, n, s)
            return
        self._add_result(t, n, s)

        t = 'info'
        n = self._get_check_name(key)
        s = 'OK'
        if not isinstance(self.snap_yaml[key], dict):
            t = 'error'
            s = "invalid %s entry: %s (not a dict)" % (key,
                                                       self.snap_yaml[key])
            self._add_result(t, n, s)
            return
        elif len(self.snap_yaml[key].keys()) < 1:
            t = 'error'
            s = "invalid %s entry (empty)" % (key)
            self._add_result(t, n, s)
            return
        self._add_result(t, n, s)

        for app in self.snap_yaml[key]:
            t = 'info'
            n = self._get_check_name('%s_entry' % key, app=app)
            s = 'OK'

            if not isinstance(self.snap_yaml[key][app], dict):
                t = 'error'
                s = "invalid entry: %s (not a dict)" % (
                    self.snap_yaml[key][app])
                self._add_result(t, n, s)
                continue
            elif len(self.snap_yaml[key][app].keys()) < 1:
                t = 'error'
                s = "invalid entry for '%s' (empty)" % (app)
                self._add_result(t, n, s)
                continue
            self._add_result(t, n, s)

            for field in self.apps_required:
                t = 'info'
                n = self._get_check_name('%s_required' % key, app=app)
                s = 'OK'
                if field not in self.snap_yaml[key][app]:
                    t = 'error'
                    s = "required field '%s' not specified" % field
                self._add_result(t, n, s)

            t = 'info'
            n = self._get_check_name('%s_unknown' % key, app=app)
            s = 'OK'
            unknown = []
            for field in self.snap_yaml[key][app]:
                if field not in self.apps_required + self.apps_optional:
                    unknown.append(field)
            if len(unknown) > 0:
                t = 'warn'
                s = "unknown fields: '%s'" % (",".join(sorted(unknown)))
            self._add_result(t, n, s)

    def _verify_value_is_file(self, app, key):
            t = 'info'
            n = self._get_check_name('%s' % key, app=app)
            s = 'OK'
            if not isinstance(self.snap_yaml['apps'][app][key], str):
                t = 'error'
                s = "%s '%s' (not a str)" % (key,
                                             self.snap_yaml['apps'][app][key])
                self._add_result(t, n, s)
            elif len(self.snap_yaml['apps'][app][key]) < 1:
                t = 'error'
                s = "invalid %s (empty)" % (key)
                self._add_result(t, n, s)
            else:
                fn = self._path_join(self._get_unpack_dir(),
                                     self.snap_yaml['apps'][app][key])
                if fn not in self.pkg_files:
                    t = 'error'
                    s = "%s does not exist" % (
                        self.snap_yaml['apps'][app][key])
            self._add_result(t, n, s)

    def check_apps_command(self):
        '''Check apps - command'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        for app in self.snap_yaml['apps']:
            key = 'command'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            self._verify_value_is_file(app, key)

    def check_apps_stop(self):
        '''Check apps - stop'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        for app in self.snap_yaml['apps']:
            key = 'stop'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            self._verify_value_is_file(app, key)

    def check_apps_poststop(self):
        '''Check apps - poststop'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        for app in self.snap_yaml['apps']:
            key = 'poststop'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            self._verify_value_is_file(app, key)

    def check_apps_stop_timeout(self):
        '''Check apps - stop-timeout'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        for app in self.snap_yaml['apps']:
            key = 'stop-timeout'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            t = 'info'
            n = self._get_check_name('%s' % key, app=app)
            s = "OK"
            if not isinstance(self.snap_yaml['apps'][app][key], int) and \
                    not isinstance(self.snap_yaml['apps'][app][key], str):
                t = 'error'
                s = "'%s' is not a string or integer" % key
            elif not re.search(r'[0-9]+[ms]?$',
                               str(self.snap_yaml['apps'][app][key])):
                t = 'error'
                s = "'%s' is not of form NN[ms] (%s)" % \
                    (self.snap_yaml['apps'][app][key], key)
            self._add_result(t, n, s)

            if t == 'error':
                continue

            t = 'info'
            n = self._get_check_name('%s_range' % key, app=app)
            s = "OK"
            st = int(str(self.snap_yaml['apps'][app][key]).rstrip(r'[ms]'))
            if st < 0 or st > 60:
                t = 'error'
                s = "stop-timeout '%d' out of range (0-60)" % \
                    self.snap_yaml['apps'][app][key]
            self._add_result(t, n, s)

    def _verify_valid_values(self, app, key, valid):
        '''Verify valid values for key in app'''
        t = 'info'
        n = self._get_check_name('%s' % key, app=app)
        s = 'OK'
        if not isinstance(self.snap_yaml['apps'][app][key], str):
            t = 'error'
            s = "%s '%s' (not a str)" % (key,
                                         self.snap_yaml['apps'][app][key])
            self._add_result(t, n, s)
        elif len(self.snap_yaml['apps'][app][key]) < 1:
            t = 'error'
            s = "invalid %s (empty)" % (key)
            self._add_result(t, n, s)
        elif self.snap_yaml['apps'][app][key] not in valid:
            t = 'error'
            s = "invalid %s: '%s'" % (key, self.snap_yaml['apps'][app][key])
        self._add_result(t, n, s)

    def check_apps_daemon(self):
        '''Check apps - daemon'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        valid = ["simple",
                 "forking",
                 "oneshot",
                 "dbus",
                 ]

        for app in self.snap_yaml['apps']:
            key = 'daemon'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            self._verify_valid_values(app, key, valid)

    def check_apps_nondaemon(self):
        '''Check apps - non-daemon'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        # Certain options require 'daemon' so list the keys that are shared
        # by services and binaries
        ok_keys = ['command', 'uses']

        for app in self.snap_yaml['apps']:
            needs_daemon = []
            for key in self.snap_yaml['apps'][app]:
                if key not in self.apps_optional or \
                        key == 'daemon' or \
                        key in ok_keys or \
                        'daemon' in self.snap_yaml['apps'][app]:
                    continue
                needs_daemon.append(key)

            t = 'info'
            n = self._get_check_name('daemon_required', app=app)
            s = "OK"
            if len(needs_daemon) > 0:
                t = 'error'
                s = "'%s' must be used with 'daemon'" % ",".join(needs_daemon)
            self._add_result(t, n, s)

    def check_apps_restart_condition(self):
        '''Check apps - restart-condition'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        valid = ["always",
                 "never",
                 "on-abnormal",
                 "on-abort",
                 "on-failure",
                 "on-success",
                 ]

        for app in self.snap_yaml['apps']:
            key = 'restart-condition'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            self._verify_valid_values(app, key, valid)

    def check_apps_busname(self):
        '''Check apps - bus-name'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        for app in self.snap_yaml['apps']:
            key = 'bus-name'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            t = 'info'
            n = self._get_check_name('%s_framework' % key, app=app)
            s = 'OK'
            if 'type' in self.snap_yaml and \
                    self.snap_yaml['type'] != 'framework':
                t = 'error'
                s = "Use of bus-name requires package be of 'type: framework'"
            self._add_result(t, n, s)

            t = 'info'
            n = self._get_check_name('%s' % key, app=app)
            s = 'OK'
            l = None
            if not isinstance(self.snap_yaml['apps'][app][key], str):
                t = 'error'
                s = "%s '%s' (not a str)" % (key,
                                             self.snap_yaml['apps'][app][key])
            elif len(self.snap_yaml['apps'][app][key]) < 1:
                t = 'error'
                s = "invalid %s (empty)" % (key)
            elif not re.search(
                    r'^[A-Za-z0-9][A-Za-z0-9_-]*(\.[A-Za-z0-9][A-Za-z0-9_-]*)+$',
                    self.snap_yaml['apps'][app][key]):
                t = 'error'
                l = 'http://dbus.freedesktop.org/doc/dbus-specification.html'
                s = "'%s' is not of form '^[A-Za-z0-9][A-Za-z0-9_-]*(\\.[A-Za-z0-9][A-Za-z0-9_-]*)+$'" % \
                    (self.snap_yaml['apps'][app][key])
            self._add_result(t, n, s, l)
            if t == 'error':
                continue

            t = 'info'
            n = self._get_check_name('%s_matches_name' % key, app=app)
            s = 'OK'
            suggested = [self.snap_yaml['name'],
                         "%s.%s" % (self.snap_yaml['name'], app)
                         ]
            found = False
            for name in suggested:
                if self.snap_yaml['apps'][app][key].endswith(name):
                    found = True
                    break
            if not found:
                t = 'error'
                s = "'%s' doesn't end with one of: %s" % \
                    (self.snap_yaml['apps'][app][key], ", ".join(suggested))
            self._add_result(t, n, s)

    def check_apps_ports(self):
        '''Check apps - ports'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        valid_keys = ['internal', 'external']
        valid_subkeys = ['port', 'negotiable']
        for app in self.snap_yaml['apps']:
            if 'ports' not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            t = 'info'
            n = self._get_check_name('ports', app=app)
            s = 'OK'
            l = None
            if not isinstance(self.snap_yaml['apps'][app]['ports'], dict):
                t = 'error'
                s = "ports '%s' (not a dict)" % (
                    self.snap_yaml['apps'][app]['ports'])
            elif len(self.snap_yaml['apps'][app]['ports'].keys()) < 1:
                t = 'error'
                s = "'ports' must contain 'internal' and/or 'external'"
            self._add_result(t, n, s, l)
            if t == 'error':
                continue

            # unknown
            unknown = []
            for key in self.snap_yaml['apps'][app]['ports']:
                if key not in valid_keys:
                    unknown.append(key)
            if len(unknown) > 0:
                t = 'error'
                n = self._get_check_name('ports_unknown_key', extra=key,
                                         app=app)
                s = "Unknown '%s' for ports" % (",".join(unknown))
                self._add_result(t, n, s)

            port_pat = re.compile(r'^[0-9]+/[a-z0-9\-]+$')
            for key in valid_keys:
                if key not in self.snap_yaml['apps'][app]['ports']:
                    continue

                if len(self.snap_yaml['apps'][app]['ports'][key].keys()) < 1:
                    t = 'error'
                    n = self._get_check_name('ports', extra=key, app=app)
                    s = 'Could not find any %s ports' % key
                    self._add_result(t, n, s)
                    continue

                for tagname in self.snap_yaml['apps'][app]['ports'][key]:
                    entry = self.snap_yaml['apps'][app]['ports'][key][tagname]
                    if len(entry.keys()) < 1 or ('negotiable' not in entry and
                                                 'port' not in entry):
                        t = 'error'
                        n = self._get_check_name('ports', extra=key, app=app)
                        s = "Could not find 'port' or 'negotiable' in '%s'" % \
                            tagname
                        self._add_result(t, n, s)
                        continue

                    # unknown
                    unknown = []
                    for subkey in entry:
                        if subkey not in valid_subkeys:
                            unknown.append(subkey)
                    if len(unknown) > 0:
                        t = 'error'
                        n = self._get_check_name('ports_unknown_subkey',
                                                 extra=key, app=app)
                        s = "Unknown '%s' for %s" % (",".join(unknown),
                                                     tagname)
                        self._add_result(t, n, s)

                    # port
                    subkey = 'port'
                    t = 'info'
                    n = self._get_check_name('ports_%s_format' % tagname,
                                             extra=subkey)
                    s = 'OK'
                    if subkey not in entry:
                        s = 'OK (skipped, not found)'
                    elif not isinstance(entry[subkey], str):
                        t = 'error'
                        s = "invalid entry: %s (not a str)" % (entry[subkey])
                    else:
                        tmp = entry[subkey].split('/')
                        if not port_pat.search(entry[subkey]) or \
                           int(tmp[0]) < 1 or int(tmp[0]) > 65535:
                            t = 'error'
                            s = "'%s' should be of form " % entry[subkey] + \
                                "'port/protocol' where port is an integer " + \
                                "(1-65535) and protocol is found in " + \
                                "/etc/protocols"
                    self._add_result(t, n, s)

                    # negotiable
                    subkey = 'negotiable'
                    t = 'info'
                    n = self._get_check_name('ports_%s_format' % tagname,
                                             extra=subkey)
                    s = 'OK'
                    if subkey not in entry:
                        s = 'OK (skipped, not found)'
                    elif not isinstance(entry[subkey], bool):
                        t = 'error'
                        s = "'%s: %s' should be either 'yes' or 'no'" % \
                            (subkey, entry[subkey])
                    self._add_result(t, n, s)

    def check_apps_socket(self):
        '''Check apps - socket'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        for app in self.snap_yaml['apps']:
            key = 'socket'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            t = 'info'
            n = self._get_check_name(key, app=app)
            s = 'OK'
            if not isinstance(self.snap_yaml['apps'][app][key], bool):
                t = 'error'
                s = "'%s: %s' should be either 'yes' or 'no'" % (
                    key, self.snap_yaml['apps'][app][key])
            elif 'listen-stream' not in self.snap_yaml['apps'][app]:
                t = 'error'
                s = "'socket' specified without 'listen-stream'"
            self._add_result(t, n, s)

    def check_apps_listen_stream(self):
        '''Check apps - listen-stream'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        for app in self.snap_yaml['apps']:
            key = 'listen-stream'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            t = 'info'
            n = self._get_check_name(key, app=app)
            s = 'OK'
            if not isinstance(self.snap_yaml['apps'][app][key], str):
                t = 'error'
                s = "invalid entry: %s (not a str)" % (
                    self.snap_yaml['apps'][app][key])
            elif len(self.snap_yaml['apps'][app][key]) == 0:
                t = 'error'
                s = "'%s' is empty" % key
            self._add_result(t, n, s)
            if t == 'error':
                continue

            t = 'info'
            n = self._get_check_name('%s_matches_name' % key, app=app)
            s = 'OK'
            sock = self.snap_yaml['apps'][app][key]
            pkgname = self.snap_yaml['name']
            if sock.startswith('@'):
                if sock != '@%s' % pkgname and \
                        not sock.startswith('@%s_' % pkgname):
                    t = 'error'
                    s = ("abstract socket '%s' is neither '%s' nor starts "
                         "with '%s'" % (sock, '@%s' % pkgname,
                                        '@%s_' % pkgname))
            elif sock.startswith('/'):
                found = False
                for path in ["/tmp/",
                             "/var/lib/snaps/%s/" % pkgname,
                             "/var/lib/snaps/%s." % pkgname,
                             "/run/shm/snaps/%s/" % pkgname,
                             "/run/shm/snaps/%s." % pkgname]:
                    if sock.startswith(path):
                        found = True
                        break
                if not found:
                    t = 'error'
                    s = ("named socket '%s' should be in a writable "
                         "app-specific area or /tmp" % sock)
            else:
                t = 'error'
                s = ("'%s' does not specify an abstract socket (starts "
                     "with '@') or absolute filename" % (sock))
            self._add_result(t, n, s)

    def _verify_valid_socket(self, app, key):
        '''Verify valid values for socket key'''
        t = 'info'
        n = self._get_check_name(key, app=app)
        s = 'OK'
        if not isinstance(self.snap_yaml['apps'][app][key], str):
            t = 'error'
            s = "invalid entry: %s (not a str)" % (
                self.snap_yaml['apps'][app][key])
        elif len(self.snap_yaml['apps'][app][key]) == 0:
            t = 'error'
            s = "'%s' is empty" % key
        elif 'listen-stream' not in self.snap_yaml['apps'][app]:
            t = 'error'
            s = "'%s' specified without 'listen-stream'" % key
        self._add_result(t, n, s)
        if t == 'error':
            return

        t = 'error'
        n = self._get_check_name('%s_reserved' % key, app=app)
        s = "'%s' should not be used until snappy supports per-app users" \
            % key
        self._add_result(t, n, s)

        t = 'info'
        n = self._get_check_name("%s_matches_name" % key, app=app)
        s = 'OK'
        if self.snap_yaml['apps'][app][key] != self.snap_yaml['name']:
            t = 'error'
            s = "'%s' != '%s'" % (self.snap_yaml['apps'][app][key],
                                  self.snap_yaml['name'])
        self._add_result(t, n, s)

    def check_apps_socket_user(self):
        '''Check apps - socket-user'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        for app in self.snap_yaml['apps']:
            key = 'socket-user'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            self._verify_valid_socket(app, key)

    def check_apps_socket_group(self):
        '''Check apps - socket-group'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        for app in self.snap_yaml['apps']:
            key = 'socket-group'
            if key not in self.snap_yaml['apps'][app]:
                # We check for required elsewhere
                continue

            self._verify_valid_socket(app, key)

    def check_uses(self):
        '''Check uses'''
        if not self.is_snap2 or 'uses' not in self.snap_yaml:
            return

        for slot in self.snap_yaml['uses']:
            # If the 'type' name is the same as the 'slot' name, then 'type'
            # is optional since the type name and the slot name are the same
            skill_type = slot
            if 'type' in self.snap_yaml['uses'][slot]:
                skill_type = self.snap_yaml['uses'][slot]['type']

                key = 'type'
                t = 'info'
                n = self._get_check_name(key, extra=slot)
                s = 'OK'
                if not isinstance(self.snap_yaml['uses'][slot][key], str):
                    t = 'error'
                    s = "invalid %s: %s (not a str)" % \
                        (key, self.snap_yaml['uses'][slot][key])
                elif len(self.snap_yaml['uses'][slot][key]) == 0:
                    t = 'error'
                    s = "'%s' is empty" % key
                self._add_result(t, n, s)
                if t == 'error':
                    continue

            t = 'info'
            n = self._get_check_name(skill_type, extra=slot)
            s = 'OK'
            if skill_type not in self.skill_types:
                t = 'error'
                s = "unknown skill type '%s'" % skill_type
            self._add_result(t, n, s)
            if t == 'error':
                continue

            min = 1
            if 'type' in self.snap_yaml['uses'][slot]:
                min = 2
            t = 'info'
            n = self._get_check_name('attributes')
            s = 'OK'
            if len(self.snap_yaml['uses'][slot]) < min:
                t = 'error'
                s = "'%s' has no attributes" % slot
            self._add_result(t, n, s)
            if t == 'error':
                continue

            for attrib in self.snap_yaml['uses'][slot]:
                if attrib == 'type':
                    continue
                t = 'info'
                n = self._get_check_name('attributes', app=slot, extra=attrib)
                s = "OK"
                if attrib not in self.skill_types[skill_type]:
                    t = 'error'
                    s = "unknown attribute '%s' for type '%s'" % (attrib,
                                                                  skill_type)
                elif not isinstance(self.snap_yaml['uses'][slot][attrib],
                                    type(self.skill_types[skill_type][attrib])):
                    t = 'error'
                    s = "'%s' is not '%s'" % \
                        (attrib,
                         type(self.skill_types[skill_type][attrib]).__name__)
                self._add_result(t, n, s)

    def check_apps_uses(self):
        '''Check uses'''
        if not self.is_snap2 or 'apps' not in self.snap_yaml:
            return

        for app in self.snap_yaml['apps']:
            key = 'uses'
            if key not in self.snap_yaml['apps'][app]:
                continue

            t = 'info'
            n = self._get_check_name("app_%s" % key, app=app)
            s = "OK"
            if not isinstance(self.snap_yaml['apps'][app][key], list):
                t = 'error'
                s = "invalid '%s' entry: '%s' (not a list)" % (
                    key, self.snap_yaml['apps'][app][key])
            elif len(self.snap_yaml['apps'][app][key]) < 1:
                t = 'error'
                s = "invalid %s entry (empty)" % (key)
            self._add_result(t, n, s)
            if t == 'error':
                continue

            # The skill referenced in the app's 'uses' field can either be a
            # known skill type (when the type name and the name of the skill is
            # the same) or can reference a name in the snap's toplevel 'uses'
            # mapping
            for slot_ref in self.snap_yaml['apps'][app][key]:
                t = 'info'
                n = self._get_check_name('app_uses_slot_reference',
                                         app=app,
                                         extra=slot_ref)
                s = "OK"
                if not isinstance(slot_ref, str):
                    t = 'error'
                    s = "invalid slot skill name reference: '%s' (not a str)" \
                        % slot_ref
                elif slot_ref not in self.skill_types and \
                        'uses' not in self.snap_yaml or \
                        slot_ref not in self.snap_yaml['uses']:
                    t = 'error'
                    s = "unknown slot skill name reference '%s'" % slot_ref
                self._add_result(t, n, s)

    def check_external_symlinks(self):
        '''Check snap for external symlinks'''
        if not self.is_snap2:
            return

        # Note: unclear if kernel and gadget snaps can legitimately have
        # external symlinks, but err on side of caution
        if 'type' in self.snap_yaml and self.snap_yaml['type'] == 'os':
            return

        t = 'info'
        n = self._get_check_name('external_symlinks')
        s = 'OK'
        links = find_external_symlinks(self._get_unpack_dir(), self.pkg_files)
        if len(links) > 0:
            t = 'error'
            s = 'package contains external symlinks: %s' % ', '.join(links)
        self._add_result(t, n, s)

    def check_architecture_all(self):
        '''Check if actually architecture all'''
        if not self.is_snap2:
            return

        if 'architectures' in self.snap_yaml and \
                'all' not in self.snap_yaml['architectures']:
            return

        t = 'info'
        n = self._get_check_name('valid_contents_for_architecture')
        s = 'OK'

        # look for compiled code
        x_binaries = []
        for i in self.pkg_bin_files:
            x_binaries.append(os.path.relpath(i, self._get_unpack_dir()))
        if len(x_binaries) > 0:
            t = 'error'
            s = "found binaries for architecture 'all': %s" % \
                ", ".join(x_binaries)
        self._add_result(t, n, s)

    def check_architecture_specified_needed(self):
        '''Check if the specified architecture is actually needed'''
        if not self.is_snap2 or 'architectures' not in self.snap_yaml:
            return

        if 'all' in self.snap_yaml['architectures']:
            return

        for arch in self.snap_yaml['architectures']:
            t = 'info'
            n = self._get_check_name('architecture_specified_needed',
                                     extra=arch)
            s = 'OK'
            if len(self.pkg_bin_files) == 0:
                t = 'warn'
                s = "Could not find compiled binaries for architecture '%s'" \
                    % arch
            self._add_result(t, n, s)

    def check_vcs(self):
        '''Check for VCS files in the package'''
        if not self.is_snap2:
            return

        t = 'info'
        n = self._get_check_name('vcs_files')
        s = 'OK'
        found = []
        for d in self.vcs_files:
            entries = glob.glob("%s/%s" % (self._get_unpack_dir(), d))
            if len(entries) > 0:
                for i in entries:
                    found.append(os.path.relpath(i, self.unpack_dir))
        if len(found) > 0:
            t = 'warn'
            s = 'found VCS files in package: %s' % ", ".join(found)
        self._add_result(t, n, s)
