#!/usr/bin/python3

# ------------------------------------------------------------------
#
#    Copyright (C) 2013 Canonical Ltd.
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License published by the Free Software Foundation.
#
# ------------------------------------------------------------------

# not needed for python3, but pyflakes is grumpy without it
from __future__ import print_function

from apparmor import click
from apparmor.click import AppName
import apparmor.easyprof
import json
import os
import shutil
import sys
import tempfile
import unittest


class ClickState(object):
    def __init__(self, rootdir):
        self.rootdir = rootdir
        # note: this contains a '_' to trip up the name parsing code
        # if not properly filtered off
        self.click_dir = os.path.join(self.rootdir, "click_apparmor")
        os.mkdir(self.click_dir)
        self.packages_tree = os.path.join(self.rootdir, "click.ubuntu.com")
        os.mkdir(self.packages_tree)
        self.profiles_dir = os.path.join(self.rootdir, "apparmor-profiles")
        os.mkdir(self.profiles_dir)

    def _stub_json(self, name, version):
        j = dict()
        j['name'] = name
        j['version'] = version
        j['hooks'] = dict()
        j['title'] = "Some silly demo app called %s" % (name)
        j['maintainer'] = 'Buggy McJerkerson <buggy@bad.soft.warez>'
        return j

    def _add_app_to_json(self, app):
        j = self.manifest_json
        j['hooks'][app] = dict()
        j['hooks'][app]['apparmor'] = "apparmor/%s.json" % (app)
        with open(self.click_manifest, "w") as f:
            json.dump(j, f, indent=2, separators=(',', ': '), sort_keys=True)

    def add_package(self, name, version, framework=None):
        self.package = name
        self.version = version
        self.click_pkgdir = os.path.join(self.packages_tree, name, version,
                                         ".click", "info")
        self.pkg_dir = os.path.join(self.packages_tree, name, version)
        os.makedirs(self.click_pkgdir)
        self.manifest_json = self._stub_json(name, version)
        if framework:
            self.manifest_json['framework'] = framework
        self.click_manifest = os.path.join(self.click_pkgdir, "%s.manifest" %
                                           (name))
        with open(self.click_manifest, "w+") as f:
            json.dump(self.manifest_json, f, indent=2, separators=(',', ': '),
                      sort_keys=True)

    def add_app(self, appname, manifest=None):
        os.mkdir(os.path.join(self.click_pkgdir, "apparmor"))
        aa_json = os.path.join(self.click_pkgdir, "apparmor", "%s.json" %
                               (appname))
        self._add_app_to_json(appname)

        with open(aa_json, "w+") as f:
            if manifest:
                f.write(manifest)
        os.symlink(aa_json, os.path.join(self.click_dir,
                   "%s_%s_%s.json" % (self.package, appname, self.version)))


class T(unittest.TestCase):
    def setUp(self):
        self.tmpdir = tempfile.mkdtemp(prefix="aa-click-manifest-")
        self.clickstate = ClickState(self.tmpdir)
        self.maxDiff = None

    def tearDown(self):
        if os.path.exists(self.tmpdir):
            shutil.rmtree(self.tmpdir)

    def test_manifest_name_parsing_1(self):
        '''simple acceptance test for parsing click manifest name'''
        ex_app = "com.ubuntu.developer.username.myapp"
        ex_name = "myapp"
        ex_vers = "0.1"
        fname = "%s_%s_%s.json" % (ex_app, ex_name, ex_vers)

        (app, appname, version) = click.parse_manifest_name(fname)

        self.assertEquals(ex_app, app, "expected app %s, got %s" %
                                       (ex_app, app))
        self.assertEquals(ex_name, appname, "expected appname %s, got %s" %
                                            (ex_name, appname))
        self.assertEquals(ex_vers, version, "expected version %s, got %s" %
                                            (ex_vers, version))

    def test_bad_manifest_name(self):
        '''test manifest name with not enough elements'''
        ex_app = "com.ubuntu.developer.username.myapp"
        ex_name = "myapp"
        ex_vers = "0.1"
        fname = "%s_%s%s" % (ex_app, ex_name, ex_vers)

        try:
            (app, appname, version) = click.parse_manifest_name(fname)
        except click.AppArmorException:
            return
        except Exception:
            raise
        raise Exception("name %s should be invalid" % (fname))

    def test_bad_manifest_name_2(self):
        '''test manifest name with too many elements'''
        ex_app = "com.ubuntu.developer.username.myapp"
        ex_name = "myapp"
        ex_vers = "0.1"
        fname = "%s_%s_%s_err" % (ex_app, ex_name, ex_vers)

        try:
            (app, appname, version) = click.parse_manifest_name(fname)
        except click.AppArmorException:
            return
        except Exception:
            raise
        raise Exception("name %s should be invalid" % (fname))

    def test_click_name_to_profile(self):
        '''test click name conversion to profile name'''

        profile = "com.ubuntu.developer.username.myapp_myapp_0.3"
        expected = "%s.json" % (profile)
        orig = "click_%s" % (profile)
        app = click.AppName(profilename=orig)
        self.assertEquals(expected, app.clickname,
                          "expected click name %s, got %s" % (expected,
                                                              app.clickname))

    def test_profile_name_to_click(self):
        '''test profile name conversion to click name'''

        orig = "com.ubuntu.developer.username.myapp_myapp_0.3.json"
        expected = "click_com.ubuntu.developer.username.myapp_myapp_0.3"
        app = click.AppName(orig)
        self.assertEquals(expected, app.profilename,
                          "expected profile name %s, got %s" %
                          (expected, app.profilename))

    def test_find_manifest_file(self):
        '''test being given a symlink and finding the main package manifest'''
        c = self.clickstate
        c.add_package("package", "version")
        c.add_app("app")

        (result, pkg_dir) = apparmor.click.get_package_manifest(
            os.path.join(c.click_dir, "package_app_version.json"), "package")
        self.assertEquals(c.click_manifest, result,
                          "Expected to get %s, got %s" %
                          (c.click_manifest, result))
        self.assertEquals(c.pkg_dir, pkg_dir, "Expected to get %s, got %s" %
                                              (c.pkg_dir, pkg_dir))

    def test_parse_security_manifest(self):
        '''test being given a symlink and parsing the manifests'''
        c = self.clickstate
        c.add_package("com.ubuntu.developer.username.myapp", "0.1")
        security_json = '''{
  "policy_groups": [ "networking" ],
  "policy_version": "1.0"
}'''
        c.add_app('sample-app', manifest=security_json)

        n = "com.ubuntu.developer.username.myapp_sample-app_0.1.json"
        nbase = n.strip(".json")
        cm = apparmor.click.ClickManifest(os.path.join(c.click_dir, n))
        easyprof_manifest = apparmor.click.transform(cm)

        dbus_id = apparmor.click.dbus_path(nbase)

        expected = json.loads('''{
  "profiles": {
    "com.ubuntu.developer.username.myapp_sample-app_0.1": {
      "policy_groups": [ "networking" ],
      "policy_vendor": "ubuntu",
      "policy_version": "1.0",
      "template": "ubuntu-sdk",
      "template_variables": {
        "APP_ID_DBUS": "%s",
        "APP_PKGNAME": "com.ubuntu.developer.username.myapp",
        "APP_VERSION": "0.1",
        "CLICK_DIR": "%s"
      }
    }
  }
}''' % (dbus_id, c.packages_tree))
        self.assertEquals(expected, easyprof_manifest,
                          "Expected to get %s, got %s" %
                          (expected, easyprof_manifest))

        # verify we're testing the json structures deeply
        expected['profiles'][nbase]['template_variables']['BOGUS_VAR'] = \
            'bogus'
        self.assertNotEquals(expected, easyprof_manifest,
                             "Expected %s and %s to differ" %
                             (expected, easyprof_manifest))

    # the top level of the security manifest is not *supposed* to
    # be a single dict entry of the app, but apparently this happens.
    # Accept it and move on
    def test_parse_security_manifest_lenient(self):
        '''test being given a symlink and parsing the manifests, leniently'''
        c = self.clickstate
        c.add_package("com.ubuntu.developer.username.myapp", "0.3")
        security_json = '''{
  "sample-app": {
    "policy_groups": [ "networking" ],
    "policy_version": "1.0"
  }
}'''
        c.add_app('sample-app', manifest=security_json)

        n = "com.ubuntu.developer.username.myapp_sample-app_0.3.json"
        nbase = n.strip(".json")
        cm = apparmor.click.ClickManifest(os.path.join(c.click_dir, n))
        easyprof_manifest = apparmor.click.transform(cm)

        dbus_id = apparmor.click.dbus_path(nbase)

        expected = json.loads('''{
  "profiles": {
    "%s": {
      "policy_groups": [ "networking" ],
      "policy_vendor": "ubuntu",
      "policy_version": "1.0",
      "template": "ubuntu-sdk",
      "template_variables": {
        "APP_ID_DBUS": "%s",
        "APP_PKGNAME": "com.ubuntu.developer.username.myapp",
        "APP_VERSION": "0.3",
        "CLICK_DIR": "%s"
      }
    }
  }
}''' % (nbase, dbus_id, c.packages_tree))
        self.assertEquals(expected, easyprof_manifest,
                          "Expected to get %s, got %s" %
                          (expected, easyprof_manifest))

    def test_parse_security_manifest_with_framework(self):
        '''test being given a symlink and parsing the manifests w/framework
           setting'''
        c = self.clickstate
        c.add_package("com.ubuntu.developer.username.myapp", "0.1",
                      framework='ubuntu-sdk-13.10')
        security_json = '''{
  "policy_groups": [ "networking" ],
  "policy_version": "1.0"
}'''
        c.add_app('sample-app', manifest=security_json)

        n = "com.ubuntu.developer.username.myapp_sample-app_0.1.json"
        nbase = n.strip(".json")
        cm = apparmor.click.ClickManifest(os.path.join(c.click_dir, n))
        easyprof_manifest = apparmor.click.transform(cm)

        dbus_id = apparmor.click.dbus_path(nbase)

        expected = json.loads('''{
  "profiles": {
    "%s": {
      "policy_groups": [ "networking" ],
      "policy_vendor": "ubuntu",
      "policy_version": "1.0",
      "template": "ubuntu-sdk",
      "template_variables": {
        "APP_ID_DBUS": "%s",
        "APP_PKGNAME": "com.ubuntu.developer.username.myapp",
        "APP_VERSION": "0.1",
        "CLICK_DIR": "%s"
      }
    }
  }
}''' % (nbase, dbus_id, c.packages_tree))
        self.assertEquals(expected, easyprof_manifest,
                          "Expected to get %s, got %s" %
                          (expected, easyprof_manifest))

    def test_parse_security_manifest_unknown_framework(self):
        '''test unknown framework raises exception'''
        c = self.clickstate
        c.add_package("com.ubuntu.developer.username.myapp", "0.1",
                      framework='unknown-sdk-13.10')
        security_json = '''{
  "policy_groups": [ "networking" ],
  "policy_version": "1.0"
}'''
        c.add_app('sample-app', manifest=security_json)

        n = "com.ubuntu.developer.username.myapp_sample-app_0.1.json"
        try:
            cm = apparmor.click.ClickManifest(os.path.join(c.click_dir, n))
            apparmor.click.transform(cm)
        except click.AppArmorException:
            return
        except Exception:
            raise
        raise Exception("Framework should be invalid")

    # the security policy is supposed to have a policy_groups entry even
    # if the list inside it is empty. However, I'm seeing click apps
    # without it so we'll accept it and move_on
    def test_parse_security_manifest_no_policy_groups(self):
        '''test being given a symlink and parsing the manifests, leniently'''
        c = self.clickstate
        package = "com.ubuntu.developer.username.yourapp"
        c.add_package(package, "0.3")
        security_json = '{ "policy_version": "1.0" }'
        c.add_app('sample-app', manifest=security_json)

        cm = apparmor.click.ClickManifest(os.path.join(c.click_dir,
                                          "%s_sample-app_0.3.json" % package))
        easyprof_manifest = apparmor.click.transform(cm)

        dbus_id = apparmor.click.dbus_path("%s_sample-app_0.3" % package)

        expected = json.loads('''{
  "profiles": {
    "%s_sample-app_0.3": {
      "policy_groups": [ ],
      "policy_vendor": "ubuntu",
      "policy_version": "1.0",
      "template": "ubuntu-sdk",
      "template_variables": {
        "APP_ID_DBUS": "%s",
        "APP_PKGNAME": "%s",
        "APP_VERSION": "0.3",
        "CLICK_DIR": "%s"
      }
    }
  }
}''' % (package, dbus_id, package, c.packages_tree))
        self.assertEquals(expected, easyprof_manifest,
                          "Expected to get %s, got %s" %
                          (expected, easyprof_manifest))

    def test_parse_security_manifest_no_entries(self):
        '''test being given a symlink and parsing the manifests, no policy
           whatsoever'''
        c = self.clickstate
        package = "com.ubuntu.newbie.username.yourapp"
        c.add_package(package, "0.3")
        security_json = '{ }'
        c.add_app('sample-app', manifest=security_json)

        cm = apparmor.click.ClickManifest(os.path.join(c.click_dir,
                                          "%s_sample-app_0.3.json" % package))
        easyprof_manifest = apparmor.click.transform(cm)

        dbus_id = apparmor.click.dbus_path("%s_sample-app_0.3" % package)

        expected = json.loads('''{
  "profiles": {
    "%s_sample-app_0.3": {
      "policy_groups": [ ],
      "policy_vendor": "ubuntu",
      "policy_version": "1.0",
      "template": "ubuntu-sdk",
      "template_variables": {
        "APP_ID_DBUS": "%s",
        "APP_PKGNAME": "%s",
        "APP_VERSION": "0.3",
        "CLICK_DIR": "%s"
      }
    }
  }
}''' % (package, dbus_id, package, c.packages_tree))
        self.assertEquals(expected, easyprof_manifest,
                          "Expected to get %s, got %s" % (expected,
                                                          easyprof_manifest))

    def test_parse_security_manifest_unconfined(self):
        '''test being given a symlink and parsing the manifests for an
           unconfined app'''

        c = self.clickstate
        appname = "com.ubuntu.developer.trusteddev.terminal"
        c.add_package(appname, "0.99.9~123")
        security_json = '''{
  "template": "unconfined",
  "policy_groups": [ ],
  "policy_version": "1.0"
}'''
        c.add_app('0wnzered', manifest=security_json)

        cm = apparmor.click.ClickManifest(os.path.join(c.click_dir,
                                          "%s_0wnzered_0.99.9~123.json" %
                                          (appname)))
        easyprof_manifest = apparmor.click.transform(cm)

        dbus_id = apparmor.click.dbus_path("%s_0wnzered_0.99.9~123" % appname)

        expected = json.loads('''{
  "profiles": {
    "%s_0wnzered_0.99.9~123": {
      "policy_groups": [ ],
      "policy_vendor": "ubuntu",
      "policy_version": "1.0",
      "template": "unconfined",
      "template_variables": {
        "APP_ID_DBUS": "%s",
        "APP_PKGNAME": "%s",
        "APP_VERSION": "0.99.9~123",
        "CLICK_DIR": "%s"
      }
    }
  }
}''' % (appname, dbus_id, appname, c.packages_tree))
        self.assertEquals(expected, easyprof_manifest,
                          "Expected to get %s, got %s" % (expected,
                                                          easyprof_manifest))

    def test_parse_security_manifest_alternative_policy_vendor(self):
        '''Test that an alternative vendor is ok'''

        c = self.clickstate
        appname = "com.ubuntu.developer.trusteddev.terminal"
        c.add_package(appname, "0.99.9~123")
        security_json = '''{
  "template": "unconfined",
  "policy_groups": [ ],
  "policy_version": "1.0",
  "policy_vendor": "somevendor"
}'''
        c.add_app('0wnzered', manifest=security_json)

        cm = apparmor.click.ClickManifest(os.path.join(c.click_dir,
                                          "%s_0wnzered_0.99.9~123.json" %
                                          (appname)))
        easyprof_manifest = apparmor.click.transform(cm)

        dbus_id = apparmor.click.dbus_path("%s_0wnzered_0.99.9~123" % appname)

        expected = json.loads('''{
  "profiles": {
    "%s_0wnzered_0.99.9~123": {
      "policy_groups": [ ],
      "policy_vendor": "somevendor",
      "policy_version": "1.0",
      "template": "unconfined",
      "template_variables": {
        "APP_ID_DBUS": "%s",
        "APP_PKGNAME": "%s",
        "APP_VERSION": "0.99.9~123",
        "CLICK_DIR": "%s"
      }
    }
  }
}''' % (appname, dbus_id, appname, c.packages_tree))
        self.assertEquals(expected, easyprof_manifest,
                          "Expected to get %s, got %s" % (expected,
                                                          easyprof_manifest))


class AppArmorManifestSynchronization(unittest.TestCase):
    def setUp(self):
        self.tmpdir = tempfile.mkdtemp(prefix="aa-manifest-consistency-")
        self.clickstate = ClickState(self.tmpdir)
        self.maxDiff = None

    def tearDown(self):
        if os.path.exists(self.tmpdir):
            shutil.rmtree(self.tmpdir)

    def test_two_equal_directories(self):
        '''test two equal directories'''
        c = self.clickstate
        clicks = ["alpha_beta_gamma", "click_click_version",
                  "wat_no-really_wat"]
        for cname in clicks:
            with open(os.path.join(c.click_dir, '%s.json' %
                      (cname)), 'w+') as f:
                f.write('invalid json here')
            with open(os.path.join(c.profiles_dir, 'click_%s' %
                                   (cname)), 'w+') as f:
                f.write('profile %s { }' % (cname))

        expected = []
        result = click.get_missing_profiles(c.click_dir, c.profiles_dir)
        self.assertEquals(expected, result,
                          "Expected to get no profiles, got %s" % (result))
        result = click.get_missing_clickhooks(c.click_dir, c.profiles_dir)
        self.assertEquals(expected, result,
                          "Expected to get no click hooks, got %s" % (result))

    def test_two_empty_directories_for_profiles(self):
        '''test two empty directories returns no new needed profiles'''
        c = self.clickstate

        expected = []
        result = click.get_missing_profiles(c.click_dir, c.profiles_dir)
        self.assertEquals(expected, result,
                          "Expected to get no profiles, got %s" % (result))

    def test_two_empty_directories_for_clicks(self):
        '''test two empty directories returns no removed click hooks'''
        c = self.clickstate

        expected = []
        result = click.get_missing_clickhooks(c.click_dir, c.profiles_dir)
        self.assertEquals(expected, result,
                          "Expected to get no clickhooks, got %s" % (result))

    def test_versus_empty_profiles_directory(self):
        '''test against empty profiles directory'''
        c = self.clickstate
        clicks = ["alpha_beta_gamma.json", "click_click_version.json",
                  "wat_no-really_wat.json"]
        for cname in clicks:
            with open(os.path.join(c.click_dir, cname), 'w+') as f:
                f.write('invalid json here')

        expected = set(clicks)
        result = set(click.get_missing_profiles(c.click_dir, c.profiles_dir))
        self.assertEquals(expected, result,
                          "Expected to get %s profiles, got %s" % (expected,
                                                                   result))

    def test_versus_empty_clicks_directory(self):
        '''test against empty clicks directory'''
        c = self.clickstate
        clicks = ["alpha_beta_gamma", "click_click_version",
                  "wat_no-really_wat"]
        for cname in clicks:
            with open(os.path.join(c.profiles_dir, 'click_%s' %
                                   (cname)), 'w+') as f:
                f.write('profile %s { }' % (cname))

        expected = set(["%s%s" % (AppName._CLICK_PREFIX, x) for x in clicks])
        result = set(click.get_missing_clickhooks(c.click_dir, c.profiles_dir))
        self.assertEquals(expected, result,
                          "Expected to get %s hooks, got %s" % (expected,
                                                                result))

    def test_two_unequal_directories(self):
        '''test two equal directories'''
        c = self.clickstate
        expected_clicks = ['missing_click_profile.json',
                           'another-missing_click_profile.json']
        expected_profiles = ['%sremoved_click_package' %
                             (AppName._CLICK_PREFIX)]
        clicks = ["alpha_beta_gamma.json", "click_click_version.json",
                  "wat_no-really_wat.json"]
        profiles = ["%s%s" % (AppName._CLICK_PREFIX,
                              x[:-len('.json')]) for x in clicks]
        clicks.extend(expected_clicks)
        profiles.extend(expected_profiles)
        for cname in clicks:
            with open(os.path.join(c.click_dir, cname), 'w+') as f:
                f.write('invalid json here')
        for pname in profiles:
            with open(os.path.join(c.profiles_dir, pname), 'w+') as f:
                f.write('profile %s { }' % (pname))

        result = set(click.get_missing_profiles(c.click_dir, c.profiles_dir))
        self.assertEquals(set(expected_clicks), result,
                          "Expected to get '%s' missing profiles, got %s" %
                          (set(expected_clicks), result))
        result = set(click.get_missing_clickhooks(c.click_dir, c.profiles_dir))
        self.assertEquals(set(expected_profiles), result,
                          "Expected to get '%s' missing click hooks, got %s" %
                          (set(expected_profiles), result))


class AppArmorPolicyModificationTests(unittest.TestCase):

    def setUp(self):
        self.remove_profiles = []

    def tearDown(self):
        apparmor_remove = "/sys/kernel/security/apparmor/.remove"
        for p in self.remove_profiles:
            with open(apparmor_remove, 'w') as f:
                f.write(p)

    def test_load_simple_policy(self):
        '''test basic load policy function'''
        policy = "profile invalid_policy_dont_use { }"
        with tempfile.NamedTemporaryFile(prefix="aa-clicktest-",
                                         delete=False) as profile:
            profile.write(bytes(policy, 'utf-8'))
            name = profile.name

        try:
            rc, output = click.load_profile(name)
        except:
            raise

        self.remove_profiles.append('invalid_policy_dont_use')
        os.remove(name)

    def test_load_simple_policies(self):
        '''test basic load polices function'''
        policy = "profile invalid_policy_dont_use { }"
        with tempfile.NamedTemporaryFile(prefix="aa-clicktest-",
                                         delete=False) as profile:
            profile.write(bytes(policy, 'utf-8'))
            name = profile.name

        try:
            rc, output = click.load_profiles([name])
        except:
            raise

        self.remove_profiles.append('invalid_policy_dont_use')
        os.remove(name)

    def test_unload_simple_policy(self):
        policy = "profile invalid_policy_dont_use { }"
        with tempfile.NamedTemporaryFile(prefix="aa-clicktest-",
                                         delete=False) as profile:
            profile.write(bytes(policy, 'utf-8'))
            name = profile.name

        try:
            rc, output = click.load_profile(name)
        except:
            raise

        click.unload_profile('invalid_policy_dont_use')
        os.remove(name)

    def test_unload_simple_policies(self):
        policies = ['more_invalid_policy_%s' % (suffix)
                    for suffix in ['blart', 'blort', 'blat']]
        names = []
        for p in policies:
            policy = "profile %s { }" % (p)
            with tempfile.NamedTemporaryFile(prefix="aa-clicktest-",
                                             delete=False) as profile:
                profile.write(bytes(policy, 'utf-8'))
                names.append(profile.name)

        try:
            rc, output = click.load_profiles(names)
        except:
            raise

        click.unload_profiles(policies)
        for name in names:
            os.remove(name)

    def test_unload_nonexistent_policy(self):
        profile = 'invalid_policy_does_not_exist_dont_use'
        click.unload_profile(profile)

    def test_apparmor_not_available(self):
        '''Test apparmor not available'''
        try:
            click.apparmor_available(parser="/nonexistent",
                                     apparmor_dirs=['/nonexistent.dir'])
        except click.AppArmorException:
            return
        except Exception:
            raise
        raise Exception("apparmor_available should have failed")


if __name__ == '__main__':
    suite = unittest.TestSuite()
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(T))
    if os.geteuid() == 0:
        suite.addTest(unittest.TestLoader().loadTestsFromTestCase(
                      AppArmorPolicyModificationTests))
    else:
        print('Not running as root, skipping tests that load/unload ' +
              'AppArmor policy', file=sys.stderr)
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(
                  AppArmorManifestSynchronization))

    rc = unittest.TextTestRunner(verbosity=2).run(suite)

    if not rc.wasSuccessful():
        sys.exit(1)
