# ------------------------------------------------------------------
#
#    Copyright (C) 2013 Canonical Ltd.
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License published by the Free Software Foundation.
#
# ------------------------------------------------------------------

# from __future__ import with_statement

from apparmor import easyprof
import apparmor
import errno
import json
import os
import re
import shutil
import sys
import tempfile

# from apparmor.easyprof import AppArmorException, error
# or apparmor.common ?

# TODO: again, move this into the common library
apparmor_fs = "/sys/kernel/security/apparmor"

#
# TODO: move this out to the common library
#
#from apparmor import AppArmorException
class AppArmorException(Exception):
    '''This class represents AppArmor exceptions'''
    def __init__(self, value):
        self.value = value

    def __str__(self):
        return repr(self.value)
#
# End common
#

#
# TODO: move this out to a utilities library
#
def error(out, exit_code=1, do_exit=True):
    '''Print error message and exit'''
    try:
        sys.stderr.write("ERROR: %s\n" % (out))
    except IOError:
        pass

    if do_exit:
        sys.exit(exit_code)

#
# TODO: move this to a utilities library
#
def load_profile(profile, parser="/sbin/apparmor_parser", args=['-r', '--write-cache']):

    if not os.path.exists(parser):
        rc, parser = easyprof.cmd(['which', 'apparmor_parser'])
        if rc != 0:
            raise AppArmorException("Could not find apparmor_parser. Skipping load")

    command = [parser]
    command.extend(args)
    command.append(profile)
    rc, output = easyprof.cmd(command)
    if rc != 0:
         raise AppArmorException("policy load failed with errno %d: %s" %(rc, output))
    return (rc, output)

#
# TODO: move this to a utilities library
#
def load_profiles(profiles, parser="/sbin/apparmor_parser", args=['-r', '--write-cache']):

    # it's possible we can be passed an empty list, be forgiving
    if len(profiles) == 0:
        return (0, [])

    if not os.path.exists(parser):
        rc, parser = easyprof.cmd(['which', 'apparmor_parser'])
        if rc != 0:
            raise AppArmorException("Could not find apparmor_parser. Skipping load")

    command = [parser]
    command.extend(args)
    command.extend(profiles)
    rc, output = easyprof.cmd(command)
    if rc != 0:
         raise AppArmorException("policy load failed with errno %d: %s" %(rc, output))
    return (rc, output)

#
# TODO: move this to a utilities library
#
def unload_profile(profile):
    '''unload a profile name (not a filename) from the kernel'''

    apparmor_remove = os.path.join(apparmor_fs, ".remove")
    try:
        with open(apparmor_remove, 'w') as f:
            f.write(profile)
    except (OSError, IOError) as e:
        if e.errno != errno.ENOENT:
            raise

#
# TODO: move this to a utilities library
#
def unload_profiles(profiles):
    '''unload a list of profile name (not filenames) from the kernel'''

    apparmor_remove = os.path.join(apparmor_fs, ".remove")
    for profile in profiles:
        try:
            with open(apparmor_remove, 'w') as f:
                f.write(profile)
        except (OSError, IOError) as e:
            if e.errno != errno.ENOENT:
                raise

#
# adjust easyprof output_policy to behave like this
#
def output_policy(easyp, params, count, dir, force=True):
    '''Output policy'''
    policy = easyp.gen_policy(**params)
    out_fn = None
    if not dir:
        if count:
            sys.stdout.write('### aa-easyprof profile #%d ###\n' % count)
        sys.stdout.write('%s\n' % policy)
        return None
    else:
        if 'profile_name' in params:
            out_fn = AppName(rawname=params['profile_name'])
        #elif 'binary' in params:
        #    out_fn = params['binary']
        else: # should not ever reach this
            raise AppArmorException("Could not determine output filename")

        # Generate an absolute path, convertng any path delimiters to '.'
        out_fn = os.path.join(dir, out_fn.profilename)
        # we differ here in that we want to overwrite if exists
        if not force and os.path.exists(out_fn):
            raise AppArmorException("'%s' already exists" % out_fn)

        if not os.path.exists(dir):
            os.mkdir(dir)

        if not os.path.isdir(dir):
            raise AppArmorException("'%s' is not a directory" % dir)

        f, fn = tempfile.mkstemp(prefix='aa-easyprof')
        if not isinstance(policy, bytes):
            policy = policy.encode('utf-8')
        os.write(f, policy)
        os.close(f)

        shutil.move(fn, out_fn)

    return out_fn

def walk_up(path):
    while True:
        yield path
        newpath = os.path.dirname(path)
        if newpath == path:
            return
        path = newpath

def get_package_manifest(hooklink, pkg):
    '''follow a click hook manifest symlink and return the main package manifest location'''

    json_dir = os.path.realpath(os.path.normpath(os.path.join(os.path.dirname(hooklink), os.readlink(hooklink))))
    manifest_name = "%s.manifest" %(pkg) 

    # walk up the tree to find the .click directory, then grab
    # info/manifest from below that
    for path in walk_up(json_dir):
        info = os.path.join(path, ".click", "info")
        manifest = os.path.join(info, manifest_name)
        if os.path.isdir(info) and os.path.exists(manifest):
            return (manifest, path)

    raise AppArmorException("Could not find click manifest '%s' from symlink '%s'" %(manifest_name, hooklink))

def get_click_dir(pkg_dir, pkg, ver):
    (p, last) = os.path.split(pkg_dir)
    if not ver == last:
        raise AppArmorException("package version dir '%s' does not match expected '%s'" %s(pkg_dir, ver))
    (result, last) = os.path.split(p)
    if not pkg == last:
        raise AppArmorException("package dir '%s' does not match expected '%s'" %(p, pkg))
    return result

required_click_fields = ["name", "version"]
required_apparmor_fields = ["policy_groups", "policy_version"]

def read_click_manifest(manifest):
    f = open(manifest, "r")
    j = json.load(f)
    for field in required_click_fields:
        if not field in j:
            raise AppArmorException("could not find required field '%s' in json" %(field))
    return j

def read_apparmor_manifest(manifest, app):
    f = open(manifest, "r")
    j = json.load(f)
    if (len(j.keys()) == 1) and app in j:
        j = j[app]
    for field in required_apparmor_fields:
        if not field in j:
            raise AppArmorException("could not find required field '%s' in json" %(field))
    return j

class ClickManifest(object):
    def __init__(self, clickhook):
        (package, app, version) = parse_manifest_name(os.path.basename(clickhook))
        (click_manifest, pkg_dir) = get_package_manifest(clickhook, package)
        click_json = read_click_manifest(click_manifest)
        profile_json = read_apparmor_manifest(clickhook, app)

        #if not package == click_json['name']:
        #    raise AppArmorException("package name '%s' in json doesn't match " %(field))

        self.appname = click_json['name']
        self.version = click_json['version']
        if 'framework' in click_json:
            self.framework = click_json['framework']
        else:
            self.framework = 'default'
        self.profiles = dict()
        self.profiles[app] = profile_json
        # FIXME: note that at some point in the future this may get
        # overridden by the package manifest entry, if defined
        self.click_dir = get_click_dir(pkg_dir, package, version)
        # security = json['security']
        # self.profiles = security['profiles']

class EasyprofManifest(object):
    def __init__(self):
        self.manifest = dict()
        self.manifest['profiles'] = dict()

    def add_profile(self, name, profile):
        self.manifest['profiles'][name] = profile

class AppName(object):
    _CLICK_PREFIX="click_"
    _CLICK_SUFFIX=".json"

    def __init__(self, clickname=None, profilename=None, rawname=None):
        # verify valid click tuple name?
        if clickname:
            self.clickname = clickname
        elif profilename:
            self.profilename = profilename
        elif rawname:
            self._clickname = rawname
        else:
            raise AppArmorException("AppName object needs an initial value")

    @property
    def clickname(self):
        return "%s%s" %(self._clickname, self._CLICK_SUFFIX)

    @clickname.setter
    def clickname(self, clickname):
        if not clickname.endswith(self._CLICK_SUFFIX):
            raise AppArmorException("invalid click manifest name '%s'" %(clickname))

        # strip the suffix off
        self._clickname = clickname[:-len(self._CLICK_SUFFIX)]

    @property
    def profilename(self):
        return "%s%s" %(self._CLICK_PREFIX, self._clickname)

    @profilename.setter
    def profilename(self, profilename):
        if not profilename.startswith(self._CLICK_PREFIX):
            raise AppArmorException("invalid click profile name '%s'" %(profilename))

        # strip the click prefix off of the string
        self._clickname = profilename[len(self._CLICK_PREFIX):]

class EasyprofProfile(object):
    def __init__(self, name):
        self.profile = dict()
        self.profile['template_variables'] = dict()
        self.profile['policy_groups'] = list()
        self.name = name

    def add_variable(self, var_name, var_value):
        self.profile['template_variables'][var_name] = var_value

    def add_policygroup(self, policygroup):
        self.profile['policy_groups'].append(policygroup)

    @property
    def template(self):
        if 'template' in self.profile:
            return self.profile['template']
        return None

    @template.setter
    def template(self, template):
        self.profile['template'] = template

    @property
    def policyvendor(self):
        if 'policy_vendor' in self.profile:
            return self.profile['policy_vendor']
        return None

    @policyvendor.setter
    def policyvendor(self, vendor):
        self.profile['policy_vendor'] = vendor

    @property
    def policyversion(self):
        if 'policy_version' in self.profile:
            return self.profile['policy_version']
        return None

    @policyversion.setter
    def policyversion(self, version):
        self.profile['policy_version'] = version

# ubuntu 13.10 tranformation function takes a click manifest json
# structure as input and returns an aa-easyprof json manifest

def ubuntu_1310_transform(click_manifest):
    '''Ubuntu 13.10 SDK framework transformation function'''
    m = EasyprofManifest()

    for p in click_manifest.profiles:
        newp = EasyprofProfile(p)
        newname = "%s_%s_%s" %(click_manifest.appname, p, click_manifest.version)

        profile = click_manifest.profiles[p]

        if 'template' in profile:
            newp.template = profile['template']
        else:
            newp.template = "ubuntu-sdk"

        newp.policyvendor = "ubuntu"

        if 'policy_version' in profile:
            newp.policyversion = profile['policy_version']
        else:
            newp.policyversion = "1.0"

        newp.add_variable("APPNAME", click_manifest.appname)
        newp.add_variable("APPVERSION", click_manifest.version)
        newp.add_variable("CLICK_DIR", click_manifest.click_dir)

        if 'policy_groups' in profile:
            for policygroup in profile['policy_groups']:
                newp.add_policygroup(policygroup)

        m.add_profile(newname, newp.profile)

    return m.manifest

# these define the transformation funtions given a specific framework
# definition in the ClickManifest file. This may need to grow more
# complex depending on how the framework definition grows in click
# packages

framework_functions = {
    "default": ubuntu_1310_transform,
    "ubuntu-sdk-13.10": ubuntu_1310_transform,
}

def parse_manifest_name(name):
    (n, ext) = os.path.splitext(name)
    if not ext == ".json":
        raise AppArmorException("unable to parse manifest name %s" %(name))

    out = n.split("_")
    if len(out) != 3:
        raise AppArmorException("unable to parse manifest name %s" %(name))

    return tuple(out)

# takes a ClickManifest as input and returns an
# aa-easyprof json structure
def transform(click_manifest):
    if not click_manifest.framework in framework_functions:
        raise AppArmorException("Unknown framework %s" %(click_manifest.framework))

    transform_function = framework_functions[click_manifest.framework]

    return transform_function(click_manifest)

# takes easyprof manifest and converts it into apparmor profiles
# directory is the output directory to place it into
# returns a list of paths written to
def to_profiles(easyprof_manifest, directory):

    (easy_opts, easy_args) = apparmor.easyprof.parse_args(args=[])

    profiles = apparmor.easyprof.parse_manifest(json.dumps(easyprof_manifest), easy_opts)
    files = []

    count = 0
    for (binary, options) in profiles:
        count += 1
        try:
            easyp = apparmor.easyprof.AppArmorEasyProfile(binary, options)
        except AppArmorException as e:
            error(e.value)
        except Exception:
            raise

        params = apparmor.easyprof.gen_policy_params(binary, options)
        # FIXME: easyp.gen_policy is probably preferred.
        f = output_policy(easyp, params, count, directory)
        if directory:
            files.append(f)

    return files

# returns a list of click links that are missing correlated profiles
def get_missing_profiles(hooksdir, profilesdir):
    '''find click hooks that are missing profiles'''
    result = []
    for hook in os.listdir(hooksdir):
        name = AppName(clickname=hook)
        if not os.path.exists(os.path.join(profilesdir, name.profilename)):
            result.append(name.clickname)
    return result

# returns a list of profiles that are missing correlated click links
def get_missing_clickhooks(hooksdir, profilesdir):
    '''find profiles that have had click packahes removed'''
    result = []
    for profile in os.listdir(profilesdir):
        try:
            name = AppName(profilename=profile)
        except apparmor.click.AppArmorException:
            continue
        if not os.path.exists(os.path.join(hooksdir, name.clickname)):
            result.append(name.profilename)
    return result
