/* ************************************************************************
 * Copyright 2013 Advanced Micro Devices, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ************************************************************************/

#include <cstring>
#include <gtest/gtest.h>
#include <boost/program_options.hpp>
#include "clFFT.h"
#include "clFFT.version.h"
#include "test_constants.h"
#include "../client/openCL.misc.h"
#include "unicode.compatibility.h"

namespace po = boost::program_options;
size_t number_of_random_tests;
time_t random_test_parameter_seed;
float tolerance;
bool verbose;

#if defined( MSVC_VER )
#if !defined( NOMINMAX )
	#define NOMINMAX
#endif
#define WIN32_LEAN_AND_MEAN			// Exclude rarely-used stuff from Windows headers
#include <intrin.h>

#if defined( _WIN64 )
void inline BSF( unsigned long* index, size_t& mask )
{
	_BitScanForward64( index, mask );
}
#else
void inline BSF( unsigned long* index, size_t& mask )
{
	_BitScanForward( index, mask );
}
#endif
#elif defined( __GNUC__ )
void inline BSF (unsigned long * index, size_t & mask) {
	*index = __builtin_ctz (mask);
}
#endif

// global for test use
bool suppress_output = false;

//	Globals that user can set on the command line, that need to be passed down to unit tests
cl_device_type device_type = CL_DEVICE_TYPE_GPU;
cl_uint device_gpu_list = ~0x0;
bool comparison_type = root_mean_square;

int main( int argc, char **argv )
{
	//	Define MEMORYREPORT on windows platfroms to enable debug memory heap checking
#if defined( MEMORYREPORT ) && defined( _WIN32 )
	TCHAR logPath[ MAX_PATH ];
	::GetCurrentDirectory( MAX_PATH, logPath );
	::_tcscat_s( logPath, _T( "\\MemoryReport.txt") );

	//	We leak the handle to this file, on purpose, so that the ::_CrtSetReportFile() can output it's memory
	//	statistics on app shutdown
	HANDLE hLogFile;
	hLogFile = ::CreateFile( logPath, GENERIC_WRITE,
		FILE_SHARE_READ|FILE_SHARE_WRITE, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL );

	::_CrtSetReportMode( _CRT_ASSERT, _CRTDBG_MODE_FILE | _CRTDBG_MODE_WNDW | _CRTDBG_MODE_DEBUG );
	::_CrtSetReportMode( _CRT_ERROR, _CRTDBG_MODE_FILE | _CRTDBG_MODE_WNDW | _CRTDBG_MODE_DEBUG );
	::_CrtSetReportMode( _CRT_WARN, _CRTDBG_MODE_FILE | _CRTDBG_MODE_DEBUG );

	::_CrtSetReportFile( _CRT_ASSERT, hLogFile );
	::_CrtSetReportFile( _CRT_ERROR, hLogFile );
	::_CrtSetReportFile( _CRT_WARN, hLogFile );

	int tmp = ::_CrtSetDbgFlag( _CRTDBG_REPORT_FLAG );
	tmp |= _CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF | _CRTDBG_CHECK_ALWAYS_DF;
	::_CrtSetDbgFlag( tmp );

	//	By looking at the memory leak report that is generated by this debug heap, there is a number with
	//	{} brackets that indicates the incremental allocation number of that block.  If you wish to set
	//	a breakpoint on that allocation number, put it in the _CrtSetBreakAlloc() call below, and the heap
	//	will issue a bp on the request, allowing you to look at the call stack
	//	::_CrtSetBreakAlloc( 997 );

#endif /* MEMORYREPORT */

	// Declare the supported options.
	po::options_description desc( "clFFT Runtime Test command line options" );
	desc.add_options()
		( "help,h",				"produces this help message" )
		( "verbose,v",			"print out detailed information for the tests" )
		( "noVersion",     "Don't print version information from the clFFT library" )
		( "noInfoCL",      "Don't print information from the OpenCL runtime" )
		( "cpu,c",         "Run tests on a CPU device" )
		( "gpu,g",         "Run tests on a GPU device (default)" )
		( "pointwise,p",         "Do a pointwise comparison to determine test correctness (default: use root mean square)" )
		( "tolerance,t",        po::value< float >( &tolerance )->default_value( 0.001f ),   "tolerance level to use when determining test pass/fail" )
		( "numRandom,r",        po::value< size_t >( &number_of_random_tests )->default_value( 2000 ),   "number of random tests to run" )
		( "seed",        po::value< time_t >( &random_test_parameter_seed )->default_value( time(NULL)%1308000000 ),
						"seed to use for the random test. defaults to time(NULL)" )
						// modulo lops off the first few digits of the time value to make the seed easier to type
						// even without these digits, the seed value won't wrap around until 2036 or later
		( "short,s",         "Run radix 2 tests; no random testing" )
		( "medium,m",         "Run all radices; no random testing" )
		;

	//	Parse the command line options, ignore unrecognized options and collect them into a vector of strings
	po::variables_map vm;
	po::parsed_options parsed = po::command_line_parser( argc, argv ).options( desc ).allow_unregistered( ).run( );
	po::store( parsed, vm );
	po::notify( vm );
	std::vector< std::string > to_pass_further = po::collect_unrecognized( parsed.options, po::include_positional );

	std::cout << std::endl;

	size_t mutex = ((vm.count( "gpu" ) > 0) ? 1 : 0)
		| ((vm.count( "cpu" ) > 0) ? 2 : 0);
	if ((mutex & (mutex-1)) != 0) {
		terr << _T("You have selected mutually-exclusive OpenCL device options:") << std::endl;
		if (vm.count ( "cpu" )  > 0) terr << _T("    cpu, c	Run tests on a CPU device" ) << std::endl;
		if (vm.count ( "gpu" )  > 0) terr << _T("    gpu, g	Run tests on a GPU device" ) << std::endl;
		return 1;
	}

	if( vm.count( "cpu" ) )
	{
		device_type = CL_DEVICE_TYPE_CPU;
	}

	if( vm.count( "gpu" ) )
	{
		device_type	= CL_DEVICE_TYPE_GPU;
		device_gpu_list = ~0;
	}

	//	Print version by default
	if( !vm.count( "noVersion" ) )
	{
		const int indent = countOf( "clFFT client API version: " );
		tout << std::left << std::setw( indent ) << _T( "clFFT client API version: " )
			<< clfftVersionMajor << _T( "." )
			<< clfftVersionMinor << _T( "." )
			<< clfftVersionPatch << std::endl;

		cl_uint libMajor, libMinor, libPatch;
		clfftGetVersion( &libMajor, &libMinor, &libPatch );

		tout << std::left << std::setw( indent ) << _T( "clFFT runtime version: " )
			<< libMajor << _T( "." )
			<< libMinor << _T( "." )
			<< libPatch << std::endl << std::endl;
	}

	//	Print clInfo by default
	if( !vm.count( "noInfoCL" ) )
	{
		cl_context tempContext = NULL;
		cl_command_queue tempQueue = NULL;
		cl_event tempEvent = NULL;
		std::vector< cl_device_id > device_id = ::initializeCL( device_type, device_gpu_list, tempContext, true );
		::cleanupCL( &tempContext, &tempQueue, 0, NULL, 0, NULL, &tempEvent );
	}

	if( vm.count( "help" ) )
	{
		std::cout << desc << std::endl;
		return 0;
	}

	if( vm.count( "verbose" ) )
	{
		verbose = true;
	}
	else
	{
		verbose = false;
	}

	if( vm.count( "short" ) && vm.count( "medium" ) )
	{
		terr << _T("Options 'short' and 'medium' are mutually-exclusive.  Please select only one.") << std::endl;
		return 1;
	}

	//	Create a new argc,argv to pass to InitGoogleTest
	//	First parameter of course is the name of this program
	std::vector< const char* > myArgv;

	//	Push back a pointer to the executable name
	if( argc > 0 )
		myArgv.push_back( *argv );

	//	Push into our new argv vector any parameter the user passed, except to filter their gtest_filter expressions
	std::string userFilter;
	for( int i = 1; i < argc; ++i )
	{
		if( vm.count( "short" ) || vm.count( "medium" ) )
		{
			std::string tmpStr( argv[ i ] );
			std::string::size_type pos = tmpStr.find( "gtest_filter" );
			if( pos == std::string::npos )
			{
				myArgv.push_back( argv[ i ] );
			}
			else
			{
				//  Capture the users filter, but only the regexp portion
				userFilter = argv[ i ];
				userFilter.erase( 0, 15 );
			}
		}
		else
		{
			myArgv.push_back( argv[ i ] );
		}
	}

	std::string newFilter;
	if( vm.count( "short" ) )
	{
		newFilter += "--gtest_filter=*accuracy_test_pow2*";
		if( userFilter.size( ) )
		{
			newFilter += ":";
			newFilter += userFilter;
		}
		myArgv.push_back( newFilter.c_str( ) );
	}

	if( vm.count( "medium" ) )
	{
		newFilter += "--gtest_filter=";
		if( userFilter.size( ) )
		{
			newFilter += userFilter;
			newFilter += ":";
		}
		newFilter += "-*Random*";
		myArgv.push_back( newFilter.c_str( ) );
	}

	if( vm.count( "pointwise" ) )
	{
		comparison_type = pointwise_compare;
	}
	else
	{
		comparison_type = root_mean_square;
	}

	int myArgc	= static_cast< int >( myArgv.size( ) );

	std::cout << "Result comparison tolerance is " << tolerance << std::endl;

	::testing::InitGoogleTest( &myArgc, const_cast< char** >( &myArgv[ 0 ] ) );

	return RUN_ALL_TESTS();
}
