#!/usr/bin/env python

########################################################################
# Chris's Lame File Bowser (Mimetype editor)
# Copyright 2004, Gabe Ginorio <gabe@zevallos.com.br>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
########################################################################

import string, re, os, os.path, commands, sys
import pygtk, gtk, pango
import mimetype
from xml.dom import minidom


########################################################################
# This function creates the mime editor dialog
########################################################################	
def editor(location, group, extention, image_path):

	# This will hold the entries
	global entries

	# Create a window for the editor
	mime_window = gtk.Window(gtk.WINDOW_TOPLEVEL)
	mime_window.set_title("Claw 4 Mime Type Editor")
	mime_window.connect("destroy", quit, mime_window)
	
	# Create a Table to format information in the mime editor window
	mime_table = gtk.Table()
	mime_window.add(mime_table)
	
	# Create a Frame for the icon picture association 
	iconFrame = gtk.Frame("Associated Image")
	mime_table.attach(iconFrame, 0,1,0,1, gtk.FILL, gtk.SHRINK)

	# Create an image using the associated icon
	icon_image = gtk.Image()
	
	# Get the theme from the preferences.xml file
	preferences_file = minidom.parse(os.environ['HOME'] + "/.claw/preferences.xml")
	node = preferences_file.getElementsByTagName("theme")
	theme = node[0].childNodes[0].data	
	
	# Remember that non xcf and svg images are picture previews
	if group == "image" and extention not in ["gimp", "x-3ds"]:
		icon_pixbuf = gtk.gdk.pixbuf_new_from_file(location)
		icon_pixbuf = icon_pixbuf.scale_simple(48, 48, gtk.gdk.INTERP_BILINEAR)	
		icon_image.set_from_pixbuf(icon_pixbuf)
	else:
		icon_image.set_from_file(image_path + "/Icons/" + extention + ".png")
	iconFrame.add(icon_image)
	
	# Create a Frame for the extention
	extentionFrame = gtk.Frame("Associated Extention")
	mime_table.attach(extentionFrame, 0, 1, 1, 2, gtk.FILL, gtk.SHRINK)
	
	# Create a Label for the extention
	extentionLabel = gtk.Label(extention)
	extentionFrame.add(extentionLabel)

	# Create a Frame for the applications
	applicationsFrame = gtk.Frame("Associated Applications")
	mime_table.attach(applicationsFrame, 0, 1, 2, 3, gtk.FILL, gtk.SHRINK)
	
	# Create a box to hold the applicatin entries
	applicationsBox = gtk.VBox()
	applicationsFrame.add(applicationsBox)
	
	# Get the applications with the mime type and format them for use
	applications = mimetype.get_associated_applications([group, extention])
	if len(applications) > 0:
		applications = map(lambda app: string.strip(app), applications)
	
	# This will hold the entry widgets
	global entries
	entries = []
	
	# Create the entries and editing buttons for the associated applications	
	# I created a function for this so we could add entry boxes later
	map(lambda app: createEntry(None, app, applicationsBox, image_path) ,applications)

	# Create a box to hold the add, save and cancel buttons
	button_box = gtk.HBox()
	mime_table.attach(button_box, 0, 1, 3, 4)

	# Create a tool tips item for the buttons
	button_tips = gtk.Tooltips()
	button_tips.enable()

	# Create the add button
	add_button = gtk.Button("Add")
	add_button.connect("clicked", createEntry,  "", applicationsBox, image_path)
	button_tips.set_tip(add_button, "Add An Application")
	button_box.pack_start(add_button)

	# Create the save button
	save_button = gtk.Button("Save")
	save_button.connect("clicked", save, mime_window, group, extention)
	button_tips.set_tip(save_button, "Save Changes and Exit")
	button_box.pack_start(save_button)

	# Create the cancel button	
	cancel_button = gtk.Button("Cancel")
	cancel_button.connect("clicked", quit, mime_window)
	button_tips.set_tip(cancel_button, "Change Nothing and Exit")
	button_box.pack_start(cancel_button)						
							
	# Now, we show the widget all at once						
	mime_window.show_all()


#######################################
# This function creates an entry
######################################
def createEntry(widget, app, applicationsBox, image_path):

	# Create a box for the entries and buttons
	entry_box = gtk.HBox()
	applicationsBox.pack_start(entry_box)
	
	# Create the entry
	entry = gtk.Entry()
	entry.set_text(app)
	# entry.set_state(gtk.STATE_INSENSITIVE)
	entry.set_editable(gtk.FALSE)
	entry_box.pack_start(entry)
		
	# Add the entry to the entries list
	entries.append(entry)

	# Create a tooltips item for the buttons
	tool_tips = gtk.Tooltips()
	tool_tips.enable()

	# Create the find button
	find_button = gtk.Button()
	find_image = gtk.Image()
	find_image.set_from_file(image_path + "/MimeEditor/find.png")
	find_button.add(find_image)
	find_button.connect("clicked", find, entry)
	tool_tips.set_tip(find_button, "Search For Application")
	entry_box.pack_start(find_button)		
		
	# Create the edit button
	edit_button = gtk.Button()
	edit_image = gtk.Image()
	edit_image.set_from_file(image_path + "/MimeEditor/edit.png")
	edit_button.add(edit_image)
	edit_button.connect("clicked", edit, entry)
	tool_tips.set_tip(edit_button, "Edit This Entry")
	entry_box.pack_start(edit_button)		
		
	# Create the delete button
	delete_button = gtk.Button()
	delete_image = gtk.Image()
	delete_image.set_from_file(image_path  + "/MimeEditor/delete.png")
	delete_button.add(delete_image)
	delete_button.connect("clicked", delete, entry_box)
	tool_tips.set_tip(delete_button, "Delete This Entry")
	entry_box.pack_start(delete_button)
		
	# This is only for if a box is added
	entry_box.show_all()


#######################################
# This function is the callback to find an app button
#######################################
def find(find_button, entry):
		
	# Create a file dialog
	file_get = gtk.FileSelection("Locate Application")
		
	# Set it to modal (the main window freezes while this is open)
	file_get.set_modal(gtk.TRUE)
		
	# Run the dialog
	response = file_get.run()
		
	# Get the returned application
	application = file_get.get_filename()

	# Destroy the widget when donw
	file_get.destroy()

	# If OK was not pressed then do nothing
	if response != gtk.RESPONSE_OK:
		return 1

	# Otherwise, see if the app is valid and you can execute it
	if os.path.exists(application) and os.access(application, os.X_OK):
			
		# Get the system application search path
		paths = string.split(os.environ['PATH'], ":")
			
		# See if the application is in the system path
		path = filter(lambda x : os.path.dirname(application) == x, paths)
			
		# If the application is in the path, remove the leading directories
		if path != []:
			application = string.replace(application, path[0] + "/", "")
		
		# Set the entry text to the application location/name
		entry.set_text(application)


#######################################
# This function is the callback to edit an entry
#######################################
def edit(edit_button, entry):

	# Make the entry editable (or not editable)
	if entry.get_editable() == gtk.TRUE:
		entry.set_editable(gtk.FALSE)
	else:
		entry.set_editable(gtk.TRUE)	
	

#######################################
# This function is the callback to delete an entry
#######################################
def delete(delete_button, entry_box):

	# First, we delete the entry box from the entries list
	entries.remove(entry_box.get_children()[0])

	# Then we delete the box whole box containing the entry and buttons
	entry_box.destroy()


#######################################
# This function is the callback to save the new mimetype
#######################################
def save(button, mime_window, group, extention):

	# Get the text in the entry boxes
	apps = map(lambda entry: entry.get_text(), entries)
	apps = map(string.strip, apps)
	
	# Make a comma sererated string of the associated applications
	text = None
	if apps != []:
		text = apps[0]
		if len(apps) > 1:
			for app in apps[1:]:
				if app != None:
					text += "," + app
				else:
					text += "" 

	# Open the appropriate mime-file and read it
	mimefile = file( os.environ['HOME'] + "/.claw/mime." + group, 'r')
	lines = mimefile.readlines()
	mimefile.close()

	# Open the appropriate mime-file for writing
	mimefile = file( os.environ['HOME'] + "/.claw/mime." + group, 'w')
	
	# Edit the appropriate line
	for line in lines:
		if re.findall(extention, string.split(line, "::")[0]) != []:
			ext, apps = string.split(line, "::")
			line = ext + "\t\t::\t" + text + "\n"
		
		mimefile.write(line)
	mimefile.close()	

	# Quit the mime editor
	quit(button, mime_window)


#######################################
# This function is the callback to destroy the app
#######################################
def quit(button, mime_window):

	# Destroy the window
	mime_window.destroy()

	
