// 
// Copyright (c) 2010, Benjamin Kaufmann
// 
// This file is part of Clasp. See http://www.cs.uni-potsdam.de/clasp/ 
// 
// Clasp is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// Clasp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Clasp; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
#ifndef CLASP_PARALLEL_SOLVE_H_INCLUDED
#define CLASP_PARALLEL_SOLVE_H_INCLUDED

#ifdef _MSC_VER
#pragma once
#endif

#ifndef DISABLE_MULTI_THREADING

#include <clasp/solve_algorithms.h>
#include <clasp/constraint.h>
#include <clasp/shared_context.h>
#include <clasp/util/thread.h>
#include <clasp/util/multi_queue.h>
#include <clasp/solver_types.h>

/*!
 * \file 
 * Defines classes controlling multi-threaded parallel solving.
 * 
 */
namespace Clasp { namespace mt {

class ParallelHandler;

//! A parallel algorithm for multi-threaded solving with and without search-space splitting.
/*!
 * The class adapts clasp's basic solve algorithm
 * to a parallel solve algorithm that solves
 * a problem using a given number of threads.
 * It supports guiding path based solving, portfolio based solving, as well
 * as a combination of these two approaches.
 */
class ParallelSolve : public SolveAlgorithm {
public:
	ParallelSolve();
	~ParallelSolve();
	// "setup" interface
	
	//! Returns number of threads that can run concurrently on the current hardware.
	static uint32  hardwareThreads();
	//! Used to signal an error in a client thread.
	static const   uint32 invalidId = UINT32_MAX;
	
	//! Initializes this object for solving with at most maxThreads.
	/*!
	 * \pre numThreads >= 1
	 * \pre ctx.shareCount() >= maxThreads
	 */
	SharedContext* init(int maxThreads, SharedContext& ctx);
	//! Enables/disables search-space splitting.
	void   setForceGP(bool gp);
	//! Configures nogood distribution.
	void   setIntegrate(uint32 grace, uint8 filter);
	//! Configures "global" restarts (only used if search-space splitting is active).
	void   setRestarts(uint32 maxR, const ScheduleStrategy& rs);
	//! Attaches a new thread to s and runs it in this algorithm.
	/*!
	 * Shall be called once for each solver except the master.
	 * \pre s.id() != ParallelSolve::invalidId
	 */
	void   addThread(Solver& s, SolveParams& p);
	
	// solve interface 
	
	//! Returns the number of active threads.
	uint32 numThreads()            const;
	bool   integrateUseHeuristic() const { return intHeuristic_; }
	uint32 integrateGrace()        const { return intGrace_; }
	uint32 integrateFlags()        const { return intFlags_; }
	//! Terminates current solving process and all client threads.
	bool   terminate();
	//! Requests a global restart.
	void   requestRestart();
	bool   handleMessages(Solver& s);
	void   pushWork(LitVec& gp);
private:
	ParallelSolve(const ParallelSolve&);
	ParallelSolve& operator=(const ParallelSolve&);
	typedef SingleOwnerPtr<const LitVec> PathPtr;
	enum ErrorCode { error_none = 0, error_oom = 1, error_runtime = 2, error_other = 4 };
	// -------------------------------------------------------------------------------------------
	// Thread setup 
	struct EntryPoint;
	void   reserveThreads();
	void   destroyThread(uint32 id);
	void   joinThreads();
	// -------------------------------------------------------------------------------------------
	// Algorithm steps
	void   startSolve(Solver& s, const SolveParams& p, const LitVec& assume);
	bool   endSolve(Solver& s); 
	bool   doSolve(Solver& s, const SolveParams& p, const LitVec& assume);
	bool   initOpt(Solver& s, ValueRep last);
	void   initQueue();
	bool   requestWork(Solver& s, PathPtr& out);
	bool   backtrackFromModel(Solver& s);
	void   terminate(Solver& s, bool complete);
	bool   waitOnSync(Solver& s);
	void   exception(Solver& s, PathPtr& path, ErrorCode e, const char* what);
	// -------------------------------------------------------------------------------------------
	struct SharedData;
	// SHARED DATA
	SharedData*       shared_;       // Shared control data
	ParallelHandler** thread_;       // Thread-locl control data
	// READ ONLY
	uint32            maxThreads_;   // number of threads alloacated 
	uint32            maxRestarts_;  // disable global restarts once reached 
	uint32            intGrace_;     // grace period for clauses to integrate
	uint32            intFlags_;     // bitset controlling clause integration
	bool              intHeuristic_; // use heuristic in clause integration
	bool              forceGP_;	     // force guiding path mode even if portfolio is used
};


//! A per-solver (i.e. thread) class that implements message handling and knowledge integration.
/*!
 * The class adds itself as a post propagator to the given solver. Each time
 * propagateFixpoint() is called (i.e. on each new decision level), it checks
 * for new lemmas to integrate and synchronizes the search with any new models.
 * Furthermore, it adds a second (high-priority) post propagator for message handling.
 */
class ParallelHandler : public PostPropagator {
public:
	//! Creates a new parallel handler to be used in the given solve group.
	/*!
	 * \param ctrl The object controlling the parallel solve operation.
	 */
	explicit ParallelHandler(ParallelSolve& ctrl);
	~ParallelHandler();

	//! Attaches this parallel handler to the given solver.
	/*!
	 * \param s The solver in which this object is used.
	 * \param p The solving parameters under which the solver operates.
	 */
	void attach(Solver& s, const SolveParams& p);

	//! Detaches this object from its solver.
	void detach();

	//! Executes F in a new thread.
	template <class F>
	void run(F f, Solver& s, SolveParams& p) {
		assert(!joinable() && solver_ == 0);
		std::thread(f, &s, &p).swap(thread_);
		assert(joinable());
	}
	
	void setError(int e) { error_ = e; }
	int  error() const   { return error_; }
	
	//! True if *this has an associated thread of execution, false otherwise.
	bool joinable() const { return thread_.joinable(); }
	//! Waits for the thread of execution associated with *this to finish.
	/*!
	 * \note The function is a noop of !joinable().
	 */
	int join() { if (joinable()) { thread_.join(); } return error_; }
	
	// overridden methods
	
	//! Returns a priority suited for a post propagators that is non-deterministic.
	uint32 priority() const { return priority_general + 100; }

	//! Integrates new information.
	bool propagateFixpoint(Solver& s);
	bool propagate(Solver& s) { return ParallelHandler::propagateFixpoint(s); }
	
	//! Checks whether new information has invalidated current model.
	bool isModel(Solver& s);

	// own interface
	
	// TODO: make functions virtual once necessary 
	
	//! Returns true if handler's guiding path is disjoint from all others.
	bool disjointPath() const { return gp_.split; }
	//! Returns true if handler has a guiding path.
	bool hasPath()      const { return gp_.impl != UINT32_MAX; }
	
	//! Called before solver starts to solve given guiding path.
	/*!
	 * \param gp      The new guiding path.
	 * \param restart Request restart after restart number of conflicts.
	 * \param isSplit True if gp resulted from a split.
	 */
	void prepareForGP(const LitVec& gp, uint64 restart, bool isSplit);

	/*!
	 * \name Message handlers
	 * \note 
	 *   Message handlers are intended as callbacks for ParallelSolve::handleMessages().
	 *   They shall not change the assignment of the solver object.
	 */
	//@{
	
	//! Algorithm is about to terminate.
	/*!
	 * Removes this object from the solver's list of post propagators.
	 */
	void handleTerminateMessage();

	//! Request for split.
	/*!
	 * Splits off a new guiding path and adds it to the control object.
	 * \pre The guiding path of this object is "splittable"
	 */
	void handleSplitMessage();

	//! Request for (global) restart.
	/*!
	 * \return true if restart is valid, else false.
	 */
	bool handleRestartMessage();

	SolveStats aggStats;  // aggregated statistics over all gps
	//@}  
private:
	static void threadMain(ParallelHandler* h);
	bool simplify(Solver& s, bool re);
	bool integrateClauses(Solver& s);
	void add(ClauseHead* h);
	ParallelSolve* ctrl() const { return messageHandler_.ctrl; }
	typedef LitVec::size_type size_type;
	typedef PodVector<Constraint*>::type ClauseDB;
	std::thread        thread_;     // active thread or empty for master
	ClauseDB           integrated_; // my integrated clauses
	Solver*            solver_;     // my solver
	const SolveParams* params_;     // my solving params
	size_type          intTail_;    // where to put next clause
	int                error_;      // error code or 0 if ok
	struct GP {
		LitVec      path;     // current guiding path
		uint64      restart;  // don't give up before restart number of conflicts
		size_type   pos;      // pos in trail
		uint32      impl;     // number of additional implied literals
		bool        split;    // does gp result from a split?
		void reset(uint64 r = UINT64_MAX, bool sp = false) {
			path.clear();
			restart = r;
			pos     = 0;
			impl    = 0;
			split   = sp;
		}
	} gp_;
	struct MessageHandler : PostPropagator {
		explicit MessageHandler(ParallelSolve* c) : ctrl(c) {}
		uint32 priority() const { return PostPropagator::priority_highest; }
		bool   propagateFixpoint(Solver& s) { return ctrl->handleMessages(s); }
		bool   propagate(Solver& s)         { return MessageHandler::propagateFixpoint(s); }
		ParallelSolve* ctrl; // get messages from here
	}    messageHandler_;
};

class GlobalQueue : public Distributor {
public:
	GlobalQueue(uint32 maxShare, uint32 typesToShare, uint32 maxLbd);
	~GlobalQueue();
	uint32  receive(const Solver& in, SharedLiterals** out, uint32 maxOut);
protected:
	void    doPublish(const Solver& source, SharedLiterals* lits);
private:
	void release();
	struct ClauseNode {
		ClauseNode()
			: targetMask(0), lits(0) {}
		uint64          targetMask;
		SharedLiterals* lits;
	};
	class Queue : public MultiQueue<ClauseNode> {
	public:
		typedef MultiQueue<ClauseNode> base_type;
		using base_type::publish;
		Queue(uint32 m) : base_type(m) {}
	};
	struct ThreadInfo {
		Queue::ThreadId id;
		char pad[64 - sizeof(Queue::ThreadId)];
	};
	Queue::ThreadId& getThreadId(uint32 sId) const {
		return threadId_[sId].id;
	}
	Queue*               queue_;
	ThreadInfo*          threadId_;
};

} }
#endif

#endif
