// 
// Copyright (c) 2006-2010, Benjamin Kaufmann
// 
// This file is part of Clasp. See http://www.cs.uni-potsdam.de/clasp/ 
// 
// Clasp is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// Clasp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Clasp; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
#include <clasp/solve_algorithms.h>
#include <clasp/solver.h>
#include <clasp/minimize_constraint.h>
#include <clasp/enumerator.h>
#include <clasp/lookahead.h>
#include <cmath>
using std::log;
namespace Clasp { 
/////////////////////////////////////////////////////////////////////////////////////////
// SolveParams
/////////////////////////////////////////////////////////////////////////////////////////
SolveParams::SolveParams() 
	: randFreq_(0.0)
	, shuffleFirst_(0), shuffleNext_(0) {
}

uint32 ReduceParams::init() const {
	if (disable) return UINT32_MAX;
	uint32 ret = static_cast<uint32>(base_/frac_);
	if      (ret < iMin_) ret = iMin_;
	else if (ret > iMax_) ret = iMax_;
	return ret;
}
/////////////////////////////////////////////////////////////////////////////////////////
// Schedule
/////////////////////////////////////////////////////////////////////////////////////////
void ScheduleStrategy::init(uint32 base, double grow, uint64 outer, bool arith) {
	grow_ = grow == 0.0 || grow >= 1.0 ? grow : 1.0;
	outer_= (outer ? outer : UINT64_MAX);
	base_ = base;
	arith_= static_cast<uint32>(arith);
	idx_  = 0;
}
uint64 ScheduleStrategy::current() const {
	uint64 x;
	if      (base_ == 0)      x = UINT64_MAX;
	else if (grow_ == 0)      x = lubyR();
	else                      x = growR();
	return x;
}
uint64 ScheduleStrategy::next() {
	++idx_;
	uint64 x = current();
	if (base_ != 0 && grow_ != 0 && x > outer_) {
		idx_    = 0;
		outer_  = arith_ == 0 ? static_cast<uint64>(outer_*grow_) : static_cast<uint64>(outer_+grow_);
		x       = growR();
	}
	return x;
}
uint64 ScheduleStrategy::growR() const { 
	return arith_ == 0
		? static_cast<uint64>(base_ * pow(grow_, (double)idx_))
		: static_cast<uint64>(base_ + (grow_*idx_));
}

uint64 ScheduleStrategy::lubyR() const {
	uint32 k = idx_+1;
	while (k) {
		uint32 nk = static_cast<uint32>(log((double)k) / log(2.0)) + 1;
		if (k == ((uint32(1) << nk) - 1)) {
			return base_ * (uint32(1) << (nk-1));
		}
		k -= uint32(1) << (nk-1);
		++k;
	}
	return base_;
}
/////////////////////////////////////////////////////////////////////////////////////////
// solve
/////////////////////////////////////////////////////////////////////////////////////////
bool solve(SharedContext& ctx, const SolveParams& p) {
	return SimpleSolve().solve(ctx, p, LitVec());
}

bool solve(SharedContext& ctx, const SolveParams& p, const LitVec& assumptions) {
	return SimpleSolve().solve(ctx, p, assumptions);
}

/////////////////////////////////////////////////////////////////////////////////////////
// SolveAlgorithm
/////////////////////////////////////////////////////////////////////////////////////////
SolveAlgorithm::SolveAlgorithm()  {
}
SolveAlgorithm::~SolveAlgorithm() {}

bool SolveAlgorithm::backtrackFromModel(Solver& s) { 
	return s.sharedContext()->enumerator()->backtrackFromModel(s) == Enumerator::enumerate_continue;
}

void SolveAlgorithm::reportProgress(int t, Solver& s, uint64 maxCfl, uint32 maxL) {
	return s.sharedContext()->enumerator()->reportProgress(Enumerator::ProgressType(t), s, maxCfl, maxL);
}
bool SolveAlgorithm::solve(SharedContext& ctx, const SolveParams& p, LitVec assume) {
	assert(ctx.master() && "SharedContext not initialized!\n");
	if (!isSentinel(ctx.tagLiteral())) {
		assume.push_back(ctx.tagLiteral());
	}
	bool r = doSolve(*ctx.master(), p, assume);
	ctx.detach(*ctx.master());
	return r;
}
bool SolveAlgorithm::initPath(Solver& s, const LitVec& path, InitParams& params) {
	assert(!s.hasConflict() && s.decisionLevel() == 0);
	SingleOwnerPtr<Lookahead> look(0);
	if (params.initLook != 0 && params.lookType != Lookahead::no_lookahead) {
		look = new Lookahead(static_cast<Lookahead::Type>(params.lookType));
		look->init(s);
		s.addPost(look.release());
		--params.initLook;
	}
	bool ok = s.propagate() && s.simplify();
	if (look.get()) { 
		s.removePost(look.get());
		look = look.get(); // restore ownership
	}
	if (!ok) { return false; }
	// setup path
	for (LitVec::size_type i = 0, end = path.size(); i != end; ++i) {
		Literal p = path[i];
		if (s.value(p.var()) == value_free) {
			s.assume(p); --s.stats.choices;
			// increase root level - assumption can't be undone during search
			s.pushRootLevel();
			if (!s.propagate())  return false;
		}
		else if (s.isFalse(p)) return false;
	}
	// do random probings if any
	if (uint32 i = params.randRuns) {
		params.randRuns = 0;
		do {
			if (s.search(params.randConf, UINT32_MAX, false, 1.0) != value_free) { return !s.hasConflict(); }
			s.undoUntil(0);
		} while (--i);
	}
	// do initial lookahead choices if requested
	if (uint32 i = params.initLook) {
		params.initLook = 0;
		assert(look.get());
		RestrictedUnit::decorate(s, i, look.release());
	}
	return true;
}

ValueRep SolveAlgorithm::solvePath(Solver& s, const SolveParams& p) {
	if (s.hasConflict()) return false;
	double maxLearnts         = p.reduce.init();
	const double boundLearnts = p.reduce.bound();
	if (maxLearnts < s.numLearntConstraints()) {
		maxLearnts = static_cast<double>(s.numLearntConstraints()) + p.reduce.initMin();
		maxLearnts = std::min(maxLearnts, (double)UINT32_MAX);
	}
	typedef Enumerator::ProgressType ProgressType;
	ScheduleStrategy rs = p.restart.sched;
	ScheduleStrategy ds = p.reduce.sched;
	ValueRep result     = value_free;
	uint32 shuffle      = p.shuffleBase();
	ProgressType t      = Enumerator::progress_restart;
	SolveLimits gLimit  = p.limits;     // global limit
	gLimit.restarts     = std::max(gLimit.restarts, uint64(1));
	uint64 rsLimit      = rs.current(); // current restart limit
	uint64 dsLimit      = ds.current(); // current deletion limit
	uint64 minLimit     = 0;            // min of all limits
	SearchLimits sLimit;
	while (result == value_free && !gLimit.reached()) {
		if  (!p.restart.local) { minLimit = std::min(rsLimit, std::min(dsLimit, gLimit.conflicts)); }
		else                   { minLimit = std::min(dsLimit, gLimit.conflicts); sLimit.local = rsLimit; }
		sLimit.conflicts = minLimit;
		sLimit.learnts   = (uint32)maxLearnts;
		reportProgress(t, s, std::min(sLimit.conflicts, sLimit.local), sLimit.learnts);
		result    = s.search(sLimit, p.randomProbability());
		minLimit  = (minLimit - sLimit.conflicts); // number of actual conflicts
		if (gLimit.conflicts != UINT64_MAX) { gLimit.conflicts -= minLimit; }
		if (result == value_true) {
			if (!backtrackFromModel(s)) {
				break; // No more models requested
			}
			else {
				result   = value_free; // continue enumeration
				t        = Enumerator::progress_model;
				if (p.restart.resetOnModel) {
					rs.reset();
				}
				// After the first solution was found, we allow further restarts only if this
				// is compatible with the enumerator used. 
				rsLimit  = !p.restart.bounded && s.backtrackLevel() > s.rootLevel()
					? static_cast<uint64>(-1)
					: rs.current();
				dsLimit  = ds.current();
			}
		}
		else if (result == value_free){  // limit reached
			rsLimit -= (!p.restart.local) * minLimit;
			dsLimit -= minLimit;
			if (dsLimit == 0 || s.numLearntConstraints() >= sLimit.learnts) {
				s.reduceLearnts(p.reduce.reduceFrac());
				dsLimit = dsLimit != 0 ? ds.current() : ds.next();
				t       = Enumerator::progress_reduce;
				if (s.numLearntConstraints() >= sLimit.learnts) { maxLearnts += s.numLearntConstraints(); }
			}
			else if (rsLimit == 0 || sLimit.local == 0) {
				// restart reached - do restart
				if (s.numFreeVars() != 0) { s.undoUntil(0); }
				t = Enumerator::progress_restart;
				rsLimit = rs.next();
				if (p.reduce.reduceOnRestart) { s.reduceLearnts(.33f); }
				if (maxLearnts != (double)UINT32_MAX && maxLearnts < boundLearnts && (s.numLearntConstraints()+rsLimit) > maxLearnts) {
					maxLearnts    = std::min(maxLearnts*p.reduce.inc(), (double)UINT32_MAX);
				}
				if (++s.stats.restarts == shuffle) {
					shuffle += p.shuffleNext();
					s.shuffleOnNextSimplify();
				}
				--gLimit.restarts;
			}
		}
	}
	p.limits = gLimit;
	return result;
}
/////////////////////////////////////////////////////////////////////////////////////////
// SimpleSolve
/////////////////////////////////////////////////////////////////////////////////////////
bool SimpleSolve::terminate() { return false; }
bool SimpleSolve::doSolve(Solver& s, const SolveParams& p, const LitVec& assume) {
	s.stats.reset();
	Enumerator*  enumerator = s.sharedContext()->enumerator();
	bool hasWork   = true, complete = true;
	InitParams init= p.init;
	// Remove any existing assumptions and restore solver to a usable state.
	// If this fails, the problem is unsat, even under no assumptions.
	while (s.clearAssumptions() && hasWork) {
		// Add assumptions - if this fails, the problem is unsat 
		// under the current assumptions but not necessarily unsat.
		if (initPath(s, assume, init)) {
			complete = (solvePath(s, p) != value_free && s.decisionLevel() == s.rootLevel());
		}
		// finished current work item
		hasWork    = complete && enumerator->optimizeNext();
	} 
	enumerator->reportResult(complete);
	return !complete;
}
}
