# ClamTk, copyright (C) 2004-2010 Dave M
#
# This file is part of ClamTk.
#
# ClamTk is free software; you can redistribute it and/or modify it
# under the terms of either:
#
# a) the GNU General Public License as published by the Free Software
# Foundation; either version 1, or (at your option) any later version, or
#
# b) the "Artistic License".
package ClamTk::App;

use strict;
#use warnings;    # disabled upon release
$|++;

use Carp qw/croak cluck/;
use Net::DNS;

use encoding 'utf8';

sub get_TK_version {

    # Stick with %.2f format - 4.50 vice 4.5
    return '4.26';
}

sub get_path {
    shift;    # throw one away
    my $wanted = shift;
    my $path;

    # These are directories and files necessary for
    # preferences, storing AV signatures and more

    # First, determine home directory
    $path->{directory} = $ENV{HOME} || ( ( getpwuid $< )[-2] );

    # Default personal clamtk directory
    $path->{clamtk} = $path->{directory} . '/.clamtk';

    # For storing quarantined files
    $path->{viruses} = $path->{clamtk} . '/viruses';

    # Store history logs here
    $path->{history} = $path->{clamtk} . '/history';

    # Plain text file for preferences
    $path->{prefs} = $path->{clamtk} . '/prefs';

    # Plain text file for restoring quarantined files
    $path->{restore} = $path->{clamtk} . '/restore';

    # The db directory stores virus defs/freshclam-related stuff
    $path->{db} = $path->{clamtk} . '/db';

    # Default variables
    $path->{whitelist_dir} =
        join( ';', $path->{viruses}, '/sys', '/dev', '/proc;' );

    # Most times freshclam is under /usr/bin
    $path->{freshclam} =
          ( -e '/usr/bin/freshclam' )       ? '/usr/bin/freshclam'
        : ( -e '/usr/local/bin/freshclam' ) ? '/usr/local/bin/freshclam'
        : ( -e '/opt/local/bin/freshclam' ) ? '/opt/local/bin/freshclam'
        :                                     '';

    # Most times clamscan is under /usr/bin
    # We'll use clampath as the actual path
    # and clamscan as clampath + scan options
    $path->{clampath} =
          ( -e '/usr/bin/clamscan' )       ? '/usr/bin/clamscan'
        : ( -e '/usr/local/bin/clamscan' ) ? '/usr/local/bin/clamscan'
        : ( -e '/opt/local/bin/clamscan' ) ? '/opt/local/bin/clamscan'
        :                                    '';

    $path->{clamscan} = $path->{clampath};

    $path->{clamscan} .= ' --no-summary --block-encrypted ';

    return ( $wanted eq 'all' ) ? $path : $path->{$wanted};
}

sub def_paths {

    # Returns (path_of_daily.c?d, path_of_main.c?d)
    # There are 3 formats:
    # 1. The .cld files
    # 2. The .cvd files
    # 3. The {daily,main}.info directories
    # As of 4.23, we're no longer looking foro the .info dirs.
    # The .cvd is the compressed database, while .cld is a
    # previous .cvd/.cld with incremental updates.

    my ( $DAILY_PATH, $MAIN_PATH );

    my @dirs = qw#
        /var/lib/clamav
        /var/clamav
        /opt/local/share/clamav
        /usr/share/clamav
        /usr/local/share/clamav
        /var/db/clamav
        #;

    my $user_set = 0;
    if ( ClamTk::Prefs->get_preference('Update') eq 'single' ) {
        $user_set = 1;
        my $paths = ClamTk::App->get_path('db');
        unshift( @dirs, $paths );
    }

    for my $dir_list (@dirs) {
        if ( -e "$dir_list/daily.cld" ) {
            $DAILY_PATH = "$dir_list/daily.cld";
        }
        elsif ( -e "$dir_list/daily.cvd" ) {
            $DAILY_PATH = "$dir_list/daily.cvd";
        }
        if ( -e "$dir_list/main.cld" ) {
            $MAIN_PATH = "$dir_list/main.cld";
        }
        elsif ( -e "$dir_list/main.cvd" ) {
            $MAIN_PATH = "$dir_list/main.cvd";
        }
        last if ( $DAILY_PATH && $MAIN_PATH );

        # the user may have set single - may need to update db
        last if ($user_set);
    }

    return ( $DAILY_PATH, $MAIN_PATH );
}

sub get_AV_version {

    # simple 'clamscan -V'
    my $paths = ClamTk::App->get_path('clampath');
    my ($version) = qx/$paths -V/;
    $version =~ s/ClamAV\s(.*?)\/.*/$1/;
    chomp($version);
    return $version;
}

sub get_AV_remote {

    # The user may have set the preference to not check this
    # on startup using 'AVCheck'
    return 'undef' if ( !ClamTk::Prefs->get_preference('AVCheck') );

    # Host for version
    my $rmt_host = 'current.cvd.clamav.net';

    # No longer need the alarm to "die" since Net::DNS
    # does it for us.
    # Just to be sure, we'll use OpenDNS servers here
    # and set a UDP timeout.
    my ( $get, $res );

    $res = Net::DNS::Resolver->new( udp_timeout => 3, );
    if ( ClamTk::Prefs->get_preference('OpenDNS') ) {
        $res->nameservers(
            qw(208.67.222.222 208.67.220.220
                208.67.222.220 208.67.220.222)
        );
    }

    ($get) = $res->send( $rmt_host, "TXT" );
    if ( !$get ) {
        warn $res->errorstring, "\n";
        return 'undef';
    }
    chomp($get);

  # This is a sample of the line that is returned ($get->string):
  # current.cvd.clamav.net.	439	IN	TXT	"0.95.3:51:10091:1259420373:1:44:12728"
  # We just need the version, held between the " and the :
    if ( $get->string =~ /IN\s+TXT\s+\"(.*?):/ ) {
        return $1;
    }
    else {
        return 'undef';
    }
}

sub get_num_sigs {

    # Adds the main + daily for total # of signatures
    my ( undef, $daily, $main ) = get_sig_info();
    return ( $daily + $main );
}

sub get_date_sigs {

    # Gets the date, preferably from daily.c?d.
    # Useful for informing user of outdated sigs.
    my ( $date, undef, undef ) = get_sig_info();
    return $date;
}

sub get_sig_info {
    my ( $daily_path, $main_path ) = def_paths();
    cluck "I'm missing daily.{cld,cvd} and/or main.{cld,cvd}.\n"
        unless ( $daily_path && $main_path );

    my ( $INFO_DATE, $INFO_DAILY, $INFO_MAIN );

    # example of what we need to parse
    # ClamAV-VDB:07 Nov 2009 03-23 -0500:9999:102368:44:X:X:ccordes:1257582210
    # ClamAV-VDB:14 May 2009 10-28 -0400:51:545035:42:
    # We'll store the date of main.{cld,cvd}; if daily.{cld,cvd} does not
    # exist, use that date as the date of signatures instead of returning
    # 'None found'. If daily.{cld,cvd} does exist, use that date instead.
    if ($main_path) {
        my $FILE;
        if ( open( $FILE, '<', $main_path ) ) {
            binmode($FILE);
            while (<$FILE>) {
                if (/ClamAV-VDB:(\S+\s+\S+\s+\S+).*?[\-\+]\d+:\d+:(\d+)/) {
                    $INFO_DATE = $1;
                    $INFO_MAIN = $2;
                    last;
                }
                close($FILE)
                    or warn "Couldn't close $main_path: $!\n";
            }
        }
        else {
            $INFO_MAIN = 0;
        }
    }
    else {
        $INFO_MAIN = 0;
    }

    if ($daily_path) {
        my $FILE;
        if ( open( $FILE, '<', $daily_path ) ) {
            binmode($FILE);
            while (<$FILE>) {
                if (/ClamAV-VDB:(\S+\s+\S+\s+\S+).*[\-\+]\d+:\d+:(\d+)/) {
                    $INFO_DATE  = $1;
                    $INFO_DAILY = $2;
                    last;
                }
            }
            close($FILE)
                or warn "Couldn't close $daily_path: $!\n";
        }
        else {
            $INFO_DATE  = '01 Jan 1970';
            $INFO_DAILY = 0;
        }
    }
    return ( $INFO_DATE, $INFO_DAILY, $INFO_MAIN );
}

1;
