package ClamTk::GUI;

use strict;
use Carp qw/croak/;
$|++;

use encoding 'utf8';

use ClamTk::App;
use ClamTk::Prefs;
use ClamTk::Results;
use ClamTk::Update;

use Cwd qw/abs_path/;
use File::Basename;
use POSIX qw/locale_h strftime/;
use File::Find::Rule;
use Date::Calc qw/Delta_Days/;
use Locale::gettext;
use File::Copy;

textdomain("clamtk");
setlocale( LC_MESSAGES, "" );
bind_textdomain_codeset( "clamtk", "UTF-8" );

use Gtk2;
use Glib qw/TRUE FALSE/;

my $window;
my $top_label;

my ( $save_log, $hidden, $size_set, $thorough ) = (0) x 4;
my ( @files, @quoted, @new );
my %dirs_scanned;
my ( $step, $pb );
my $tt;
my $SCAN;
my $scan_pid;
my $stopped = 0;
my $scan_option;
my $scan_frame;
my $start_time;
my ( $left_status, $mid_status, $right_status );
my $found;
my ( $found_count, $num_scanned, $num_so_far, $current ) = (0) x 4;

my ( $engine_version_text, $defs_version_text, $last_date_text );
my ($last_infect_date);
my ( $engine_gui_img, $defs_gui_img, $status_gui_img );

my ( $hidden_box, $deep_box, $log_box, $size_box );
my ($scan_status_label);
my $av_remote = '';

my $actions;
my @option_entries;

my ( @q_files,   @h_files );
my ( $q_label,   $h_label );
my ( $new_slist, $new_hlist );

sub start_gui {
    my ($class) = shift;
    my ($self)  = shift;

    INIT { Gtk2->init; }
    $window = Gtk2::Window->new();
    $window->signal_connect( destroy => sub { Gtk2->main_quit; } );
    $window->set_title( gettext('Virus Scanner') );
    $window->set_border_width(5);
    $window->set_resizable(FALSE);
    if ( -e "/usr/share/pixmaps/clamtk.png" ) {
        $window->set_default_icon_from_file("/usr/share/pixmaps/clamtk.png");
    } elsif ( -e "/usr/share/pixmaps/clam.xpm" ) {
        $window->set_default_icon_from_file("/usr/share/pixmaps/clam.xpm");
    }

    my $vbox = Gtk2::VBox->new( FALSE, 0 );
    $window->add($vbox);

    my @entries = (
        [ "FileMenu",       undef, gettext("_Scan") ],
        [ "ViewMenu",       undef, gettext("_View") ],
        [ "OptionsMenu",    undef, gettext("_Options") ],
        [ "QuarantineMenu", undef, gettext("_Quarantine") ],
        [ "HelpMenu",       undef, gettext("_Help") ],

        [   "Scan_File",        'gtk-copy',
            gettext("A _File"), "<control>F",
            gettext("Scan a file"), sub { getfile('file') }
        ],
        [   "Quick_Home",             'gtk-go-down',
            gettext("Home (_Quick)"), "<control>Q",
            gettext("Quick Home Scan"), sub { getfile('home') }
        ],
        [   "Full_Home",                 'gtk-goto-bottom',
            gettext("Home (Recursive)"), "<control>Z",
            gettext("Full Home Scan"), sub { getfile('full-home') }
        ],
        [   "Scan_Directory",        'gtk-zoom-in',
            gettext("A _Directory"), "<control>D",
            gettext("Scan a Directory"), sub { getfile('dir') }
        ],
        [   "Recursive_Scan",
            'gtk-zoom-fit',
            gettext("_Recursive Scan"),
            "<control>R",
            gettext("Recursively scan a directory"),
            sub { getfile('recur') }
        ],
        [   "Exit",           'gtk-quit',
            gettext("E_xit"), "<control>X",
            gettext("Quit this program"), sub { Gtk2->main_quit }
        ],
        [   "Status", 'gtk-edit', gettext("_Status"), "<control>S",
            gettext("See how many files are quarantined"),
            \&quarantine_check
        ],
        [   "Maintenance",
            'gtk-preferences',
            gettext("_Maintenance"),
            "<control>M",
            gettext("View files that have been quarantined"),
            \&maintenance,
        ],
        [   "Empty",
            'gtk-delete',
            gettext("_Empty Quarantine Folder"),
            "<control>E",
            gettext("Delete all files that have been quarantined"),
            \&del_quarantined
        ],
        [   "UpdateSig",
            'gtk-network',
            gettext("_Update Signatures"),
            "<control>U",
            gettext("Update your virus signatures"),
            sub { ClamTk::Update->updater() }
        ],
        [   "About",                          'gtk-about',
            gettext("_About"),                "<control>A",
            gettext("About this program..."), \&about
        ],
    );

    my @view_entries = (
        [   "ManageHistories",
            'gtk-index',
            gettext("Manage _Histories"),
            "<control>H",
            gettext("Select Histories to Delete"),
            sub { history('delete') },
            FALSE
        ],
        [   "ClearOutput",                'gtk-clear',
            gettext("Clear _Output"),     "<control>O",
            gettext("Clear the Display"), \&clear_output,
            FALSE
        ],
        [   "LoadPrefs",                      'gtk-revert-to-saved',
            gettext("Load Scan Preferences"), "<control>L",
            gettext("Load Scan Preferences"), \&load_prefs,
            FALSE
        ],
        [   "SavePrefs",                      'gtk-save',
            gettext("Save Scan Preferences"), "<control>P",
            gettext("Save Scan Preferences"), \&save_prefs,
            FALSE
        ],
    );

    my $ui_info = "<ui>
        <menubar name='MenuBar'>
         <menu action='FileMenu'>
          <menuitem action='Scan_File'/>
          <menuitem action='Scan_Directory'/>
          <menuitem action='Recursive_Scan'/>
          <menuitem action='Quick_Home'/>
          <menuitem action='Full_Home'/>
          <separator/>
          <menuitem action='Exit'/>
         </menu>
          <menu action='ViewMenu'>
          <menuitem action='ManageHistories'/>
          <menuitem action='ClearOutput'/>
          <menuitem action='LoadPrefs'/>
          <menuitem action='SavePrefs'/>
         </menu>
         <menu action='QuarantineMenu'>
          <menuitem action='Status'/>
          <menuitem action='Maintenance'/>
          <menuitem action='Empty'/>
         </menu>
         <menu action='HelpMenu'>
          <menuitem action='UpdateSig'/>
          <menuitem action='About'/>
         </menu>
        </menubar>
</ui>";

    $actions = Gtk2::ActionGroup->new("Actions");
    $actions->add_actions( \@entries,      undef );
    $actions->add_actions( \@view_entries, undef );

    my $ui = Gtk2::UIManager->new;
    $ui->insert_action_group( $actions, 0 );

    $window->add_accel_group( $ui->get_accel_group );
    $ui->add_ui_from_string($ui_info);
    $vbox->pack_start( $ui->get_widget("/MenuBar"), FALSE, FALSE, 0 );

    $top_label = Gtk2::Label->new;

    $tt = Gtk2::Tooltips->new();
    $vbox->pack_start( $top_label, FALSE, FALSE, 0 );

###################
## Actions Frame ##
###################

    my $gui_frame = Gtk2::Frame->new( gettext('Actions') );
    $vbox->pack_start( $gui_frame, FALSE, FALSE, 0 );

    my $gui_box = Gtk2::VBox->new();
    $gui_frame->add($gui_box);

    my $bbox = Gtk2::HButtonBox->new;
    $gui_box->pack_start( $bbox, TRUE, TRUE, 0 );

    my $home_img = Gtk2::Image->new_from_stock( 'gtk-home', 'small-toolbar' );
    my $home_btn = Gtk2::Button->new();
    $home_btn->signal_connect( clicked => sub { getfile('home') } );
    $home_btn->set_property( 'image' => $home_img );
    $home_btn->set_label( gettext('Home') );
    $tt->set_tip( $home_btn, gettext('Scan your home directory') );
    $bbox->add($home_btn);

    my $file_img = Gtk2::Image->new_from_stock( 'gtk-copy', 'small-toolbar' );
    my $file_btn = Gtk2::Button->new();
    $file_btn->signal_connect( clicked => sub { getfile('file') } );
    $file_btn->set_property( 'image' => $file_img );
    $file_btn->set_label( gettext('File') );
    $tt->set_tip( $file_btn, gettext('Scan a file') );
    $bbox->add($file_btn);

    my $dir_img =
        Gtk2::Image->new_from_stock( 'gtk-directory', 'small-toolbar' );
    my $dir_btn = Gtk2::Button->new();
    $dir_btn->signal_connect( clicked => sub { getfile('dir') } );
    $dir_btn->set_property( 'image' => $dir_img );
    $dir_btn->set_label( gettext('Directory') );
    $tt->set_tip( $dir_btn, gettext('Scan a directory') );
    $bbox->add($dir_btn);

    my $quit_img = Gtk2::Image->new_from_stock( 'gtk-quit', 'small-toolbar' );
    my $quit_btn = Gtk2::Button->new();
    $quit_btn->set_property( 'image' => $quit_img );
    $quit_btn->set_label( gettext('Exit') );
    $quit_btn->signal_connect( clicked => sub { Gtk2->main_quit } );
    $tt->set_tip( $quit_btn, gettext('Exit this program') );
    $bbox->add($quit_btn);

    my $opt_box = Gtk2::HBox->new( FALSE, 0 );
    $gui_box->pack_start( $opt_box, FALSE, FALSE, 0 );

    $hidden_box = Gtk2::CheckButton->new_with_label( gettext('Scan hidden') );
    $opt_box->add($hidden_box);
    $tt->set_tip( $hidden_box,
        gettext('Scan files beginning with a dot (.*)') );
    $hidden_box->signal_connect( toggled => sub { $hidden ^= 1; } );

    $deep_box = Gtk2::CheckButton->new_with_label( gettext('Thorough') );
    $opt_box->add($deep_box);
    $tt->set_tip( $deep_box, 'Enable extra scan settings' );
    $deep_box->signal_connect( toggled => sub { $thorough ^= 1; } );

    $size_box = Gtk2::CheckButton->new_with_label( gettext('Ignore size') );
    $opt_box->add($size_box);
    $tt->set_tip( $size_box, gettext('Scan files larger than 20 MB') );
    $size_box->signal_connect( toggled => sub { $size_set ^= 1; } );

    $log_box = Gtk2::CheckButton->new_with_label( gettext('Save a log') );
    $opt_box->add($log_box);
    $tt->set_tip( $log_box, gettext('Keep a record of this scan') );
    $log_box->signal_connect( toggled => sub { $save_log ^= 1; } );

##################
## Status frame ##
##################

    my $status_frame = Gtk2::Frame->new( gettext('Status') );
    $vbox->pack_start( $status_frame, FALSE, FALSE, 0 );

    my $status_table = Gtk2::Table->new( 5, 3, FALSE );
    $status_frame->add($status_table);

    my $engine_gui_box = Gtk2::HBox->new( TRUE, 0 );
    my $engine_text = Gtk2::Label->new( gettext('Antivirus engine') );
    $engine_version_text = Gtk2::Label->new( gettext('Unknown') );
    $engine_gui_img =
        Gtk2::Image->new_from_stock( 'gtk-apply', 'small-toolbar' );
    $tt->set_tip( $engine_gui_img, gettext('Current') );
    $engine_gui_box->pack_start( $engine_gui_img,      TRUE, TRUE, 0 );
    $engine_gui_box->pack_start( $engine_text,         TRUE, TRUE, 0 );
    $engine_gui_box->pack_start( $engine_version_text, TRUE, TRUE, 0 );
    $status_table->attach_defaults( $engine_gui_box, 0, 3, 0, 1 );

    my $status_gui_box   = Gtk2::HBox->new( TRUE, 0 );
    my $gui_text         = Gtk2::Label->new( gettext('GUI version') );
    my $gui_version_text = Gtk2::Label->new( $self->{VERSION} );
    $status_gui_img =
        Gtk2::Image->new_from_stock( 'gtk-apply', 'small-toolbar' );
    $tt->set_tip( $status_gui_img, gettext('Current') );
    $status_gui_box->pack_start( $status_gui_img,   TRUE, TRUE, 0 );
    $status_gui_box->pack_start( $gui_text,         TRUE, TRUE, 0 );
    $status_gui_box->pack_start( $gui_version_text, TRUE, TRUE, 0 );
    $status_table->attach_defaults( $status_gui_box, 0, 3, 1, 2 );

    my $defs_gui_box = Gtk2::HBox->new( TRUE, 0 );
    my $defs_text = Gtk2::Label->new( gettext('Virus definitions') );
    $defs_version_text = Gtk2::Label->new( gettext('Unknown') );
    $defs_gui_img =
        Gtk2::Image->new_from_stock( 'gtk-apply', 'small-toolbar' );
    $tt->set_tip( $defs_gui_img, gettext('Current') );
    $defs_gui_box->pack_start( $defs_gui_img,      TRUE, TRUE, 0 );
    $defs_gui_box->pack_start( $defs_text,         TRUE, TRUE, 0 );
    $defs_gui_box->pack_start( $defs_version_text, TRUE, TRUE, 0 );
    $status_table->attach_defaults( $defs_gui_box, 0, 3, 2, 3 );

    my $last_gui_box = Gtk2::HBox->new( TRUE, 0 );
    my $last_gui_img =
        Gtk2::Image->new_from_stock( 'gtk-print-preview', 'small-toolbar' );
    my $last_text = Gtk2::Label->new( gettext('Last virus scan') );
    $last_date_text = Gtk2::Label->new( gettext('Unknown') );
    $tt->set_tip( $last_gui_img, gettext('Date of your last virus scan') );
    $last_gui_box->pack_start( $last_gui_img,   TRUE, TRUE, 0 );
    $last_gui_box->pack_start( $last_text,      TRUE, TRUE, 0 );
    $last_gui_box->pack_start( $last_date_text, TRUE, TRUE, 0 );
    $status_table->attach_defaults( $last_gui_box, 0, 3, 3, 4 );

    my $last_infect_box = Gtk2::HBox->new( TRUE, 0 );
    my $last_infect_img =
        Gtk2::Image->new_from_stock( 'gtk-dnd', 'small-toolbar' );
    my $last_infect_text = Gtk2::Label->new( gettext('Last infected file') );
    $last_infect_date = Gtk2::Label->new( gettext('Unknown') );
    $tt->set_tip( $last_infect_img, gettext('Date of last known infection') );
    $last_infect_box->pack_start( $last_infect_img,  TRUE, TRUE, 0 );
    $last_infect_box->pack_start( $last_infect_text, TRUE, TRUE, 0 );
    $last_infect_box->pack_start( $last_infect_date, TRUE, TRUE, 0 );
    $status_table->attach_defaults( $last_infect_box, 0, 3, 4, 5 );

################
## Scan frame ##
################

    $scan_frame = Gtk2::Frame->new( gettext("Scan") );
    $vbox->pack_start( $scan_frame, FALSE, FALSE, 0 );

    my $scan_box = Gtk2::VBox->new();
    $scan_frame->add($scan_box);

    $scan_status_label = Gtk2::Label->new("");
    $scan_status_label->set_justify('center');
    $scan_status_label->set_ellipsize('middle');
    $scan_box->pack_start( $scan_status_label, FALSE, FALSE, 0 );

    my $scan_table = Gtk2::Table->new( 1, 2, FALSE );
    $scan_box->pack_start( $scan_table, FALSE, FALSE, 0 );

    $pb = Gtk2::ProgressBar->new;
    $scan_table->attach( $pb, 0, 1, 0, 1, [qw/fill expand/], [qw/shrink/], 0,
        0 );
    $pb->set_fraction(0);

    my $stop_image =
        Gtk2::Image->new_from_stock( 'gtk-stop', 'small-toolbar' );
    my $stop_btn = Gtk2::Button->new();
    $stop_btn->set_property( image => $stop_image );
    $stop_btn->set_relief('none');
    $tt->set_tip( $stop_btn, gettext('Stop scanning now') );
    $scan_table->attach( $stop_btn, 1, 2, 0, 1, [qw/shrink shrink/],
        [qw/shrink/], 0, 0 );
    $stop_btn->signal_connect(
        'clicked' => sub {

            if ( !scalar(@files) ) {
                $scan_frame->hide();
            }
            @quoted = ();
            kill 15, $scan_pid if ($scan_pid);
            $scan_status_label->set_text( gettext("Please wait...") );
            waitpid( $scan_pid, 0 );

            # this close returns the stupid readline() error.
            # not sure how to fix it yet, besides commenting
            # out 'use warnings' :) it's the only way to immediately
            # stop the $SCAN so far...
            close($SCAN);    # or warn "Unable to close scanner! $!\n";
            $scan_status_label->set_text("");
            $stopped = 1;
        }
    );

    my $bottom_box = Gtk2::HBox->new( FALSE, 0 );
    $scan_box->pack_start( $bottom_box, FALSE, FALSE, 0 );

    $left_status = Gtk2::Label->new( gettext('Files Scanned: ') );
    $bottom_box->pack_start( $left_status, TRUE, TRUE, 0 );

    $mid_status = Gtk2::Label->new( gettext('Viruses Found: ') );
    $bottom_box->pack_start( $mid_status, TRUE, TRUE, 0 );

    $right_status = Gtk2::Label->new( gettext('Ready') );
    $bottom_box->pack_start( $right_status, TRUE, TRUE, 0 );

    $window->show_all();
    $scan_frame->hide();

    load_prefs();

    if (@ARGV) {
        update_status_frame();
        if ( -d $ARGV[0] ) {
            my $d = abs_path( $ARGV[0] );
            if ( not $d ) {    # no permissions
                $found->{$found_count}->{name} = $ARGV[0];
                $found->{$found_count}->{status} =
                    gettext("Could not scan (permissions)");
                $found->{$found_count}->{action} = gettext("None");
                $found_count++;
                ClamTk::Results->display( $main::scanner, 1, $found );
                clean_up();
            } elsif ( $ARGV[0] =~ m#^/(proc|sys|dev)# ) {
                $found->{$found_count}->{name} = $ARGV[0];
                $found->{$found_count}->{status} =
                    gettext("Directory excluded from scan");
                $found->{$found_count}->{action} = gettext("None");
                $found_count++;
                ClamTk::Results->display( $main::scanner, 1, $found );
                clean_up();
            } else {
                getfile( 'cmd-scan', $d );
            }
        } else {
            my $pos = 0;
            for my $f (@ARGV) {
                $f = abs_path($f);
                if ( $f =~ m#^/(proc|sys|dev)#
                    or not -r $f )
                {
                    splice( @ARGV, $pos, 1 );
                    $pos++;
                }
            }
            getfile( 'cmd-scan', @ARGV );
        }
    } else {
        update_status_frame();
    }

    Gtk2->main();

    return $window;
}

sub update_status_frame {
    $top_label->set_text( gettext("Please wait, initializing...") );
    Gtk2->main_iteration while ( Gtk2->events_pending );

    # ClamAV version
    my ($version) = ClamTk::App->get_AV_version($main::scanner);
    $engine_version_text->set_text($version);

    $av_remote = ClamTk::App->get_AV_remote($main::scanner);

    #warn "version = >$version<, av_remote = >$av_remote<\n";

    if ( $av_remote eq "undef" || $av_remote eq "unavailable" ) {
        $engine_gui_img->set_from_stock( 'gtk-dialog-question',
            'small-toolbar' );
        $tt->set_tip( $engine_gui_img, gettext('Unable to check') );
        $window->queue_draw;
        Gtk2->main_iteration while ( Gtk2->events_pending );
    } elsif ( ( $version cmp $av_remote ) == -1 ) {
        $engine_gui_img->set_from_stock( 'gtk-dialog-error',
            'small-toolbar' );
        $tt->set_tip( $engine_gui_img,
            gettext('The antivirus engine is reporting an outdated engine') );
        $window->queue_draw;
        Gtk2->main_iteration while ( Gtk2->events_pending );
    } elsif ( ( $version cmp $av_remote ) == 0 ) {
        $engine_gui_img->set_from_stock( 'gtk-apply', 'small-toolbar' );
        $tt->set_tip( $engine_gui_img, gettext('Current') );
        $window->queue_draw;
        Gtk2->main_iteration while ( Gtk2->events_pending );
    } elsif ( ( $version cmp $av_remote ) == 1 ) {    # shouldn't happen
        $engine_gui_img->set_from_stock( 'gtk-apply', 'small-toolbar' );
        $tt->set_tip( $engine_gui_img, gettext('Current') );
        $window->queue_draw;
        Gtk2->main_iteration while ( Gtk2->events_pending );
    } else {
        $engine_gui_img->set_from_stock( 'gtk-dialog-question',
            'small-toolbar' );
        $tt->set_tip( $engine_gui_img, gettext('Unable to check') );
        $window->queue_draw;
        Gtk2->main_iteration while ( Gtk2->events_pending );
    }

    # ClamTk version
    Gtk2->main_iteration while ( Gtk2->events_pending );
    my ($ret) = ClamTk::Update->update_gui();
    if ( $ret == 0 ) {
        $status_gui_img->set_from_stock( 'gtk-dialog-question',
            'small-toolbar' );
        $tt->set_tip( $status_gui_img, gettext('Unable to check') );
    } elsif ( $ret == 1 || $ret == 3 ) {
        $status_gui_img->set_from_stock( 'gtk-apply', 'small-toolbar' );
        $tt->set_tip( $status_gui_img, gettext('Current') );
    } else {
        $status_gui_img->set_from_stock( 'gtk-dialog-error',
            'small-toolbar' );
        $tt->set_tip( $status_gui_img,
            gettext('A newer version is available') );
    }
    Gtk2->main_iteration while ( Gtk2->events_pending );

    # Signatures
    set_sig_status();

    # Last scan
    my $Config = ClamTk::Prefs->get_all_prefs($main::scanner);
    return unless $Config;
    my $last_scan;
    if ( $Config->{_}->{LastScan} ) {
        $last_scan = $Config->{_}->{LastScan};
        $last_date_text->set_text($last_scan);
    } else {
        $last_date_text->set_text( gettext('Never') );
    }

    # Last infection
    if ( $Config->{_}->{LastInfection} ) {
        my $last_infection = $Config->{_}->{LastInfection};
        $last_infect_date->set_text($last_infection);
    } else {
        $last_infect_date->set_text( gettext('Never') );
    }
    $top_label->set_text("");
}

sub set_sig_status {
    my ($self) = shift;
    my ($app)  = shift;
    my ($ret)  = shift;

    my $sig_date = ClamTk::App->get_date_sigs();
    if ($sig_date) {
        $defs_version_text->set_text($sig_date);
    } else {
        $defs_version_text->set_text( gettext("None found") );
    }

    my ( $d, $m, $y ) = split / /, $sig_date;
    my $date_ret = date_diff( $d, $m, $y );
    if ( $date_ret eq 'outdated' ) {
        $defs_gui_img->set_from_stock( 'gtk-dialog-error', 'small-toolbar' );
        $tt->set_tip( $defs_gui_img,
            gettext('Your antivirus signatures are out-of-date') );
    } else {
        $defs_gui_img->set_from_stock( 'gtk-apply', 'small-toolbar' );
        $tt->set_tip( $defs_gui_img, 'Current' );
    }
}

sub set_tk_status {
    my ($self) = shift;
    my ($app)  = shift;
    my ($ret)  = shift;

    if ( $ret == 0 ) {
        $status_gui_img->set_from_stock( 'gtk-dialog-question',
            'small-toolbar' );
        $tt->set_tip( $status_gui_img, gettext('Unable to check') );
    } elsif ( $ret == 1 || $ret == 3 ) {
        $status_gui_img->set_from_stock( 'gtk-apply', 'small-toolbar' );
        $tt->set_tip( $status_gui_img, gettext('Current') );
    } else {
        $status_gui_img->set_from_stock( 'gtk-dialog-error',
            'small-toolbar' );
        $tt->set_tip( $status_gui_img,
            gettext('A newer version is available') );
    }
    Gtk2->main_iteration while ( Gtk2->events_pending );
    $window->queue_draw;
}

sub date_diff {
    my ( $day2, $month2, $year2 ) = @_;
    my ( $day1, $month1, $year1 ) = split / /,
        strftime( '%d %m %Y', localtime );
    my %months = (
        'Jan' => 1,
        'Feb' => 2,
        'Mar' => 3,
        'Apr' => 4,
        'May' => 5,
        'Jun' => 6,
        'Jul' => 7,
        'Aug' => 8,
        'Sep' => 9,
        'Oct' => 10,
        'Nov' => 11,
        'Dec' => 12,
    );
    return unless ( $day2 && $month2 && $year2 );

    my $diff =
        Delta_Days( $year1, $month1, $day1, $year2, $months{$month2}, $day2 );
    $diff *= -1;
    if ( $diff >= 7 ) {
        return 'outdated';
    } else {
        return 'current';
    }
}

sub getfile {
    my ($option) = shift;
    my @cmd_input = @_;

    # these lines are for 'thorough' :)
    if ($thorough) {
        $main::scanner->{clamscan} .= " --detect-pua --detect-broken";
    } elsif ( !$thorough ) {
        $main::scanner->{clamscan} =~ s/\s--detect-pua --detect-broken//;
    }

    $scan_option = $option;

    # disable File::Find warnings for this scope.
    no warnings 'File::Find';

    # $option will be either "home", "full-home", "file", "dir",
    # "recur", or "cmd-scan"
    $pb->set_fraction(0);
    Gtk2->main_iteration while ( Gtk2->events_pending );
    clear_output();
    chdir( $main::scanner->{directory} ) or chdir("/tmp");

    my ( $filename, $dir, $dialog );
    Gtk2->main_iteration while ( Gtk2->events_pending );
    my $rule = File::Find::Rule->new;
    $rule->file;
    $rule->exists;
    $rule->readable;
    $rule->maxdepth(1)
        unless ( $option eq 'recur' or $option eq 'full-home' );

    my $title =
          ( $option eq 'file' ) ? gettext('Select File')
        : ( $option eq 'dir' )
        ? gettext('Select a Directory (directory scan)')
        : ( $option eq 'recur' )
        ? gettext('Select a Directory (recursive scan)')
        : '';

    if ( $option eq 'home' ) {
        $scan_status_label->set_text( gettext("Please wait...") );
        Gtk2->main_iteration while ( Gtk2->events_pending );
        @files = $rule->in( $main::scanner->{directory} );
    } elsif ( $option eq 'file' ) {
        $dialog = Gtk2::FileChooserDialog->new(
            gettext($title), undef, 'open',
            'gtk-cancel' => 'cancel',
            'gtk-ok'     => 'ok',
        );
        $dialog->set_select_multiple(TRUE);
        if ( "ok" eq $dialog->run ) {
            $scan_status_label->set_text( gettext("Please wait...") );
            $window->queue_draw;
            Gtk2->main_iteration while ( Gtk2->events_pending );
            @files = $dialog->get_filenames;
            $window->queue_draw;
            $dialog->destroy;
            $window->queue_draw;
            Gtk2->main_iteration while ( Gtk2->events_pending );
        } else {
            $dialog->destroy;
            return;
        }
    } elsif ( $option eq 'full-home' ) {
        $scan_status_label->set_text( gettext("Please wait...") );
        Gtk2->main_iteration while ( Gtk2->events_pending );
        @files = $rule->in( $main::scanner->{directory} );
    } elsif ( $option eq 'dir' or $option eq 'recur' ) {
        $dialog = Gtk2::FileChooserDialog->new(
            gettext($title), undef,
            'select-folder',
            'gtk-cancel' => 'cancel',
            'gtk-ok'     => 'ok',
        );
        if ( "ok" eq $dialog->run ) {
            $dir = $dialog->get_filename;
            if ( $dir =~ m#^/(proc|sys|dev)#
                or not -r $dir )
            {
                $dialog->destroy;
                $found->{$found_count}->{name} = $dir;
                $found->{$found_count}->{status} =
                    ( not -r $dir )
                    ? gettext("Could not scan (permissions)")
                    : gettext("Will not scan that directory");
                $found->{$found_count}->{action} = gettext("None");
                $found_count++;
                ClamTk::Results->display( $main::scanner, 1, $found );
                clean_up();
                return;
            }
            $scan_status_label->set_text( gettext("Please wait...") );
            Gtk2->main_iteration while ( Gtk2->events_pending );
            $window->queue_draw;
            $dialog->destroy;
            $window->queue_draw;
            $dir ||= $main::scanner->{directory};
            Gtk2->main_iteration while ( Gtk2->events_pending );
            @files = $rule->in($dir);
        } else {
            $dialog->destroy;
            return;
        }
    } elsif ( $option eq 'cmd-scan' ) {
        if ( -d $cmd_input[0] ) {
            $scan_status_label->set_text( gettext("Please wait...") );
            $window->queue_draw;

            Gtk2->main_iteration while ( Gtk2->events_pending );
            @files = $rule->in( $cmd_input[0] );
        } else {
            $scan_status_label->set_text( gettext("Please wait...") );
            $window->queue_draw;
            Gtk2->main_iteration while ( Gtk2->events_pending );
            @files = @cmd_input;
        }
    } else {
        die gettext("Shouldn't reach this."), "\n";
    }

    # open the scan frame for display
    $scan_frame->show_all();
    Gtk2->main_iteration while Gtk2->events_pending;
    $window->queue_draw;
    Gtk2->main_iteration while ( Gtk2->events_pending );

    # start the timer - replaces the "Ready"
    $start_time = time;
    $right_status->set_text( gettext("Elapsed time: ") );

    # reset %$found
    $found = {};

    # Attempt to remove dangling symlinks
    my $index = 0;
    while ( $index <= $#files ) {
        if ( -l $files[$index] && !-e $files[$index] ) {
            splice( @files, $index, 1 );
        } else {
            $index++;
        }
    }

    # only a single file
    if ( $option eq 'file' ) {
        scan(@files);
    } else {
        if ( $hidden == 0 && $option ne 'recur' && $option ne 'full-home' ) {
            @files = grep { basename($_) !~ /^\./ } @files;
        }

        if ( scalar(@files) == 1 ) {
            $step = 1;
        } elsif ( scalar(@files) > 1 ) {
            $step = 1 / scalar(@files);
        } else {
            $step = 1;
        }

        if ( !$size_set ) {
            my @large;
            foreach my $foo (@files) {
                if ( -s $foo >= 20_000_000 ) {
                    push( @large, $foo );
                } else {
                    push( @new, $foo );
                }
            }

            if (@large) {
                foreach my $too_big (@large) {
                    $scan_status_label->set_text(
                        sprintf gettext("Scanning %s..."),
                        dirname($too_big) );
                    $num_scanned++;
                    timer();
                    next;
                }
            }
        } else {
            @new = @files;
        }

        if (@new) {
            my @send;
            while ( my $t = pop(@new) ) {
                last if ($stopped);
                push( @send, $t );
                if ( scalar(@send) == 255 ) {
                    scan(@send);
                    @send = ();
                } else {
                    next;
                }
            }
            scan(@send) if (@send);
        }
    }
    clean_up();
}

sub scan {
    my @get = @_;
    @quoted = map { quotemeta($_) } @get;
    timer();

    my $command = $main::scanner->{clamscan};

    if ( ClamTk::Prefs->get_property( $main::scanner, "Update" ) eq 'single' )
    {
        $command .= " --database=$main::scanner->{db}";
    }

    my $pid = open( $SCAN, "-|", "$command @quoted 2>&1" );
    defined($pid) or die gettext("couldn't fork: "), "$!\n";
    my $scan_count = 0;
    $scan_status_label->set_text(
        sprintf gettext("Scanning %s..."),
        $scan_option ne 'file'
        ? dirname( $get[$scan_count] )
        : $get[$scan_count]
    );
    $scan_pid = $pid;    # this is for the 'stop button'

    Gtk2->main_iteration while Gtk2->events_pending;
    while (<$SCAN>) {
        Gtk2->main_iteration while Gtk2->events_pending;
        if ( $av_remote eq 'unavailable' && /ClamAV engine is outdated/ ) {
            $engine_gui_img->set_from_stock( 'gtk-dialog-error',
                'small-toolbar' );
            $engine_version_text->set_text( gettext('Outdated') );
            $tt->set_tip( $engine_gui_img,
                gettext('The antivirus is reporting an outdated engine') );
            $window->queue_draw;
            Gtk2->main_iteration while ( Gtk2->events_pending );
        }

        #my ( $file, $status ) = split /:/;
        my ( $file, $status );
        if (/(.*?): (.*?) FOUND/) {
            $file   = $1;
            $status = $2;
        } elsif (/(.*?): (OK)/) {
            $file   = $1;
            $status = $2;
        }    #else {
             #	warn "something else: file = <$file>, stat = <$status>\n";
             #}

        chomp($file)   if ( defined $file );
        chomp($status) if ( defined $status );
        next unless ( -e $file && $status );
        next if ( $status =~ /module failure/ );

        $dirs_scanned{ dirname($file) } = 1
            unless ( dirname($file) =~ /\/tmp\/clamav/
            || dirname($file) eq "." );

        $status =~ s/\s+FOUND$//;

        # do not show files in archives - we just want the end-result.
        # it still scans and we still show the result.
        next if ( $file =~ /\/tmp\/clamav/ );

        timer();
        my $clean_words = join( '|',
            "OK",                          "Zip module failure",
            "RAR module failure",          "Encrypted.RAR",
            "Encrypted.Zip",               "Empty file",
            "Excluded",                    "Input/Output error",
            "Files number limit exceeded", "handler error",
            "Broken.Executable",           "Oversized.Zip" );

        if ( $status !~ /$clean_words/ ) {    # a virus
            $found->{$found_count}->{name}   = $file;
            $found->{$found_count}->{status} = $status;
            $found->{$found_count}->{action} = gettext("None");
            $found_count++;
        }

        $num_so_far = keys %$found;
        if ( $num_so_far > 0 ) {
            $mid_status->set_markup(
                sprintf gettext("<b>Viruses Found: %d</b>"), $num_so_far );
        } else {
            $mid_status->set_text( sprintf gettext("Viruses Found: %d"),
                $num_so_far );
        }
        $num_scanned++;

        # Pulse!
        if ( $current + $step <= 0 || $current + $step >= 1.0 ) {
            $current = .99;
        } else {
            $current += $step;
        }

        $pb->set_fraction($current);
        $pb->set_text( sprintf gettext("Percent complete: %2d"),
            $current * 100 );
        Gtk2->main_iteration while ( Gtk2->events_pending );

        $scan_count++;
        if ( defined( $quoted[$scan_count] ) ) {
            Gtk2->main_iteration while ( Gtk2->events_pending );
            $scan_status_label->set_text(
                sprintf gettext("Scanning %s..."),
                $scan_option ne 'file'
                ? dirname( $get[$scan_count] )
                : basename( $get[$scan_count] )
            );
            Gtk2->main_iteration while ( Gtk2->events_pending );
        }
    }
    if ( !@new ) {
        close($SCAN);    # or warn "Unable to close scanner! $!\n";
        $pb->set_text( gettext("Percent complete: 100") );
    }
    Gtk2->main_iteration while ( Gtk2->events_pending );
}

sub clear_output {
    return if ( scalar(@files) > 0 );
    $pb->set_fraction(0);
    $window->queue_draw;
    $top_label->set_text("");
    $scan_status_label->set_text("");

    $left_status->set_text( gettext("Files Scanned: ") );
    $mid_status->set_text( gettext("Viruses Found: ") );
    $right_status->set_text( gettext("Ready") );
    $scan_frame->hide();
}

sub timer {
    Gtk2->main_iteration while ( Gtk2->events_pending );
    my $now     = time;
    my $seconds = $now - $start_time;
    my $s       = sprintf "%02d", ( $seconds % 60 );
    my $m       = sprintf "%02d", ( $seconds - $s ) / 60;
    $right_status->set_text( sprintf gettext("Elapsed time: %s"), "$m:$s" );
    $left_status->set_text( sprintf gettext("Files Scanned: %d"),
        $num_scanned );
    $window->queue_draw;
    Gtk2->main_iteration while ( Gtk2->events_pending );
}

sub clean_up {
    $pb->set_fraction(1.0);
    $pb->set_text("");

    my $db_total = ClamTk::App->get_num_sigs();
    my $REPORT;    # filehandle for histories log
    my ( $mon, $day, $year ) = split / /, strftime( '%b %d %Y', localtime );

    # Save date of scan
    my $Config = ClamTk::Prefs->get_all_prefs($main::scanner);
    return unless $Config;

    my $li;
    if ( $num_so_far > 0 ) {
        $li = "$day $mon $year";
        $last_infect_date->set_text("$day $mon $year");
        $window->queue_draw;
    } elsif ( $Config->{_}->{LastInfection} ) {
        $li = $Config->{_}->{LastInfection};
    } else {
        $li = gettext("Never");
    }

    my $update = $Config->{_}->{Update}     || 'shared';
    my $sl     = $Config->{_}->{SaveToLog}  || 0;
    my $sh     = $Config->{_}->{ScanHidden} || 0;
    my $limit  = $Config->{_}->{SizeLimit}  || 0;
    my $t      = $Config->{_}->{Thorough}   || 0;
    my $ls     = "$day $mon $year";
    ClamTk::Prefs->set_all( $main::scanner, $sl, $sh, $limit, $t, $ls, $li,
        $update );

    $last_date_text->set_text("$day $mon $year");
    $window->queue_draw;
    Gtk2->main_iteration while ( Gtk2->events_pending );

    if ($save_log) {
        my $virus_log =
            $main::scanner->{history} . "/" . "$mon-$day-$year" . ".log";

        # sort the directories scanned for display
        my @sorted = sort { $a cmp $b } keys %dirs_scanned;
        if ( open $REPORT, '>>', $virus_log ) {
            print $REPORT "\nClamTk, v$main::scanner->{VERSION}\n",
                scalar localtime, "\n";
            print $REPORT sprintf gettext("ClamAV Signatures: %d\n"),
                $db_total;
            print $REPORT gettext("Directories Scanned:\n");
            for my $list (@sorted) {
                print $REPORT "$list\n";
            }
            printf $REPORT gettext(
                "\nFound %d possible %s (%d %s scanned).\n\n"), $num_so_far,
                $num_so_far == 1 ? gettext("virus") : gettext("viruses"),
                $num_scanned,
                $num_scanned == 1 ? gettext("file") : gettext("files");
        } else {
            $scan_status_label->set_text(
                gettext("Could not write to logfile. Check permissions.") );
            $save_log = 0;
        }
    }
    $db_total =~ s/(\w+)\s+$/$1/;
    $scan_status_label->set_text(
        sprintf gettext("Scanning complete (%d signatures)"), $db_total );
    $left_status->set_text( sprintf gettext("Files Scanned: %d"),
        $num_scanned );
    if ( $num_so_far != 0 ) {
        $mid_status->set_text( sprintf gettext("Viruses Found: %d"),
            $num_so_far );
    }
    $right_status->set_text( gettext("Ready") );
    $window->queue_draw;

    if ( $num_so_far == 0 ) {
        print $REPORT gettext("No viruses found.\n") if ($save_log);
    } else {
        if ($save_log) {
            for my $num ( sort keys %$found ) {
                printf $REPORT "%-38s %38s\n",
                    $found->{$num}->{name}, $found->{$num}->{status};
            }
        }
    }
    if ($save_log) {
        print $REPORT "-" x 77, "\n";
        close($REPORT) if ( fileno($REPORT) );
    }

    if ($num_so_far) {
        ClamTk::Results->display( $main::scanner, 0, $found );
    } elsif ( !keys %$found && $scan_option eq 'cmd-scan' ) {
        for my $f (@files) {
            $found->{$found_count}->{name}   = $f;
            $found->{$found_count}->{status} = gettext("Nothing detected");
            $found->{$found_count}->{action} = gettext("None");
            $found_count++;
        }
        ClamTk::Results->display( $main::scanner, 1, $found );
    }

    # reset things
    $num_so_far   = 0;
    $num_scanned  = 0;
    $found_count  = 0;
    %dirs_scanned = ();
    @files        = ();
    @new          = ();
    @quoted       = ();
    $current      = 0;
    $stopped      = 0;
}

sub show_message_dialog {
    my ( $parent, $type, $button, $message ) = @_;

    my $dialog;
    $dialog =
        Gtk2::MessageDialog->new_with_markup( $parent,
        [qw(modal destroy-with-parent)],
        $type, $button, $message );

    $dialog->run;
    $dialog->destroy;
    return;
}

sub maintenance {
    my $main_win = Gtk2::Window->new;
    $main_win->signal_connect( destroy => sub { $main_win->destroy; } );
    $main_win->set_default_size( 250, 200 );
    $main_win->set_title( gettext("Quarantine") );

    my $new_vbox = Gtk2::VBox->new;
    $main_win->add($new_vbox);

    @q_files = glob "$main::scanner->{viruses}/*";
    my $s_win = Gtk2::ScrolledWindow->new;
    $s_win->set_shadow_type('etched-in');
    $s_win->set_policy( 'automatic', 'automatic' );
    $new_vbox->pack_start( $s_win, TRUE, TRUE, 0 );

    $new_slist = Gtk2::SimpleList->new( gettext('File') => 'text', );
    $s_win->add($new_slist);

    my $new_hbox = Gtk2::HButtonBox->new;
    $new_vbox->pack_start( $new_hbox, FALSE, FALSE, 0 );

    my $pos_quit = Gtk2::Button->new_with_label( gettext("Close Window") );
    $new_hbox->add($pos_quit);
    $pos_quit->signal_connect( clicked => sub { $main_win->destroy } );
    my $false_pos = Gtk2::Button->new_with_label( gettext("False Positive") );
    $new_hbox->add($false_pos);
    $false_pos->signal_connect(
        clicked => \&main_false_pos,
        "false_pos"
    );
    my $del_pos = Gtk2::Button->new_with_label( gettext("Delete") );
    $new_hbox->add($del_pos);
    $del_pos->signal_connect( clicked => \&main_del_pos, "false_pos" );

    $q_label = Gtk2::Label->new();
    $new_vbox->pack_start( $q_label, FALSE, FALSE, 2 );

    for my $opt (@q_files) {
        push @{ $new_slist->{data} }, basename($opt);
    }
    $main_win->set_position('mouse');
    $main_win->show_all;
}

sub main_false_pos {
    my @sel = $new_slist->get_selected_indices;
    return if ( !@sel );
    my $deref = $sel[0];
    return if ( not exists $q_files[$deref] );
    my $base = basename( $q_files[$deref] );
    system( "mv", $q_files[$deref], $main::scanner->{directory} );
    my $new_name = $base;
    $new_name =~ s/.VIRUS$//;
    rename(
        "$main::scanner->{directory}/$base",
        "$main::scanner->{directory}/$new_name"
    );

    if ( -e $q_files[$deref] ) {
        $q_label->set_text( gettext("Operation failed.") );
        return;
    }
    splice @{ $new_slist->{data} }, $deref, 1;
    $q_label->set_text( gettext("Moved to home directory.") );
    @q_files = glob "$main::scanner->{viruses}/*";
}

sub main_del_pos {
    my @sel = $new_slist->get_selected_indices;
    return if ( !@sel );
    my $deref = $sel[0];
    return if ( not exists $q_files[$deref] );
    my $base = basename( $q_files[$deref] );
    unlink $q_files[$deref];
    if ( -e $q_files[$deref] ) {
        $q_label->set_text( gettext("Operation failed.") );
        return;
    }

    splice @{ $new_slist->{data} }, $deref, 1;
    $q_label->set_text( gettext("Deleted.") );
    @q_files = glob "$main::scanner->{viruses}/*";
}

sub quarantine_check {
    if ( !-d $main::scanner->{viruses} ) {
        show_message_dialog( $window, 'error', 'close',
            gettext("No virus directory available.") );
        return;
    }
    my @trash;
    unless ( opendir( DIR, $main::scanner->{viruses} ) ) {
        show_message_dialog( $window, 'error', 'close',
            gettext("Unable to open the virus directory.") );
        return;
    }
    @trash = grep { -f "$main::scanner->{viruses}/$_" } readdir(DIR);
    closedir(DIR);
    my $del = scalar(@trash);
    if ( !$del ) {
        show_message_dialog( $window, 'info', 'ok',
            gettext("No items currently quarantined.") );
    } else {
        my $notice = sprintf gettext("%d item(s) currently quarantined."),
            $del;
        show_message_dialog( $window, 'info', 'ok', $notice );
    }
}

sub del_quarantined {
    unless ( -e $main::scanner->{viruses} ) {
        show_message_dialog( $window, 'error', 'close',
            gettext("There is no quarantine directory to empty.") );
        return;
    } else {
        my $confirm_message = gettext("Really delete all quarantined files?");
        my $confirm =
            Gtk2::MessageDialog->new( $window,
            [qw(modal destroy-with-parent)],
            'question', 'ok-cancel', $confirm_message );

        if ( "cancel" eq $confirm->run ) {
            $confirm->destroy;
            return;
        } else {
            $confirm->destroy;
            my @trash;
            unless ( opendir( DIR, $main::scanner->{viruses} ) ) {
                show_message_dialog( $window, 'error', 'close',
                    gettext("Unable to open the virus directory.") );
                return;
            }
            @trash = grep { -f "$main::scanner->{viruses}/$_" } readdir(DIR);
            closedir(DIR);
            if ( scalar(@trash) == 0 ) {
                show_message_dialog( $window, 'info', 'close',
                    gettext("There are no quarantined items to delete.") );
            } else {
                my $del = 0;
                foreach (@trash) {
                    unlink "$main::scanner->{viruses}/$_" and $del++;
                }
                my $notice = sprintf gettext("Removed %d item(s)."), $del;
                show_message_dialog( $window, 'info', 'close', $notice );
            }
        }
    }
}

sub history {
    @h_files = glob "$main::scanner->{history}/*.log";
    my $new_win = Gtk2::Window->new;
    $new_win->signal_connect( destroy => sub { $new_win->destroy } );
    $new_win->set_default_size( 260, 200 );
    $new_win->set_title( gettext("Scanning Histories") );

    my $new_vbox = Gtk2::VBox->new;
    $new_win->add($new_vbox);

    my $s_win = Gtk2::ScrolledWindow->new;
    $s_win->set_shadow_type('etched-in');
    $s_win->set_policy( 'automatic', 'automatic' );
    $new_vbox->pack_start( $s_win, TRUE, TRUE, 0 );

    $new_hlist = Gtk2::SimpleList->new( gettext('Histories') => 'text', );
    $s_win->add($new_hlist);

    my $new_hbox = Gtk2::HButtonBox->new;
    $new_vbox->pack_start( $new_hbox, FALSE, FALSE, 0 );

    my $hist_view = Gtk2::Button->new_with_label( gettext("View") );
    $new_hbox->add($hist_view);
    $hist_view->signal_connect( clicked => \&view_box, "viewer" );

    my $pos_quit = Gtk2::Button->new_with_label( gettext("Close Window") );
    $new_hbox->add($pos_quit);
    $pos_quit->signal_connect( clicked => sub { $new_win->destroy } );
    my $del_single = Gtk2::Button->new_with_label( gettext("Delete") );
    $new_hbox->add($del_single);
    $del_single->signal_connect(
        clicked => \&history_del_single,
        "del_single"
    );
    my $del_all = Gtk2::Button->new_with_label( gettext("Delete All") );
    $new_hbox->add($del_all);
    $del_all->signal_connect(
        clicked => \&history_del_all,
        "del_all"
    );

    $h_label = Gtk2::Label->new();
    $new_vbox->pack_start( $h_label, FALSE, FALSE, 2 );

    for my $opt (@h_files) {
        push @{ $new_hlist->{data} }, basename($opt);
    }
    $new_win->set_position('mouse');
    $new_win->show_all;
}

sub history_del_single {
    my @sel = $new_hlist->get_selected_indices;
    return if ( !@sel );
    my $deref = $sel[0];
    return if ( not exists $h_files[$deref] );
    unlink $h_files[$deref];
    if ( -e $h_files[$deref] ) {
        my $notice = sprintf gettext("Unable to delete %s!"),
            $h_files[$deref];
        show_message_dialog( $window, 'error', 'ok', $notice );
        return;
    }
    splice @{ $new_hlist->{data} }, $deref, 1;
    my $base = basename( $h_files[$deref] );
    $h_label->set_text( sprintf gettext("Deleted %s."), $base );
    @h_files = glob "$main::scanner->{history}/*";
}

sub history_del_all {
    return unless (@h_files);
    my $confirm_message = gettext("Really delete all history logs?");
    my $confirm =
        Gtk2::MessageDialog->new( $window, [qw(modal destroy-with-parent)],
        'question', 'ok-cancel', $confirm_message );

    if ( "cancel" eq $confirm->run ) {
        $confirm->destroy;
        return;
    } else {
        $confirm->destroy;
        my @not_del;
        my $size = @h_files;
        foreach (@h_files) {
            unlink($_) or push( @not_del, $_ );
        }
        if ( scalar(@not_del) >= 1 ) {
            $h_label->set_text(
                sprintf gettext("Could not delete files: %s!"), @not_del );
        } else {
            show_message_dialog( $window, 'info', 'ok',
                gettext("Successfully removed history logs.") );
        }
        splice @{ $new_hlist->{data} }, 0, $size;
        @h_files = glob "$main::scanner->{history}/*";
    }
}

sub view_box {
    my @sel = $new_hlist->get_selected_indices;
    return if ( !@sel );
    my $deref = $sel[0];
    return if ( not exists $h_files[$deref] );

    my $base = basename( $h_files[$deref] );

    my $view_win = Gtk2::Dialog->new( sprintf( gettext("Viewing %s"), $base ),
        undef, [], 'gtk-close' => 'close' );
    $view_win->set_default_response('close');
    $view_win->signal_connect( response => sub { $view_win->destroy } );
    $view_win->set_default_size( 600, 350 );

    my $textview = Gtk2::TextView->new;
    $textview->set( editable => FALSE );

    my $FILE;    # filehandle for histories log
    unless ( open( $FILE, '<', $h_files[$deref] ) ) {
        my $notice = sprintf gettext("Problems opening %s..."),
            $h_files[$deref];
        show_message_dialog( $window, 'error', 'ok', $notice );
        return;
    }
    my $text;
    $text = do {
        local $/ = undef;
        $text = <$FILE>;
    };
    close($FILE)
        or warn sprintf gettext("Unable to close FILE %s! %s\n"),
        $h_files[$deref];

    my $textbuffer = $textview->get_buffer;
    $textbuffer->create_tag( 'mono', family => 'Monospace' );
    $textbuffer->insert_with_tags_by_name( $textbuffer->get_start_iter, $text,
        'mono' );

    my $scroll_win = Gtk2::ScrolledWindow->new;
    $scroll_win->set_border_width(5);
    $scroll_win->set_shadow_type('etched-in');
    $scroll_win->set_policy( 'automatic', 'automatic' );

    $view_win->vbox->pack_start( $scroll_win, TRUE, TRUE, 0 );

    $scroll_win->add($textview);
    $view_win->show_all();
}

sub load_prefs {
    my $Config = ClamTk::Prefs->get_all_prefs($main::scanner);
    return unless $Config;
    my $changer;
    if ( $Config->{_}->{SaveToLog} ) {
        $log_box->set_active(TRUE);
    }
    if ( $Config->{_}->{ScanHidden} ) {
        $hidden_box->set_active(TRUE);
    }
    if ( $Config->{_}->{SizeLimit} ) {
        $size_box->set_active(TRUE);
    }
    if ( $Config->{_}->{Thorough} ) {
        $deep_box->set_active(TRUE);
    }
    $scan_status_label->set_text(
        gettext("\nLoaded your scanning preferences.") );
}

sub save_prefs {
    my $Config = ClamTk::Prefs->get_all_prefs($main::scanner);
    return unless $Config;

    my ( $ls, $li, $update );
    if ( $Config->{_}->{Update} ) {
        $update = $Config->{_}->{Update};
    } else {
        $update = 'undef';
    }

    if ( $Config->{_}->{LastScan} ) {
        $ls = $Config->{_}->{LastScan};
    } else {
        $ls = gettext("Never");
    }
    if ( $Config->{_}->{LastInfection} ) {
        $li = $Config->{_}->{LastInfection};
    } else {
        $li = gettext("Never");
    }

    my $ret_val =
        ClamTk::Prefs->set_all( $main::scanner, $save_log, $hidden, $size_set,
        $thorough, $ls, $li, $update );

    my $msg =
        ( $ret_val == 0 )
        ? gettext("Could not save your preferences.")
        : gettext("Your preferences were saved.");

    $scan_status_label->set_text($msg);
}

sub about {
    my $about = Gtk2::AboutDialog->new;
    $about->set_authors("Dave M, dave.nerd <at> gmail.com");
    $about->set_version( $main::scanner->{VERSION} );
    my @translators = (
	'Nawaf AsSulami, Arabic (ar_SA)',
        'Karel Hudan, Czech (cs_CZ)',
        'Jimmy Christensen, Danish (da_DK)',
        'Ronny Steiner, German (de_DE)',
        'Alexandros Chatzicharalambidis, Greek (el_GR)',
        'Mariano Rojo, Spanish (es_ES)',
        'Heriberto  Cantú, Spanish (es_ES)',
        'Alain Bernard, French (fr_FR)',
        'David Garcia Rojo, French (fr_FR)',
        'Viale Fabrice, French (fr_FR)',
        'Arnaud Dubois, French (fr_FR)',
        'Sébastien, French (fr_FR)',
        'Román Pena, Galician (gl_ES)',
        'Edoardo Tosca, Italian (it_IT)',
        'Tobia Fasciati, Italian (it_IT)',
        'Alessandro Volturno, Italian (it_IT)',
        'Gina C, Korean (ko_KR)',
        'Eun Joon Kim, Korean (ko_KR)',
        'Frederik Mattelaere, Dutch (nl_BE)',
        'Rob van den Berg, Dutch (nl_NL)',
        'Robert Tomasik, Polish (pl_PL)',
        'Piotr Makowski, Polish (pl_PL)',
        'Bruno Diniz, Portugese (pt_BR)',
        'Felipe Augusto, Portugese (pt_BR)',
        'Veronica B., Romanian (ro_RO)',
        'Adrian Mowrey, Romanian (ro_RO)',
        'Vitaly Lipatov, Russian (ru_RU)',
        'Tomáš Vadina, Slovak (sk)',
        'Martin McDowell, Slovene (sl_SI)',
        'Petter Viklund, Swedish (sv_SE)',
        'Gürkan Gür, Turkish (tr_TR)',
        'Tao Wei, Chinese (zh_CN)',
        'Aron Xu, Chinese (zh_CN)',
    );
    my @artists =
        ( 'Edoardo Tosca (Website Design)', 'Gerald Ganson (Icon Design)', );
    my $t_list = join "\n", @translators;
    $about->set_translator_credits($t_list);
    my $a_list = join "\n", @artists;
    $about->set_artists($a_list);
    my $logo =
          -e '/usr/share/pixmaps/clamtk.png' ? '/usr/share/pixmaps/clamtk.png'
        : 'usr/share/pixmaps/clamtk.xpm'     ? '/usr/share/pixmaps/clamtk.xpm'
        :                                      '';
    my $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($logo);
    $about->set_logo($pixbuf);
    $about->set_website('http://clamtk.sf.net');
    $about->set_comments(
        "ClamTk is a GUI front-end for the ClamAV antivirus using gtk2-perl."
    );
    $about->set_license( "ClamTk, (c) 2004-2009. All rights reserved.\n\n"
            . "This program is free software; you can redistribute it\n"
            . "and/or modify it under the same terms as Perl itself." );
    $about->run;
    $about->destroy;
}

1;
