package ClamTk::App;

use strict;
use Carp qw/croak/;
use Net::DNS;
$|++;

use encoding 'utf8';

my ( $num_of_sigs, $date_of_sigs );
my ( $DAILY_PATH,  $MAIN_PATH );
my ( $INFO_MAIN,   $INFO_DATE, $INFO_DAILY );

sub new {
    my ( $class, $self ) = @_;

    $self->{directory} = $ENV{HOME} || glob "~";
    $self->{clamtk}    = $self->{directory} . "/.clamtk";
    $self->{viruses}   = $self->{clamtk} . "/viruses";
    $self->{history}   = $self->{clamtk} . "/history";
    $self->{prefs}     = $self->{clamtk} . "/prefs";
    $self->{db}        = $self->{clamtk} . "/db";

    $self->{freshclam} =
          ( -e '/usr/bin/freshclam' )       ? '/usr/bin/freshclam'
        : ( -e '/usr/local/bin/freshclam' ) ? '/usr/local/bin/freshclam'
        : ( -e '/opt/local/bin/freshclam' ) ? '/opt/local/bin/freshclam'
        :                                     croak "freshclam not found!\n";

    $self->{clampath} =
          ( -e '/usr/bin/clamscan' )       ? '/usr/bin/clamscan'
        : ( -e '/usr/local/bin/clamscan' ) ? '/usr/local/bin/clamscan'
        : ( -e '/opt/local/bin/clamscan' ) ? '/opt/local/bin/clamscan'
        :                                    croak "clamscan not found!\n";

    $self->{clamscan} = $self->{clampath};

    $self->{clamscan} .= " --no-summary --block-encrypted ";

    return bless( $self, $class );
}

sub def_paths {

    # There are 3 formats:
    # 1. The .cld files
    # 2. The .cvd files
    # 3. The {daily,main}.info directories
    # The .cld seems to be popular now, so check for that first.

    my @dirs = qw#
        /var/lib/clamav
        /var/clamav
        /opt/local/share/clamav
        /usr/share/clamav
        /usr/local/share/clamav
        #;

    my $user_set = 0;
    if ( ClamTk::Prefs->get_property( $main::scanner, "Update" ) eq 'single' )
    {
        $user_set = 1;
        unshift( @dirs, $main::scanner->{db} );
    }

    for my $dir_list (@dirs) {
        if ( -e "$dir_list/daily.cld" ) {
            $DAILY_PATH = "$dir_list/daily.cld";
        } elsif ( -e "$dir_list/daily.cvd" ) {
            $DAILY_PATH = "$dir_list/daily.cvd";
        }
        if ( -e "$dir_list/main.cld" ) {
            $MAIN_PATH = "$dir_list/main.cld";
        } elsif ( -e "$dir_list/main.cvd" ) {
            $MAIN_PATH = "$dir_list/main.cvd";
        }
        last if ( $DAILY_PATH && $MAIN_PATH );

        # the user may have set single - may need to update db
        last if ($user_set);
    }

    if ( !$DAILY_PATH && !$MAIN_PATH ) {
        for my $dir_list (
            '/var/lib/clamav',         '/var/clamav',
            '/opt/local/share/clamav', '/usr/share/clamav',
            '/usr/local/share/clamav'
            )
        {
            if ( -e "$dir_list/daily.inc/daily.info" ) {
                $DAILY_PATH = "$dir_list/daily.inc/daily.info";
            }
            if ( -e "$dir_list/main.inc/main.info" ) {
                $MAIN_PATH = "$dir_list/main.inc/main.info";
            }
        }

    }
}

sub get_AV_version {
    my ($class)   = shift;
    my ($self)    = shift;
    my ($version) = `$self->{clampath} -V`;
    $version =~ s/ClamAV\s(.*?)\/.*/$1/;
    chomp($version);
    return $version;
}

sub get_AV_remote {
    my ($class)  = shift;
    my ($self)   = shift;
    my $rmt_host = 'current.cvd.clamav.net';

    # No longer need the alarm to "die" since Net::DNS
    # does it for us.
    # Just to be sure, we'll use OpenDNS servers here
    # and set a UDP timeout.
    my ( $get, $res );
    $res = Net::DNS::Resolver->new(
        nameservers => [
            qw(208.67.222.222 208.67.220.220
                208.67.222.220 208.67.220.222)
        ],
        udp_timeout => 3,
    );

    ($get) = $res->send( $rmt_host, "TXT" );
    if ( !$get ) {
        warn $res->errorstring, "\n";
        return 'undef';
    }
    chomp($get);

    if ( $get->string =~ /IN\s+TXT\s+\"(.*?):/ ) {
        return $1;
    } else {
        return 'undef';
    }
}

sub get_num_sigs {
    get_sig_info();
    return ( $INFO_MAIN + $INFO_DAILY );
}

sub get_date_sigs {
    get_sig_info();
    return $INFO_DATE;
}

sub get_sig_info {
    def_paths();
    warn "I don't have any paths!\n" unless ( $MAIN_PATH && $DAILY_PATH );
    if ($MAIN_PATH) {
        my $FILE;
        if ( open( $FILE, "<", $MAIN_PATH ) ) {
            while (<$FILE>) {
                if (/ClamAV-VDB:\S+\s+\S+\s+\S+.*?[\-\+]\d+:\d+:(\d+)/) {
                    $INFO_MAIN = $1;
                    last;
                }
                close($FILE);
            }
        } else {
            $INFO_MAIN = 0;
        }
    } else {
        $INFO_MAIN = 0;
    }

    if ($DAILY_PATH) {
        my $FILE;
        if ( open( $FILE, "<", $DAILY_PATH ) ) {
            while (<$FILE>) {
                if (/ClamAV-VDB:(\S+\s+\S+\s+\S+).*[\-\+]\d+:\d+:(\d+)/) {
                    $INFO_DATE  = $1;
                    $INFO_DAILY = $2;
                    last;
                }
            }
            close($FILE);
        } else {
            $INFO_DATE  = '01 Jan 1970';
            $INFO_DAILY = 0;
        }
    }
}

1;
