#!/bin/sh

# This script freely provided by Bill Landry (bill@inetmsg.com).
# Comments, suggestions, and recommendations for improving this
# script are always welcome.
#
# Script documentation and updates can be viewed/downloaded from:
#
#     http://www.inetmsg.com/pub/
#
# The latest version will always be named: clamav-unofficial-sigs.tar.gz
# Older versions can be found in the "archive" directory.

################################################################################
#                                                                              #
#            THERE ARE NO USER CONFIGURABLE OPTIONS IN THIS SCRIPT             #
#                                    * * *                                     #
#   ALL CONFIGURATION OPTIONS ARE LOCATED IN THE INCLUDED CONFIGURATION FILE   #
#                                                                              #
################################################################################

default_config="/etc/clamav-unofficial-sigs.conf"

version="v3.3 (updated 2009-05-19)"
output_ver="
   `basename $0` $version
"

usage="
ClamAV Unofficial Signature Databases Update Script - $version

   Usage: `basename $0` [OPTION] [PATH|FILE]

        -b      Add a bypass signature entry to local.ign in order
                to temporarily resolve a false-positive issue with
                a third-party signature.  The local.ign file will
                automatically be deleted and the databases reloaded
                once the local.ign file's timestamp is 24 hours old.

        -c      Direct script to use a specific configuration file
                e.g.: '-c /path/to/`basename "$default_config"`'

        -d      Decode a third-party signature either by signature name
                (e.g: Sanesecurity.Junk.15248) or hexadecimal string.
                This flag will 'NOT' decode image signatures.

        -e      Hexadecimal encode an entire input string that can be
                used in any '*.ndb' signature database file.

        -f      Hexadecimal encode a formatted input string containing
                signature spacing fields '{}, (), *', without encoding 
                the spacing fields, so that the encoded signature can
                be used in any '*.ndb' signature database file.

        -g      GPG verify a specific Sanesecurity database file
                e.g.: '-g filename.ext' (do not include file path)

        -h      Display this script's help and usage information

        -i      Output system and configuration information for
                viewing or possible debugging purposes

        -r      Remove the clamav-unofficial-sigs script and all of
                its associated files and databases from the system.

        -s      Clamscan integrity test a specific database file
                e.g.: '-s filename.ext' (do not include file path)

        -v      Output script version and date information

Alternative to using '-c': Place config file in /etc ($default_config)
"

# Function to handle general response if script cannot find the config file in /etc.
no_default_config () {
if [ ! -s "$default_config" ] ; then
   echo ""
   echo "Cannot find your configuration file - place a copy in /etc and try again..."
   echo ""
   echo "     e.g.: $default_config"
   echo ""
   exit
fi
. "$default_config"
echo ""
}

# Function to support user config settings for applying file and directory access permissions.
perms () {
   if [ -n "$clam_user" -a -n "$clam_group" ] ; then
      "${@:-}"
   fi
}

# Take input from the commandline and process.
while getopts 'bc:defg:hirs:v' option ; do
   case $option in
      b)  no_default_config
          echo "Input a third-party signature name that you wish to bypass due to false-positives"
          echo "and press enter (do not  include '.UNOFFICIAL' in the signature name nor add quote"
          echo "marks to any input string):"
          echo ""
          read input
          if [ -n "$input" ]
             then
                cd "$clam_dbs"
                input=`echo "$input" | tr -d "'" | tr -d '"'`
                sig_ign=`grep -n "$input:" *.ndb | cut -d ":" -f-3`
                if [ -n "$sig_ign" ]
                   then
                      if ! grep "$sig_ign" local.ign > /dev/null 2>&1
                         then
                            echo "$sig_ign" >> local.ign
                            perms chown $clam_user:$clam_group local.ign
                            chmod 0644 local.ign
                            $reload_opt
                            echo ""                      
                            echo "Signature '$input' has been added to local.ign and databases have been reloaded."
                            echo "The local.ign file will automatically be deleted once the file's timestamp is 24 hour old."
                         else
                            echo ""
                            echo "Signature '$input' already exists in local.ign - no action taken."
                      fi
                   else
                      echo ""
                      echo "Signature '$input' could not be found."
                      echo ""
                      echo "This script will only create a bypass entry in local.ign for ClamAV"
                      echo "'UNOFFICIAL' third-Party signatures as found in the *.ndb databases."
                fi
             else
                echo "No input detected - no action taken."
          fi
          echo ""
          exit
          ;;
      c)  conf_file="$OPTARG"
          ;;
      d)  no_default_config
          echo "Input a third-party signature name to decode (e.g: Sanesecurity.Junk.15248) or"
          echo "a hexadecimal encoded data string and press enter (do not include '.UNOFFICIAL'"
          echo "in the signature name nor add quote marks to any input string):"
          echo ""
          read input
          input=`echo "$input" | tr -d "'" | tr -d '"'`
          echo ""
          if `echo "$input" | grep "\." > /dev/null`
             then
                cd "$clam_dbs"
                sig=`grep "$input:" *.ndb`
                if [ -n "$sig" ]
                   then
                      db_file=`echo "$sig" | cut -d ':' -f1`
                      echo "$input found in: $db_file"
                      echo "$input signature decodes to:"
                      echo ""
                      echo "$sig" | cut -d ":" -f5 | perl -pe 's/([a-fA-F0-9]{2})|(\{[^}]*\}|\([^)]*\))/defined $2 ? $2 : chr(hex $1)/eg'
                   else
                      echo "Signature '$input' could not be found."
                      echo ""
                      echo "This script will only decode ClamAV 'UNOFFICIAL' third-Party,"
                      echo "non-image based, signatures as found in the *.ndb databases."
                fi
            else
               echo "Here is the decoded hexadecimal input string:"
               echo ""
               echo "$input" | perl -pe 's/([a-fA-F0-9]{2})|(\{[^}]*\}|\([^)]*\))/defined $2 ? $2 : chr(hex $1)/eg'
          fi
          echo ""
          exit
          ;;
      e)  no_default_config
          echo "Input the data string that you want to hexadecimal encode and then press enter.  Do not include"
          echo "any quotes around the string unless you want them included in the hexadecimal encoded output:"
          echo ""
          read input
          echo ""
          echo "Here is the hexadecimal encoded input string:"
          echo ""
          echo "$input" | perl -pe 's/(.)/sprintf("%02lx", ord $1)/eg'
          echo ""
          exit
          ;;
      f)  no_default_config
          echo "Input a formated data string containing spacing fields '{}, (), *' that you want to hexadecimal"
          echo "encode, without encoding the spacing fields, and then press enter.  Do not include any quotes"
          echo "around the string unless you want them included in the hexadecimal encoded output:"
          echo ""
          read input
          echo ""
          echo "Here is the hexadecimal encoded input string:"
          echo ""
          echo "$input" | perl -pe 's/(\{[^}]*\}|\([^)]*\)|\*)|(.)/defined $1 ? $1 : sprintf("%02lx", ord $2)/eg'
          echo ""
          exit
          ;;
      g)  no_default_config
          db_file=`echo "$OPTARG" | awk -F '/' '{print $NF}'`
          if [ -s "$ss_dir/$db_file" ]
             then
                echo "GPG signature testing database file: $ss_dir/$db_file"
                echo ""
                if ! gpg --trust-model always -q --no-default-keyring --homedir $gpg_dir --keyring $gpg_dir/ss-keyring.gpg --verify $ss_dir/$db_file.sig $ss_dir/$db_file
                   then
                     gpg --always-trust -q --no-default-keyring --homedir $gpg_dir --keyring $gpg_dir/ss-keyring.gpg --verify $ss_dir/$db_file.sig $ss_dir/$db_file
                fi
             else
                echo "File '$db_file' cannot be found or is not a Sanesecurity database file."
                echo "Only the following Sanesecurity and OITC databases can be GPG signature tested:"
                echo "$ss_dbs"
                echo "Check the file name and try again..."
          fi
          echo ""
          exit
          ;;
      h)  echo "$usage"
          exit
          ;;
      i)  no_default_config
          echo "*** SCRIPT VERSION ***"
          echo "`basename $0` $version"
          echo ""
          echo "*** SYSTEM INFORMATION ***"
          uname=`which uname`
          $uname -a
          echo ""
          echo "*** CLAMSCAN LOCATION & VERSION ***"
          clamscan=`which clamscan`
          echo "$clamscan"
          $clamscan --version | head -1
          echo ""
          echo "*** RSYNC LOCATION & VERSION ***"
          rsync=`which rsync`
          echo "$rsync"
          $rsync --version | head -1
          echo ""
          echo "*** CURL LOCATION & VERSION ***"
          curl=`which curl`
          echo "$curl"
          $curl --version | head -1
          echo ""
          echo "*** GPG LOCATION & VERSION ***"
          gpg=`which gpg`
          echo "$gpg"
          $gpg --version | head -1
          echo ""
          echo "*** SCRIPT WORKING DIRECTORY INFORMATION ***"
          ls -ld $work_dir
          echo ""
          ls -lR $work_dir | grep -v total
          echo ""
          echo "*** CLAMAV DIRECTORY INFORMATION ***"
          ls -ld $clam_dbs
          echo "---"
          ls -l $clam_dbs | grep -v total
          echo ""
          echo "*** SCRIPT CONFIGURATION SETTINGS ***"
          egrep -v "^#|^$" $default_config
          echo ""
          exit
          ;;
      r)  no_default_config
          if [ -n "$pkg_mgr" -a -n "$pkg_rm" ]
             then
                echo "  This script (clamav-unofficial-sigs) was installed on the system"
                echo "  via '$pkg_mgr', use '$pkg_rm' to remove the script"
                echo "  and all of its associated files and databases from the system."
                echo ""
             else
                echo "  Are you sure you want to remove the clamav-unofficial-sigs script and all of its"
                echo -n "  associated files, third-party databases, and work directories from the system? (y/n): "
                read response
                if [ "$response" = "y" -o "$response" = "Y" ]
                   then
                      if [ -s "$config_dir/purge.txt" ] 
                         then
                            echo ""
                            for file in `cat $config_dir/purge.txt` ; do
                               rm -f -- "$file"
                               echo "     Removed file: $file"
                            done
                            cron_file=`find /etc/ -name clamav-unofficial-sigs.cron`
                            if [ -s "$cron_file" ] ; then
                               rm -f "$cron_file"
                               echo "     Removed file: $cron_file"
                            fi
                            log_rotate_file=`find /etc/ -name clamav-unofficial-sigs.logrotate`
                            if [ -s "$log_rotate_file" ] ; then
                               rm -f "$log_rotate_file"
                               echo "     Removed file: $log_rotate_file"
                            fi
                            rm -f -- "$default_config" && echo "     Removed file: $default_config"
                            rm -f -- "$0" && echo "     Removed file: $0"
                            rm -rf -- "$work_dir" && echo "     Removed script working directories: $work_dir"
                            echo ""
                            echo "  The clamav-unofficial-sigs script and all of its associated files, third-party"
                            echo "  databases, and work directories have been successfully removed from the system."
                            echo ""
                         else
                            echo "  Cannot locate 'purge.txt' file in $config_dir."
                            echo "  Files and signature database will need to be removed manually."
                            echo ""
                      fi
                   else
                      echo "$usage"
                fi
          fi
          exit
          ;;
      s)  no_default_config
          input=`echo "$OPTARG" | awk -F '/' '{print $NF}'`
          db_file=`find $work_dir -name $input`
          if [ -s "$db_file" ]
             then
                echo "Clamscan integrity testing: $db_file"
                echo ""
                if clamscan --quiet -d $db_file $config_dir/scan-test.txt ; then
                   echo "Clamscan reports that '$input' database integrity tested GOOD"
                fi
             else
                echo "File '$input' cannot be found."
                echo "Here is a list of third-party databases that can be clamscan integrity tested:"
                echo ""
                echo "Sanesecurity $ss_dbs""MSRBL $msrbl_dbs""SecuriteInfo $si_dbs""MalwarePatrol $mbl_dbs"
                echo "Check the file name and try again..."
          fi
          echo ""
          exit
          ;;
      v)  echo "$output_ver"
          exit
          ;;
      *)  echo "$usage"
          exit
          ;;
   esac
done

# Handle '-c' config file location issues.
if [ "$1" = -c ] ;
   then
      if [ ! -s "$conf_file" ] ; then
         echo ""
         echo "   Config file does not exist at: $2"
         echo "   Check the config file path and try again..."
         echo "$usage"
         exit
      fi
      if [ "`basename "$conf_file"`" != "`basename "$default_config"`" ] ; then
         echo ""
         echo "   Invalid config file: $2"
         echo "   Config file must be named: `basename $default_config`"
         echo "$usage"
         exit
      fi
      config_source="$conf_file"
   else
      if [ $# -ne 0 ] ; then
         echo ""
         echo "   Invalid option: $1"
         echo "$usage"
         exit
      fi
      if [ ! -s "$default_config" ] ; then
         echo ""
         echo "   Cannot find default config file at: $default_config"
         echo "$usage"
         exit
      fi
      config_source="$default_config"
fi

. "$config_source"

################################################################################

# Using functions here to handle config settings for script comments and logging.
comment () {
   test "$comment_silence" = "no" && echo "${@:-}"
}

log () {
   test "$enable_logging" = "yes" && echo `date "+%b %d %T"` "${@:-}" >> "$log_file_path/$log_file_name"
}

# Check to see if the script's "USER CONFIGURATION FILE" has been completed.
if [ "$user_configuration_complete" != "yes" ]
   then
      echo ""
      echo "              *** SCRIPT CONFIGURATION HAS NOT BEEN COMPLETED ***"
      echo "   Please review the script configuration file: `basename $default_config`."
      echo "       Once the user configuration has been completed, rerun the script."
      echo ""
      log "ALERT - SCRIPT HALTED, user configuration not completed"
   exit 1
fi

# Check to see if the working directories have been created.
# If not, create them.  Otherwise, ignore and proceed with script.
mkdir -p "$ss_dir" "$msrbl_dir" "$si_dir" "$mbl_dir" "$config_dir" "$gpg_dir" "$add_dir"

# Set secured access permissions to the GPG directory
chmod 0700 "$gpg_dir"

# If we haven't done so yet, download Sanesecurity public GPG key and import to custom keyring.
if [ ! -s "$gpg_dir/publickey.gpg" ] ; then
   if ! curl -s -S $curl_proxy --connect-timeout 15 --max-time 60 -L -R http://www.sanesecurity.net/publickey.gpg -o $gpg_dir/publickey.gpg
      then
         echo ""
         echo "Could not download Sanesecurity public GPG key"
         log "ALERT - Could not download Sanesecurity public GPG key"
         exit 1
      else
         comment ""
         comment "Sanesecurity public GPG key successfully downloaded"
         comment ""
         log "INFO - Sanesecurity public GPG key successfully downloaded"
         rm -f -- "$gpg_dir/ss-keyring.gp*"
         if ! gpg -q --no-options --no-default-keyring --homedir $gpg_dir --keyring $gpg_dir/ss-keyring.gpg --import $gpg_dir/publickey.gpg 2>/dev/null
            then
               echo "Could not import Sanesecurity public GPG key to custom keyring"
               log "ALERT - Could not import Sanesecurity public GPG key to custom keyring"
               exit 1
            else
               chmod 0644 $gpg_dir/*.*
               comment "Sanesecurity public GPG key successfully imported to custom keyring"
               log "INFO - Sanesecurity public GPG key successfully imported to custom keyring"
         fi
   fi
fi

# If custom keyring is missing, try to re-import Sanesecurity public GPG key.
if [ ! -s "$gpg_dir/ss-keyring.gpg" ] ; then
   rm -f -- "$gpg_dir/ss-keyring.gp*"
   if ! gpg -q --no-options --no-default-keyring --homedir $gpg_dir --keyring $gpg_dir/ss-keyring.gpg --import $gpg_dir/publickey.gpg
      then
         echo "Custom keyring MISSING or CORRUPT!  Could not import Sanesecurity public GPG key to custom keyring"
         log "ALERT - Custom keyring MISSING or CORRUPT!  Could not import Sanesecurity public GPG key to custom keyring"
         exit 1
      else
         chmod 0644 $gpg_dir/*.*
         comment "Sanesecurity custom keyring MISSING!  GPG key successfully re-imported to custom keyring"
         comment ""
         log "INFO - Sanesecurity custom keyring MISSING!  GPG key successfully re-imported to custom keyring"
   fi
fi

# Database update check, time randomization section.  This script now
# provides support for both bash and non-bash enabled system shells.
if [ "$enable_random" = "yes" ] ; then
   if [ -n "$RANDOM" ]
      then
         sleep_time=$(($RANDOM * $(($max_sleep_time - $min_sleep_time)) / 32767 + $min_sleep_time))
      else
         sleep_time=0
         while [ "$sleep_time" -lt "$min_sleep_time" -o "$sleep_time" -gt "$max_sleep_time" ] ; do
            sleep_time=`head -1 /dev/urandom | cksum | awk '{print $2}'`
         done
   fi
   if [ ! -t 0 ]
      then
         comment "`date` - Pausing database file updates for $sleep_time seconds..."
         log "INFO - Pausing database file updates for $sleep_time seconds..."
         sleep $sleep_time
         comment ""
         comment "`date` - Pause complete, checking for new database files..."
      else
         curl_silence="no"
         rsync_silence="no"
         gpg_silence="no"
         comment_silence="no"
         log "INFO - Script was run manually"
   fi
fi

# Create "scan-test.txt" file for clamscan database integrity testing.
if [ ! -s "$config_dir/scan-test.txt" ] ; then
   echo "This is the clamscan test file..." > "$config_dir/scan-test.txt"
fi

# Unofficial ClamAV database provider URLs
ss_url="rsync.sanesecurity.net"
msrbl_url="rsync.mirror.msrbl.com"
si_url="clamav.securiteinfo.com"
mbl_url="www.malwarepatrol.net"

# Create the Sanesecurity rsync "include" file (defines which files to download).
ss_include_dbs="$config_dir/ss-include-dbs.txt"
if [ -n "$ss_dbs" ] ; then
   rm -f -- "$ss_include_dbs" "$ss_dir/*.sha256"
   for db_name in $ss_dbs ; do
      echo "$db_name" >> "$ss_include_dbs"
      echo "$db_name.sig" >> "$ss_include_dbs"
   done
fi

# Create the MSRBL rsync "include" file (defines which files to download).
msrbl_include_dbs="$config_dir/msrbl-include-dbs.txt"
if [ -n "$msrbl_dbs" ] ; then
   rm -f -- "$msrbl_include_dbs"
   for db_name in $msrbl_dbs ; do
      echo "$db_name" >> "$msrbl_include_dbs"
   done
fi

# If rsync proxy is defined in the config file, then export it for use.
if [ -n "$rsync_proxy" ]; then
   RSYNC_PROXY="$rsync_proxy"
   export RSYNC_PROXY
fi

# Create files containing lists of current and previously active 3rd-party databases
# so that databases and/or backup files that are no longer being used can be removed.
current_dbs="$config_dir/current-dbs.txt"
previous_dbs="$config_dir/previous-dbs.txt"
mv -f "$current_dbs" "$previous_dbs" 2>/dev/null
clamav_files () {
   echo "$clam_dbs/$db" >> "$current_dbs"
   if [ "$keep_db_backup" = "yes" ] ; then
      echo "$clam_dbs/$db-bak" >> "$current_dbs"
   fi
}
if [ -n "$ss_dbs" ] ; then
   for db in $ss_dbs ; do
      echo "$ss_dir/$db" >> "$current_dbs"
      echo "$ss_dir/$db.sig" >> "$current_dbs"
      clamav_files
   done
fi
if [ -n "$msrbl_dbs" ] ; then
   for db in $msrbl_dbs ; do
      echo "$msrbl_dir/$db" >> "$current_dbs"
      clamav_files
   done
fi
if [ -n "$si_dbs" ] ; then
   for db in $si_dbs ; do
      echo "$si_dir/$db" >> "$current_dbs"
      echo "$si_dir/$db.gz" >> "$current_dbs"
      clamav_files
   done
fi
if [ -n "$mbl_dbs" ] ; then
   for db in $mbl_dbs ; do
      echo "$mbl_dir/$db" >> "$current_dbs"
      clamav_files
   done
fi
if [ -n "$add_dbs" ] ; then
   for db in $add_dbs ; do
      echo "$add_dir/$db" >> "$current_dbs"
      clamav_files
   done
fi

# Remove 3rd-party databases and/or backup files that are no longer being used.
db_changes="$config_dir/db-changes.txt"
diff "$current_dbs" "$previous_dbs" 2>/dev/null | grep '>' | awk '{print $2}' > "$db_changes"
if [ -s "$db_changes" ] ; then
   if grep -vq "bak" $db_changes 2> /dev/null ; then
      do_clamd_reload=2
   fi
   comment ""
   for file in `cat $db_changes` ; do
      rm -f -- "$file"
      comment "File removed: $file"
      log "INFO - File removed: $file"
   done
fi

# Create "purge.txt" file for package maintainers to support package uninstall.
purge="$config_dir/purge.txt"
cp -f "$current_dbs" "$purge"
echo "$config_dir/current-dbs.txt" >> $purge
echo "$config_dir/db-changes.txt" >> $purge
echo "$config_dir/last-mbl-update.txt" >> $purge
echo "$config_dir/last-si-update.txt" >> $purge
echo "$config_dir/msrbl-include-dbs.txt" >> $purge
echo "$config_dir/previous-dbs.txt" >> $purge
echo "$config_dir/scan-test.txt" >> $purge
echo "$config_dir/ss-include-dbs.txt" >> $purge
echo "$gpg_dir/publickey.gpg" >> $purge
echo "$gpg_dir/secring.gpg" >> $purge
echo "$gpg_dir/ss-keyring.gpg*" >> $purge
echo "$gpg_dir/trustdb.gpg" >> $purge
echo "$log_file_path/$log_file_name*" >> $purge
echo "$purge" >> "$purge"

# Silence rsync output and only report errors - useful if script is run via cron.
if [ "$rsync_silence" = "yes" ] ; then
   rsync_output_level="-q"
fi

# If the local rsync client supports the '--no-motd' flag, then enable it.
if rsync --help | grep 'no-motd' > /dev/null ; then
   no_motd="--no-motd"
fi

# If the local rsync client supports the '--contimeout' flag, then enable it.
if rsync --help | grep 'contimeout' > /dev/null ; then
   contimeout="--contimeout=30"
fi

# Silence curl output and only report errors - useful if script is run via cron.
if [ "$curl_silence" = "yes" ] ; then
   curl_output_level="-s -S"
fi

# If ClamD status check is enabled ("clamd_socket" variable is uncommented
# and the socket path is correctly specified in "User Edit" section above),
# then test to see if clamd is running or not.
if [ -n "$clamd_socket" ] ; then
   if [ "`perl -e 'use IO::Socket::UNIX; print $IO::Socket::UNIX::VERSION,"\n"' 2>/dev/null`" ]
      then
         io_socket1=1
         if [ "`perl -MIO::Socket::UNIX -we '$s = IO::Socket::UNIX->new(shift); $s->print("PING"); \
            print $s->getline; $s->close' "$clamd_socket" 2>/dev/null`" = "PONG" ] ; then
            io_socket2=1
            comment "===================="
            comment "= ClamD is running ="
            comment "===================="
            log "INFO - ClamD is running"
         fi
      else
         socat="`which socat 2>/dev/null`"
         if [ -n "$socat" -a -x "$socat" ] ; then
            socket_cat1=1
            if [ "`(echo "PING"; sleep 1;) | socat - "$clamd_socket" 2>/dev/null`" = "PONG" ] ; then
               socket_cat2=1
               comment "===================="
               comment "= ClamD is running ="
               comment "===================="
               log "INFO - ClamD is running"
            fi
         fi
   fi
   if [ -z "$io_socket1" -a -z "$socket_cat1" ]
      then
         echo ""
         echo "                         --- WARNING ---"
         echo "   It appears that neither 'SOcket CAT' (socat) nor the perl module"
         echo "   'IO::Socket::UNIX' are installed on the system.  In order to run"
         echo "   the ClamD socket test to determine whether ClamD is running or"
         echo "   or not, either 'socat' or 'IO::Socket::UNIX' must be installed."
         echo ""
         echo "   You can silence this warning by either installing 'socat' or the"
         echo "   'IO::Socket::UNIX' perl module, or by simply commenting out the"
         echo "   'clamd_socket' variable in the clamav-unofficial-sigs.conf file."
         log "WARNING - Neither socat nor IO::Socket::UNIX perl module found, cannot test whether ClamD is running"
      else
         if [ -z "$io_socket2" -a -z "$socket_cat2" ] ; then
            echo ""
            echo "     *************************"
            echo "     *     !!! ALERT !!!     *"
            echo "     * CLAMD IS NOT RUNNING! *"
            echo "     *************************"
            echo ""
            log "ALERT - ClamD is not running"
            if [ -n "$start_clamd" ] ; then
               echo "    Attempting to start ClamD..."
               echo ""
               if [ -n "$io_socket1" ]
                  then
                     rm -f -- "$clamd_pid" "$clamd_lock" "$clamd_socket" 2>/dev/null
                     $start_clamd > /dev/null && sleep 5
                     if [ "`perl -MIO::Socket::UNIX -we '$s = IO::Socket::UNIX->new(shift); $s->print("PING"); \
                        print $s->getline; $s->close' "$clamd_socket" 2>/dev/null`" = "PONG" ]
                        then
                           echo "=================================="
                           echo "= ClamD was successfully started ="
                           echo "=================================="
                           log "NOTICE - ClamD was successfuly started"
                        else
                           echo "     *************************"
                           echo "     *     !!! PANIC !!!     *"
                           echo "     * CLAMD FAILED TO START *"
                           echo "     *************************"
                           echo ""
                           echo "Check to confirm that the clamd start process defined for"
                           echo "the 'start_clamd' variable in the 'USER EDIT SECTION' is"
                           echo "set correctly for your particular distro.  If it is, then"
                           echo "check your logs to determine why clamd failed to start."
                           echo ""
                           log "CRITICAL - ClamD failed to start"
                        exit 1
                     fi
                  else
                     if [ -n "$socket_cat1" ] ; then
                        rm -f -- "$clamd_pid" "$clamd_lock" "$clamd_socket" 2>/dev/null
                        $start_clamd > /dev/null && sleep 5
                        if [ "`(echo "PING"; sleep 1;) | socat - "$clamd_socket" 2>/dev/null`" = "PONG" ]
                           then
                              echo "=================================="
                              echo "= ClamD was successfully started ="
                              echo "=================================="
                              log "NOTICE - ClamD was successfuly started"
                           else
                              echo "     *************************"
                              echo "     *     !!! PANIC !!!     *"
                              echo "     * CLAMD FAILED TO START *"
                              echo "     *************************"
                              echo ""
                              echo "Check to confirm that the clamd start process defined for"
                              echo "the 'start_clamd' variable in the 'USER EDIT SECTION' is"
                              echo "set correctly for your particular distro.  If it is, then"
                              echo "check your logs to determine why clamd failed to start."
                              echo ""
                              log "CRITICAL - ClamD failed to start"
                           exit 1
                        fi
                     fi
               fi
            fi
         fi
   fi
fi

# Check and save current system time since epoch for time related database downloads.
# However, if unsuccessful, issue a warning that we cannot calculate times since epoch.
if [ -n "$si_dbs" -o -n "mbl_dbs" ]
   then
      if [ `date +%s` -gt 0 2>/dev/null ]
         then
            current_time=`date +%s`
         else
            if [ `perl -le print+time 2>/dev/null` ] ; then
               current_time=`perl -le print+time`
            fi
      fi
   else
      echo ""
      echo "                           --- WARNING ---"
      echo "The system's date function does not appear to support 'date +%s', nor was 'perl' found"
      echo "on the system.  The SecuriteInfo and MalwarePatrol updates were bypassed at this time."
      echo ""
      echo "You can silence this warning by either commenting out the 'si_dbs' and 'mbl_dbs'"
      echo "variables in the 'USER CONFIGURATION' section of the script, or by installing perl or"
      echo "the GNU date utility, either of which can calculate the needed seconds since epoch."
      log "WARNING - Systems does not support calculating time since epoch, SecuriteInfo and MalwarePatrol updates bypassed"
      si_dbs=""
      mbl_dbs=""
fi

# Check to see if local.ign file exists and if it's timestamp is 24 hours old, delete the file.
if [ -e "$clam_dbs/local.ign" -a -n "$current_time" ] ; then
   if [ -e "$clam_dbs/local.ign" -a -n "$current_time" ] ; then
      file_time=`stat -c %Y "$clam_dbs/local.ign"`
      file_time_plus=$(($file_time + 86400))
      if [ "$current_time" -gt "$file_time_plus" ] ; then
         rm -f "$clam_dbs/local.ign"
         comment ""
         comment "File 'local.ign' timestamp is older than 24 hours - file deleted"
         log "INFO - File 'local.ign' timestamp is older than 24 hours - file deleted"
         do_clamd_reload=3
      fi
   fi
fi

################################################################
# Check for Sanesecurity database & GPG signature file updates #
################################################################
if [ -n "$ss_dbs" ] ; then
   db_file=""
   comment ""
   comment "======================================================================"
   comment "Sanesecurity Database & GPG Signature File Updates"
   comment "======================================================================"
   ss_mirror_ips=`host -T $ss_url | awk '{print $NF}'`
   for ss_mirror_ip in $ss_mirror_ips ; do
      ss_mirror_name=`host $ss_mirror_ip | awk '{print $NF}' | cut -d "(" -f2 | sed 's/\(.*\)./\1/'`
      ss_mirror_site_info="$ss_mirror_name $ss_mirror_ip"
      comment ""
      comment "Sanesecurity mirror site used: $ss_mirror_site_info"
      log "INFO - Sanesecurity mirror site used: $ss_mirror_site_info"
      if rsync $rsync_output_level $no_motd --files-from=$ss_include_dbs -ctz $contimeout \
         --timeout=30 --stats rsync://$ss_mirror_ip/sanesecurity $ss_dir 2>/dev/null
         then
            ss_rsync_success="1"
            for db_file in $ss_dbs ; do
               if ! cmp -s $ss_dir/$db_file $clam_dbs/$db_file ; then
                  comment ""
                  comment "Testing updated Sanesecurity database file: $db_file"
                  log "INFO - Testing updated Sanesecurity database file: $db_file"
                  if ! gpg --trust-model always -q --no-default-keyring --homedir $gpg_dir --keyring $gpg_dir/ss-keyring.gpg \
                       --verify $ss_dir/$db_file.sig $ss_dir/$db_file 2>/dev/null
                     then
                        gpg --always-trust -q --no-default-keyring --homedir $gpg_dir --keyring $gpg_dir/ss-keyring.gpg \
                        --verify $ss_dir/$db_file.sig $ss_dir/$db_file 2>/dev/null
                  fi
                  if [ "$?" = "0" ]
                     then
                        test "$gpg_silence" = "no" && echo "Sanesecurity GPG Signature tested good on $db_file database"
                        log "INFO - Sanesecurity GPG Signature tested good on $db_file database" ; true
                     else
                        echo "Sanesecurity GPG Signature test FAILED on $db_file database - SKIPPING"
                        log "WARNING - Sanesecurity GPG Signature test FAILED on $db_file database - SKIPPING" ; false
                  fi
                  if [ "$?" = "0" ] ; then
                     if clamscan --quiet -d $ss_dir/$db_file $config_dir/scan-test.txt 2>/dev/null
                        then
                           comment "Clamscan reports Sanesecurity $db_file database integrity tested good"
                           log "INFO - Clamscan reports Sanesecurity $db_file database integrity tested good" ; true
                        else
                           echo "Clamscan reports Sanesecurity $db_file database integrity tested BAD - SKIPPING"
                           log "WARNING - Clamscan reports Sanesecurity $db_file database integrity tested BAD - SKIPPING" ; false
                     fi && \
                     (test "$keep_db_backup" = "yes" && cp -f $clam_dbs/$db_file $clam_dbs/$db_file-bak 2>/dev/null ; true) && \
                     if rsync -cqt $ss_dir/$db_file $clam_dbs
                        then
                           perms chown $clam_user:$clam_group $clam_dbs/$db_file
                           comment "Successfully updated Sanesecurity production database file: $db_file"
                           log "INFO - Successfully updated Sanesecurity production database file: $db_file"
                           ss_update=1
                           do_clamd_reload=1
                        else
                           echo "Failed to successfully update Sanesecurity production database file: $db_file - SKIPPING"
                           log "WARNING - Failed to successfully update Sanesecurity production database file: $db_file - SKIPPING"
                     fi
                  fi
               fi
            done
            if [ "$ss_update" != "1" ]
               then
                  comment ""
                  comment "No Sanesecurity database file updates found"
                  log "INFO - No Sanesecurity database file updates found"
                  break
               else
                  break
            fi
         else
            comment "Connection to $ss_mirror_site_info failed - Trying next mirror site..."
            log "WARNING - Connection to $ss_mirror_site_info failed - Trying next mirror site..."
      fi
   done
   if [ "$ss_rsync_success" != "1" ] ; then
      echo ""
      echo "Access to all Sanesecurity mirror sites failed - Check for connectivity issues."
      log "WARNING - Access to all Sanesecurity mirror sites failed - Check for connectivity issues."
   fi
fi

#########################################
# Check for MSRBL database file updates #
#########################################
if [ -n "$msrbl_dbs" ] ; then
   db_file=""
   comment ""
   comment "======================================================================"
   comment "MSRBL Database File Updates"
   comment "======================================================================"
   msrbl_mirror_ips=`host -T $msrbl_url | awk '{print $NF}'`
   for msrbl_mirror_ip in $msrbl_mirror_ips ; do
      msrbl_mirror_name=`host $msrbl_mirror_ip | awk '{print $NF}' | cut -d "(" -f2 | sed 's/\(.*\)./\1/'`
      msrbl_mirror_site_info="$msrbl_mirror_name $msrbl_mirror_ip"
      comment ""
      comment "MSRBL mirror site used: $msrbl_mirror_site_info"
      log "INFO - MSRBL mirror site used: $msrbl_mirror_site_info"
      if rsync $rsync_output_level $no_motd --files-from=$msrbl_include_dbs -ctz --stats \
         $contimeout --timeout=30 rsync://$msrbl_mirror_ip/msrbl $msrbl_dir 2>/dev/null
         then
            msrbl_rsync_success="1"
            for db_file in $msrbl_dbs ; do
               if ! cmp -s $msrbl_dir/$db_file $clam_dbs/$db_file ; then
                  comment ""
                  comment "Testing updated MSRBL database file: $db_file"
                  log "INFO - Testing updated MSRBL database file: $db_file"
                  if clamscan --quiet -d $msrbl_dir/$db_file $config_dir/scan-test.txt 2>/dev/null
                     then
                        comment "Clamscan reports MSRBL $db_file database integrity tested good"
                        log "INFO - Clamscan reports MSRBL $db_file database integrity tested good" ; true
                     else
                        echo "Clamscan reports MSRBL $db_file database integrity tested BAD - SKIPPING"
                        log "WARNING - Clamscan reports MSRBL $db_file database integrity tested BAD - SKIPPING" ; false
                  fi && \
                  (test "$keep_db_backup" = "yes" && cp -f $clam_dbs/$db_file $clam_dbs/$db_file-bak 2>/dev/null ; true) && \
                  if rsync -cqt $msrbl_dir/$db_file $clam_dbs
                     then
                        perms chown $clam_user:$clam_group $clam_dbs/$db_file
                        comment "Successfully updated MSRBL production database file: $db_file"
                        log "INFO - Successfully updated MSRBL production database file: $db_file"
                        msrbl_update=1
                        do_clamd_reload=1
                     else
                        echo "Failed to successfully update MSRBL production database file: $db_file - SKIPPING"
                        log "WARNING - Failed to successfully update MSRBL production database file: $db_file - SKIPPING"
                  fi
               fi
            done
            if [ "$msrbl_update" != "1" ]
               then
                  comment ""
                  comment "No MSRBL database file updates found"
                  log "INFO - No MSRBL database file updates found"
                  break
               else
                  break
            fi
         else
            comment "Connection to $msrbl_mirror_site_info failed - Trying next mirror site..."
            log "WARNING - Connection to $msrbl_mirror_site_info failed - Trying next mirror site..."
      fi
   done
   if [ "$msrbl_rsync_success" != "1" ] ; then
      echo ""
      echo "Access to all MSRBL mirror sites failed - Check for connectivity issues."
      log "WARNING - Access to all MSRBL mirror sites failed - Check for connectivity issues."
   fi
fi

#######################################################################
# Check for updated SecuriteInfo database files every set number of   #
# hours as defined in the "USER CONFIGURATION" section of this script #
#######################################################################
if [ -n "$si_dbs" ] ; then
   if [ -s "$config_dir/last-si-update.txt" ]
      then
         last_si_update=`cat $config_dir/last-si-update.txt`
      else
         last_si_update="0"
   fi
   db_file=""
   loop=""
   update_interval=$(($si_update_hours * 3600))
   time_interval=$(($current_time - $last_si_update))
   if [ "$time_interval" -ge $(($update_interval - 600)) ]
      then
         echo "$current_time" > "$config_dir"/last-si-update.txt
         comment ""
         comment "======================================================================"
         comment "SecuriteInfo Database File Updates"
         comment "======================================================================"
         log "INFO - Checking for SecuriteInfo updates..."
         si_updates="0"
         for db_file in $si_dbs ; do
            z_opt="" DT1="" DT2=""
            test -s $si_dir/$db_file.gz && DT1=`ls -l $si_dir/$db_file.gz | awk '{print $6,$7}'` && z_opt="-z $si_dir/$db_file.gz"
            if [ "$loop" = "1" ]
               then
                  comment "---"
               else
                  comment ""
            fi
            comment "Checking for updated SecuriteInfo database file: $db_file"
            comment ""
            si_db_update="0"
            if curl $curl_proxy $curl_output_level --connect-timeout 15 --max-time 60 \
               -L -R $z_opt -o $si_dir/$db_file.gz http://$si_url/$db_file.gz
               then
                  loop="1"
                  test -s $si_dir/$db_file.gz && DT2=`ls -l $si_dir/$db_file.gz | awk '{print $6,$7}'` && \
                  test "$DT1" != "$DT2" -o ! -s "$si_dir/$db_file" && gunzip -cdfq $si_dir/$db_file.gz > $si_dir/$db_file
                  if ! cmp -s $si_dir/$db_file $clam_dbs/$db_file ; then
                     comment ""
                     comment "Testing updated SecuriteInfo database file: $db_file"
                     log "INFO - Testing updated SecuriteInfo database file: $db_file"
                     if clamscan --quiet -d $si_dir/$db_file $config_dir/scan-test.txt 2>/dev/null
                        then
                           comment "Clamscan reports SecuriteInfo $db_file database integrity tested good"
                           log "INFO - Clamscan reports SecuriteInfo $db_file database integrity tested good" ; true
                        else
                           echo "Clamscan reports SecuriteInfo $db_file database integrity tested BAD - SKIPPING"
                           log "WARNING - Clamscan reports SecuriteInfo $db_file database integrity tested BAD - SKIPPING" ; false
                     fi && \
                     (test "$keep_db_backup" = "yes" && cp -f $clam_dbs/$db_file $clam_dbs/$db_file-bak 2>/dev/null ; true) && \
                     if rsync -cqt $si_dir/$db_file $clam_dbs
                        then
                           perms chown $clam_user:$clam_group $clam_dbs/$db_file
                           comment "Successfully updated SecuriteInfo production database file: $db_file"
                           log "INFO - Successfully updated SecuriteInfo production database file: $db_file"
                           si_updates=1
                           si_db_update=1
                           do_clamd_reload=1
                        else
                           echo "Failed to successfully update SecuriteInfo production database file: $db_file - SKIPPING"
                           log "WARNING - Failed to successfully update SecuriteInfo production database file: $db_file - SKIPPING"
                     fi
                  fi
               else
                 log "WARNING - Failed curl connection to $si_url - SKIPPED SecuriteInfo $db_file update"
            fi
            if [ "$si_db_update" != "1" ] ; then
               comment ""
               comment "No updated SecuriteInfo $db_file database file found"
            fi
         done
         if [ "$si_updates" != "1" ] ; then
            log "INFO - No SecuriteInfo database file updates found"
         fi
      else
         comment ""
         comment "======================================================================"
         comment "SecuriteInfo Database File Updates"
         comment "======================================================================"
         comment ""
         time_remaining=$(($update_interval - $time_interval))
         hours_left=$(($time_remaining / 3600))
         minutes_left=$(($time_remaining % 3600 / 60))
         comment "$si_update_hours hours have not yet elapsed since the last SecuriteInfo update check"
         comment ""
         comment "     --- No update check was performed at this time ---"
         comment ""
         comment "Next check will be performed in approximately $hours_left hour(s), $minutes_left minute(s)"
         log "INFO - Next SecuriteInfo check will be performed in approximately $hours_left hour(s), $minutes_left minute(s)"
   fi
fi

#####################################################################
# Download MalwarePatrol database file(s) every set number of hours #
# as defined in the "USER CONFIGURATION" section of this script.    #
#####################################################################
if [ -n "$mbl_dbs" ] ; then
   if [ -s "$config_dir/last-mbl-update.txt" ]
      then
         last_mbl_update=`cat $config_dir/last-mbl-update.txt`
      else
         last_mbl_update="0"
   fi
   db_file=""
   update_interval=$(($mbl_update_hours * 3600))
   time_interval=$(($current_time - $last_mbl_update))
   if [ "$time_interval" -ge $(($update_interval - 600)) ]
      then
         echo "$current_time" > "$config_dir"/last-mbl-update.txt
         log "INFO - Checking for MalwarePatrol updates..."
         for db_file in $mbl_dbs ; do
            # Delete the old MBL (mbl.db) database file if it exists and start using the newer
            # format (mbl.ndb) database file instead.
            test -e $clam_dbs/$db_file -o -e $clam_dbs/$db_file-bak && rm -f -- "$clam_dbs/mbl.d*"
            comment ""
            comment "======================================================================"
            comment "MalwarePatrol $db_file Database File Update"
            comment "======================================================================"
            comment ""
            if curl $curl_proxy $curl_output_level --connect-timeout 15 --max-time 60 -R \
               -o $mbl_dir/$db_file http://$mbl_url/cgi/submit?action=list_clamav_ext
               then
                  if ! cmp -s $mbl_dir/$db_file $clam_dbs/$db_file 
                     then
                        comment ""
                        comment "Testing updated MalwarePatrol database file: $db_file"
                        log "INFO - Testing updated database file: $db_file"
                        if clamscan --quiet -d $mbl_dir/$db_file $config_dir/scan-test.txt 2>/dev/null
                           then
                              comment "Clamscan reports MalwarePatrol $db_file database integrity tested good"
                              log "INFO - Clamscan reports MalwarePatrol $db_file database integrity tested good" ; true
                           else
                              echo "Clamscan reports MalwarePatrol $db_file database integrity tested BAD - SKIPPING"
                              log "WARNING - Clamscan reports MalwarePatrol $db_file database integrity tested BAD - SKIPPING" ; false
                        fi && \
                        (test "$keep_db_backup" = "yes" && cp -f $clam_dbs/$db_file $clam_dbs/$db_file-bak 2>/dev/null ; true) && \
                        if rsync -cqt $mbl_dir/$db_file $clam_dbs
                           then
                              perms chown $clam_user:$clam_group $clam_dbs/$db_file
                              comment "Successfully updated MalwarePatrol production database file: $db_file"
                              log "INFO - Successfully updated MalwarePatrol production database file: $db_file"
                              mbl_update=1
                              do_clamd_reload=1
                           else
                              echo "Failed to successfully update MalwarePatrol production database file: $db_file - SKIPPING"
                              log "WARNING - Failed to successfully update MalwarePatrol production database file: $db_file - SKIPPING"
                        fi
                     else
                        comment ""
                        comment "MalwarePatrol signature database ($db_file) did not change - skipping"
                        log "INFO - MalwarePatrol signature database ($db_file) did not change - skipping"
                  fi
               else
                  log "WARNING - Failed curl connection to $mbl_url - SKIPPED MalwarePatrol $db_file update"
            fi
         done
      else
         comment ""
         comment "======================================================================"
         comment "MalwarePatrol Database File Update"
         comment "======================================================================"
         comment ""
         time_remaining=$(($update_interval - $time_interval))
         hours_left=$(($time_remaining / 3600))
         minutes_left=$(($time_remaining % 3600 / 60))
         comment "$mbl_update_hours hours have not yet elapsed since the last MalwarePatrol download"
         comment ""
         comment "     --- No database download was performed at this time ---"
         comment ""
         comment "Next download will be performed in approximately $hours_left hour(s), $minutes_left minute(s)"
         log "INFO - Next MalwarePatrol download will be performed in approximately $hours_left hour(s), $minutes_left minute(s)"
   fi
fi

###################################################
# Check for user added signature database updates #
###################################################
if [ -n "$add_dbs" ] ; then
   comment ""
   comment "======================================================================"
   comment "User Added Signature Database File Update(s)"
   comment "======================================================================"
   comment ""
   for db_url in $add_dbs ; do
      base_url=`echo $db_url | cut -d "/" -f3`
      db_file=`basename $db_url`
      if [ "`echo $db_url | cut -d ":" -f1`" = "rsync" ]
         then
            if ! rsync $rsync_output_level $no_motd $contimeout --timeout=30 -ctz --exclude=*.txt \
                 --stats --exclude=*.sha256 --exclude=*.sig --exclude=*.gz $db_url $add_dir ; then
               echo "Failed rsync connection to $base_url - SKIPPED $db_file update"
               log "WARNING - Failed rsync connection to $base_url - SKIPPED $db_file update"
            fi
         else
            if [ -s "$add_dir/$db_file" ] ; then
               z_opt="-z $add_dir/$db_file"
               if ! curl $curl_output_level --connect-timeout 15 --max-time 60 \
                    -L -R $z_opt -o $add_dir/$db_file $db_url ; then
                  echo "Failed curl connection to $base_url - SKIPPED $db_file update"
                  log "WARNING - Failed curl connection to $base_url - SKIPPED $db_file update"
               fi
            fi
      fi
   done
   db_file=""
   for db_file in `ls $add_dir`; do
      if ! cmp -s $add_dir/$db_file $clam_dbs/$db_file ; then
         comment ""
         comment "Testing updated database file: $db_file"
         clamscan --quiet -d $add_dir/$db_file $config_dir/scan-test.txt 2>/dev/null
         if [ "$?" = "0" ]
            then
               comment "Clamscan reports $db_file database integrity tested good"
               log "INFO - Clamscan reports $db_file database integrity tested good" ; true
            else
               echo "Clamscan reports $db_file database integrity tested BAD - SKIPPING"
               log "WARNING - Clamscan reports $db_file database integrity tested BAD - SKIPPING" ; false
         fi && \
         (test "$keep_db_backup" = "yes" && cp -f $clam_dbs/$db_file $clam_dbs/$db_file-bak 2>/dev/null ; true) && \
         if rsync -cqt $add_dir/$db_file $clam_dbs
            then
               perms chown $clam_user:$clam_group $clam_dbs/$db_file
               comment "Successfully updated User-Added production database file: $db_file"
               log "INFO - Successfully updated User-Added production database file: $db_file"
               add_update=1
               do_clamd_reload=1
            else
               echo "Failed to successfully update User-Added production database file: $db_file - SKIPPING"
               log "WARNING - Failed to successfully update User-Added production database file: $db_file - SKIPPING"
         fi
      fi
   done
   if [ "$add_update" != "1" ] ; then
      comment ""
      comment "No User-Defined database file updates found"
      log "INFO - No User-Defined database file updates found"
   fi
fi

# Set appropriate directory and file access permissions
perms chown -R $clam_user:$clam_group "$clam_dbs"
if ! find "$clam_dbs" "$work_dir" -type f -exec chmod 0644 {} + 2>/dev/null ; then
   if ! find "$clam_dbs" "$work_dir" -type f -print0 | xargs -0 chmod 0644 2>/dev/null ; then
      if ! find "$clam_dbs" "$work_dir" -type f | xargs chmod 0644 2>/dev/null ; then
         find "$clam_dbs" -type f -exec chmod 0644 {} \;
         find "$work_dir" -type f -exec chmod 0644 {} \;
      fi
   fi
fi

# Reload all clamd databases if updates detected and $reload_dbs" is
# set to "yes", and neither $reload_opt nor $do_clamd_reload are null.
if [ "$reload_dbs" = "yes" -a -z "$reload_opt" ] ; then
      echo ""
      echo "********************************************************************************************"
      echo "* Check the script's configuration file, 'reload_dbs' enabled but no 'reload_opt' selected *"
      echo "********************************************************************************************"
      log "WARNING - Check the script's configuration file, 'reload_dbs' enabled but no 'reload_opt' selected"
   elif [ "$reload_dbs" = "yes" -a "$do_clamd_reload" = "1" -a -n "$reload_opt" ] ; then
      comment ""
      comment "=================================================="
      comment "= Update(s) detected, reloading ClamAV databases ="
      comment "=================================================="
      log "INFO - Update(s) detected, reloading ClamAV databases"
      $reload_opt
   elif [ "$reload_dbs" = "yes" -a "$do_clamd_reload" = "2" -a -n "$reload_opt" ] ; then
      comment ""
      comment "============================================================"
      comment "= Database removal(s) detected, reloading ClamAV databases ="
      comment "============================================================"
      log "INFO - Database removal(s) detected, reloading ClamAV databases"
      $reload_opt
   elif [ "$reload_dbs" = "yes" -a "$do_clamd_reload" = "3" -a -n "$reload_opt" ] ; then
      comment ""
      comment "============================================================"
      comment "= File 'local.ign' was deleted, reloading ClamAV databases ="
      comment "============================================================"
      log "INFO - File 'local.ign' was deleted, reloading ClamAV databases"
      $reload_opt
   elif [ "$reload_dbs" = "yes" -a -z "$do_clamd_reload" ] ; then
      comment ""
      comment "========================================================="
      comment "= No update(s) detected, NOT reloading ClamAV databases ="
      comment "========================================================="
      log "INFO - No update(s) detected, NOT reloading ClamAV databases"
   else
      comment ""
      comment "======================================================================"
fi

exit $?
