(in-package :contextl)

(defun parse-method-body (form body)
  (let* ((in-layerp (eq (car body) :in-layer))
         (layer-spec (if in-layerp (cadr body) 't)))
    (when (consp layer-spec)
      (unless (null (cddr layer-spec))
        (error "Incorrect :in-layer specification in ~S." form)))
    (loop with layer = (if (atom layer-spec)
                         layer-spec
                         (cadr layer-spec))
          with layer-arg = (if (atom layer-spec)
                             (gensym "LAYER-ARG-")
                             (car layer-spec))
          for tail = (if in-layerp (cddr body) body) then (cdr tail)
          until (listp (car tail))
          collect (car tail) into qualifiers
          finally
          (when (member :in-layer qualifiers)
            (error "Incorrect occurrence of :in-layer in ~S. Must occur before qualifiers." form))
          (return (values layer-arg layer qualifiers (car tail) (cdr tail))))))

(defun prepare-layer (layer)
  (if (symbolp layer)
    (defining-layer layer)
    layer))

(defun prepare-layered-method-body (name form layer-arg body)
  (loop for tail = body then (cdr tail)
        for (first . rest) = tail
        while tail
        while (or (stringp first)
                  (and (consp first) (eq (car first) 'declare)))
        count (stringp first) into nof-seen-strings
        collect first into declarations
        finally
        (when (> nof-seen-strings 1)
          (error "Too many documentation strings in ~S." form))
        (return `(,@declarations
                  (block ,(plain-function-name name)
                    (flet ((call-next-layered-method (&rest args)
                             (declare (dynamic-extent args))
                             (if args
                               (apply #'call-next-method ,layer-arg args)
                               (call-next-method))))
                      #-lispworks
                      (declare (inline call-next-layered-method)
                               (ignorable (function call-next-layered-method)))
                      ,first ,@rest))))))

(defclass layered-function (standard-generic-function) ()
  (:metaclass funcallable-standard-class)
  (:default-initargs :method-class (find-class 'layered-method)))

(defmethod print-object ((object layered-function) stream)
  (print-unreadable-object (object stream :type t :identity t)
    (princ (get-layered-function-caller-name (generic-function-name object)) stream)))

(defclass layered-method (standard-method) ())

(defmacro define-layered-function (name (&rest args) &body options)
  (let ((definer (get-layered-function-definer-name name)))
    (with-unique-names (layer-arg rest-arg)
      `(progn
         (defgeneric ,definer (,layer-arg ,@args)
           ,@(unless (member :generic-function-class options :key #'car)
               '((:generic-function-class layered-function)))
           (:argument-precedence-order 
            ,@(let ((argument-precedence-order (assoc :argument-precedence-order options)))
                (if argument-precedence-order
                  (cdr argument-precedence-order)
                  (required-args args)))
            ,layer-arg)
           ,@(loop for option in (remove :argument-precedence-order options :key #'car)
                   if (eq (car option) :method)
                   collect (multiple-value-bind
                               (layer-arg layer qualifiers args method-body)
                               (parse-method-body option (cdr option))
                             `(:method ,@qualifiers ((,layer-arg ,(prepare-layer layer)) ,@args)
                               ,@(prepare-layered-method-body name option layer-arg method-body)))
                   else collect option))
         (declaim (inline ,name))
         (defun ,name (&rest ,rest-arg)
           (declare #-openmcl (dynamic-extent ,rest-arg)
                    (optimize (speed 3) (debug 0) (safety 0)
                              (compilation-speed 0)))
           (apply #',definer (layer-context-prototype *active-context*) ,rest-arg))
         (eval-when (:compile-toplevel :load-toplevel :execute)
           (bind-layered-function-names ',name))
         #',definer))))

(defmacro define-layered-method (&whole form name &body body)
  (multiple-value-bind
      (layer-arg layer qualifiers args method-body)
      (parse-method-body form body)
    `(defmethod ,(get-layered-function-definer-name name)
                ,@qualifiers ((,layer-arg ,(prepare-layer layer)) ,@args)
       ,@(prepare-layered-method-body name form layer-arg method-body))))
