;;;;
;;;; AspectL
;;;;
;;;; Copyright (c) 2005 Pascal Costanza
;;;;
;;;; Permission is hereby granted, free of charge, to any person
;;;; obtaining a copy of this software and associated documentation
;;;; files (the "Software"), to deal in the Software without
;;;; restriction, including without limitation the rights to use,
;;;; copy, modify, merge, publish, distribute, sublicense, and/or
;;;; sell copies of the Software, and to permit persons to whom the
;;;; Software is furnished to do so, subject to the following
;;;; conditions:
;;;;
;;;; The above copyright notice and this permission notice shall be
;;;; included in all copies or substantial portions of the Software.
;;;;
;;;; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
;;;; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
;;;; OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
;;;; NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
;;;; HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
;;;; WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
;;;; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
;;;; OTHER DEALINGS IN THE SOFTWARE.
;;;;

(in-package #:al.dynascope)

(defparameter *dynascope-special-symbols-package*
  (find-package '#:aspectl.dynascope.special-symbols))

(defun make-special-symbol ()
  "creates a fresh unique special symbol"
  (gentemp "SPECIAL-SYMBOL-" *dynascope-special-symbols-package*))

(defun special-symbol-p (symbol)
  "checks whether a symbol is special, as created by make-special-symbol"
  (and (symbolp symbol)
       (eq (symbol-package symbol) *dynascope-special-symbols-package*)))

(defvar *symbol-access* nil
  "set/get a place's special symbol instead of its symbol value
   when this is set to a non-nil value")

(defmacro with-symbol-access (&body body)
  "executes body in an environment with *symbol-access* set to t"
  `(let ((*symbol-access* t))
     ,@body))

(defmacro without-symbol-access (&body body)
  "executes body in an environment with *symbol-access* set to nil"
  `(let ((*symbol-access* nil))
     ,@body))

(defun prepare-binding (binding env)
  "ensure that a binding form is 'well-formed' to ease further processing"
  (labels ((list-length-bound-p (list length)
	     (if (atom list) (<= 0 length)
	       (list-length-bound-p (cdr list) (1- length)))))
    (assert (list-length-bound-p binding 2) ()
      "Bad initialization form: ~S." binding)
    (when (symbolp binding)
      (setf binding (list binding nil)))
    (when (symbolp (car binding))
      (setf binding `(,(macroexpand (car binding) env) ,@(cdr binding))))
    binding))

(defmacro checked-progv (symbols values &body body)
  "like progv, only that symbols must all be special symbols"
  (with-unique-names (symbol-list)
    `(let (,symbol-list)
       (assert (progn (setq ,symbol-list ,symbols)
                 (every #'special-symbol-p ,symbol-list))
           () "Attempt at rebinding a non-special-place.")
       (progv ,symbol-list ,values ,@body))))

(defmacro dletf* (bindings &body body &environment env)
  "sequentially bind places to new values with dynamic scope,
   and execute body in that new dynamic environment"
  (reduce (lambda (binding body)
	    (setf binding (prepare-binding binding env))
	    (etypecase (car binding)
	      (symbol `(let (,binding)
			 (declare (special ,(car binding)))
			 ,body))
	      (cons   `(checked-progv
			   (list (with-symbol-access ,(car binding)))
			   (list ,(cadr binding))
			 ,body))))
	  bindings :from-end t :initial-value `(progn ,@body)))

(defmacro dletf (bindings &body body &environment env)
  "bind places to new values with dynamic scope in parallel,
   and execute body in that new dynamic environment"
  (loop for binding in bindings
        do (setf binding (prepare-binding binding env))
        collect (if (symbolp (car binding))
                    `',(car binding)
                  (car binding)) into symbol-forms
        when (symbolp (car binding)) collect (car binding) into variables
        collect (cadr binding) into value-forms
        finally (return `(checked-progv
                             (with-symbol-access
                              (list ,@symbol-forms))
                             (list ,@value-forms)
                           (locally (declare (special ,@variables))
                             ,@body)))))
