;;;;
;;;; AspectL
;;;;
;;;; Copyright (c) 2005 Pascal Costanza
;;;;
;;;; Permission is hereby granted, free of charge, to any person
;;;; obtaining a copy of this software and associated documentation
;;;; files (the "Software"), to deal in the Software without
;;;; restriction, including without limitation the rights to use,
;;;; copy, modify, merge, publish, distribute, sublicense, and/or
;;;; sell copies of the Software, and to permit persons to whom the
;;;; Software is furnished to do so, subject to the following
;;;; conditions:
;;;;
;;;; The above copyright notice and this permission notice shall be
;;;; included in all copies or substantial portions of the Software.
;;;;
;;;; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
;;;; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
;;;; OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
;;;; NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
;;;; HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
;;;; WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
;;;; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
;;;; OTHER DEALINGS IN THE SOFTWARE.
;;;;

(in-package #:al.clos-mop)

(defun the-class (class)
  "If class is a class, return it. If it is a symbol, find the class."
  (ctypecase class
    (class class)
    (symbol (find-class class))))

(defun the-direct-slot-definition (class slot)
  "If slot is a direct-slot-definition, return it.
   If it is a symbol, find it in (the-class class)."
  (ctypecase slot
    (direct-slot-definition slot)
    (symbol (find slot (class-direct-slots (the-class class))
                  :key #'slot-definition-name))))

(defun the-effective-slot-definition (class slot)
  "If slot is an effective-slot-definition, return it.
   If it is a symbol, find it in (the-class class)."
  (ctypecase slot
    (slot-definition slot)
    (symbol (find slot (class-slots (the-class class))
                  :key #'slot-definition-name))))

#+lispworks
(defgeneric find-method-combination (generic-function name options)
  (:method ((generic-function standard-generic-function) name options)
	   "LispWorks doesn't provide find-method-combination.
            Therefore, we use an internal function instead."
	   (declare (ignore options))
	   (clos::find-a-method-combination-type name)))

(defvar *preliminary-modified-classes* (make-hash-table :test #'equal)
  "Classes that have forward referenced classes as direct superclasses but no
   direct superclass that is a subtype of the required superclass are
   preliminary modified to have the required superclass as a direct superclass.
   This table maps those classes to the original list of direct superclasses
   so that the preliminary modification can be undone in case one of the
   forward referenced classes turns out as a subtype of the required
   superclass.")

(defun initialize-class-metaobject
       (call-next class metaclass required-superclass
                  &rest initargs
                  &key direct-superclasses
                  &allow-other-keys)
  "Can be used in an initialize-instace method specialized on a metaclass.
   Ensures that the required superclass will become part of the class's
   class precedence list, just below standard-object, by appending it to the
   list of direct superclasses if required. Correctly handles forward
   referenced classes. If used, reinitalize-instance must also be
   specialized and call reinitialize-class-metaobject accordingly. Should be
   used as follows, with appropriate replacements for 'metaclass' and
   'required-superclass':

   (defmethod initialize-instance :around
     ((class metaclass) &rest initargs)
     (declare (dynamic-extent initargs))
     (apply #'initialize-class-metaobject #'call-next-method
            class (find-class 'metaclass) (find-class 'required-superclass)
            initargs))"
  (declare (dynamic-extent initargs))
  (let ((direct-superclasses (remove required-superclass direct-superclasses))
        (class-key (list class metaclass)))
    (remhash class-key *preliminary-modified-classes*)
    (if (some (lambda (direct-superclass)
                (typep direct-superclass metaclass))
              direct-superclasses)
        (apply call-next class
               :direct-superclasses direct-superclasses
               initargs)
      (progn
	#-clisp
        (when (some (lambda (direct-superclass)
                      #-clisp (typep direct-superclass 'forward-referenced-class)
                      #+clisp (symbolp direct-superclass))
                    direct-superclasses)
          (setf (gethash class-key *preliminary-modified-classes*)
                direct-superclasses))
        (apply call-next class
               :direct-superclasses
               (append (remove (find-class 'standard-object)
                               direct-superclasses :test #'eq)
                       (list required-superclass))
               initargs)))))

(defun reinitialize-class-metaobject
       (call-next class metaclass required-superclass
                  &rest initargs
                  &key (direct-superclasses nil direct-superclasses-p)
                  &allow-other-keys)
  "Can be used in a reinitialize-instace method specialized on a metaclass.
   Ensures that the required superclass will become part of the class's
   class precedence list, just below standard-object, by appending it to the
   list of direct superclasses if required. Correctly handles forward
   referenced classes. If used, initalize-instance must also be
   specialized and call initialize-class-metaobject accordingly. Should be
   used as follows, with appropriate replacements for 'metaclass' and
   'required-superclass':

   (defmethod reinitialize-instance :around
     ((class metaclass) &rest initargs)
     (declare (dynamic-extent initargs))
     (apply #'reinitialize-class-metaobject #'call-next-method
            class (find-class 'metaclass) (find-class 'required-superclass)
            initargs))"
  (declare (dynamic-extent initargs)
           (ignore direct-superclasses))
  (prog1
      (if direct-superclasses-p
          (apply #'initialize-class-metaobject call-next
                 class metaclass required-superclass initargs)
        (let* ((class-key (list class metaclass))
               (org-direct-superclasses (gethash class-key *preliminary-modified-classes*)))
          (if org-direct-superclasses
              (apply #'initialize-class-metaobject call-next
                     class metaclass required-superclass
                     :direct-superclasses org-direct-superclasses initargs)
            (apply call-next class initargs))))
    (mapc #'reinitialize-instance (class-direct-subclasses class))))

(defvar *generic-function* nil
  "Some method objects must know about the generic functions
   to which they are about to be added. Usually, this can be
   achieved by specializing make-method-lambda, but many MOP
   implementations don't handle this correctly. Therefore,
   defmethod* is provided as an alternative to defmethod. It
   binds *generic-function* accordingly.")

(defmacro defmethod* (name &rest args)
  "Some method objects must know about the generic functions
   to which they are about to be added. Usually, this can be
   achieved by specializing make-method-lambda, but many MOP
   implementations don't handle this correctly. Therefore,
   defmethod* is provided as an alternative to defmethod. It
   binds *generic-function* accordingly."
  `(let ((*generic-function* (if (fboundp ',name)
                                 (fdefinition ',name)
                               (ensure-generic-function ',name))))
     (defmethod ,name ,@args)))
