<?php
/*
 +--------------------------------------------------------------------+
 | Copyright CiviCRM LLC. All rights reserved.                        |
 |                                                                    |
 | This work is published under the GNU AGPLv3 license with some      |
 | permitted exceptions and without any warranty. For full license    |
 | and copyright information, see https://civicrm.org/licensing       |
 +--------------------------------------------------------------------+
 */

/**
 *
 * @package CRM
 * @copyright CiviCRM LLC https://civicrm.org/licensing
 */

/**
 * This class provides the functionality to email a group of contacts.
 */
class CRM_Contribute_Form_Task_Status extends CRM_Contribute_Form_Task {

  /**
   * Are we operating in "single mode", i.e. updating the task of only
   * one specific contribution?
   *
   * @var bool
   */
  public $_single = FALSE;

  protected $_rows;

  /**
   * Build all the data structures needed to build the form.
   */
  public function preProcess() {
    $id = CRM_Utils_Request::retrieve('id', 'Positive',
      $this, FALSE
    );

    if ($id) {
      $this->_contributionIds = [$id];
      $this->_componentClause = " civicrm_contribution.id IN ( $id ) ";
      $this->_single = TRUE;
      $this->assign('totalSelectedContributions', 1);
    }
    else {
      parent::preProcess();
    }

    // check that all the contribution ids have pending status
    $query = "
SELECT count(*)
FROM   civicrm_contribution
WHERE  contribution_status_id != 2
AND    {$this->_componentClause}";
    $count = CRM_Core_DAO::singleValueQuery($query);
    if ($count != 0) {
      CRM_Core_Error::statusBounce(ts('Please select only online contributions with Pending status.'));
    }

    // we have all the contribution ids, so now we get the contact ids
    parent::setContactIDs();
    $this->assign('single', $this->_single);
  }

  /**
   * Build the form object.
   */
  public function buildQuickForm() {
    $status = CRM_Contribute_BAO_Contribution_Utils::getContributionStatuses(
      'contribution', $this->_contributionIds[0]
    );
    $byName = CRM_Contribute_PseudoConstant::contributionStatus(NULL, 'name');
    // FIXME: if it's invalid to transition from Pending to
    // In Progress or Overdue, we should move that logic to
    // CRM_Contribute_BAO_Contribution_Utils::getContributionStatuses.
    foreach (['Pending', 'In Progress', 'Overdue'] as $suppress) {
      unset($status[CRM_Utils_Array::key($suppress, $byName)]);
    }
    $this->add('select', 'contribution_status_id',
      ts('Contribution Status'),
      $status,
      TRUE
    );
    $this->add('checkbox', 'is_email_receipt', ts('Send e-mail receipt'));
    $this->setDefaults(['is_email_receipt' => 1]);

    $contribIDs = implode(',', $this->_contributionIds);
    $query = "
SELECT c.id            as contact_id,
       co.id           as contribution_id,
       c.display_name  as display_name,
       co.total_amount as amount,
       co.receive_date as receive_date,
       co.source       as source,
       co.payment_instrument_id as paid_by,
       co.check_number as check_no
FROM   civicrm_contact c,
       civicrm_contribution co
WHERE  co.contact_id = c.id
AND    co.id IN ( $contribIDs )";
    $dao = CRM_Core_DAO::executeQuery($query);

    // build a row for each contribution id
    $this->_rows = [];
    $attributes = CRM_Core_DAO::getAttribute('CRM_Contribute_DAO_Contribution');
    $defaults = [];
    $now = date("Y-m-d");
    $paidByOptions = ['' => ts('- select -')] + CRM_Contribute_PseudoConstant::paymentInstrument();

    while ($dao->fetch()) {
      $row['contact_id'] = $dao->contact_id;
      $row['contribution_id'] = $dao->contribution_id;
      $row['display_name'] = $dao->display_name;
      $row['amount'] = $dao->amount;
      $row['source'] = $dao->source;
      $row['trxn_id'] = &$this->addElement('text', "trxn_id_{$row['contribution_id']}", ts('Transaction ID'));
      $this->addRule("trxn_id_{$row['contribution_id']}",
        ts('This Transaction ID already exists in the database. Include the account number for checks.'),
        'objectExists',
        ['CRM_Contribute_DAO_Contribution', $dao->contribution_id, 'trxn_id']
      );

      $row['fee_amount'] = &$this->add('text', "fee_amount_{$row['contribution_id']}", ts('Fee Amount'),
        $attributes['fee_amount']
      );
      $this->addRule("fee_amount_{$row['contribution_id']}", ts('Please enter a valid amount.'), 'money');
      $defaults["fee_amount_{$row['contribution_id']}"] = 0.0;

      $row['trxn_date'] = $this->add('datepicker', "trxn_date_{$row['contribution_id']}", ts('Transaction Date'), [], FALSE, ['time' => FALSE]);
      $defaults["trxn_date_{$row['contribution_id']}"] = $now;

      $this->add("text", "check_number_{$row['contribution_id']}", ts('Check Number'));
      $defaults["check_number_{$row['contribution_id']}"] = $dao->check_no;

      $this->add("select", "payment_instrument_id_{$row['contribution_id']}", ts('Payment Method'), $paidByOptions);
      $defaults["payment_instrument_id_{$row['contribution_id']}"] = $dao->paid_by;

      $this->_rows[] = $row;
    }

    $this->assign_by_ref('rows', $this->_rows);
    $this->setDefaults($defaults);
    $this->addButtons([
      [
        'type' => 'next',
        'name' => ts('Update Pending Status'),
        'isDefault' => TRUE,
      ],
      [
        'type' => 'back',
        'name' => ts('Cancel'),
      ],
    ]);

    $this->addFormRule(['CRM_Contribute_Form_Task_Status', 'formRule']);
  }

  /**
   * Global validation rules for the form.
   *
   * @param array $fields
   *   Posted values of the form.
   *
   * @return array
   *   list of errors to be posted back to the form
   */
  public static function formRule($fields) {
    $seen = $errors = [];
    foreach ($fields as $name => $value) {
      if (strpos($name, 'trxn_id_') !== FALSE) {
        if ($fields[$name]) {
          if (array_key_exists($value, $seen)) {
            $errors[$name] = ts('Transaction ID\'s must be unique. Include the account number for checks.');
          }
          $seen[$value] = 1;
        }
      }

      if ((strpos($name, 'check_number_') !== FALSE) && $value) {
        $contribID = substr($name, 13);

        if ($fields["payment_instrument_id_{$contribID}"] != CRM_Core_PseudoConstant::getKey('CRM_Contribute_BAO_Contribution', 'payment_instrument_id', 'Check')) {
          $errors["payment_instrument_id_{$contribID}"] = ts("Payment Method should be Check when a check number is entered for a contribution.");
        }
      }
    }
    return empty($errors) ? TRUE : $errors;
  }

  /**
   * Process the form after the input has been submitted and validated.
   */
  public function postProcess() {
    $params = $this->controller->exportValues($this->_name);

    // submit the form with values.
    self::processForm($this, $params);

    CRM_Core_Session::setStatus(ts('Contribution status has been updated for selected record(s).'), ts('Status Updated'), 'success');
  }

  /**
   * Process the form with submitted params.
   *
   * Also supports unit test.
   *
   * @param CRM_Core_Form $form
   * @param array $params
   *
   * @throws \Exception
   */
  public static function processForm($form, $params) {
    $statusID = $params['contribution_status_id'] ?? NULL;
    $baseIPN = new CRM_Core_Payment_BaseIPN();

    // get the missing pieces for each contribution
    $contribIDs = implode(',', $form->_contributionIds);
    $details = self::getDetails($contribIDs);
    $template = CRM_Core_Smarty::singleton();

    // for each contribution id, we just call the baseIPN stuff
    foreach ($form->_rows as $row) {
      $input = $ids = $objects = [];
      $input['component'] = $details[$row['contribution_id']]['component'];

      $ids['contact'] = $row['contact_id'];
      $ids['contribution'] = $row['contribution_id'];
      $ids['contributionRecur'] = NULL;
      $ids['contributionPage'] = NULL;
      $ids['membership'] = $details[$row['contribution_id']]['membership'] ?? NULL;
      $ids['participant'] = $details[$row['contribution_id']]['participant'] ?? NULL;
      $ids['event'] = $details[$row['contribution_id']]['event'] ?? NULL;

      if (!$baseIPN->validateData($input, $ids, $objects, FALSE)) {
        CRM_Core_Error::statusBounce('Supplied data was not able to be validated');
      }

      $contribution = &$objects['contribution'];

      $contributionStatuses = CRM_Contribute_PseudoConstant::contributionStatus(NULL,
        'name'
      );

      if ($statusID == array_search('Cancelled', $contributionStatuses)) {
        $transaction = new CRM_Core_Transaction();
        $baseIPN->cancelled($objects, $transaction);
        $transaction->commit();
        continue;
      }
      elseif ($statusID == array_search('Failed', $contributionStatuses)) {
        $transaction = new CRM_Core_Transaction();
        $baseIPN->failed($objects, $transaction);
        $transaction->commit();
        continue;
      }

      // status is not pending
      if ($contribution->contribution_status_id != array_search('Pending',
          $contributionStatuses
        )
      ) {
        continue;
      }

      // set some fake input values so we can reuse IPN code
      $input['amount'] = $contribution->total_amount;
      $input['is_test'] = $contribution->is_test;
      $input['fee_amount'] = $params["fee_amount_{$row['contribution_id']}"];
      $input['check_number'] = $params["check_number_{$row['contribution_id']}"];
      $input['payment_instrument_id'] = $params["payment_instrument_id_{$row['contribution_id']}"];
      $input['net_amount'] = $contribution->total_amount - $input['fee_amount'];

      if (!empty($params["trxn_id_{$row['contribution_id']}"])) {
        $input['trxn_id'] = trim($params["trxn_id_{$row['contribution_id']}"]);
      }
      else {
        $input['trxn_id'] = $contribution->invoice_id;
      }
      $input['trxn_date'] = $params["trxn_date_{$row['contribution_id']}"] . ' ' . date('H:i:s');
      $input['is_email_receipt'] = !empty($params['is_email_receipt']);

      // @todo calling CRM_Contribute_BAO_Contribution::completeOrder like this is a pattern in it's last gasps. Call contribute.completetransaction api.
      CRM_Contribute_BAO_Contribution::completeOrder($input, $ids, $objects);

      // reset template values before processing next transactions
      $template->clearTemplateVars();
    }
  }

  /**
   * @param string $contributionIDs
   *
   * @return array
   */
  public static function &getDetails($contributionIDs) {
    if (empty($contributionIDs)) {
      return [];
    }
    $query = "
SELECT    c.id              as contribution_id,
          c.contact_id      as contact_id     ,
          mp.membership_id  as membership_id  ,
          pp.participant_id as participant_id ,
          p.event_id        as event_id
FROM      civicrm_contribution c
LEFT JOIN civicrm_membership_payment  mp ON mp.contribution_id = c.id
LEFT JOIN civicrm_participant_payment pp ON pp.contribution_id = c.id
LEFT JOIN civicrm_participant         p  ON pp.participant_id  = p.id
WHERE     c.id IN ( $contributionIDs )";

    $rows = [];
    $dao = CRM_Core_DAO::executeQuery($query);

    while ($dao->fetch()) {
      $rows[$dao->contribution_id]['component'] = $dao->participant_id ? 'event' : 'contribute';
      $rows[$dao->contribution_id]['contact'] = $dao->contact_id;
      if ($dao->membership_id) {
        if (!array_key_exists('membership', $rows[$dao->contribution_id])) {
          $rows[$dao->contribution_id]['membership'] = [];
        }
        $rows[$dao->contribution_id]['membership'][] = $dao->membership_id;
      }
      if ($dao->participant_id) {
        $rows[$dao->contribution_id]['participant'] = $dao->participant_id;
      }
      if ($dao->event_id) {
        $rows[$dao->contribution_id]['event'] = $dao->event_id;
      }
    }
    return $rows;
  }

}
