;;; citar.el --- Citation-related commands for org, latex, markdown -*- lexical-binding: t; -*-

;; Copyright (C) 2021-2022 Bruce D'Arcus

;; Author: Bruce D'Arcus <https://github.com/bdarcus>
;; Maintainer: Bruce D'Arcus <https://github.com/bdarcus>
;; Created: February 27, 2021
;; SPDX-License-Identifier: GPL-3.0-or-later
;; Version: 1.0
;; Homepage: https://github.com/emacs-citar/citar
;; Package-Requires: ((emacs "27.1") (parsebib "4.2") (org "9.5") (citeproc "0.9"))

;; This file is not part of GNU Emacs.
;;
;;; Commentary:

;;  A completing-read front-end to browse, filter and act on BibTeX, BibLaTeX,
;;  and CSL JSON bibliographic data, including LaTeX, markdown, and org-cite
;;  citation editing support.
;;
;;; Code:

(eval-when-compile
  (require 'cl-lib)
  (require 'subr-x))
(require 'seq)
(require 'map)
(require 'browse-url)
(require 'citar-cache)
(require 'citar-format)
(require 'citar-file)

;;; pre-1.0 API cleanup

;; make public
;; (make-obsolete 'citar--get-candidates 'citar-get-candidates "1.0")

;; Renamed in 1.0
(make-obsolete 'citar-has-file #'citar-has-files "1.0")
(make-obsolete 'citar-has-note #'citar-has-notes "1.0")
(make-obsolete 'citar-open-library-file #'citar-open-files "1.0")
(make-obsolete 'citar-attach-library-file #'citar-attach-files "1.0")
(make-obsolete 'citar-open-link #'citar-open-links "1.0")
(make-obsolete 'citar-get-link #'citar-get-links "1.0") ; now returns list
(make-obsolete 'citar-display-value 'citar-get-display-value "1.0")

;; make all these private
(make-obsolete 'citar-shorten-names 'citar--shorten-names "1.0")
(make-obsolete 'citar-get-template 'citar--get-template "1.0")
(make-obsolete 'citar-open-multi 'citar--open-multi "1.0")
(make-obsolete 'citar-select-group-related-resources
               'citar--select-group-related-resources "1.0")
(make-obsolete 'citar-select-resource 'citar--select-resource "1.0")

;; also rename
(make-obsolete 'citar-has-a-value 'citar-get-field-with-value "0.9.5") ; now returns cons pair
(make-obsolete 'citar-field-with-value 'citar-get-field-with-value "1.0") ; now returns cons pair
(make-obsolete 'citar--open-note 'citar-file--open-note "1.0")

;;(make-obsolete-variable 'citar-format-note-function "1.0")

;;; Declare variables and functions for byte compiler

(defvar embark-default-action-overrides)
(declare-function citar-org-format-note-default "citar-org")

;;; Variables

(defvar-local citar--entries nil
  "Override currently active citar entries.

When non-nil, should be a hash table mapping citation keys to
entries, as returned by `citar-get-entries'. Then all citar
functions will use that hash table as the source of bibliography
data instead of accessing the cache.

This variable should only be let-bound locally for the duration
of individual functions or operations. This is useful when using
multiple Citar functions in quick succession, to guarantee that
all potential cache accesses and updates are performed up-front.
In such cases, use a pattern like this:

  (let ((citar--entries (citar-get-entries)))
    ...)

Note that this variable is buffer-local, since Citar has a
different list of bibliographies (and hence entries) for each
buffer.")

;;;; Faces

(defgroup citar nil
  "Citations and bibliography management."
  :group 'editing)

(defface citar
  '((t :inherit font-lock-doc-face))
  "Default Face for `citar' candidates."
  :group 'citar)

(defface citar-highlight
  '((t :weight bold))
  "Face used to highlight content in `citar' candidates."
  :group 'citar)

(defface citar-selection
  '((t :inherit highlight :slant italic))
  "Face used for the currently selected candidates."
  :group 'citar)

;;;; Bibliography, file, and note paths

(defcustom citar-bibliography nil
  "A list of bibliography files."
  :group 'citar
  :type '(repeat file))

(defcustom citar-library-paths nil
  "A list of files paths for related PDFs, etc."
  :group 'citar
  :type '(repeat directory))

(defcustom citar-library-file-extensions nil
  "List of file extensions to filter for related files.

These are the extensions the `citar-file-open-function'
will open, via `citar-file-open'.

When nil, the function will not filter the list of files."
  :group 'citar
  :type '(repeat string))

(defcustom citar-notes-paths nil
  "A list of file paths for bibliographic notes."
  :group 'citar
  :type '(repeat directory))

(defcustom citar-crossref-variable "crossref"
  "The bibliography field to look for cross-referenced entries.

When non-nil, find associated files and notes not only in the
original entry, but also in entries specified in the field named
by this variable."
  :group 'citar
  :type '(choice (const "crossref")
                 (string :tag "Field name")
                 (const :tag "Ignore cross-references" nil)))

(defcustom citar-additional-fields nil
  "A list of fields to add to parsed data.

By default, citar filters parsed data based on the fields
specified in `citar-templates', `citar-file-variable'
`citar-crossref-variable', and `citar-link-fields'. This
specifies additional fields to include."
  :group 'citar
  :type '(repeat string))

;;;; Displaying completions and formatting

(defcustom citar-templates
  '((main . "${author editor:30}     ${date year issued:4}     ${title:48}")
    (suffix . "          ${=key= id:15}    ${=type=:12}    ${tags keywords keywords:*}")
    (preview . "${author editor} (${year issued date}) ${title}, \
${journal journaltitle publisher container-title collection-title}.\n")
    (note . "Notes on ${author editor}, ${title}"))
  "Configures formatting for the bibliographic entry.

The main and suffix templates are for candidate display, and note
for the title field for new notes."
  :group 'citar
  :type  '(alist :key-type symbol
                 :value-type string
                 :options (main suffix preview note)))

(defcustom citar-ellipsis nil
  "Ellipsis string to mark ending of truncated display fields.

If t, use the value of `truncate-string-ellipsis'.  If nil, no
ellipsis will be used.  Otherwise, this should be a non-empty
string specifying the ellipsis."
  :group 'citar
  :type '(choice (const :tag "Use `truncate-string-ellipsis'" t)
                 (const :tag "No ellipsis" nil)
                 (const "…")
                 (const "...")
                 (string :tag "Ellipsis string")))

(defcustom citar-format-reference-function
  #'citar-format-reference
  "Function used to render formatted references.

This function is called by `citar-insert-reference' and
`citar-copy-reference'. The default value,
`citar-format-reference', formats references using the `preview'
template set in `citar-template'. To use `citeproc-el' to format
references according to CSL styles, set the value to
`citar-citeproc-format-reference'. Alternatively, set to a custom
function that takes a list of (KEY . ENTRY) and returns formatted
references as a string."
  :group 'citar
  :type '(choice (function-item :tag "Use 'citar-template'" citar-format-reference)
                 (function-item :tag "Use 'citeproc-el'" citar-citeproc-format-reference)
                 (function :tag "Other")))

(defcustom citar-display-transform-functions
  ;; TODO change this name, as it might be confusing?
  '((("author" "editor") . citar--shorten-names))
  "Configure transformation of field display values from raw values.

All functions that match a particular field are run in order."
  :group 'citar
  :type '(alist :key-type   (choice (const t) (repeat string))
                :value-type function))

(defcustom citar-symbols
  `((file  .  ("F" . " "))
    (note .   ("N" . " "))
    (link .   ("L" . " ")))
  "Configuration alist specifying which symbol or icon to pick for a bib entry.
This leaves room for configurations where the absense of an item
may be indicated with the same icon but a different face.

To avoid alignment issues make sure that both the car and cdr of a symbol have
the same width."
  :group 'citar
  :type '(alist :key-type symbol
                :value-type (cons (string :tag "Present")
                                  (string :tag "Absent"))
                :options (file note link)))

(defcustom citar-symbol-separator " "
  "The padding between prefix symbols."
  :group 'citar
  :type 'string)

;;;; Citar actions and other miscellany

(defcustom citar-default-action #'citar-open
  "The default action for the `citar-at-point' command.
Should be a function that takes one argument, a list with each
entry being either a citation KEY or a (KEY . ENTRY) pair."
  :group 'citar
  :type 'function)

(defcustom citar-at-point-fallback 'prompt
  "Fallback action for `citar-at-point'.
The action is used when no citation key is found at point.
`prompt' means choosing entries via `citar-select-keys'
and nil means no action."
  :group 'citar
  :type '(radio (const :tag "Prompt" prompt)
                (const :tag "Ignore" nil)))

(defcustom citar-open-prompt (list #'citar-open #'citar-attach-files #'citar-open-note)
  "Always prompt to open files, notes, or links.

If nil, when chosen keys have a single resource, it will be
selected without prompting. When t, `citar-open',
`citar-open-files', `citar-attach-files', `citar-open-links',
`citar-open-notes', and `citar-open-note' will always prompt to
select a resource.

Otherwise, the value should be a list of command names that will
always prompt to select."
  :group 'citar
  :type '(choice (const :tag "Always prompt" t)
                 (const :tag "Prompt only for multiple resources" nil)
                 (set :tag "Commands that prompt for multiple resources"
                      (function-item citar-open)
                      (function-item citar-open-files)
                      (function-item citar-attach-files)
                      (function-item citar-open-links)
                      (function-item citar-open-notes)
                      (function-item citar-open-note))))

;;;; File, note, and URL handling

(defcustom citar-open-resources '(:files :links :notes :create-notes)
  "Types of resources that `citar-open' offers to open."
  :group 'citar
  :type '(set (const :tag "Library files" :files)
              (const :tag "Links" :links)
              (const :tag "Notes" :notes)
              (const :tag "Create notes" :create-notes)))

(defcustom citar-open-always-create-notes nil
  "Offer to create notes even for keys that already have notes.

If nil, `citar-open' and `citar-open-notes' will only offer to
create new notes for keys that have no existing notes. When t,
offer to create new notes for all chosen keys.

Otherwise, the value should be a list of command names that will
offer to create new notes unconditionally."
  :group 'citar
  :type '(choice (const :tag "Always offer to create notes" t)
                 (const :tag "Create notes only if none exist" nil)
                 (set :tag "Create notes for commands"
                      (function-item citar-open)
                      (function-item citar-open-notes))))

(defcustom citar-file-sources (list (list :items #'citar-file--get-from-file-field
                                          :hasitems #'citar-file--has-file-field)
                                    (list :items #'citar-file--get-library-files
                                          :hasitems #'citar-file--has-library-files))
  "List of backends used to get library files for bibliography references.

Should be a list of plists, where each plist has the following properties:

  :items Function that takes a list of citation keys and returns
    a hash table mapping each of those keys to a list of files.

  :hasitems Function that takes a citation key and returns
    non-nil if it has associated files."
  :group 'citar
  :type '(repeat (plist :value-type function :options (:items :hasitems))))

(defcustom citar-notes-sources
  `((citar-file .
                ,(list :name "Notes"
                       :category 'file
                       :items #'citar-file--get-notes
                       :hasitems #'citar-file--has-notes
                       :open #'find-file
                       :create #'citar-file--create-note
                       :transform #'file-name-nondirectory)))
  "The alist of notes backends available for configuration.

The format of the cons should be (NAME . PLIST), where the
plist has the following properties:

  :name the group display name

  :category the completion category

  :hasitems function to test for keys with notes

  :open function to open a given note candidate

  :items function to return candidate strings for keys

  :annotate annotation function (optional)

  :transform transformation function (optional)"
  :group 'citar
  :type '(alist :key-type symbol :value-type plist))

(defcustom citar-notes-source 'citar-file
  "The notes backend."
  :group 'citar
  :type 'symbol)

;; TODO should this be a major mode function?
(defcustom citar-note-format-function #'citar-org-format-note-default
  "Function used by `citar-file' note source to format new notes."
  :group 'citar
  :type 'function)

(defcustom citar-link-fields '((doi . "https://doi.org/%s")
                               (pmid . "https://www.ncbi.nlm.nih.gov/pubmed/%s")
                               (pmcid . "https://www.ncbi.nlm.nih.gov/pmc/articles/%s")
                               (url . "%s"))
  "Bibliography fields to parse into links.

Association list whose keys are symbols naming bibliography
fields and values are URL strings. In each URL, \"%s\" is
replaced by the contents of the corresponding field."
  :group 'citar
  :type '(alist :key-type symbol :value-type string))


;;;; Major mode functions

;; TODO Move this to `citar-org', since it's only used there?
;; Otherwise it seems to overlap with `citar-default-action'
(defcustom citar-at-point-function #'citar-dwim
  "The function to run for `citar-at-point'."
  :group 'citar
  :type 'function)

(defcustom citar-major-mode-functions
  '(((org-mode) .
     ((local-bib-files . citar-org-local-bib-files)
      (insert-citation . citar-org-insert-citation)
      (insert-edit . citar-org-insert-edit)
      (key-at-point . citar-org-key-at-point)
      (citation-at-point . citar-org-citation-at-point)
      (list-keys . citar-org-list-keys)))
    ((latex-mode) .
     ((local-bib-files . citar-latex-local-bib-files)
      (insert-citation . citar-latex-insert-citation)
      (insert-edit . citar-latex-insert-edit)
      (key-at-point . citar-latex-key-at-point)
      (citation-at-point . citar-latex-citation-at-point)
      (list-keys . citar-latex-list-keys)))
    ((markdown-mode) .
     ((insert-keys . citar-markdown-insert-keys)
      (insert-citation . citar-markdown-insert-citation)
      (insert-edit . citar-markdown-insert-edit)
      (key-at-point . citar-markdown-key-at-point)
      (citation-at-point . citar-markdown-citation-at-point)
      (list-keys . citar-markdown-list-keys)))
    (t .
       ((insert-keys . citar--insert-keys-comma-separated))))
  "The variable determining the major mode specific functionality.

It is alist with keys being a list of major modes.

The value is an alist with values being functions to be used for
these modes while the keys are symbols used to lookup them up.
The keys are:

local-bib-files: the corresponding functions should return the list of
local bibliography files.

insert-keys: the corresponding function should insert the list of keys given
to as the argument at point in the buffer.

insert-citation: the corresponding function should insert a
complete citation from a list of keys at point.  If the point is
in a citation, new keys should be added to the citation.

insert-edit: the corresponding function should accept an optional
prefix argument and interactively edit the citation or key at
point.

key-at-point: the corresponding function should return the
citation key at point or nil if there is none.  The return value
should be (KEY . BOUNDS), where KEY is a string and BOUNDS is a
pair of buffer positions indicating the start and end of the key.

citation-at-point: the corresponding function should return the
keys of the citation at point, or nil if there is none.  The
return value should be (KEYS . BOUNDS), where KEYS is a list of
strings and BOUNDS is pair of buffer positions indicating the
start and end of the citation.

list-keys: the corresponding function should return the keys
of all citations in the current buffer."
  :group 'citar
  :type 'alist)

;;;; History, including future history list.

(defvar citar-history nil
  "Search history for `citar'.")

(defcustom citar-presets nil
  "List of predefined searches."
  :group 'citar
  :type '(repeat string))

(defcustom citar-select-multiple t
  "Use `completing-read-multiple' for selecting citation keys.
When nil, all citar commands will use `completing-read'."
  :type 'boolean
  :group 'citar)

;;;; Keymaps

(defvar citar-map
  (let ((map (make-sparse-keymap)))
    (define-key map (kbd "c") #'citar-insert-citation)
    (define-key map (kbd "k") #'citar-insert-keys)
    (define-key map (kbd "r") #'citar-copy-reference)
    (define-key map (kbd "R") #'citar-insert-reference)
    (define-key map (kbd "b") #'citar-insert-bibtex)
    (define-key map (kbd "o") #'citar-open)
    (define-key map (kbd "e") #'citar-open-entry)
    (define-key map (kbd "l") #'citar-open-links)
    (define-key map (kbd "n") #'citar-open-notes)
    (define-key map (kbd "f") #'citar-open-files)
    (define-key map (kbd "RET") #'citar-run-default-action)
    map)
  "Keymap for Embark minibuffer actions.")

(defvar citar-citation-map
  (let ((map (make-sparse-keymap)))
    (define-key map (kbd "i") #'citar-insert-edit)
    (define-key map (kbd "o") #'citar-open)
    (define-key map (kbd "e") #'citar-open-entry)
    (define-key map (kbd "l") #'citar-open-links)
    (define-key map (kbd "n") #'citar-open-notes)
    (define-key map (kbd "f") #'citar-open-files)
    (define-key map (kbd "r") #'citar-copy-reference)
    (define-key map (kbd "RET") #'citar-run-default-action)
    map)
  "Keymap for Embark citation-key actions.")

;;; Bibliography cache

(defun citar--bibliography-files (&rest buffers)
  "Bibliography file names for BUFFERS.
The elements of BUFFERS are either buffers or the symbol global.
Returns the absolute file names of the bibliographies in all
these contexts.

When BUFFERS is nil, return local bibliographies for the current
buffer and global bibliographies."
  (citar-file--normalize-paths
   (mapcan (lambda (buffer)
             (if (eq buffer 'global)
                 (if (listp citar-bibliography) citar-bibliography
                   (list citar-bibliography))
               (with-current-buffer buffer
                 (citar--major-mode-function 'local-bib-files #'ignore))))
           (or buffers (list (current-buffer) 'global)))))

(defun citar--bibliographies (&rest buffers)
  "Return bibliographies for BUFFERS."
  (delete-dups
   (mapcan
    (lambda (buffer)
      (citar-cache--get-bibliographies (citar--bibliography-files buffer) buffer))
    (or buffers (list (current-buffer) 'global)))))

;;; Completion functions

(defun citar--completion-table (candidates &optional filter &rest metadata)
  "Return a completion table for CANDIDATES.

CANDIDATES is a hash with references CAND as key and CITEKEY as value,
  where CAND is a display string for the bibliography item.

FILTER, if non-nil, should be a predicate function taking
  argument KEY. Only candidates for which this function returns
  non-nil will be offered for completion.

By default the metadata of the table contains the category and
affixation function. METADATA are extra entries for metadata of
the form (KEY . VAL).

The returned completion table can be used with `completing-read'
and other completion functions."
  (let ((metadata `(metadata . ((category . citar-candidate)
                                . ((affixation-function . ,#'citar--ref-affix)
                                   . ,metadata)))))
    (lambda (string predicate action)
      (if (eq action 'metadata)
          metadata
        ;; REVIEW this now works, but probably needs refinement
        (let ((predicate
               (when (or filter predicate)
                 (lambda (_ key)
                   (and (or (null filter) (funcall filter key))
                        (or (null predicate) (funcall predicate string)))))))
          (complete-with-action action candidates string predicate))))))

(cl-defun citar-select-refs (&key (multiple t) filter)
  "Select bibliographic references.

A wrapper around `completing-read' that returns (KEY . ENTRY),
where ENTRY is a field-value alist.  Therefore `car' of the
return value is the cite key, and `cdr' is an alist of structured
data.

Takes the following optional keyword arguments:

MULTIPLE: if t, calls `completing-read-multiple' and returns an
  alist of (KEY . ENTRY) pairs.

FILTER: if non-nil, should be a predicate function taking
  arguments KEY and ENTRY.  Only candidates for which this
  function returns non-nil will be offered for completion.  For
  example:

  (citar-select-ref :filter (citar-has-note))

  (citar-select-ref :filter (citar-has-file))"
  (let* ((candidates (or (citar--format-candidates)
                         (user-error "No bibliography set")))
         (chosen (if (and multiple citar-select-multiple)
                     (citar--select-multiple "References: " candidates
                                             filter 'citar-history citar-presets)
                   (completing-read "Reference: " (citar--completion-table candidates filter)
                                    nil nil nil 'citar-history citar-presets nil))))
    ;; If CAND is not in CANDIDATES, treat it as a citekey (e.g. inserted into the minibuffer by `embark-act')
    (cl-flet ((candkey (cand) (or (gethash cand candidates) cand)))
      ;; Return a list of keys regardless of 1 or many
      (if (listp chosen)
          (mapcar #'candkey chosen)
        (list (candkey chosen))))))

(cl-defun citar-select-ref (&key filter)
  "Select bibliographic references.

Call `citar-select-ref' with optional FILTER; see its
documentation for the return value."
  (car (citar-select-refs :multiple nil :filter filter)))

(defun citar--multiple-completion-table (selected-hash candidates filter)
  "Return a completion table for multiple selection.
SELECTED-HASH is the hash-table containing selected candidates.
CANDIDATES is the list of completion candidates, FILTER is the function
to filter them."
  (citar--completion-table
   candidates filter
   `(group-function . (lambda (cand transform)
                        (pcase (list (not (not transform))
                                     (gethash (substring-no-properties cand) ,selected-hash))
                          ('(nil nil) "Select Multiple")
                          ('(nil t)   "Selected")
                          ('(t nil) cand)
                          ('(t t)
                           (add-face-text-property 0 (length cand) 'citar-selection nil (copy-sequence cand))
                           cand))))))

(defvar citar--multiple-setup '("TAB" . "RET")
  "Variable whose value should be a cons (SEL . EXIT)
SEL is the key which should be used for selection. EXIT is the key which
is used for exiting the minibuffer during completing read.")

(defun citar--multiple-exit ()
  "Exit with the currently selected candidates."
  (interactive)
  (setq unread-command-events (listify-key-sequence (kbd (car citar--multiple-setup)))))

(defun citar--setup-multiple-keymap ()
  "Make a keymap suitable for `citar--select-multiple'."
  (let ((keymap (make-composed-keymap nil (current-local-map)))
        (kbdselect (kbd (car citar--multiple-setup)))
        (kbdexit (kbd (cdr citar--multiple-setup))))
    (define-key keymap kbdselect (lookup-key keymap kbdexit))
    (define-key keymap kbdexit #'citar--multiple-exit)
    (use-local-map keymap)))

(defun citar--select-multiple (prompt candidates &optional filter history def)
  "Select multiple CANDIDATES with PROMPT.
HISTORY is the `completing-read' history argument."
  ;; Because completing-read-multiple just does not work for long candidate
  ;; strings, and IMO is a poor UI.
  (let* ((selected-hash (make-hash-table :test 'equal)))
    (while (let ((initial-history (symbol-value history))
                 (item (minibuffer-with-setup-hook #'citar--setup-multiple-keymap
                         (completing-read
                          (format "%s (%s/%s): " prompt
                                  (hash-table-count selected-hash)
                                  (hash-table-count candidates))
                          (citar--multiple-completion-table selected-hash candidates filter)
                          nil t nil history `("" . ,def)))))
             (unless (string-empty-p item)
               (if (not (gethash item selected-hash))
                   (puthash item t selected-hash)
                 (remhash item selected-hash)
                 (set history initial-history)))
             (not (or (eq last-command #'citar--multiple-exit)
                      (string-empty-p item)))))
    (hash-table-keys selected-hash)))

(cl-defun citar--get-resource-candidates (keys &key files links notes create-notes)
  "Return related resource candidates for KEYS.

Return a list (CATEGORY . CANDIDATES), where CATEGORY is a
completion category and CANDIDATES is a list of resources
associated with KEYS. Return nil if there are no associated
resources.

The resources include:
 * FILES: a list of files or t to use `citar-get-files'.
 * LINKS: a list of links or t to use `citar-get-links'.
 * NOTES: a list of notes or t to use `citar-get-notes'.
 * CREATE-NOTES: a list of cite keys for which to create notes,
     or t to use KEYS. See `citar-open-always-create-notes'.

If any of FILES, LINKS, NOTES, or CREATE-NOTES is nil, that
resource type is omitted from CANDIDATES.

CATEGORY is one of:
 * `file' when returning only files
 * `url' when returning only links
 * the `:category' property of `citar-notes-source' if returning
   only notes
 * `citar-reference' when returning notes to create.
 * `multi-category' when CANDIDATES has resources of multiple
   types. The `multi-category' text property is applied to each
   element of CANDIDATES."
  (cl-flet ((getresources (table) (when table
                                    (delete-dups (apply #'append (hash-table-values table)))))
            (keycands (type keys) (let ((format (citar-format--parse (citar--get-template 'completion)))
                                        (width (- (frame-width) 2)))
                                    (mapcar (lambda (key)
                                              (let* ((entry (citar-get-entry key))
                                                     (cand (citar-format--entry format entry width
                                                                                :ellipsis citar-ellipsis))
                                                     (keycand (citar--prepend-candidate-citekey key cand))
                                                     (target (cons 'citar-reference
                                                                   (propertize key 'citar--resource type))))
                                                (propertize keycand 'multi-category target)))
                                            keys)))
            (withtype (type cat cands) (when cands
                                         (cons cat (mapcar (lambda (cand)
                                                             (propertize cand 'citar--resource type))
                                                           cands)))))
    (let* ((citar--entries (citar-get-entries))
           (files (if (listp files) files (getresources (citar-get-files keys))))
           (links (if (listp links) links (getresources (citar-get-links keys))))
           (keynotes (unless (and (listp notes) (listp create-notes)) (citar-get-notes keys)))
           (notes (if (listp notes) notes (getresources keynotes)))
           (create-notes (keycands 'create-note
                                   (cond ((listp create-notes) create-notes)
                                         ((or (eq t citar-open-always-create-notes)
                                              (memq this-command citar-open-always-create-notes)
                                              (not keynotes))
                                          keys)
                                         (t (seq-remove (lambda (key) (gethash key keynotes)) keys)))))
           (notecat (citar--get-notes-config :category))
           (sources (delq nil (list (withtype 'file 'file files)
                                    (withtype 'url 'url links)
                                    (withtype 'note notecat notes)
                                    (withtype 'create-note 'citar-candidate create-notes)))))
      (if (null (cdr sources))          ; if sources is nil or singleton list,
          (car sources)                 ; return either nil or the only source.
        (cons 'multi-category           ; otherwise, combine all sources
              (mapcan
               (pcase-lambda (`(,cat . ,cands))
                 (if (not cat)
                     cands
                   (mapcar (lambda (cand)
                             (if (get-text-property 0 'multi-category cand)
                                 cand
                               (propertize cand 'multi-category (cons cat cand))))
                           cands)))
               sources))))))

(defun citar--annotate-note (candidate)
  "Annotate note CANDIDATE."
  (when-let (((eq 'note (get-text-property 0 'citar--resource candidate)))
             (annotate (citar--get-notes-config :annotate)))
    (funcall annotate (substring-no-properties candidate))))

(cl-defun citar--select-resource (keys &key files links notes create-notes)
  "Select related FILES, NOTES, or LINKS resource for KEYS.

Return (TYPE . RESOURCE), where TYPE is `file', `link', `note',
or `create-note' and RESOURCE is the selected resource string.
Return nil if there are no resources.

Use `completing-read' to prompt for a resource, unless there is
only one resource and `citar-open-prompt' is t or contains
`this-command'. Return nil if the user declined to choose."
  (when-let ((resources (citar--get-resource-candidates keys :files files :links links
                                                        :notes notes :create-notes create-notes)))
    (pcase-let ((`(,category . ,cands) resources))
      (when-let ((selected
                  (if (not (or (cdr cands) (eq t citar-open-prompt) (memq this-command citar-open-prompt)))
                      (car cands)
                    (let* ((metadata `(metadata
                                       (group-function . ,#'citar--select-group-related-resources)
                                       (annotation-function . ,#'citar--annotate-note)
                                       ,@(when category `((category . ,category)))))
                           (table (lambda (string predicate action)
                                    (if (eq action 'metadata)
                                        metadata
                                      (complete-with-action action cands string predicate))))
                           (selected (completing-read "Select resource: " table nil t)))
                      (car (member selected cands))))))
        (pcase (get-text-property 0 'citar--resource selected)
          ('create-note (cons 'create-note (citar--extract-candidate-citekey selected)))
          (type (cons type (substring-no-properties selected))))))))

(defun citar--select-group-related-resources (resource transform)
  "Group RESOURCE by type or TRANSFORM."
  (pcase (get-text-property 0 'citar--resource resource)
    ('file (if transform (file-name-nondirectory resource) "Library Files"))
    ('url (if transform resource "Links"))
    ('note
     (if transform
         (funcall (or (citar--get-notes-config :transform) #'identity) resource)
       (or (citar--get-notes-config :name) "Notes")))
    ('create-note
     (if transform
         resource
       (format "Create %s" (or (citar--get-notes-config :name) "Notes"))))
    (_ (if transform
           resource
         nil))))

(defun citar--format-candidates ()
  "Format completion candidates for bibliography entries.

Return a hash table with the keys being completion candidate
strings and values being citation keys.

Return nil if `citar-bibliographies' returns nil."
  ;; Populate bibliography cache.
  (when-let ((bibs (citar--bibliographies)))
    (let* ((citar--entries (citar-cache--entries bibs))
           (preformatted (citar-cache--preformatted bibs))
           (hasfilesp (citar-has-files))
           (hasnotesp (citar-has-notes))
           (haslinksp (citar-has-links))
           (hasfilestag (propertize " has:files" 'invisible t))
           (hasnotestag (propertize " has:notes" 'invisible t))
           (haslinkstag (propertize " has:links" 'invisible t))
           (symbolswidth (string-width (citar--symbols-string t t t)))
           (width (- (frame-width) symbolswidth 2))
           (completions (make-hash-table :test 'equal :size (hash-table-count citar--entries))))
      (prog1 completions
        (maphash
         (lambda (citekey _entry)
           (let* ((hasfiles (and hasfilesp (funcall hasfilesp citekey)))
                  (hasnotes (and hasnotesp (funcall hasnotesp citekey)))
                  (haslinks (and haslinksp (funcall haslinksp citekey)))
                  (preform (or (gethash citekey preformatted)
                               (error "No preformatted candidate string: %s" citekey)))
                  (display (citar-format--star-widths
                            (- width (car preform)) (cdr preform)
                            t citar-ellipsis))
                  (tagged (if (not (or hasfiles hasnotes haslinks))
                              display
                            (concat display
                                    (when hasfiles hasfilestag)
                                    (when hasnotes hasnotestag)
                                    (when haslinks haslinkstag)))))
             (puthash tagged citekey completions)))
         citar--entries)))))

(defun citar--prepend-candidate-citekey (citekey candidate)
  "Prepend invisible CITEKEY to CANDIDATE string.
CITEKEY is quoted if necessary and can be extracted using
`citar--extract-candidate-citekey'."
  (let* ((keyquoted (if (or (string-empty-p citekey)          ; quote citekey if it's empty,
                            (= ?\" (aref citekey 0))          ; or starts with ",
                            (seq-contains-p citekey ?\s #'=)) ; or has a space.
                        (prin1-to-string citekey)
                      citekey))
         (prefix (propertize (concat keyquoted
                                     (when (and candidate (not (string-empty-p candidate))) " "))
                             'invisible t)))
    (concat prefix candidate)))

(defun citar--extract-candidate-citekey (candidate)
  "Extract the citation key from string CANDIDATE."
  (unless (string-empty-p candidate)
    (if (= ?\" (aref candidate 0))
        (read candidate)
      (substring-no-properties candidate 0 (seq-position candidate ?\s #'=)))))

(defun citar--key-at-point ()
  "Return bibliography key at point in current buffer, along with its bounds.
Return (KEY . BOUNDS), where KEY is a string and BOUNDS is either
nil or a (BEG . END) pair indicating the location of KEY in the
buffer. Return nil if there is no key at point or the current
major mode is not supported."
  (citar--major-mode-function 'key-at-point #'ignore))

(defun citar--citation-at-point ()
  "Return citation at point in current buffer, along with its bounds.
Return (KEYS . BOUNDS), where KEYS is a list of citation keys and
BOUNDS is either nil or a (BEG . END) pair indicating the
location of the citation in the buffer. Return nil if there is no
citation at point or the current major mode is not supported."
  (citar--major-mode-function 'citation-at-point #'ignore))

(defun citar-key-at-point ()
  "Return the citation key at point in the current buffer.
Return nil if there is no key at point or the major mode is not
supported."
  (car (citar--key-at-point)))

(defun citar-citation-at-point ()
  "Return a list of keys comprising the citation at point in the current buffer.
Return nil if there is no citation at point or the major mode is
not supported."
  (car (citar--citation-at-point)))

;;; Major-mode functions

(defun citar--get-major-mode-function (key &optional default)
  "Return function associated with KEY in `major-mode-functions'.
If no function is found matching KEY for the current major mode,
return DEFAULT."
  (alist-get
   key
   (cdr (seq-find
         (pcase-lambda (`(,modes . ,_functions))
           (or (eq t modes)
               (apply #'derived-mode-p (if (listp modes) modes (list modes)))))
         citar-major-mode-functions))
   default))

(defun citar--major-mode-function (key default &rest args)
  "Function for the major mode corresponding to KEY applied to ARGS.
If no function is found, the DEFAULT function is called."
  (apply (citar--get-major-mode-function key default) args))

;;; Data access functions

(defun citar-get-entry (key)
  "Return entry for reference KEY, as an association list.
Note: this function accesses the bibliography cache and should
not be used for retreiving a large number of entries. Instead,
prefer `citar--get-entries'."
  (if citar--entries
      (gethash key citar--entries)
    (citar-cache--entry key (citar--bibliographies))))

(defun citar-get-entries ()
  "Return all entries for currently active bibliographies.
Return a hash table whose keys are citation keys and values are
the corresponding entries."
  (or citar--entries (citar-cache--entries (citar--bibliographies))))

(defun citar-get-value (field key-or-entry)
  "Return value of FIELD in reference KEY-OR-ENTRY.
KEY-OR-ENTRY should be either a string key, or an entry alist as
returned by `citar-get-entry'. Return nil if the FIELD is not
present in KEY-OR-ENTRY."
  (let ((entry (if (stringp key-or-entry)
                   (citar-get-entry key-or-entry)
                 key-or-entry)))
    (cdr (assoc-string field entry 'case-fold))))

(defun citar-get-field-with-value (fields key-or-entry)
  "Find the first field among FIELDS that has a value in KEY-OR-ENTRY.
Return (FIELD . VALUE), where FIELD is the element of FIELDS that
was found to have a value, and VALUE is its value."
  (let ((entry (if (stringp key-or-entry)
                   (citar-get-entry key-or-entry)
                 key-or-entry)))
    (seq-some (lambda (field)
                (when-let ((value (citar-get-value field entry)))
                  (cons field value)))
              fields)))

(defun citar-get-display-value (fields key-or-entry)
  "Return the first non nil value for KEY-OR-ENTRY among FIELDS .

The value is transformed using `citar-display-transform-functions'"
  (let ((fieldvalue (citar-get-field-with-value fields key-or-entry)))
    (seq-reduce (lambda (string fun)
                  (if (or (eq t (car fun))
                          (seq-contains-p (car fun) (car fieldvalue) #'string=))
                      (funcall (cdr fun) string)
                    string))
                citar-display-transform-functions
                ;; Make sure we always return a string, even if empty.
                (or (cdr fieldvalue) ""))))

;;;; File, notes, and links

(defun citar--get-notes-config (property)
  "Return PROPERTY value for configured notes backend."
  (plist-get
   (alist-get citar-notes-source citar-notes-sources) property))

(defun citar-register-notes-source (name config)
  "Register note backend.

NAME is a symbol, and CONFIG is a plist."
  (citar--check-notes-source name config)
  (setf (alist-get name citar-notes-sources) config))

(defun citar-remove-notes-source (name)
  "Remove note backend NAME."
  (cl-callf2 assq-delete-all name citar-notes-sources))

(cl-defun citar-get-notes (&optional (key-or-keys nil filter-p))
  "Return notes associated with KEY-OR-KEYS.

KEY-OR-KEYS should be either a list KEYS or a single key. Return
a hash table mapping elements of KEYS to lists of associated
notes found using `citar-notes-source'. Include notes associated
with cross-referenced keys.

If KEY-OR-KEYS is omitted, return notes for all entries. If it is
nil, return nil."
  (when (or key-or-keys (not filter-p))
    (citar--get-resources key-or-keys
                          (citar--get-notes-config :items))))

(defun citar-create-note (key &optional entry)
  "Create a note for KEY and ENTRY.
If ENTRY is nil, use `citar-get-entry' with KEY."
  (interactive (list (citar-select-ref)))
  (funcall (citar--get-notes-config :create) key (or entry (citar-get-entry key))))

(cl-defun citar-get-files (&optional (key-or-keys nil filter-p))
  "Return files associated with KEY-OR-KEYS.

KEY-OR-KEYS should be either a list KEYS or a single key. Return
a hash table mapping elements of KEYS to lists of associated
files found using `citar-file-sources'. Include files associated
with cross-referenced keys.

If KEY-OR-KEYS is omitted, return files for all entries. If it is
nil, return nil."
  (when (or key-or-keys (not filter-p))
    (citar--get-resources key-or-keys
                          (mapcar (lambda (source)
                                    (plist-get source :items))
                                  citar-file-sources))))

(cl-defun citar-get-links (&optional (key-or-keys nil filter-p))
  "Return links associated with KEY-OR-KEYS.

KEY-OR-KEYS should be either a list KEYS or a single key. Return
a hash table mapping elements of KEYS to lists of associated
links found using `citar-link-fields'. Include links associated
with cross-referenced keys.

If KEY-OR-KEYS is omitted, return notes for all entries. If it is
nil, return nil."
  (when (or key-or-keys (not filter-p))
    (citar--get-resources key-or-keys
                          (apply-partially
                           #'citar--get-resources-using-function
                           (lambda (_citekey entry)
                             (let (keylinks)
                               (when entry
                                 (pcase-dolist (`(,fieldname . ,urlformat) citar-link-fields)
                                   (when-let ((fieldvalue (citar-get-value fieldname entry)))
                                     (push (format urlformat fieldvalue) keylinks))))
                               (nreverse keylinks)))))))

(defun citar-has-files ()
  "Return predicate testing whether entry has associated files.

Return a function that takes KEY and returns non-nil when the
corresponding bibliography entry has associated files. The
returned predicated may by nil if no entries have associated
files.

For example, to test whether KEY has associated files:

  (when-let ((hasfilesp (citar-has-files)))
    (funcall hasfilesp KEY))

When testing many keys, call this function once and use the
returned predicate repeatedly.

Files are detected using `citar-file-sources', which see. Also
check any bibliography entries that are cross-referenced from the
given KEY; see `citar-crossref-variable'."
  (citar--has-resources
   (mapcar (lambda (source)
             (when-let ((hasitems (plist-get source :hasitems)))
               (funcall hasitems)))
           citar-file-sources)))


(defun citar-has-notes ()
  "Return predicate testing whether entry has associated notes.

Return a function that takes KEY and returns non-nil when the
corresponding bibliography entry has associated notes. The
returned predicate may be nil if no entries have associated
notes.

For example, to test whether KEY has associated notes:

  (let ((hasnotesp (citar-has-notes)))
    (funcall hasnotesp KEY))

When testing many keys, call this function once and use the
returned predicate repeatedly.

Notes are detected using `citar-has-notes-functions', which see.
Also check any bibliography entries that are cross-referenced
from the given KEY; see `citar-crossref-variable'."
  (citar--has-resources
   (funcall (citar--get-notes-config :hasitems))))


(defun citar-has-links ()
  "Return predicate testing whether entry has links.

Return a function that takes KEY and returns non-nil when the
corresponding bibliography entry has associated links. See the
documentation of `citar-has-files' and `citar-has-notes', which
have similar usage."
  (let ((linkfields (mapcar (lambda (field) (symbol-name (car field))) citar-link-fields)))
    (citar--has-resources
     (apply-partially #'citar-get-field-with-value linkfields))))


(defun citar--has-resources (predicates)
  "Combine PREDICATES into a single resource predicate.

PREDICATES should be a list of functions that take a bibliography
KEY and return non-nil if the item has a resource. It may also be
a single such function.

Return a predicate that returns non-nil for a given KEY when any
of the elements of PREDICATES return non-nil for that KEY. If
PREDICATES is empty or all its elements are nil, then the
returned predicate is nil.

When `citar-crossref-variable' is the name of a crossref field,
the returned predicate also tests if an entry cross-references
another entry in ENTRIES that has associated resources."
  (when-let ((hasresourcep (if (functionp predicates)
                               predicates
                             (let ((predicates (remq nil predicates)))
                               (if (null (cdr predicates))
                                   ;; optimization for single predicate; just use it directly
                                   (car predicates)
                                 ;; otherwise, call all predicates until one returns non-nil
                                 (lambda (citekey)
                                   (seq-some (lambda (predicate)
                                               (funcall predicate citekey))
                                             predicates)))))))
    (if-let ((xref citar-crossref-variable))
        (lambda (citekey)
          (or (funcall hasresourcep citekey)
              (when-let ((xkey (citar-get-value xref citekey)))
                (funcall hasresourcep xkey))))
      hasresourcep)))

(defun citar--get-resources (key-or-keys functions)
  "Return hash table mapping each element of KEY-OR-KEYS to a list of resources.

KEY-OR-KEYS should be either a list KEYS or a single key.
FUNCTIONS should be a list of functions, each of which takes a
list of bibliography keys and returns a hash table mapping each
of those keys to a list of resources. FUNCTIONS may also be a
single such function.

Return a hash table mapping each element of KEYS to the
concatenated list of resources returned by all the FUNCTIONS.
Also include resources associated with cross-references from
KEYS."
  (let* ((citar--entries (citar-get-entries))
         (keys (if (listp key-or-keys) (delete-dups key-or-keys) (list key-or-keys)))
         (functions (if (functionp functions) (list functions) (remq nil functions)))
         (xref citar-crossref-variable)
         (getxref (apply-partially #'citar-get-value xref))
         (xkeys (if (not xref)
                    keys
                  (delete-dups (append keys (delq nil (mapcar getxref keys))))))
         (resources (delq nil (mapcar (lambda (func) (funcall func xkeys)) functions))))
    (cl-flet* ((getreslists (citekey) (delq nil (mapcar (apply-partially #'gethash citekey) resources)))
               (xresources (citekey entry) (apply #'append
                                                  (nconc (getreslists citekey)
                                                         (when-let ((xkey (and xref
                                                                               (citar-get-value xref entry))))
                                                           (getreslists xkey))))))
      (citar--get-resources-using-function #'xresources keys))))

(defun citar--get-resources-using-function (func &optional keys)
  "Collect resources for KEYS returned by FUNC.

Return a hash table mapping each element of KEYS to the result of
calling FUNC on that key and corresponding bibliography entry. If
KEYS is nil, call FUNC on every key and entry returned by
`citar-get-entries'.

Note: This is a helper function to make it easier to write
getters for file, note, and link resources."
  (let ((resources (make-hash-table :test 'equal)))
    (prog1 resources
      (cl-flet ((putresult (citekey entry) (when-let ((result (funcall func citekey entry)))
                                             (puthash citekey result resources))))
        (if (null keys)
            (maphash #'putresult (citar-get-entries))
          (dolist (citekey keys)
            (putresult citekey (citar-get-entry citekey))))))))

;;; Format and display field values

(defun citar--shorten-names (names)
  "Return a list of family names from a list of full NAMES.

To better accommodate corporate names, this will only shorten
personal names of the form \"family, given\"."
  (when (stringp names)
    (mapconcat
     (lambda (name)
       (if (eq 1 (length name))
           (cdr (split-string name " "))
         (car (split-string name ", "))))
     (split-string names " and ") ", ")))

(defun citar--fields-for-format (template)
  "Return list of fields for TEMPLATE."
  (mapcan (lambda (fieldspec) (when (consp fieldspec) (cdr fieldspec)))
          (citar-format--parse template)))

(defun citar--fields-in-formats ()
  "Find the fields to mentioned in the templates."
  (seq-mapcat #'citar--fields-for-format
              (list (citar--get-template 'main)
                    (citar--get-template 'suffix)
                    (citar--get-template 'preview)
                    (citar--get-template 'note))))

(defun citar--fields-to-parse ()
  "Determine the fields to parse from the template."
  (delete-dups `(,@(citar--fields-in-formats)
                 ,@(when citar-file-variable
                     (list citar-file-variable))
                 ,@(when citar-crossref-variable
                     (list citar-crossref-variable))
                 ,@(mapcar (lambda (field) (symbol-name (car field))) citar-link-fields)
                 . ,citar-additional-fields)))

;;; Affixations and annotations

(defun citar--ref-affix (cands)
  "Add affixation prefix to CANDS."
  (seq-map
   (lambda (candidate)
     (let ((symbols (citar--ref-make-symbols candidate)))
       (list candidate symbols "")))
   cands))

(defun citar--ref-make-symbols (cand)
  "Make CAND annotation or affixation string for has-symbols."
  (let ((candidate-symbols (citar--symbols-string
                            (string-match-p "has:files" cand)
                            (string-match-p "has:notes" cand)
                            (string-match-p "has:links" cand))))
    candidate-symbols))

(defun citar--ref-annotate (cand)
  "Add annotation to CAND."
  ;; REVIEW/TODO we don't currently use this, but could, for Emacs 27.
  (citar--ref-make-symbols cand))

(defun citar--symbols-string (has-files has-note has-link)
  "String for display from booleans HAS-FILES HAS-LINK HAS-NOTE."
  (cl-flet ((thing-string (has-thing thing-symbol)
                          (if has-thing
                              (cadr (assoc thing-symbol citar-symbols))
                            (cddr (assoc thing-symbol citar-symbols)))))
    (seq-reduce (lambda (constructed newpart)
                  (let* ((str (concat constructed newpart
                                      citar-symbol-separator))
                         (pos (length str)))
                    (put-text-property (- pos 1) pos 'display
                                       (cons 'space (list :align-to (string-width str)))
                                       str)
                    str))
                (list (thing-string has-files 'file)
                      (thing-string has-note 'note)
                      (thing-string has-link 'link)
                      "")
                "")))

(defun citar--get-template (template-name)
  "Return template string for TEMPLATE-NAME."
  (or
   (cdr (assq template-name citar-templates))
   (when (eq template-name 'completion)
     (concat (propertize (citar--get-template 'main) 'face 'citar-highlight)
             (propertize (citar--get-template 'suffix) 'face 'citar)))
   (error "No template for \"%s\" - check variable 'citar-templates'" template-name)))

;;;###autoload
(defun citar-insert-preset ()
  "Prompt for and insert a predefined search."
  (interactive)
  (unless (minibufferp)
    (user-error "Command can only be used in minibuffer"))
  (when-let ((enable-recursive-minibuffers t)
             (search (completing-read "Preset: " citar-presets)))
    (insert search)))

(defun citar--stringify-keys (keys)
  "Encode a list of KEYS as a single string."
  (combine-and-quote-strings (if (listp keys) keys (list keys)) " & "))

(defun citar--unstringify-keys (keystring)
  "Split KEYSTRING into a list of keys."
  (split-string-and-unquote keystring " & "))

;;; Commands

;;;###autoload
(defun citar-open (keys)
  "Open related resources (links, files, or notes) for KEYS."
  (interactive (list (citar-select-refs)))
  (pcase (let ((embark-default-action-overrides
                (cons (cons t #'citar--open-resource)
                      (bound-and-true-p embark-default-action-overrides))))
           (apply #'citar--select-resource keys
                  (mapcan (lambda (type) (list type t)) citar-open-resources)))
    (`(,type . ,resource) (citar--open-resource resource type))
    (_ (error "No associated resources: %s" keys))))

(defun citar--open-resource (resource &optional type)
  "Open RESOURCE of TYPE.
TYPE should be `file', `url', `note', or `create-note'. If TYPE
is nil, then RESOURCE must have a `citar--resource' text property
specifying TYPE."
  ;; IMPORTANT: This function must not have an `interactive' specification. When called as an Embark action,
  ;; it relies on RESOURCE having the `citar--resource' text property to decide which action to take. However,
  ;; `embark-act' strips text properties for interactive commands, for which it injects the target text as
  ;; minibuffer input.
  (if-let ((opener (pcase (or type (get-text-property 0 'citar--resource resource))
                     ('file #'citar-file-open)
                     ('url #'browse-url)
                     ('note #'citar-open-note)
                     ('create-note #'citar-create-note))))
      (funcall opener (substring-no-properties resource))
    (error "Could not open resource of type `%s': %S" type resource)))

;;;###autoload
(defun citar-open-files (key-or-keys)
  "Open library file associated with KEY-OR-KEYS."
  (interactive (list (citar-select-refs)))
  ;; TODO filter to refs have files?
  (citar--library-file-action key-or-keys #'citar-file-open))

;;;###autoload
(defun citar-attach-files (key-or-keys)
  "Attach library file associated with KEY-OR-KEYS to outgoing MIME message."
  (interactive (list (citar-select-ref)))
  (citar--library-file-action key-or-keys #'mml-attach-file))

(defun citar--library-file-action (key-or-keys action)
  "Run ACTION on file associated with KEY-OR-KEYS.
Use `citar--select-resource' to choose a file."
  (let ((citar--entries (citar-get-entries)))
    (pcase (let ((embark-default-action-overrides
                  (cons (cons `(file . ,this-command) action)
                        (bound-and-true-p embark-default-action-overrides))))
             (citar--select-resource key-or-keys :files t))
      (`(file . ,file) (funcall action file))
      (`(,type . ,resource) (error "Expected resource of type `file', got `%s': %S" type resource))
      ('nil
       (ignore
        ;; If some key had files according to the `:hasitems' function, but `:items' returned nothing, then
        ;; don't print the following message. The `:items' function is responsible for telling the user why it
        ;; failed, and we want that explanation to appear in the echo area.
        (let ((keys (if (listp key-or-keys) key-or-keys (list key-or-keys)))
              (hasfilep (citar-has-files)))
          (unless (and hasfilep (seq-some hasfilep keys))
            (message "No associated files for %s" key-or-keys))))))))

;;;###autoload
(defun citar-open-note (note)
  "Open NOTE, which should be a string returned by `citar-get-notes'.
When called interactively, prompt for a note to open from a list
of all notes."
  (interactive (list (when-let* ((notes (citar-get-notes))
                                 (allnotes (delete-dups (apply #'append (hash-table-values notes)))))
                       (cdr (citar--select-resource nil :notes allnotes)))))
  (when note
    (funcall (citar--get-notes-config :open) note)))

;;;###autoload
(defun citar-open-notes (keys)
  "Open notes associated with the KEYS."
  (interactive (list (citar-select-refs)))
  (pcase (let ((embark-default-action-overrides
                (cons (cons t #'citar--open-resource)
                      (bound-and-true-p embark-default-action-overrides))))
           (citar--select-resource keys :notes t :create-notes t))
    (`(note . ,note) (citar-open-note note))
    (`(create-note . ,key) (citar-create-note key))))

;;;###autoload
(defun citar-open-links (key-or-keys)
  "Open URL or DOI link associated with KEY-OR-KEYS in a browser."
  (interactive (list (citar-select-refs)))
  (pcase (let ((embark-default-action-overrides
                (cons (cons `(url . ,this-command) #'browse-url)
                      (bound-and-true-p embark-default-action-overrides))))
           (citar--select-resource key-or-keys :links t))
    (`(url . ,url) (browse-url url))
    (`(,type . ,resource) (error "Expected resource of type `url', got `%s': %S" type resource))
    ('nil (message "No link found for %s" key-or-keys))))

;;;###autoload
(defun citar-open-entry (key)
  "Open bibliographic entry associated with the KEY."
  (interactive (list (citar-select-ref)))
  (when-let ((bib-files (citar--bibliography-files)))
    (citar--open-entry key bib-files)))

(defun citar--open-entry (key bib-files)
  "Open entry for KEY in the relevant BIB-FILES."
  ;; Adapted from 'bibtex-completion-show-entry'.
  (catch 'break
    (dolist (bib-file bib-files)
      (let ((buf (or (get-file-buffer bib-file)
                     (find-buffer-visiting bib-file))))
        (find-file bib-file)
        (widen)
        (goto-char (point-min))
        (when (re-search-forward
               (concat "^@\\(" parsebib--bibtex-identifier
                       "\\)[[:space:]]*[\\(\\{][[:space:]]*"
                       (regexp-quote key) "[[:space:]]*,") nil t)
          (throw 'break t))
        (unless buf
          (kill-buffer))))))

;;;###autoload
(defun citar-insert-bibtex (keys)
  "Insert bibliographic entry associated with the KEYS."
  (interactive (list (citar-select-refs)))
  (dolist (key keys)
    (citar--insert-bibtex key)))

(defun citar--insert-bibtex (key)
  "Insert the bibtex entry for KEY at point."
  (let* ((bibtex-files
          (citar--bibliography-files))
         (entry
          (with-temp-buffer
            (bibtex-set-dialect)
            (dolist (bib-file bibtex-files)
              (insert-file-contents bib-file))
            (bibtex-search-entry key)
            (let ((beg (bibtex-beginning-of-entry))
                  (end (bibtex-end-of-entry)))
              (buffer-substring-no-properties beg end)))))
    (unless (equal entry "")
      (insert entry "\n\n"))))

;;;###autoload
(defun citar-export-local-bib-file ()
  "Create a new bibliography file from citations in current buffer.

The file is titled \"local-bib\", given the same extention as
the first entry in `citar-bibliography', and created in the same
directory as current buffer."
  (interactive)
  (let* ((keys (citar--major-mode-function 'list-keys #'ignore))
         (ext (file-name-extension (car citar-bibliography)))
         (file (format "%slocal-bib.%s" (file-name-directory buffer-file-name) ext)))
    (with-temp-file file
      (dolist (key keys)
        (citar--insert-bibtex key)))))

;;;###autoload
(defun citar-insert-citation (keys &optional arg)
  "Insert citation for the KEYS.

Prefix ARG is passed to the mode-specific insertion function. It
should invert the default behaviour for that mode with respect to
citation styles. See specific functions for more detail."
  (interactive
   (if (citar--get-major-mode-function 'insert-citation)
       (list (citar-select-refs) current-prefix-arg)
     (error "Citation insertion is not supported for %s" major-mode)))
  (citar--major-mode-function
   'insert-citation
   #'ignore
   keys
   arg))

(defun citar-insert-edit (&optional arg)
  "Edit the citation at point.
ARG is forwarded to the mode-specific insertion function given in
`citar-major-mode-functions'."
  (interactive "P")
  (citar--major-mode-function
   'insert-edit
   (lambda (&rest _)
     (message "Citation editing is not supported for %s" major-mode))
   arg))

;;;###autoload
(defun citar-insert-reference (keys)
  "Insert formatted reference(s) associated with the KEYS."
  (interactive (list (citar-select-refs)))
  (insert (funcall citar-format-reference-function keys)))

;;;###autoload
(defun citar-copy-reference (keys)
  "Copy formatted reference(s) associated with the KEYS."
  (interactive (list (citar-select-refs)))
  (let ((references (funcall citar-format-reference-function keys)))
    (if (not (equal "" references))
        (progn
          (kill-new references)
          (message (format "Copied:\n%s" references)))
      (message "Key not found."))))

(defun citar-format-reference (keys)
  "Return formatted reference(s) for the elements of KEYS."
  (let* ((entries (mapcar #'citar-get-entry keys))
         (template (citar--get-template 'preview)))
    (with-temp-buffer
      (dolist (entry entries)
        (insert (citar-format--entry template entry)))
      (buffer-string))))

;;;###autoload
(defun citar-insert-keys (keys)
  "Insert KEYS citekeys."
  (interactive (list (citar-select-refs)))
  (citar--major-mode-function
   'insert-keys
   #'citar--insert-keys-comma-separated
   keys))

(defun citar--insert-keys-comma-separated (keys)
  "Insert comma separated KEYS."
  (insert (string-join keys ", ")))

;;;###autoload
(defun citar-add-file-to-library (key)
  "Add a file to the library for KEY.
The FILE can be added from an open buffer, a file path, or a
URL."
  (interactive (list (citar-select-ref)))
  (citar--check-configuration 'citar-library-paths)
  (unless citar-library-paths
    (user-error "Make sure `citar-library-paths' is non-nil"))
  (let* ((directory (if (cdr citar-library-paths)
                        (completing-read "Directory: " citar-library-paths)
                      (car citar-library-paths)))
         (filepath (expand-file-name key directory))
         (withext (lambda (extension)
                    (let* ((extension (or extension (read-string "File extension: "))))
                      (if (string-empty-p extension)
                          filepath
                        (concat filepath "." extension))))))
    (pcase (read-char-choice "Add file from [b]uffer, [f]ile, or [u]rl? " '(?b ?f ?u))
      (?b
       (with-current-buffer (read-buffer "Add file buffer: " (current-buffer))
         (let ((destfile (funcall withext (and buffer-file-name (file-name-extension buffer-file-name)))))
           (write-file destfile 'confirm))))
      (?f
       (let* ((file (read-file-name "Add file: " nil nil t))
              (destfile (funcall withext (file-name-extension file))))
         (copy-file file destfile 1)))  ; last arg integer means to confirm before overwriting
      (?u
       (let* ((url (read-string "Add file URL: "))
              ;; TODO: Use Content-Type HTTP response header to guess file extension
              (ext (url-file-extension url))
              (destfile (funcall withext (when (> (length ext) 1) (substring ext 1)))))
         (url-copy-file url destfile 1))))))

;;;###autoload
(defun citar-run-default-action (keys)
  "Run the default action `citar-default-action' on KEYS."
  (funcall citar-default-action keys))

;;;###autoload
(defun citar-dwim ()
  "Run the default action on citation keys found at point."
  (interactive)
  (if-let ((keys (or (citar-key-at-point) (citar-citation-at-point))))
      (citar-run-default-action (if (listp keys) keys (list keys)))
    (user-error "No citation keys found")))

(defun citar--check-configuration (&rest variables)
  "Signal error if any VARIABLES have values of the wrong type.
VARIABLES should be the names of Citar customization variables."
  (dolist (variable variables)
    (unless (boundp variable)
      (error "Unbound variable in citar--check-configuration: %s" variable))
    (let ((value (symbol-value variable)))
      (pcase variable
        ((or 'citar-library-paths 'citar-notes-paths)
         (unless (and (listp value)
                      (seq-every-p #'stringp value))
           (error "`%s' should be a list of directories: %S" variable `',value)))
        ((or 'citar-library-file-extensions 'citar-file-note-extensions)
         (unless (and (listp value)
                      (seq-every-p #'stringp value))
           (error "`%s' should be a list of strings: %S" variable `',value)))
        ((or 'citar-has-files-functions 'citar-get-files-functions 'citar-file-parser-functions)
         (unless (and (listp value) (seq-every-p #'functionp value))
           (error "`%s' should be a list of functions: %S" variable `',value)))
        ((or 'citar-note-format-function)
         (unless (functionp value)
           (error "`%s' should be a function: %S" variable `',value)))
        (_
         (error "Unknown variable in citar--check-configuration: %s" variable))))))

(defun citar--check-notes-source (name config)
  "Signal error if notes source plist CONFIG has incorrect keys or values.
SOURCE must be a plist representing a notes source with NAME. See
`citar-notes-sources' for the list of valid keys and types."

  (let ((required '(:items :hasitems :open))
        (optional '(:name :category :create :transform :annotate))
        (keys (map-keys config)))
    (when-let ((missing (cl-set-difference required keys)))
      (error "Note source `%s' missing required keys: %s" name missing))
    (when-let ((extra (cl-set-difference keys (append required optional))))
      (warn "Note source `%s' has unknown keys: %s" name extra)))

  (pcase-dolist (`(,type . ,props)
                 '((functionp :items :hasitems :open :create :transform :annotate)
                   (stringp :name)
                   (symbolp :category)))
    (when-let ((wrongtype (seq-filter (lambda (prop)
                                        (when-let ((value (plist-get config prop)))
                                          (not (funcall type value)))) props)))
      (error "Note source `%s' keys must be of type %s: %s" name type wrongtype))))

(provide 'citar)
;;; citar.el ends here
