#include "cp_types.h"
#include "cp_proto.h"
#include "cp_post.h"

/* Drawing flag scheme: flags indicate drawing instructions
   for objects -- circles/lines/faces. Bits set as follows:
      1  ->  draw object?
      2  ->  fill? (4 and 16 imply 2, also)
      4  ->  off = foreground, on = background 
             (applies to interior, not border, overriden by bit 16)
      8  ->  border color? (default=foreground, else recorded color)
      16 ->  interior color? (default set by bit 4, on --> recorded color)
      32 ->  display label?

Eg.  flag=3: filled object, in foreground
     flag=9: open object, border in (recorded) color
             (for 'edge', this gives colored edge)
     flag=19: filled in color, border in foreground
     flag=27: filled, border and interior in color
     flag=15: filled with background, border in color
     flag=32: label only (face or circle)

Normally, flag for each type of object; often passed on to
subroutine, so may need color code with it:
Eg. (cflag, ccol, ecol) for circle flag, int color, border color.
*/

int post_facelist(FILE *fp,struct p_data *p,int fflag,
		  struct Vertlist **facelist,int fix,int l_face)
     /* lay out faces from facelist. Standard flag for colors.
fix=1 means to locate faces successively as drawn (changing stored
centers). facelist is cleared. l_face="live face" may be given; */
{
  int ecol=FG_COLOR,fcol=BG_COLOR,n,v1,v2,v3,vv=0;
  double o1,o2,o3;
  struct Vertlist *trace;
  struct K_data *pK_ptr;
  struct R_data *pR_ptr;

  pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
  if ((*facelist)==NULL) return 0;
  if (fflag & 4) fcol=BG_COLOR;
  trace=*facelist;
  /* post first */
  if (l_face>0 && l_face<=p->facecount) vv=l_face;
  do {
    /* if fix, draw successive ones as contig. */
    if (fix)
      {
	if ( (n=nghb_tri(p,vv,trace->v))<0 )
	  n=p->faces[trace->v].index_flag;
	/* if not contig, use index_flag */
	v1=p->faces[trace->v].vert[n];
	v2=p->faces[trace->v].vert[(n+1)%3];
	v3=p->faces[trace->v].vert[(n+2)%3];
	      
	if (p->overlap_status) 
	  {
	    o1=pK_ptr[v1].overlaps[nghb(p,v1,v2)];
	    o2=pK_ptr[v2].overlaps[nghb(p,v2,v3)];
	    o3=pK_ptr[v3].overlaps[nghb(p,v3,v2)];
	  }
	else o1=o2=o3=1.0;
	any_compcenter(p->hes,
		       pR_ptr[v1].center,pR_ptr[v2].center,&pR_ptr[v3].center,
		       pR_ptr[v1].rad,pR_ptr[v2].rad,&pR_ptr[v3].rad,o1,o2,o3);
	/* compute and store new center */
      }

    while (trace!=NULL && trace->v==vv) trace=trace->next;
    if (trace==NULL) break;
    vv=trace->v;
    if ( vv>0 && vv<=(p->facecount) )
      {
	if (fflag & 16) fcol=p->faces[vv].color;
	if (fflag & 8) ecol=p->faces[vv].color;
	post_any_face(fp,p,vv,fflag,ecol,fcol);
	if (fflag & 32) post_face_label(fp,p,vv);
      }
    trace=trace->next;
  } while (trace!=NULL);
  vert_free(facelist);
  return vv;
} /* post_facelist */

int post_bdry_seg(FILE *fp,struct p_data *p,int n,int eflag,int ecol)
     /* draw boundary segment, given starting face and index of begin 
vert in an EdgePair structure. */
{
  int indx=0,colr;
  complex v_cent,w_cent;
  struct RedList *trace;

  if (!(trace=p->edge_pair[n].edge)) return 0;
  if (trace->next->face==trace->prev->face
      && trace->v_flag!=(indx=p->edge_pair[n].edge_indx)) 
    /* blue face, second edge, get data from prev face. */
    w_cent=trace->prev->center;
  else w_cent=trace->center;
  colr=FG_COLOR;
  if (eflag & 8) colr=ecol;
  fprintf(fp,"gs\n%f ourlinewidth\n",3*ps_linewidth);
  do
    {
      v_cent=w_cent;
      if (trace->next->face==trace->prev->face
	  && trace->v_flag==indx) /* blue face, want second edge */
	{
	  indx=(indx+1) %3;
	  w_cent=trace->prev->center;
	}
      else 
	{
	  trace=trace->next_edge;
	  indx=(nghb_tri(p,trace->prev->face,trace->face)+2) % 3;
	  w_cent=trace->center;
	}
      post_any_geo(fp,p,v_cent,w_cent,9,colr);
    } while (!(trace->corner_flag[indx] & 3));
  fprintf(fp,"gr\n");
  return 1;
} /* post_bdry_seg */

int post_bdry_seg_num(FILE *fp,struct p_data *p,int n)
     /* put segment number at first circle */
{
  complex ctr;
  struct RedList *trace;

  if (!(trace=p->edge_pair[n].edge)) return 0;
  if (trace->next->face==trace->prev->face
      && trace->v_flag!=p->edge_pair[n].edge_indx) 
    /* blue face, second edge, get data from prev face. */
    ctr=trace->prev->center;
  else ctr=trace->center;
  post_num(fp,ctr,n,p->screen->box);
  return 1;
} /* post_bdry_seg_num */

int post_any_face(FILE *fp,struct p_data *p,int face,
		  int fflag,int ecol,int fcol)
{
  int front;
  complex p1,p2,p3;
  hyp_geodesic hg1,hg2,hg3;
  struct R_data *pR_ptr;

  if (face<1 || face > p->facecount ||
      !p->faces[face].plot_flag) 
    return 0;
  pR_ptr=p->packR_ptr;
  p1=pR_ptr[p->faces[face].vert[0]].center;
  p2=pR_ptr[p->faces[face].vert[1]].center;
  p3=pR_ptr[p->faces[face].vert[2]].center;
  if (p->hes< 0)
    {
      h_geodesic(p1,p2,&hg1);
      h_geodesic(p2,p3,&hg2);
      h_geodesic(p3,p1,&hg3); /* set triangle structures */
      if (!geo_ck(hg1,p->screen->box) && 
	  !geo_ck(hg2,p->screen->box) && 
	  !geo_ck(hg3,p->screen->box)) 
	return 1;  /* off screen */
      fprintf(fp,"gs %f %f m\n",p1.re,p1.im);
      if (hg1.line_flag)
	fprintf(fp,"%f %f l\n",p2.re,p2.im);
      else 
	{
	  if (hg1.z1.re==p1.re && hg1.z1.im==p1.im) /* right direction*/
	    fprintf(fp,"%f %f %f %f %f a\n",
		    hg1.c.re,hg1.c.im,hg1.rad,
		    hg1.arg1*degPI,hg1.arg2*degPI);
	  else fprintf(fp,"%f %f %f %f %f an\n",
		       hg1.c.re,hg1.c.im,hg1.rad,
		       hg1.arg2*degPI,hg1.arg1*degPI);
	}
      if (hg2.line_flag)
	fprintf(fp,"%f %f l\n",p3.re,p3.im);
      else 
	{
	  if (hg2.z1.re==p2.re && hg2.z1.im==p2.im) /* right direction*/
	    fprintf(fp,"%f %f %f %f %f a\n",
		    hg2.c.re,hg2.c.im,hg2.rad,
		    hg2.arg1*degPI,hg2.arg2*degPI);
	  else fprintf(fp,"%f %f %f %f %f an\n",
		       hg2.c.re,hg2.c.im,hg2.rad,
		       hg2.arg2*degPI,hg2.arg1*degPI);
	}
      if (hg3.line_flag)
	fprintf(fp,"%f %f l\n",p1.re,p1.im);
      else 
	{
	  if (hg3.z1.re==p3.re && hg3.z1.im==p3.im) /* right direction*/
	    fprintf(fp,"%f %f %f %f %f a\n",
		    hg3.c.re,hg3.c.im,hg3.rad,
		    hg3.arg1*degPI,hg3.arg2*degPI);
	  else fprintf(fp,"%f %f %f %f %f an\n",
		       hg3.c.re,hg3.c.im,hg3.rad,
		       hg3.arg2*degPI,hg3.arg1*degPI);
	}
      if (ecol!=FG_COLOR) /* edge colors? */
	fprintf(fp,"%f %f %f srgb ",
		(double)red[ecol]/255,(double)green[ecol]/255,
		(double)blue[ecol]/255);
      if ((fflag & 2) && fcol==BG_COLOR)
	fprintf(fp,"gs 1 1 1 srgb fill gr ");
      else if ((fflag & 2) && fcol!=FG_COLOR) 
	fprintf(fp,"gs %f %f %f srgb fill gr ",
		(double)red[fcol]/255,(double)green[fcol]/255,
		(double)blue[fcol]/255);
      else if (fflag & 2)
	fprintf(fp,"gs gry  sg fill gr\n"); 
      fprintf(fp,"s gr\n"); /* finish drawing */
    } /* end of hyp case */
  else if (p->hes == 0)
    {
      if (line_ck(p1,p2,p->screen->box) || 
	  line_ck(p2,p3,p->screen->box) || 
	  line_ck(p3,p1,p->screen->box))
	{
	  fprintf(fp,"gs %f %f m %f %f l %f %f l cp\n",
		  p1.re,p1.im,p2.re,p2.im,p3.re,p3.im);
	  if (ecol!=FG_COLOR) /* edge colors? */
	    fprintf(fp,"%f %f %f srgb ",
		    (double)red[ecol]/255,(double)green[ecol]/255,
		    (double)blue[ecol]/255);
	  if ((fflag & 2) && fcol!=FG_COLOR) 
	    fprintf(fp,"gs %f %f %f srgb fill gr ",
		    (double)red[fcol]/255,(double)green[fcol]/255,
		    (double)blue[fcol]/255);
	  else if (fflag & 2)
	    fprintf(fp,"gs gry  sg fill gr\n"); 
	  fprintf(fp,"s gr\n"); /* finish drawing */
	} /* end of eucl case */
    }
  else if (p->hes>0) 
    post_s_tri(fp,
	       ss_view(p->screen,p1,1,&front),
	       ss_view(p->screen,p2,1,&front),
	       ss_view(p->screen,p3,1,&front),fflag,ecol,fcol);
  if (fflag & 32) post_face_label(fp,p,face);
  return 1;
} /* post_any_face */

