#include "cp_types.h"
#include "cp_proto.h"

/* routines for circle packing with specified Gaussian curvature. */

int kappa_calc(struct p_data *p,int v,double *kap)
     /* compute the square root of curvature for that model in which 
label for v gives angle sum 2pi. return = 0 means no pos curv will 
work. return =-1 means center had hyp radius infinity. */
{
  double Theta,Phi,Top,Bottom, Middle, Difference,*R;
  int i,n;
  double kap_upper_bd,rad,bound,dum;
  struct K_data *pK_ptr;
  struct R_data *pR_ptr;

  pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
  n=pK_ptr[v].num;
  /* set up label */
  R=(double *)malloc((n+2)*sizeof(double));
  if (p->hes<0) /* hyp data, stored as s-radii */
    {
      if ((rad=pR_ptr[v].rad)<=0)
	{
	  *kap=0.0;
	  free(R);
	  return -1;
	}
      R[0]=-log(rad);
      for (i=0;i<=n;i++)
	{
	  if ((rad=pR_ptr[pK_ptr[v].flower[i]].rad)<=0) 
	    R[i+1]=10.0; /* arbitrary large value */
	  else R[i+1]=-log(rad);
	}
    }
  else
    {
      R[0]=pR_ptr[v].rad;
      for (i=0;i<=n;i++) R[i+1]=pR_ptr[pK_ptr[v].flower[i]].rad;
    }
  /* First get eucl angle sum to see if curvature should be +,-, or 0 */
  Theta= 0.0;
  for (i=1;i<=n;i++)
    Theta += acos (((R[0] + R[i])*(R[0]+R[i]) + 
		    (R[0] + R[i+1])*(R[0]+R[i+1])-
		    (R[i]+R[i+1])*(R[i]+R[i+1]))/
		   (2*(R[0] + R[i])*(R[0] + R[i+1])));
  /* close to euclidean */
  if (fabs(Theta-2.0*M_PI)<okerr) 
    {
      free(R);
      *kap=0.0;
      return 1;
    }	
  /* Positive curvature case */
  if (Theta <= (2.0 * M_PI))
    {
      bound=0.0;
      for (i=0;i<n;i++)
	{
	  if ((dum=R[0]+R[i+1]+R[i+2])>bound) bound=dum;
	}
      kap_upper_bd=M_PI/bound; 
      /* kap too small means labels would be inconsistent */
      Phi=pos_k_angle_sum(kap_upper_bd-okerr,R,n);
      if (Phi< 2.0*M_PI-okerr) 
	/* no curvature will work for this label */
	{
	  free(R);
	  *kap=0.0;
	  return 0;
	}
      if (1.0< kap_upper_bd/2.0) Top= 1.0;
      else Top= kap_upper_bd/2.0;
      Phi=pos_k_angle_sum(Top,R,n);
      while (Phi <= (2*M_PI) && Top<kap_upper_bd-okerr)
	{
	  Top=(2.0*Top<kap_upper_bd-okerr) ? 
	    2.0*Top : kap_upper_bd-okerr;
	  Phi=pos_k_angle_sum(Top,R,n);
	}
      Bottom=0;
      Middle=0;  /* to handle Theta = 2*M_PI */
      Difference= Theta - 2.0* M_PI;
      while (fabs(Difference)>okerr)
	{
	  Middle= (Top + Bottom)/2.0;
	  Difference= pos_k_angle_sum(Middle,R,n)-2.0*M_PI;
	  if (Difference>=0) Top=Middle;
	  else Bottom=Middle;
	}
      free(R);
      *kap=Middle;
      return 1;
    }
  /* Negative curvature case */
  else 
    {
      Top=-1;
      Phi=neg_k_angle_sum(Top,R,n);
      while (Phi >= (2.0*M_PI))
	{
	  Top= 2.0*Top;
	  Phi= neg_k_angle_sum(Top,R,n);
	}
      Bottom= Middle=0.0;
      Difference= Theta - 2.0*M_PI;
      while (fabs(Difference)>okerr)
	{
	  Middle=(Top + Bottom)/2.0;
	  Difference=neg_k_angle_sum(Middle,R,n)-2.0*M_PI;
	  if (Difference<=0) Top=Middle;
	  else Bottom=Middle;
	}
      free(R);
      *kap=Middle;
      return 1;
    }
} /* kappa_calc */

double pos_k_angle_sum(double c,double *R,int m)
     /* ang sum comp for curvature + (c*c) */
{ 
  int i;
  double Sum;

  Sum=0.0;
  for (i=1;i<=m;i++)
    Sum += acos((cos(c*(R[i]+R[i+1]))-(cos(c*(R[0]+R[i]))*
				       cos(c*(R[0]+R[i+1]))))/
		(sin(c*(R[0]+R[i]))*sin(c*(R[0]+R[i+1]))));
  return (Sum);
} /* pos_k_angle_sum */
			
double neg_k_angle_sum(double c,double *R,int m)
     /* ang sum comp for curvature - (c*c) */
{
  int i;
  double Sum;
 
  Sum=0.0;
  c=c*(-1.0);  /* to simplify formula below */
  for (i=1;i<=m;i++)
    Sum += acos((cosh(c*(R[0]+R[i]))*cosh(c*(R[0]+R[i+1])) -
		 cosh(c*(R[i]+R[i+1])))/(sinh(c*(R[0]+R[i]))*
					 sinh(c*(R[0]+R[i+1]))));
  return (Sum);
} /* neg_k_angle_sum */

int color_kappa(struct p_data *p,double base)
     /* set circle colors; compare model curvature to base. 
Red==>kappa > base, Blue==> < base, white==> = base. */
{
  int v;
  double kap;
  struct K_data *pK_ptr;

  pK_ptr=p->packK_ptr;
  for (v=1;v<=p->nodecount;v++)
    {
      if (pK_ptr[v].bdry_flag || kappa_calc(p,v,&kap)<=0) 
	{
	  pK_ptr[v].color=FG_COLOR;
	  return 1;
	}
      kap=kap-base;
      if (kap<=0)
	{
	  if (fabs(kap)<okerr) pK_ptr[v].color=0;
	  if (kap>= -1.0) 
	    pK_ptr[v].color=(int)(kap*40 + 100);
	  else if (kap>= -2.0) 
	    pK_ptr[v].color=(int)((kap+1)*25+60);
	  else if (kap>= -3.0)
	    pK_ptr[v].color=(int)((kap+2)*20+35);	
	  else if (kap>= -4.0)
	    pK_ptr[v].color=(int)((kap+3)*14+15);
	  else 
	    pK_ptr[v].color=1;
	}
      else 
	{
	  if (kap<=1.0)
	    pK_ptr[v].color=(int)(kap*40+100);
	  else if (kap<=2.0)
	    pK_ptr[v].color=(int)((kap-1)*25+140);
	  else if (kap<=3.0)
	    pK_ptr[v].color=(int)((kap-2)*20+165);
	  else if (kap<=4.0)
	    pK_ptr[v].color=(int)((kap-3)*14+185);
	  else 
	    pK_ptr[v].color=199;
	}
    }
  return 1;
} /* color_kappa */
	
