/* cp_call.c    CirclePack dependent, bureaucratic routines for
   calling more general routines in the libraries. */

#include "cp_head.h"

char next[BUFSIZE],buff[BUFSIZE];

struct Vertlist *node_link_parse(struct p_data *p,char *dpoint,
				 char **endptr,int *hits)
/* call to more general routine */
{
  return (Node_link_parse(p,dpoint,endptr,hits,&Vlist,&Elist,&Flist,
			  &region,pathlist,pathlength));
} /* node_link_parse */

struct Vertlist *face_link_parse(struct p_data *p,char *dpoint,
				 char **endptr,int *hits)
/* call to more general routine */
{
  return (Face_link_parse(p,dpoint,endptr,hits,&Vlist,&Elist,&Flist,
			  &region,pathlist,pathlength));
} /* face_link_parse */

struct Edgelist *node_pair_link(struct p_data *p,char *dpoint,
				char **endptr,int *hits)
/* call to more general routine */
{
  return (Node_pair_link(p,dpoint,endptr,hits,&Vlist,&Elist,&Flist,
			 &region,pathlist,pathlength));
} /* node_pair_link */

int cmd_search(char *datastr,int *shift)
     /* Process datastr to pick out full cmd; takes care of line 
continuations, etc. Shift gives end of cmd. */
{
  int flag=0,done=0,high_water=0;
  char *nextpoint;

  *shift=0;
  nextpoint=datastr;
  while (!done)
    {
      while (nextpoint[flag]!='\n' 
	     && nextpoint[flag]!='\0'
	     && nextpoint[flag]!='\\') flag++;
      if (nextpoint[flag]=='\0') 
	{
	  *shift=flag;
	  if (flag==0) return 0; /* vacuous command */
	  done=1;
	}
      else if (nextpoint[flag]=='\n')
	{
	  nextpoint[flag]='\0';
	  done=*shift=flag;
	  if (flag==0) return 0; /* vacuous command */
	}
      else /* continuation indicator */
	{
	  high_water=flag+1;
	  while (nextpoint[high_water]==' '
		 || nextpoint[high_water]=='\t') high_water++;
	  if (nextpoint[high_water]=='\n') flag=high_water+1;
	  else /* improper termination of cmd, but okay */
	    {
	      nextpoint[high_water]='\0';
	      *shift=high_water;
	      done=1;
	    }
	}
    }
  flag=0;
  while (flag<*shift) /* replace line breaks/continues by spaces */
    {
      if (nextpoint[flag]=='\\' || nextpoint[flag]=='\n') 
	nextpoint[flag]=' ';
      flag++;
    }
  return flag;
} /* cmd_search */

int match_call(struct p_data *p,char *datastr)
     /* call to put two packs in register */
{
  int qnum=0,v=0,w=0,V=0,W=0,err_flag=0;
  double err=0.0;
  char *ptr;
  Mobius M;
  struct p_data *q=NULL;
  
  ptr=datastr;
  stripsp(ptr);
  if (ptr[0]!='-' || ptr[1]!='q' || (qnum=atoi(ptr+2))<0 || qnum>=NUM_PACKS
      || sscanf(ptr+4,"%d %d %d %d",&v,&w,&V,&W)<2
      || v<1 || v>p->nodecount || w<1 || w>p->nodecount
      || V>(q=&packdata[qnum])->nodecount || W>q->nodecount
      || (V<1 && (V=vertex_translation(p->vertex_map,v))<1)
      || (W<1 && (W=vertex_translation(p->vertex_map,w))<1) )
    {
      sprintf(msgbuf,"match usage: -q<pack#> v w <V W>. "
	      "No pack number given or vertices improper.\n");
      emsg();
      return 0;
    }
  if (!match(p,q,v,w,V,W,&err,&err_flag,&M) || err_flag)
    {
      sprintf(msgbuf,"match: something went wrong in generating the "
	      "Mobius transformation.");
      emsg();
      return 0;
    }
  apply_Mobius(p,"a",1,M);
  return 1;
} /* match_call */

int blend_call(struct p_data *p,char *datastr)
{
  int qnum,n,v;
  char *nextpoint;

  nextpoint=datastr;
  if (!grab_next(&nextpoint,next) || strlen(next)<3
      || next[0]!='-' || next[1]!='q' || (qnum=atoi(next+2))<0
      || qnum>=NUM_PACKS || sscanf(nextpoint,"%d %d",&v,&n)!=2)
    {
      sprintf(msgbuf,"blend usage: -q<pack#> v n.");
      emsg();
      return 0;
    }
  return (blend(&packdata[qnum],p,v,n));
} /* blend_call */

int path_construct_call(struct p_data *p,char *datastr)
{
  int flag=0,v=0;
  char next[256],*nextpoint;

  nextpoint=datastr;
  stripsp(nextpoint);
  if (*nextpoint=='-')
    {
      grab_next(&nextpoint,next);
      if (next[1]=='i') flag=1;
    }
  if (!(v=grab_one_vert(p,&nextpoint)))
    return 0;
  if (Vlist) vert_free(&Vlist);
  Vlist=path_construct(p,flag,nextpoint);
  return 1;
} /* path_construct_call */

int get_put_data_call(struct p_data *p,char *datastr,int rev_flag)
/* to transfer various data from p to target pack. datastr
must begin with -q<pnum>, where pnum is the target pack, then have 
flags: -t means to use the vertex_map to translate indices (default
is not to translate), -<f> indicating desired data. */
{
  int qnum,trans_flag=0,hits=0;
  char *ptr,flag;
  char next[BUFSIZE];

  ptr=datastr;
  if (!grab_next(&ptr,next) || next[0]!='-' || next[1]!='q' 
      || (qnum=atoi(next+2)) <0 || qnum>=NUM_PACKS)
    {
      sprintf(msgbuf,"get/put_data usage: -q<pack#> -<flag> <v>. "
	      "No pack number given.\n");
      emsg();
      return 0;
    }
  stripsp(ptr);
  while (grab_next(&ptr,next) && next[0]=='-')
    {
      if ((flag=next[1])=='t') trans_flag=1; /* want to translate indices */
      else 
	if (get_put_data(p,&packdata[qnum],ptr,flag,
			 rev_flag,trans_flag))
	  hits++;
    } /* end of looking for data specifications */
  if (!hits) 
    {
      sprintf(msgbuf,"get/put_data usage: -q<pack#> -<flag> <v>. "
	      "The flag for 'translation' is -t, data flags are:\n"
	      "  r=radii,z=centers,a=aims,s=angle sums,c=circle "
	      "colors,m=mark,f=face colors.");
      emsg();
      return 0;
    }
  return (hits);
} /* get_put_data_call */
  

int cir_invert_call(struct p_data *p,char *datastr)
     /* map outside of one circle onto inside of second target circle */
{
  int err_flag=0,v1,v2,hits;
  double rad1,rad2;
  complex ctr1,ctr2;
  char *nextpoint,*endptr;
  struct Vertlist *vlist;
  Mobius mob;

  nextpoint=datastr;
  if (!grab_next(&nextpoint,next))
    return 0;
  if (next[0]=='-') /* option -u means target is the unit circle */
    {
      if (next[1]!='u') return 0; 
      else 
	{
	  ctr2.re=ctr2.im=0.0;
	  rad2=1.0;
	}
      if (!(vlist=node_link_parse(p,nextpoint,&endptr,&hits)))
	return 0;
      v1=vlist->v;
      vert_free(&vlist);
      if (p->hes<0) geom_to_e_pack_convert(p); 
               /* have to convert to euclidean */
      ctr1=p->packR_ptr[v1].center;
      rad1=p->packR_ptr[v1].rad;
      if (p->hes>0) /* spherical */
	s_to_e_data(ctr1,rad1,&ctr1,&rad1);
    }
  else /* target circle is vert v2 */
    {
      if (!(vlist=node_link_parse(p,nextpoint,&endptr,&hits))
	  || !(vlist->next) || (v1=vlist->v)==(v2=vlist->next->v)) 
	{
	  vert_free(&vlist); /* didn't get two distinct verts */
	  return 0; 
	}
      vert_free(&vlist);
      if (p->hes<0) geom_to_e_pack_convert(p); 
               /* have to convert to euclidean */
      ctr1=p->packR_ptr[v1].center;
      ctr2=p->packR_ptr[v2].center;
      rad1=p->packR_ptr[v1].rad;
      rad2=p->packR_ptr[v2].rad;
      if (p->hes>0) /* spherical */
	{
	  s_to_e_data(ctr1,rad1,&ctr1,&rad1);
	  s_to_e_data(ctr2,rad2,&ctr2,&rad2);
	}
    }
      
  mob=cir_invert(ctr1,rad1,ctr2,rad2,&err_flag);
  if (err_flag || !apply_Mobius(p,"a",1,mob)) 
    return 0;
  return 1;
} /* cir_invert_call */

int molify_call(struct p_data *p,char *datastr)
{
  int n=1,hits;
  char next[32],*nextpoint,*endptr;
  struct Vertlist *vertlist;

  nextpoint=datastr;
  stripsp(nextpoint);
  if (nextpoint[0]=='-' && nextpoint[1]=='i')
    {
      nextpoint += 2;
      grab_next(&nextpoint,next);
      if ((n=atoi(next))<1) n=1;
    }
  vertlist=node_link_parse(p,nextpoint,&endptr,&hits);
  n=molify_pack_centers(p,n,vertlist);
  vert_free(&vertlist);
  return n;
} /* molify_call */

int cookie_call(struct p_data *p,char *datastr)
{
  int i,j,w,orig_nodecount,v,ind,ocount=0;
  int *newold=NULL,*oldnew=NULL;
  double angle;
  char *nextpoint;
  struct Overlaps *overlaps=NULL,*otrace,*btrace;
  struct Edgelist *trace;
  struct K_data *pK_ptr;

  pK_ptr=p->packK_ptr;
  nextpoint=datastr;
  orig_nodecount=p->nodecount;
  /* overlaps saved as linked list */
  if (p->overlap_status) 
    {
      overlaps=(struct Overlaps *)
	calloc(1,sizeof(struct Overlaps));
      otrace=overlaps;
      for (v=1;v<=p->nodecount;v++)
	for (j=0;j<(pK_ptr[v].num+pK_ptr[v].bdry_flag);j++)
	  if ( v<pK_ptr[v].flower[j]
	       && (angle=pK_ptr[v].overlaps[j])!=1.0 )
	    {
	      ocount++;
	      otrace=otrace->next=(struct Overlaps *)
		calloc(1,sizeof(struct Overlaps));
	      otrace->v=v;
	      otrace->w=pK_ptr[v].flower[j];
	      otrace->angle=angle;
	    }
      free_overlaps(p);
      if (!ocount) {free(overlaps);overlaps=NULL;} /* no overlaps */
    }

  if (!(i=cookie_cutter(p,nextpoint,&newold))) return 0;
  pK_ptr=p->packK_ptr;
  if (newold) /* set up vertex_map */
    {
      p->vertex_map=trace=(struct Edgelist *)
	calloc((size_t)1,sizeof(struct Edgelist));
      for (j=1;j<=p->nodecount;j++)
	if ((w=newold[j]))
	  {
	    trace=trace->next=(struct Edgelist *)
	      calloc((size_t)1,sizeof(struct Edgelist));
	    trace->v=j;
	    trace->w=w;
	  }
      trace=p->vertex_map;
      p->vertex_map=p->vertex_map->next;
      free(trace);
    }
  if (overlaps && alloc_overlaps(p)) /* reset saved overlaps */
    {
      oldnew=(int *)calloc((size_t)(orig_nodecount+1),sizeof(int));
      for (i=1;i<=p->nodecount;i++)
	oldnew[newold[i]]=i;
      /* change indices to their new values, throw out data no longer
	 in the new complex. Remember, first overlap is empty, but
         we need to keep it as handle to linked list as we prune it. */
      btrace=overlaps;
      otrace=overlaps->next;
      while (otrace)
	{
	  if (!(otrace->v=oldnew[otrace->v])
	      || !(otrace->w=oldnew[otrace->w]))
	    {
	      /* throw out? reset back trace ptr */
	      if (!otrace->next) 
		{
		  btrace->next=NULL;
		  free(otrace);
		  otrace=NULL; /* done */
		}
	      else
		{
		  btrace->next=otrace->next;
		  free(otrace);
		  otrace=btrace->next;
		}
	    }
	  else
	    {
	      btrace=otrace;
	      otrace=otrace->next;
	    }
	}
      if (!overlaps->next)
	{
	  free(overlaps);
	  overlaps=NULL;
	}
      else /* remove empty first position in overlaps */
	{
	  btrace=overlaps->next;
	  free(overlaps);
	  overlaps=btrace;
	}
      otrace=overlaps;
      while(otrace)
	{
	  ind=nghb(p,otrace->v,otrace->w);
	  pK_ptr[otrace->v].overlaps[ind]=otrace->angle;
	  if (!pK_ptr[otrace->v].bdry_flag && ind==0)
	    pK_ptr[otrace->v].overlaps[pK_ptr[otrace->v].num]=otrace->angle;
	  ind=nghb(p,otrace->w,otrace->v);
	  pK_ptr[otrace->w].overlaps[ind]=otrace->angle;
	  if (!pK_ptr[otrace->w].bdry_flag && ind==0)
	    pK_ptr[otrace->w].overlaps[pK_ptr[otrace->w].num]=otrace->angle;
	  otrace=otrace->next;
	}
      if (oldnew) free(oldnew);
    }
  if ((otrace=overlaps))
    {
      while (otrace && otrace->next) 
	{
	  overlaps=otrace->next;
	  free(otrace);
	  otrace=overlaps;
	}
      free(overlaps);
    }
  if (newold) free(newold);
  return i;
} /* cookie_call */

/* ================= diagostic calls =========================== */

int diag_call(struct p_data *p,char *datastr)
/* selecting diagnostic routines:
   c = deep combinatoric check
   i = connected interior?
   h = hanging bdry verts?
   R = red chain layout error?
   r = check red chain
   o <v> = placement data on v
   e <x> = first placement error worse than x.
   s = verts with radius < 2*(smallest radius)
 */
{
  int i,v,count=0,n;
  int pnum=pack_num(p);
  double val;
  char *nextpoint;

  nextpoint=datastr;
  if (!grab_next(&nextpoint,next)) return 1; /* no request */
  do
    {
      for (i=strlen(next);i<10;i++) next[i]='\0'; /* clear old info */
      if (next[0]!='-') return count;
      if (next[1]=='c') /* deep combinatoric check */
	{
	  if ((n=comb_deep_ck(p,&v)))
	    {
	      sprintf(msgbuf,"Diag: seems combinatorics are okay.");
	      msg();
	    }
	  else
	    {
	      sprintf(msgbuf,"Diag: deep check suggests combinatoric "
		      "error in p%d.\n Error code %d, seems to involve "
		      "vert %d.",pnum,n,v);
	      emsg();
	    }
	  count++;
	}
      if (next[1]=='i') /* connected interior? */ 
	{
	  if (!diag_int_connected(p))
	    {
	      sprintf(msgbuf,"Diag: p%d has a connected interior.",pnum);
	      msg();
	    }
	  count++;
	}
      else if (next[1]=='h') /* any hanging bdry verts? */
 	{
	  if (!diag_hanging_verts(p))
	    {
	      sprintf(msgbuf,
		"Diag: in p%d, all bdry verts have interior nghbs.",
		      pnum);
	      msg();
	    }
	  count++;
	}
      else if (next[1]=='s') /* smallest radius? */
	diag_small_rads(p);
      else if (next[1]=='R') /* check red chain layout error */
	{
	  sprintf(msgbuf,
	    "Diag: in p%d, red chain layout error %f; see "
		  "/tmp/diagnostic_log for details.",
		  pnum,check_red_layout(p));
	  emsg();
	}
      else if (next[1]=='r') /* check red chain */
	{
	  if ((n=check_red_chain(p)))
	    {
	      sprintf(msgbuf,
		"Diag: in p%d, red chain seems ok; count is %d.",
		      pnum,n);
	      msg();
	    }
	  else
	    {
	      sprintf(msgbuf,
		"Diag: in p%d, corrupt red chain; see "
		      "/tmp/diagnostic_log for details.",
		      pnum);
	      emsg();
	    }
	}
      else if (next[1]=='o' && !(v=grab_one_vert(p,&nextpoint)))
	/* feedback placement info on v */
	count+=cir_placement_data(p,v);
      else if (next[1]=='e' && grab_next(&nextpoint,next) 
	       && sscanf(next,"%lf",&val)) /* placement errors */
	count+=placement_err(p,val);
    }
  while (nextpoint!=NULL && grab_next(&nextpoint,next) );
  return count;		
} /* diag_call */

int reverse_orient_call(struct p_data *p)
/* call and process */
{
  if (!reverse_orient(p)) return 0;
  sprintf(buf,"fix -p%d -K -c",pack_num(p));
  handle_cmd(buf,&current_p);
  sprintf(buf,"fix -p%d",pack_num(p));
  handle_cmd(buf,&current_p);
  return 1;
} /* reverse_orient_call */

int puncture_call(struct p_data *p,char *datastr)
/* call and process */
{
  int hits;
  char *nextpoint,*endptr;
  struct Vertlist *vlist;

  nextpoint=datastr;
  if (!(vlist=node_link_parse(p,nextpoint,&endptr,&hits))
      || p->packK_ptr[vlist->v].bdry_flag 
      || !puncture(p,vlist->v))
    {
      if (vlist) vert_free(&vlist);
      return 0;
    }
  vert_free(&vlist);
  complex_count(p,0);
  facedraworder(p,0);
  sprintf(buf,"set_aim -p%d -d",pack_num(p));
  handle_cmd(buf,&current_p);
  fillcurves(p);
  return 1;
} /* puncture_call */

int swap_nodes_call(struct p_data *p,char *datastr)
/* call and process */
{
  int v,w,hits;
  char *nextpoint,*endptr;
  struct Vertlist *vlist;

  nextpoint=datastr;
  if (!(vlist=node_link_parse(p,nextpoint,&endptr,&hits))
      || !(vlist->next))
    return 0;
  v=vlist->v;
  w=vlist->next->v;
  vert_free(&vlist);
  if (!swap_nodes(p,v,w)) return 0;
  complex_count(p,0);
  facedraworder(p,0);
  return 1;
} /* swap_nodes_call */

int add_circles_call(struct p_data *p,char *datastr)
/* call and process */
{
  int count;

  count=add_circles(p,datastr);
  if (count)
    {
      if (!complex_count(p,0)) return 0;
      facedraworder(p,0);
      sprintf(msgbuf,"Added %d circles to p%d.",
	      count,pack_num(p));
      msg();
    }
  return count;
} /* add_circles_call */

int geom_to_e_call(struct p_data *p,char *datastr)
/* call to convert pack; -l option means to reestablish original 
radii (even though geom has changed). */
{
	int flag=0,i,pnum=pack_num(p);
	double *hold=NULL;

	if (datastr[0]=='-' && datastr[1]=='l') flag=1;
	if (flag)
	 {
		hold=(double *)malloc((p->nodecount+2)*sizeof(double));
		for (i=1;i<=p->nodecount;i++) 
		   hold[i]=p->packR_ptr[i].rad;
		if (p->hes<0) for (i=1;i<=p->nodecount;i++)
		 {
			if (hold[i]<=0) hold[i]=5.0;
			else hold[i]=(-log(hold[i]));
		 }
	 }
	geom_to_e_pack_convert(p);
	if (flag) 
	 {
		for (i=1;i<=p->nodecount;i++) 
			p->packR_ptr[i].rad=hold[i];
		free(hold);
	 }
	fillcurves(p); 
       	set_aim_default(p);
	p->screen->unitcircle=0;
	set_u_cir_flag(pnum,FALSE);
	sprintf(msgbuf,"Pack %d converted to euclidean data.   ",pnum);msg();
	strcpy(buf,p->file_name);
	strcat(buf," (eucl)\0");
	pmsg(pnum,buf);
	canvmsg(pnum);
	return 1;
} /* geom_to_e_call */

int geom_to_h_call(struct p_data *p,char *datastr)
/* call to convert pack; -l option means to reestablish original 
radii (even though geom has changed). */
{
	int flag=0,i,pnum=pack_num(p);
	double *hold=NULL;

	if (datastr[0]=='-' && datastr[1]=='l') flag=1;
	if (flag)
	 {
		hold=(double *)malloc((p->nodecount+1)*sizeof(double));
		for (i=1;i<=p->nodecount;i++) 
		   hold[i]=p->packR_ptr[i].rad;
	 }
	geom_to_h_pack_convert(p);
	if (flag) 
	 {
		for (i=1;i<=p->nodecount;i++) 
			p->packR_ptr[i].rad=hold[i];
		free(hold);
	 }
	fillcurves(p);
	set_aim_default(p);
	p->screen->unitcircle=1;
	set_u_cir_flag(pnum,TRUE);
	sprintf(msgbuf,"Pack %d converted to hyp data.   ",pnum);msg();
	strcpy(buf,p->file_name);
	strcat(buf," (hyp)\0");
	pmsg(pnum,buf);canvmsg(pnum);
	return 1;
} /* geom_to_h_call */

int geom_to_s_call(struct p_data *p,char *datastr)
/* call to convert pack; -l option means to reestablish original 
radii (even though geom has changed). */
{
	int flag=0,i,pnum=pack_num(p);
	double *hold=NULL;

	if (datastr[0]=='-' && datastr[1]=='l') flag=1;
	if (flag)
	 {
		hold=(double *)malloc((p->nodecount+1)*sizeof(double));
		for (i=1;i<=p->nodecount;i++) 
		   hold[i]=p->packR_ptr[i].rad;
		if (p->hes<0) for (i=1;i<=p->nodecount;i++)
		 {
			if (hold[i]<=0) hold[i]=5.0;
			else hold[i]=(-log(hold[i]));
		 }
	 }
	geom_to_s_pack_convert(p);
	if (flag) 
	 {
		for (i=1;i<=p->nodecount;i++) 
			p->packR_ptr[i].rad=hold[i];
		free(hold);
	 }
	fillcurves(p);
	set_aim_default(p);
	p->screen->unitcircle=0;
	set_u_cir_flag(pnum,TRUE);
	sprintf(msgbuf,"Pack %d converted to spherical data.   ",pnum);msg();
	strcpy(buf,p->file_name);
	strcat(buf," (sph)\0");
	pmsg(pnum,buf);canvmsg(pnum);
	return 1;
} /* geom_to_s_call */

int NSpole_call(struct p_data *p,char *datastr)
/* get and apply sphere normalization */
{
  int N,S,E=0,errflag=0;
  double factor=1.0,Erad=0.0;
  complex Ectr;
  char *nextpoint;
  Mobius Mob;
  struct R_data *pR_ptr;

  if (p->hes<=0) return 0;
  pR_ptr=p->packR_ptr;
  Ectr.re=Ectr.im=0.0;
  nextpoint=datastr;
  if (!(N=grab_one_vert(p,&nextpoint))
      || !(S=grab_one_vert(p,&nextpoint)) || N==S)
    return 0;
  if (grab_next(&nextpoint,next) && next[0]!='-'
      && (E=grab_one_vert(p,&nextpoint)) && E!=N && E!=S)
    {
      Ectr=pR_ptr[E].center;
      Erad=pR_ptr[E].rad;
    }
  else if (next[0]=='-' && next[1]=='t'
	   && sscanf(nextpoint,"%lf",&factor))
    {Ectr.re=Ectr.im=0.0;}
  NS_mobius(pR_ptr[N].center,pR_ptr[S].center,Ectr,
		pR_ptr[N].rad,pR_ptr[S].rad,Erad,factor,&errflag,&Mob);
  if (!errflag) 
    {
      apply_Mobius(p,"a",1,Mob);
      return 1;
    }
  else return 0;
} /* NSpole_call */

int rect_ratio_call(struct p_data *p,char *datastr)
/* Report the ratio length/height of a eucl packing, presumably 
in form of a rectangle. Give the four corners in counterclockwise 
order, lower-left first. Return values:
  0: error
  1; okay
  -1: data of lengths/heights aren't consistent with a rectangle 
  (off by > 5%). */
{
  int val,pnum=pack_num(p);
  double aspect;

  val=rect_ratio(p,datastr,&aspect);
  if (!val) return 0;
  if (val<0)
    {
      sprintf(msgbuf,"rect_ratio: p%d doesn't seem to be a rectangle.",
	      pnum);
      emsg();
      return 0;
    }
  sprintf(msgbuf,"In p%d, length/height=%f.",pnum,aspect);
  msg();
  return 1;
} /* rect_ratio_call */

int riffle_call(struct p_data *p,int passes,int lpr,int flag)
/* call to compute radii of packing p. flag indicates which
   routine to use: 0=old reliable, 1=superstep, 2=superstep2 (only
   eucl for now). */
{
  int count,hr,min,sec;
  long tic,toc;

  if (!cmd_mode) xv_set(canvas_frame[pack_num(p)],FRAME_BUSY,TRUE,0);
  tic=get_time();
  if (!flag) /* old reliable */
    {
      passes *= p->intnode;
      if (p->hes<0) count=old_h_riffle(p,passes);
      else if (p->hes==0) count=old_e_riffle(p,passes);
      else count=0; /* spherical algorithm not yet working */
    }
  else if (flag==1) /* superstep */
    {
      if (p->hes<0) count=h_riffle(p,passes);
      else if (p->hes==0) count=e_riffle(p,passes);
      else count=0; /* spherical algorithm not yet working */
    }
  else if (flag==2 && p->hes==0)
    {
      count=new_e_riffle(p,passes);
    }
  if (count && lpr)
    {
      toc=get_time();
      time_to_hms(toc-tic,&hr,&min,&sec);
      sprintf(msgbuf," RIFFLED p%d; passes=%d; "
	      "time(h,m,s) (%d,%d,%d). ",pack_num(p),count,hr,min,sec);
      msg();
    }
  if (!cmd_mode) xv_set(canvas_frame[pack_num(p)],FRAME_BUSY,FALSE,0);
  return count;
} /* riffle_call */

int gen_mark_call(struct p_data *p,char *datastr)
     /* call routine which records generations of vertices
	as measured from 'seeds' (specified in datastr). Records
	it in 'mark' and returns the last vertex marked. Command
	'-m n' tells it to stop at max generation n. */
{
  int last_vert;

  if ((last_vert=gen_mark(p,datastr)))
    {
      sprintf(msgbuf,"gen_mark: last vertex marked = %d and its "
	      "generation is %d.",
	      last_vert,p->packK_ptr[last_vert].mark);
      msg();
      return last_vert;
    }
  else
    {
      sprintf(msgbuf,"Usage: gen_mark [-m n] <v..>");
      emsg();
      return 0;
    }
} /* gen_mark_call */

int gen_cut_call(struct p_data *p,char *datastr)
/* cut out a simply connected subcomplex based on generations
   from a specified vertex (which is generation 1). 
   fixup: currently, only works for simply connected. */
{
  int v,gen;

  if (!p->status || p->locks) return 0;
  if (p->genus!=0 || (p->euler!=1 && p->euler!=2))
    {
      sprintf(msgbuf,"gen_cut: works only for simply connnected "
	      "complexes.");
      msg();
      return 1;
    }
  if (sscanf(datastr,"%d %d",&v,&gen)!=2
      || v<1 || v>p->nodecount || gen<2) return 0;
  else return (gen_cut(p,v,gen));
} /* gen_cut_call */

int spiral_call(int pnum,char *datastr)
/* call Doyle spiral routine */
{
  double a,b;
  struct p_data *p;

  p=&packdata[pnum];
  if (!p->status || sscanf(datastr,"%lf%lf",&a,&b)!=2 ||
      a<=0 || b<=0 || !spiral(p,a,b))
    {
      strcpy(msgbuf,"Data form: a b. Try again.");
      emsg();
      return 0;
    }
  return 1;
} /* spiral_call */

int marking(struct p_data *p,char *datastr)
/* set mark on circles or faces */
{
  int count=0,i,vert,flg,hits,*gen_list,tick,nv,dum;
  char *nextpoint,*endptr;
  struct Vertlist *vertlist,*trace,*facelist;
  struct K_data *pK_ptr;
  
  pK_ptr=p->packK_ptr;
  nextpoint=datastr;
  if (!grab_next(&nextpoint,next) || next[0]!='-' 
      || !p->status) return 1;
  do
    {
if (next[0]!='-') return count;
else if (next[1]=='w') /* wipe out all marks */
 {
	for (i=1;i<=p->nodecount;i++) pK_ptr[i].mark=0;
	for (i=1;i<=p->facecount;i++) p->faces[i].mark=0;
	count++;
 }
else if (next[1]=='g') /* mark circles by generations from vert */
  /* vert is gen 1 */
  {
    if (!nextpoint || !(vert=grab_one_vert(p,&nextpoint)))
	vert=p->alpha; /* alpha = default */
    hits=0;
    if ((gen_list=label_seed_generations(p,vert,NULL,0,&hits,&dum)))
      {
	for (i=1;i<=p->nodecount;i++) 
	  pK_ptr[i].mark=gen_list[i];
	free(gen_list);gen_list=NULL;
      }
  }
else if (next[1]=='c' && next[2]=='o'
	 && vert_draw_order(p))          /* mark by drawing order */ 
  {
    tick=1;
    pK_ptr[p->alpha].mark=tick++;
    nv=pK_ptr[p->alpha].next_vert;
    while (nv>0 && nv!=p->alpha)
      {
	pK_ptr[nv].mark=tick++;
	nv=pK_ptr[nv].next_vert;
	count++;
      }
  }
else if (next[1]=='c') /* mark circles */
 {
	flg= (next[2]=='w'); /* 1==>wipe out */
	if ((vertlist=node_link_parse(p,nextpoint,
		&endptr,&hits))!=NULL)
	 {
		nextpoint=endptr;
		trace=vertlist;
		do
		 {
			vert=trace->v;
			if (flg) pK_ptr[vert].mark=0;
			else pK_ptr[vert].mark++;
			count++;
			trace=trace->next;
		 } while (trace!=NULL);
		vert_free(&vertlist);
	 }
	else if (flg)
	 {
		for (i=1;i<=p->nodecount;i++) pK_ptr[i].mark=0;
		count++;
	 }
 }
else if (next[1]=='f') /* mark faces */
 {
	flg= (next[2]=='w'); /* 1==>wipe out */
	if ((facelist=face_link_parse(p,nextpoint,&endptr,&hits))!=NULL)
	 {
		nextpoint=endptr;
		trace=facelist;
		do
		 {
			vert=trace->v;
			if (flg) p->faces[vert].mark=0;
			else p->faces[vert].mark++;
			count++;
			trace=trace->next;
		 } while (trace!=NULL);
		vert_free(&facelist);
	 }
	else if (flg)
	 {
		for (i=1;i<=p->facecount;i++) p->faces[i].mark=0;
		count++;
	 }
 }
	 } /* end of do */
	while (nextpoint!=NULL && grab_next(&nextpoint,next) );
	return count;		
} /* marking */

int norm_scale_call(struct p_data *p,char *datastr)
     /* normalize a eucl packing in various ways */
{
  int v,w,qnum;
  double factor,rad;
  char *nextpoint;

  nextpoint=datastr;
  if ( !p->status || p->hes>0 || p->hes<0) return 0;
  if (!grab_next(&nextpoint,next) || next[0]!='-') return 0;
  if (next[1]=='e')     /* scale p so rad of v in p equals
				rad of w in pack q */
    {
      if (sscanf(nextpoint,"%d %d %d",&qnum,&v,&w)!=3
	  || qnum<0 || qnum>=NUM_PACKS || !packdata[qnum].status
	  || v<1 || v>p->nodecount
	  || w<1 || w>packdata[qnum].nodecount
	  || (rad=p->packR_ptr[v].rad)<=okerr) return 0;
      factor=packdata[qnum].packR_ptr[w].rad/rad;
      return eucl_scale(p,factor);
    }
  return e_norm_scale(p,datastr); /* library routine handles rest
				   of options */
} /* norm_scale_call */

int set_aim_call(struct p_data *p,char *datastr)
/* Set the intended target angle sums. */ 
{ 
  int count=0,mode;
  char *nextpoint;

  nextpoint=datastr;
  grab_next(&nextpoint,next);
  if (next[0]=='-' && next[1]=='d') mode=0; /* default mode */
  else if (next[0]=='-' && next[1]=='c') mode=1;/* current mode */
  else if (next[0]=='-' && next[1]=='%') mode=2; /* add inc mode */
  else mode=3;
  if ((count=set_aims(p,datastr)))
    {
      if (mode==0) 
	sprintf(msgbuf,"Set %d `aims' of pack %d to default.",
		count,pack_num(p));
      else if (mode==1)
	sprintf(msgbuf,"Set %d `aims' of pack %d to "
		"their current angle sums.",
		count,pack_num(p));
      else if (mode==2)
	sprintf(msgbuf,"Incremented %d `aims' of pack %d.",
		count,pack_num(p));
      else if (mode==3) 
	sprintf(msgbuf,"Set %d `aims' of pack %d.",
		count,pack_num(p));
      msg();
      return count;
    }
  strcpy(msgbuf,"No aims adjusted.");
  emsg();
  return 0;
} /* set_aim_call */

int randize(struct p_data *p,char *datastr)
/* set random radii (default) or overlaps (or random changes) */
{ 
	int mode=0,count=0,vert,hits,flag=0,vv,ww,k;
	double rn,rand_value,lowr,highr,tmprad;
	char *nextptr,*lastptr,*endptr;
	struct Vertlist *vertlist=NULL,*trace;
	struct Edgelist *edgelist=NULL,*etrace;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	lowr=10.0*okerr;highr=10; /* default radii range */
	nextptr=lastptr=datastr;
	grab_next(&nextptr,next);
	if (!strncmp(next,"-o",2)) /* overlaps, not radii */
	 {
		flag=1;
		lowr=0.0; /* default overlap range */
		grab_next(&nextptr,next);
	 }
	do
	 {
		if (next[0]=='-') 
		 {
			if (next[1]=='r') /* range */
			 {
				if (sscanf(nextptr,"%lf %lf",&lowr,&highr)!=2
					|| highr<lowr) 
				 {
					if (count) fillcurves(p);
					return count;
				 }
				grab_next(&nextptr,next);
				grab_next(&nextptr,next); /* move past data */
				if (mode==1)
				 {
					if (lowr<okerr) lowr=okerr;
					if (highr<lowr) highr=2*lowr;
				 }
			 }
			else if (next[1]=='f'
			   && (!flag || p->overlap_status)) 
					/* increment (treat as factor) */
			 {
				mode=1;
				if (lowr<okerr) lowr=okerr;
				if (highr<lowr) highr=2*lowr;
			 }
			else if (!flag && next[1]=='j')
					/* jiggle radii */
			 {
				mode=1;
				lowr=1-10*okerr;
				highr=1+10*okerr;
			 }
			lastptr=nextptr;
		 }
		else if (!flag) /* adjust radii */
		 {
			if ((vertlist=node_link_parse(p,
				lastptr,&endptr,&hits))==NULL)
			 {
				if (count) fillcurves(p);
				return count;
			 } 
					/* default = no action */
			nextptr=lastptr=endptr;
			trace=vertlist;
do
 {
	vert=trace->v;
	rn=gwrand(); /* pseudo-random in (0,1) */
	rand_value=lowr+(highr-lowr)*rn;
	if (mode==1) /* multiply by factor */
	 {
		if (p->hes>0) /* sphere */
		 {
			pR_ptr[vert].rad *= rand_value;
			if (pR_ptr[vert].rad>=M_PI)
				pR_ptr[vert].rad=M_PI-okerr;
			count++;
		 }
		if (p->hes<0) /* hyperbolic */
		 {
			if (pR_ptr[vert].rad < 0) /* infinite radius */
			 {
				if (rand_value < 1.0) /* decrease to .5 */
				   pR_ptr[vert].rad=.5;
			 }
			else
			 {
				tmprad=exp(log(pR_ptr[vert].rad)*rand_value); 
				if (tmprad < .0001) pR_ptr[vert].rad = -.1;
					/* infinite */
				else pR_ptr[vert].rad = tmprad;			
					/* s-radii */
			 }
			count++;
		 }
		else /* eucl */
		 {
			pR_ptr[vert].rad *= rand_value;
			count++;
		 }
	 }
	else
	 {
		if (p->hes>0)
		 {
			pR_ptr[vert].rad = rand_value;
			if (pR_ptr[vert].rad>=M_PI)
				pR_ptr[vert].rad=M_PI-okerr;
			count++;
		 }
		if (p->hes<0) /* hyperbolic */
		 {
			tmprad= exp((-1.0)*rand_value);	
			if (tmprad < .0001) pR_ptr[vert].rad = -.1;
				/* infinite */
			else pR_ptr[vert].rad = tmprad;			
			count++;
		 }
		else 
		 {
			pR_ptr[vert].rad = rand_value;
			count++;
		 }
	 }
	trace=trace->next;
 } /* end of inner do */
			while (trace!=NULL);
			vert_free(&vertlist);
			if (count) fillcurves(p);
			return count;
		 }
		else /* adjust overlaps */
		 {
		   if ((edgelist=node_pair_link(p,
			lastptr,&endptr,&hits))==NULL)
		    {
			if (count) fillcurves(p);
			   return count;
		    }   /* default = no action */
		   alloc_overlaps(p);
		   nextptr=lastptr=endptr;
		   etrace=edgelist;			
		   do
		    {
			vv=etrace->v;
			ww=etrace->w;
			rn=gwrand(); /* pseudo-random in (0,1) */
			rand_value=lowr+(highr-lowr)*rn;
			if ((k=nghb(p,vv,ww))>=0)
			 {
			   if (mode==1) /* factor */
			      set_overlap(p,vv,k,
				rand_value*p->packK_ptr[vv].overlaps[k]);
			   else set_overlap(p,vv,k,rand_value);
			 }
			etrace=etrace->next;
		    } /* end of inner do */
		   while (etrace!=NULL);
		   edge_free(&edgelist);
		   if (count) fillcurves(p);
		   return count;
		 }
	 } /* end of do */
	while ((lastptr=nextptr)!=NULL && grab_next(&nextptr,next) );
	if (count) fillcurves(p);
	return count;		
} /* randize */

int max_pack(struct p_data *p,char *datastr)
/* shortcut for doing maximal packing */
{
  int n,pnum;

  pnum=pack_num(p);
  if (p->intrinsic_geom < 0) /* hyperbolic case */
    {
      if (p->hes >=0)
	{
	  sprintf(buf,"geom_to_h -p%d",pnum);
	  handle_cmd(buf,&current_p);
	}
      sprintf(buf,"set_aim -p%d -d",pnum);
      handle_cmd(buf,&current_p);
      sprintf(buf,"set_rad -p%d -.1 b",pnum);
      handle_cmd(buf,&current_p);
      sprintf(buf,"repack -p%d",pnum);
      handle_cmd(buf,&current_p);
      sprintf(buf,"fix -p%d",pnum);
      handle_cmd(buf,&current_p);
      return 1;
    }
  if (p->intrinsic_geom == 0) /* must be 1-torus */
    {
      if (p->hes < 0 || p->hes > 0)
	{
	  sprintf(buf,"geom_to_e -p%d",pnum);
	  handle_cmd(buf,&current_p);
	}
      sprintf(buf,"set_aim -p%d -d",pnum);
      handle_cmd(buf,&current_p);
      sprintf(buf,"repack -p%d",pnum);
      handle_cmd(buf,&current_p);
      sprintf(buf,"fix -p%d",pnum);
      handle_cmd(buf,&current_p);
      return 1;
    }
  if (p->intrinsic_geom > 0) /* sphere */
    {
      if (sscanf(datastr,"%d",&n) && n>0 && n<= p->nodecount)
	/* can specify node at which to puncture, else max node */
	{
	  sprintf(buf,"swap -p%d %d %d",pnum,n,p->nodecount);
	  handle_cmd(buf,&current_p);
	}
      else n=p->nodecount;
      sprintf(buf,"puncture -p%d %d",pnum,p->nodecount);
      handle_cmd(buf,&current_p);
      if (p->hes >=0)
	{
	  sprintf(buf,"geom_to_h -p%d",pnum);
	  handle_cmd(buf,&current_p);
	}
      sprintf(buf,"set_aim -p%d -d",pnum);
      handle_cmd(buf,&current_p);
      sprintf(buf,"set_rad -p%d -.1 b",pnum);
      handle_cmd(buf,&current_p);
      sprintf(buf,"repack -p%d",pnum);
      handle_cmd(buf,&current_p);
      sprintf(buf,"fix -p%d",pnum);
      handle_cmd(buf,&current_p);
      sprintf(buf,"geom_to_s -p%d",pnum);
      handle_cmd(buf,&current_p);
      sprintf(buf,"add_ideal -p%d",pnum);
      handle_cmd(buf,&current_p);
      if (n != p->nodecount)
	{
	  sprintf(buf,"swap -p%d %d %d",pnum,n,p->nodecount);
	  handle_cmd(buf,&current_p);
	}
      if (n==p->alpha || nghb(p,n,p->alpha) >= 0)
	n = p->gamma; /* try to avoid normalization problem */
      sprintf(buf,"NSpole -p%d %d %d",pnum,p->alpha,n);
      handle_cmd(buf,&current_p);
      return 1;
    }
  return 0;
} /* max_pack */

int reset_screen(struct s_data *q)
     /* call for resenting screen box and panel settings */
{
  int snum=screen_num(q);

  if (cmd_mode) return 1;
  set_u_cir_flag(snum,FALSE);
  set_coord_flag(snum,FALSE);
  if (packdata[snum].hes<0)
    {
      q->unitcircle=1;
      set_u_cir_flag(snum,TRUE);
    }
  xv_set(sfac[snum],PANEL_VALUE,0,0);
  return (reset_screen_box(q,
			   (int)xv_get(canvas_frame[snum],XV_WIDTH),
			   (int)xv_get(canvas_frame[snum],XV_HEIGHT)));
} /* reset_screen */

int enfoldcall(struct p_data *p,char *datastr)
{
  int n,m,vert,i,count=0,enfold_flag=0,hits;
  struct K_data *pK_ptr;
  char *node_ptr,*endptr;
  struct Vertlist *vertlist,*trace;

  pK_ptr=p->packK_ptr;
  node_ptr=datastr;
  if (!grab_next(&node_ptr,next) || sscanf(next,"%d",&n)!=1
      || n<0 || n>(MAX_PETALS-3)) return 0;
  if ( (vertlist=node_link_parse(p,node_ptr,&endptr,&hits)) != NULL)
    {
      trace=vertlist;
      do {
	vert=trace->v;
	if (pK_ptr[vert].bdry_flag)
	  {
	    m=MAX_PETALS-pK_ptr[vert].num-1;
	    m=(n<m)? n:m;
	    for (i=1;i<=m;i++) add_vert(p,vert);
	    enfold_flag=enfold(p,vert);
	    count++;
	  }
	trace=trace->next;
      } while (trace!=NULL && enfold_flag);
      vert_free(&vertlist);
    }
  if (count)
    {
      if (!pK_ptr[p->beta].bdry_flag) choose_beta(p);
      complex_count(p,FALSE);
      facedraworder(p,0);
      fillcurves(p);
    }
  return count;
} /* enfoldcall */

int adjoincall(char *datastr)
     /* return 0 on error */
{
  int pnum1,pnum2,v1,v2,n,n1,n2,*oldnew=NULL;
  int v,j,vb,vert,b1,b2,vv,ww,offset=0,over_flag=0;
  double angle;
  struct p_data *p1,*p2;
  struct K_data *pK_ptr1;
  struct Overlaps *overlaps=NULL,*trace;
  struct Vertlist *vtrace;
  struct Edgelist *etrace;
  char *nextpoint,*newstr;

  nextpoint=datastr;
  if (!grab_next(&nextpoint,next) || !sscanf(next,"%d",&pnum1)
      || pnum1<0 || pnum1>=NUM_PACKS
      ||!grab_next(&nextpoint,next) || !sscanf(next,"%d",&pnum2)
      || pnum2<0 || pnum2>=NUM_PACKS
      || !grab_next(&nextpoint,next) || !sscanf(next,"%d",&v1)
      || !grab_next(&nextpoint,next) || !sscanf(next,"%d",&v2)
      || !grab_next(&nextpoint,next))
    return 0;
  n=0;
  newstr=(char *)calloc((size_t)strlen(next),sizeof(char));
  if ((next[0]!='(' && !sscanf(next,"%d",&n))
      || (next[0]=='(' && (!strncpy(newstr,next,strlen(next))
			   || !paren_parse(&newstr,&b1,&b2)))) return 0;
  p1=&packdata[pnum1];p2=&packdata[pnum2];
  pK_ptr1=p1->packK_ptr;
  if (!n) /* get n from 'paren' style input */
    {
      if (b1<=0 || b2>p1->nodecount 
	  || !pK_ptr1[b1].bdry_flag) return 0;
      n=1;
      vert=b1;
      vb=pK_ptr1[vert].flower[pK_ptr1[vert].num];
      while (vb!=b2 && vb!=vert && n< p1->nodecount)
	{
	  vb=pK_ptr1[vb].flower[pK_ptr1[vb].num];
	  n++;
	}
      if (n>=p1->nodecount) return 0;
    }
  if (pnum1!=pnum2) offset=p1->nodecount;
  if ((n1=bdry_comp_count(p1,v1))<n || n<0) 
    /* signal to identify as many edges as possible */
    /* fixup: needs work for all possible cases ?? */
    {
      if (n1<1 || (n2=bdry_comp_count(p2,v2))<1)
	return 0;
      if (n1==n2) n=n1-1;
      else n= (n1<n2) ? n1:n2;
    }

  if ( p1->overlap_status || p2->overlap_status )
    over_flag=1;

  /* overlaps of pnum1 saved as linked list */
  if (p1->overlap_status) 
    {
      overlaps=(struct Overlaps *)
	calloc(1,sizeof(struct Overlaps));
      trace=overlaps;
      for (v=1;v<=p1->nodecount;v++)
	for (j=0;j<(pK_ptr1[v].num+pK_ptr1[v].bdry_flag);j++)
	  {
	    if ( v<pK_ptr1[v].flower[j]
		 && (angle=pK_ptr1[v].overlaps[j])!=1.0 )
	      {
		trace->v=v;
		trace->w=pK_ptr1[v].flower[j];
		trace->angle=angle;
		trace=trace->next=(struct Overlaps *)
		  calloc(1,sizeof(struct Overlaps));
	      }
	  }
    }

  /* now try to adjoin */
  if (adjoin(p1,p2,v1,v2,n,&oldnew))
    {
      complex_count(p1,TRUE);

      if (p1->flist) vert_free(&p1->flist);
      if (pnum1==pnum2 && p1->vlist)
	{
	  vtrace=p1->vlist;
	  while (vtrace)
	    {
	      vtrace->v=oldnew[vtrace->v];
	      vtrace=vtrace->next;
	    }
	}
      if (pnum1==pnum2 && p1->elist)
	{
	  etrace=p1->elist;
	  while (etrace)
	    {
	      etrace->v=oldnew[etrace->v];
	      etrace->w=oldnew[etrace->w];
	      etrace=etrace->next;
	    }
	}

      /* restablish saved overlaps */
      if(over_flag && alloc_overlaps(p1))
	{
	  if (!offset && overlaps) /* self-adjoin, use new indices */
	    {
	      trace=overlaps;
	      while (trace && trace->next!=NULL)
		{
		  vv=oldnew[trace->v];
		  ww=oldnew[trace->w];
		  set_overlap(p1,vv,nghb(p1,vv,ww),trace->angle);
		  trace=trace->next;
		  free(overlaps);
		  overlaps=trace;
		}
	      free(overlaps);
	    }
	  else if (overlaps) /* reset pnum1 overlaps */
	    {
	      trace=overlaps;
	      while (trace && trace->next!=NULL)
		{
		  set_overlap(p1,trace->v,
			      nghb(p1,trace->v,trace->w),trace->angle);
		  trace=trace->next;
		  free(overlaps);
		  overlaps=trace;
		}
	      free(overlaps);
	      overlaps=NULL;
	    }
	  if ( offset && p2->overlap_status ) 
	    /* new overlaps from p2? */		
	    {
	      for(v=1;v<=p2->nodecount;v++)
		{
		  vv=oldnew[v];
		  for(j=0;j<p2->packK_ptr[v].num+
			p2->packK_ptr[v].bdry_flag;j++)
		    if (v<p2->packK_ptr[v].flower[j])
		      {
			ww=oldnew[p2->packK_ptr[v].flower[j]];
			if ((angle=p2->packK_ptr[v].overlaps[j])!=1.0)
			  set_overlap(p1,vv,nghb(p1,vv,ww),angle);
		      }
		}
	    }
	}


      fillcurves(p1);
      facedraworder(p1,0);
      set_aim_default(p1);
      p1->active_node=p1->beta; 
      sprintf(msgbuf,"adjoin %d %d %d %d %d;",
	      pnum1,pnum2,v1,v2,n);
      if (oldnew) free(oldnew);
      if ((trace=overlaps))
	{
	  while (trace && trace->next) 
	    {
	      overlaps=trace->next;
	      free(trace);
	      trace=overlaps;
	    }
	  free(overlaps);
	}
      msg();return 1;
    }
  strcpy(msgbuf,"Something went wrong with 'adjoin' routine.");
  emsg();
  if (oldnew) free(oldnew);
  if ((trace=overlaps))
    {
      while (trace && trace->next) 
	{
	  overlaps=trace->next;
	  free(trace);
	  trace=overlaps;
	}
      free(overlaps);
    }
  return 0;
} /* adjoincall */

int color_circles(struct p_data *p,int f,char *datastr)
{
  int cp,i,ccode,hits;
  double base;
  struct K_data *pK_ptr;
  struct Vertlist *vertlist,*trace;
  char *nextpoint,*endptr;

  pK_ptr=p->packK_ptr;
  nextpoint=datastr;
  if (!p->status) return 0;
  switch(f)
    {
    case 5: /* fill with foreground */
      {
	for (i=1;i<=p->nodecount;i++) pK_ptr[i].color=FG_COLOR;
 	return 1;
      }
    case 4: /* fill with background */
      {
	for (i=1;i<=p->nodecount;i++) pK_ptr[i].color=BG_COLOR;
 	return 1;
      }
    case 1: /* color by radius */
      {
	return (radius_col_comp(p));
      }
    case 3: /* color by comparison */
      {
	if (!sscanf(nextpoint,"%d",&cp) || cp<0 || cp>=NUM_PACKS
	    || !packdata[cp].status
	    || packdata[cp].hes!=p->hes) return 0;
	if (p->hes<0) {h_compare_cl(p,&packdata[cp]);return 1;}
	else if (p->hes==0) {e_compare_cl(p,&packdata[cp]);return 1;}
	return 0;
      }
    case 6: /* specified color, circles */
      {
	if (!grab_next(&nextpoint,next) || !sscanf(next,"%d",&ccode) 
	    || ccode<0 || ccode>255) return 0;
	if ((vertlist=node_link_parse(p,nextpoint,&endptr,&hits))==NULL) 
	  return 1;
	trace=vertlist;
	while (trace!=NULL)
	  {
	    pK_ptr[trace->v].color=ccode;
	    trace=trace->next;
	  }
	vert_free(&vertlist);
	return 1;	
      }
    case 7: /* take colors from pack cp */
      {
	if (!sscanf(nextpoint,"%d",&cp) || cp<0 || cp>=NUM_PACKS
	    || !packdata[cp].status) cp=current_p;
	for (i=1;i<=p->nodecount;i++)
	  if (i<=packdata[cp].nodecount) 
	    pK_ptr[i].color=packdata[cp].packK_ptr[i].color;
	return 1;
      }
    case 8: /* compare model curvature to value: red=pos, blue=neg */
      {
	if (!sscanf(nextpoint,"%lf",&base)) base=0.0;
	else
	  {
	    if (base<=0) base=(-1.0)*sqrt((-1.0)*base);
	    else base=sqrt(base);
	  }
	color_kappa(p,base);
      }
    } /* end of switch */
  return 0;
} /* color_circles */
	
int color_faces(struct p_data *p,int f,char *datastr)
{
  int cp,i,ccode,hits;
  struct Vertlist *vertlist,*trace;
  char *nextpoint,*endptr;

  if (!p->status) return 0;
  nextpoint=datastr;
  switch(f)
    {
    case 5: /* foreground */
      {
	for (i=1;i<=p->facecount;i++) p->faces[i].color=FG_COLOR;
 	return 1;
      }
    case 4: /* background */
      {
	for (i=1;i<=p->facecount;i++) p->faces[i].color=BG_COLOR;
 	return 1;
      }
    case 1: /* color by area */
      {
	return (area_col_comp(p));
      }
    case 3: /* color by area comparison */
      {
	if (!sscanf(nextpoint,"%d",&cp) || cp<0 || cp>=NUM_PACKS
	    || !packdata[cp].status
	    || packdata[cp].hes!=p->hes) return 0;
	if (p->hes<0) {h_compare_area(p,&packdata[cp]);return 1;}
	else if (p->hes==0) {e_compare_area(p,&packdata[cp]);return 1;}
	return 0;
      }
    case 6: /* specified color, faces */
      {
	if (!grab_next(&nextpoint,next) || !sscanf(next,"%d",&ccode) 
	    || ccode<0 || ccode>199) return 0;
	if ((vertlist=face_link_parse(p,nextpoint,&endptr,&hits))==NULL) 
	  return 1;
	trace=vertlist;
	while (trace!=NULL)
	  {
	    p->faces[trace->v].color=ccode;
	    trace=trace->next;
	  }
	vert_free(&vertlist);
	return 1;	
      }
    case 7: /* take colors from pack cp */
      {
	if (!sscanf(nextpoint,"%d",&cp) || cp<0 || cp>=NUM_PACKS
	    || !packdata[cp].status) cp=current_p;
	for (i=1;i<=p->facecount;i++)
	  if (i<=packdata[cp].facecount) 
	    p->faces[i].color=packdata[cp].faces[i].color;
	return 1;
      }
    } /* end of switch */
  return 0;
} /* color_faces */

int map_list_call(struct p_data *p,char *datastr)
/* to form Vlist or Flist using the 'vertex_map' of pack p to 
translate indices. For faces, can only do this if target pack 
is specified with -q flag. Flag -1 means to translate using the 
inverse of vertex_map. Get settings and call map_list. */
{
  int mode=1,qnum=-1,rev_flag=0;
  char *nextpoint;
  struct p_data *q;

  if (!p->status || !p->vertex_map) return 0;
  nextpoint=datastr;
  q=p;
  if (grab_next(&nextpoint,next) && next[0]=='-')
    {
      if (next[1]=='f') mode=2; /* faces */
      else if (next[1]=='c') mode=1; /* circles */
      else if (next[1]=='q') /* map to pack qnum */
	{
	  if (sscanf(next+2,"%d",&qnum) && qnum>=0 && qnum<NUM_PACKS
	      && packdata[qnum].status) 
	    q=&packdata[qnum];
	}
      else if (next[1]=='1') rev_flag=1; /* use inverse map */
    }
  else nextpoint=datastr;
  return (map_list(p,q,nextpoint,mode,rev_flag));
  /* rest of datastr for specifying indices */
} /* map_list_call */

int adjacency_call(struct p_data *p,char *datastr)
{
  int n;
  char *nextpoint,filename[NAME_MAX];
  FILE *fp;

  filename[0]='\0';
  set_packing_path();
  strcpy(filename,path);
  nextpoint=datastr;
  if (!grab_next(&nextpoint,next)) return 0; /* no filename? */
  strcpy(filename,next);
  if (!(fp=fopen(filename,"w"))) return 0;
  n=adjacency(fp,p);
  fclose(fp);
  return n;
} /* adjacency_call */

int write_light_call(struct p_data *p,char *datastr,int flag)
     /* write pack data out to file in p_light form */
{
  int i;
  char *nextpoint,filename[NAME_MAX];
  struct K_data *pK_ptr;
  struct p_light *pl=NULL;
  FILE *fp;

  pK_ptr=p->packK_ptr;
  nextpoint=datastr;
  if (!grab_next(&nextpoint,next)) /* no filename? */
    goto filename_error;
  filename[0]='\0';
  set_packing_path();
  strcpy(buff,path);
  strcat(buff,next);
  strcpy(filename,buff); /* now have full pathname */
  for (i=1;i<=p->nodecount;i++) /* mark interior */
    {
      if (pK_ptr[i].bdry_flag) pK_ptr[i].util_flag=0;
      else pK_ptr[i].util_flag=1;
    }
  if (!(fp=fopen(filename,"w")) 
      || !(pl=convert_to_p_light(p)) 
      || !write_light(fp,pl,flag))
    {
      if (fp) fclose(fp);
      if (pl) free_p_light(&pl);
      return 0;
    }
  free_p_light(&pl);
  fclose(fp);
  return 1;

filename_error: 
  sprintf(buf,"Need filename with the 'write_light' command.");
  emsg();
  return 0;
} /* write_light_call */

int write_parse(struct p_data *p,char *datastr)
/* parse options for writing packings. Caution, no confirmation 
   requested. Default is cgirzv. Note: all options in one
   continuous string directly after '-'.

A append (requires additional options)

c combinatorics
g geometry
i non-default inv_dist and aims
r radii (g needed)
z centers
a angle sums
v vertex_map
l non-default lists: verts/faces/edges
o non-default colors
f non-zero plot_flags
t edge-pairing mobius

m minimum = cgri
M max = cgriazsoflv
*/
{
  int act=00137,i,n,c,append_flag=0,script_flag=0;
  char *nextpoint,filename[NAME_MAX];
  long fspot;
  FILE *fpp,*tmpfp;

  nextpoint=datastr;
  if (!grab_next(&nextpoint,next)) /* no filename? */
    goto filename_error;
  filename[0]='\0';
  set_packing_path();
  if (next[0]=='-') /* here come the options */
    {
      act=0;
      i=1;
      while ((c=next[i++])!='\0')
	{
	  switch(c)
	    {
	    case 'A': {append_flag=1;break;}
	    case 'c': {act |= 00001;break;}
	    case 'g': {act |= 00002;break;}
	    case 'i': {act |= 00004;break;}
	    case 'r': {act |= 00010;break;}
	    case 'z': {act |= 00020;break;}
	    case 'a': {act |= 00040;break;}
	    case 'v': {act |= 00100;break;}
	    case 'l': {act |= 00200;break;}
	    case 'o': {act |= 00400;break;}
	    case 'f': {act |= 01000;break;}
	    case 't': {
	      if (p->edge_pair[1].edge) act |= 02000;
	      break;
	     }

	    case 'm': {act |= 00017;break;}
	    case 'M': {act |= 01777;break;}
	    case 'S': {script_flag=1;break;}
	    }
	} /* done with - option */
      if (!grab_next(&nextpoint,next)) /* get filename */
	goto filename_error;
      if (append_flag)
	{
	  if (!act) return 0; /* 'append' option needs additional options */
	  strcpy(filename, next);
	  strcpy(buff,path);
	  strcat(buff,filename);
	  strcpy(filename,buff); /* now have full pathname */
	  if ((fpp=fopen(filename,"r+"))!=NULL 
	      && position_before(fpp,"END")
	      && (tmpfp=fopen("/tmp/hold_tail.ps","w+")))
	    /* open and position named file for read/write */
	    {
	      fspot=ftell(fpp);
	      while ((c=getc(fpp))!=EOF) putc(c,tmpfp);
	      fseek(fpp,fspot,SEEK_SET);
      	      n=writepack(fpp,p,act,1);
	      fprintf(fpp,"\n");
	      rewind(tmpfp);
	      while ((c=getc(tmpfp))!=EOF) putc(c,fpp);
	      if (tmpfp) fclose(tmpfp);
	      if (fpp) fclose(fpp);
	      return n;
	    }
	  else
	    {
	      if (fpp) fclose(fpp);
	      sprintf(msgbuf,"Couldn't open %s for 'append'.",
		      filename);
	      emsg();
	      return 0;
	    }
	}
    }

  /* write to script file: pack data follows 'name' */
  /* NOTE: this option seems to run into xview(?) error. */
  /*  else if (script_flag) 
	{
	  pos=(Textsw_index)xv_get(script_sw,
	      TEXTSW_INSERTION_POINT);
	  if ((return_flag=writecall(p,"/tmp/script-hold.p",j)))
	    {
	      xv_set(script_sw,TEXTSW_INSERTION_POINT,
		     TEXTSW_INFINITY,NULL);
	      sprintf(buf,"\n\n%s \n",filename);
	      textsw_insert(script_sw,buf,strlen(buf));
	      xv_set(script_sw,TEXTSW_STATUS,&sw_status,
		     TEXTSW_INSERT_FROM_FILE,
		     "/tmp/script-hold.p",NULL);
	      if (sw_status!=TEXTSW_STATUS_OKAY) return_flag=0;
	      xv_set(script_sw,TEXTSW_INSERTION_POINT,pos,NULL);
	    }
	  if (return_flag==0)
	    {
	      sprintf(buf,
		"Intermediate write to or from /tmp/script-hold.p failed.");
	      emsg();
	      return 0;
	    }
	  else return (return_flag);
	  }	*/

  strcpy(p->file_name,next);
  strcpy(buff,path);
  strcat(buff,next);
  strcpy(filename,buff); /* now have full pathname */

  return writecall(p,filename,act);

filename_error: 
  sprintf(buf,"Need filename with the 'write' command.");
  emsg();
  return 0;
} /* write_parse */

int timedelay(double tm)
     /* time delay about tm seconds. */
{
  struct tms buffer;  /* for getting times */
  long starttime,elapsetime;
	
  times(&buffer);
  if (tm>0)
    {
      starttime=buffer.tms_utime;
      do
	{
	  times(&buffer);
	  elapsetime=buffer.tms_utime-starttime;
	}
      while (elapsetime< tm*12);
    }
  return 1;
} /* timedelay */

int set_overlapscall(struct p_data *p,char *datastr)
/* set overlaps or inversive distances */
{
	int j,v,count=0,hits=1,nb;
	double ang,inv_d;
	char *nextpoint,*endptr;
	struct Edgelist *edgelist,*track;
	struct Vertlist *vertlist,*trace;
	struct K_data *pK_ptr;

	pK_ptr=p->packK_ptr;
	nextpoint=datastr;
	stripsp(nextpoint);
	if (nextpoint[0]=='-' && nextpoint[1]=='d') /* set to default */
	 {
		free_overlaps(p);
		fillcurves(p);
		sprintf(msgbuf,"Overlap angles for pack %d set to 0.",
			pack_num(p));
		msg();
		return 1;
	 }
	if (nextpoint[0]=='-' && nextpoint[1]=='c') /* set current */
	 {
		nextpoint[0]=nextpoint[1]=' ';stripsp(nextpoint);
		nextpoint += notspace(nextpoint);
		if (nextpoint[0]=='\0') {nextpoint--;nextpoint[0]='a';}
		if ((edgelist=node_pair_link(p,nextpoint,
			&endptr,&hits))!=NULL)
		 {
			if (!alloc_overlaps(p)) return 0;
			track=edgelist;
			do {
			   nb=nghb(p,track->v,track->w);
			   set_overlap(p,track->v,nb,
				comp_inv_dist(p,track->v,track->w));
				/* note: inv_dist routine not very robust */
			   track=track->next;
			   count++;
			 } while (track!=NULL);
			edge_free(&edgelist);
		 }
		fillcurves(p); /* not yet robust for inversive dist */
		sprintf(msgbuf,"Set %d overlap angles for p=%d "
			"to current values.",count,pack_num(p));
		msg();
		return count;
	 }
	inv_d=1.0; /* just in case */
	if ( !grab_next(&nextpoint,next)
		|| (next[0]=='*' && (!sscanf(next+1,"%lf",&ang)
			|| ang<0.0))
	     || (next[0]!='*' && (!sscanf(next,"%lf",&ang)
/* ??			|| ang>0.5 || ang<0.0)) ) */
/* part of attempt to allow overlaps greater than pi/2 */
			|| ang>1.0 || ang<0.0)) )
	 {
		strcpy(msgbuf,"Specify first: angle in [0,1] or '*' "
		       "and inv dist [0,infty].");
		emsg();
	 	return 0;
	 }
	if (next[0]!='*') 
	 {
		if (ang>0.5)
		 {
		   strcpy(msgbuf,"WARNING: overlap > 0.5 (Pi) specified; "
			  "behavior not guaranteed.\n repack with -o option.");
		   emsg();
		 }
		inv_d=cos(ang*M_PI);	/* overlap */
	 }
	else inv_d=ang;				/* separated */
	nextpoint += notspace(nextpoint);
	if (nextpoint[0]=='-' && nextpoint[1]=='f') /* petals of given flowers */
	 {
	 	nextpoint[0]=nextpoint[1]=' ';stripsp(nextpoint);
	 	if (nextpoint[0]=='\0' || (vertlist=node_link_parse(p,nextpoint,
			&endptr,&hits))==NULL) return count;
	 	 {
			if (!alloc_overlaps(p)) return 0;
			trace=vertlist;
			do
			 {
			   v=trace->v;
			   for (j=0;j<pK_ptr[v].num+pK_ptr[v].bdry_flag;j++)
			      set_overlap(p,v,j,inv_d);		
			   trace=trace->next;
			   count++;
			 } while (trace!=NULL);
			vert_free(&vertlist);
		 }
		fillcurves(p);
		return count;
	 }			
	if (nextpoint[0]=='\0') {nextpoint--;nextpoint[0]='a';}
	if ((edgelist=node_pair_link(p,nextpoint,&endptr,&hits))!=NULL)
	 {
		if (!alloc_overlaps(p)) return 0;
		track=edgelist;
		do {
			nb=nghb(p,track->v,track->w);
			set_overlap(p,track->v,nb,inv_d);
			track=track->next;
			count++;
		 } while (track!=NULL);
		edge_free(&edgelist);
	 }
	fillcurves(p);
	if (inv_d<=1.0 && inv_d>=0.0)
	 { 
		sprintf(msgbuf,"Set %d overlap angle(s) for p=%d to %f pi.", 	
		   count,pack_num(p),acos(inv_d)/M_PI);
		msg();
	 }
	else if (inv_d>1.0)
	 { 
		sprintf(msgbuf,"Set %d inv_distance(s) for p=%d to %f.", 	
		   count,pack_num(p),inv_d);
		msg();
	 }
	return count;
} /* set_overlapscall */

int set_postfile_call(int flag,char *datastr)
     /* flag=0 means print file, else custom postscript file */
{
  char filename[NAME_MAX];

  if (sscanf(datastr,"%s",filename))
    {
      if (!flag)
	{
	  strcpy(print_file_name,filename);
	  if (!cmd_mode) set_print_file(filename);
	}
      else
	{
	  strcpy(ps_file_name,filename);
	  if (!cmd_mode) set_ps_file(filename);
	}
      return 1;
    }
  else return 0;
} /* set_postfile_call */

int find_help_packrc(char *argv0)
/* Locate help and set CPHelp, locate and read `.packrc' file. 
Check in order: current directory, home directory, CPHOME/doc (for
help), or basename of the CirclePack call. */
{
  int i,n=0;
  char CPenv[NAME_MAX+1],RC[NAME_MAX+1],*ptr=NULL;

  CPHelp[0]='\0';
  CPenv[NAME_MAX]=RC[NAME_MAX]='\0';

      /* clear the function keys */
  buff[BUFSIZE-1]='\0'; 
  for (i=0;i<=10;i++) ftn_keys[i]=NULL;
 
      /* find environment */
  if (!(ptr=getenv("CPHOME"))) /* check first for env */
    {
      n=strlen(argv0);
      while (n>0 && argv0[n]!='/') n--;
      if (n==0) strncpy(CPenv,home_dir,NAME_MAX);
      else
	{
	  if (n>NAME_MAX-1) n=NAME_MAX;
	  strncpy(CPenv,argv0,n);
	}
    }
  else strncpy(CPenv,ptr,NAME_MAX);

      /* find and read system default CPpackrc */
  strncpy(RC,CPenv,NAME_MAX);
  if (ptr && strlen(RC)<(NAME_MAX-5)) strcat(RC,"/etc");
  if (strlen(RC)<(NAME_MAX-10)) strcat(RC,"/CPpackrc");
  if (stat_file(RC)) read_packrc(RC); 
      
       /* find and read individual .packrc */
  strcpy(RC,"./.packrc"); /* current ?*/
  if (!stat_file(RC))
    {
      strncpy(RC,home_dir,NAME_MAX);
      if (strlen(RC)<(NAME_MAX-8)) strcat(RC,"/.packrc"); /* home dir? */
    }
  if (stat_file(RC)) read_packrc(RC);
  /* mh note: if no .packrc, then read_packrc is never called and no warning
     given about missing .packrc file; following 5 lines added by mh */
  else
    {
      fprintf(stderr,
	      "CirclePack did not find your '.packrc' file. Continue.\n");
    }

       /* find cp_help.info if CPHelp not set in packrc calls */
  if (strlen(CPHelp)==0)
    {
      strcpy(CPHelp,"./cp_help.info"); /* current ?*/
      if (!stat_file(CPHelp))
	{
	  strncpy(CPHelp,home_dir,(NAME_MAX-15));
	  strcat(CPHelp,"/cp_help.info"); /* home dir? */
	  if (!stat_file(CPHelp))
	    {
	      strncpy(CPHelp,CPenv,(NAME_MAX-15));
	      if (ptr) strcat(CPHelp,"/doc");
	      strcat(CPHelp,"/cp_help.info");
	    }
	  if (!stat_file(CPHelp))
	    { 
	      fprintf(stderr,"CirclePack did not "
		      "find 'cp_help.info'. Continue.\n");
	    }
	}
    }

  return 1;
} /* find_help_packrc */

int read_packrc(char *filename)
{
  FILE *fp;
  char *keyword=NULL,*next=NULL,*ptr;
  char buff[BUFSIZE],file_name[NAME_MAX];
  int cs,i;

  if (!(fp=fopen(filename,"r")))
    {
      fprintf(stderr,
	      "CirclePack did not find your '.packrc' file. Continue.");
      return 0;
    }
  keyword=(char *)malloc(256);
  next=(char *)malloc(256);
  while (fgets(keyword,BUFSIZE+255,fp)!=NULL)
    {
      stripsp(keyword);
      if( *keyword!='#' && *keyword!='\n' && *keyword!='\0')
	{
	  ptr=keyword;
	  grab_next(&ptr,next);	
	  /* directory name for reading and writing files */
	  if (!strncmp(next,"CP_PATH",7) 
	      && grab_next(&ptr,next) && (strcspn(next,"#")>0) )
	    {
	      strcpy(packing_dir_name,next);
	    }
	  /* postscript file */
	  else if (!strncmp(next,"PS_FILE",7) 
		   && grab_next(&ptr,next) && (strcspn(next,"#")>0) )
	    {
	      strcpy(print_file_name,next);
	    }
	  /* custom postscript file */
	  else if (!strncmp(next,"CUSTOM_FILE",11) 
		   && grab_next(&ptr,next) && (strcspn(next,"#")>0) )
	    {
	      strcpy(ps_file_name,next);
	    }
	  /* canvas size */
	  else if (!strncmp(next,"CANV_SIZE",9)
		   && grab_next(&ptr,next) && sscanf(next,"%d",&cs)==1)
	    canv_size=cs;		
	  /* print command */
	  else if (!strncmp(next,"PRINT_CMD",9))
	    {
	      i=strcspn(ptr,"#\t\r\0\n");
	      strncpy(next,ptr,i); next[i]='\0'; stripsp(next);
	      strcpy(print_cmd,next);
	    }
	  /* initial screen location */ 
	  else if (!strncmp(next,"SCREEN_X_Y",10)
		   && grab_next(&ptr,next) && sscanf(next,"%d",&ScrX)
		   && grab_next(&ptr,next) && sscanf(next,"%d",&i) )
	    {
	      if (ScrX<0 || ScrX>800) ScrX=0;
	      if (i>=0 && i<900) ScrY=i;
	    }	
	  /* function key pre-defined cmd strings */
	  else if (!strncmp(next,"F1_KEY:",7)
		   && grab_line(buff,&ptr,BUFSIZE-1))
	    {
	      ftn_keys[1]=
		(char *)calloc((size_t)(strlen(buff)+1),sizeof(char));
	      strcpy(ftn_keys[1],buff);
	    }
	  else if (!strncmp(next,"F2_KEY:",7)
		   && grab_line(buff,&ptr,BUFSIZE-1))
	    {
	      ftn_keys[2]=
		(char *)calloc((size_t)(strlen(buff)+1),sizeof(char));
	      strcpy(ftn_keys[2],buff);
	    }
	  else if (!strncmp(next,"F3_KEY:",7)
		   && grab_line(buff,&ptr,BUFSIZE-1))
	    {
	      ftn_keys[3]=
		(char *)calloc((size_t)(strlen(buff)+1),sizeof(char));
	      strcpy(ftn_keys[3],buff);
	    }
	  else if (!strncmp(next,"F4_KEY:",7)
		   && grab_line(buff,&ptr,BUFSIZE-1))
	    {
	      ftn_keys[4]=
		(char *)calloc((size_t)(strlen(buff)+1),sizeof(char));
	      strcpy(ftn_keys[4],buff);
	    }
	  else if (!strncmp(next,"F5_KEY:",7)
		   && grab_line(buff,&ptr,BUFSIZE-1))
	    {
	      ftn_keys[5]=
		(char *)calloc((size_t)(strlen(buff)+1),sizeof(char));
	      strcpy(ftn_keys[5],buff);
	    }
	  else if (!strncmp(next,"F6_KEY:",7)
		   && grab_line(buff,&ptr,BUFSIZE-1))
	    {
	      ftn_keys[6]=
		(char *)calloc((size_t)(strlen(buff)+1),sizeof(char));
	      strcpy(ftn_keys[6],buff);
	    }
	  else if (!strncmp(next,"F7_KEY:",7)
		   && grab_line(buff,&ptr,BUFSIZE-1))
	    {
	      ftn_keys[7]=
		(char *)calloc((size_t)(strlen(buff)+1),sizeof(char));
	      strcpy(ftn_keys[7],buff);
	    }
	  else if (!strncmp(next,"F8_KEY:",7)
		   && grab_line(buff,&ptr,BUFSIZE-1))
	    {
	      ftn_keys[8]=
		(char *)calloc((size_t)(strlen(buff)+1),sizeof(char));
	      strcpy(ftn_keys[8],buff);
	    }
	  else if (!strncmp(next,"F9_KEY:",7)
		   && grab_line(buff,&ptr,BUFSIZE-1))
	    {
	      ftn_keys[9]=
		(char *)calloc((size_t)(strlen(buff)+1),sizeof(char));
	      strcpy(ftn_keys[9],buff);
	    }
	  else if (!strncmp(next,"F10_KEY:",8)
		   && grab_line(buff,&ptr,BUFSIZE-1))
	    {
	      ftn_keys[10]=
		(char *)calloc((size_t)(strlen(buff)+1),sizeof(char));
	      strcpy(ftn_keys[10],buff);
	    }
	  
	  /* help file name */
	  else if (!strncmp(next,"CP_HELP_FILE",12))
	    {
	      if (grab_next(&ptr,next) 
		  && sscanf(next,"%s",CPHelp))
		{
		  if (CPHelp[0]=='~')
		    {
		      strcpy(file_name,home_dir);
		      strcat(file_name,CPHelp+1);
		      strcpy(CPHelp,file_name);
		    }
		}
	      if (CPHelp[0]!='\0' || !stat_file(CPHelp)) CPHelp[0]='\0'; 
	      /* look elsewhere back in calling routine */
	    }
	}
    } /* end of while */
  free(keyword);
  free(next);
  fclose(fp);
  return 1;
} /* read_packrc */

int read_workspace(int script_flag)
{
  FILE *fp;
  char keyword[80],*ptr;
  char filename[NAME_MAX],canvasname[32];
  int winopen,i;
  Rect rect;

/* read '.packwp' to specify window location/status. Look for
 default in CPHOME/etc/CPpackwp */
	
  strcpy(filename,home_dir);
  strcat(filename,"/.packwp"); /* home dir */
  if (!(fp=fopen(filename,"r")))
    {
      if (!(ptr=getenv("CPHOME"))) return 0;
      strcpy(filename,ptr);
      strcat(filename,"/etc/CPpackwp");
      if (!(fp=fopen(filename,"r"))) return 0;
    }
  while ( fscanf(fp,"%63s",keyword)!=EOF )
    {
      if (!strncmp(keyword,"base_frame:",11)
        && (fscanf(fp,"%d %d %d %d %d",&winopen,
        &rect.r_left,&rect.r_top,&rect.r_width,&rect.r_height)==5))
	{
	  frame_set_rect(base_frame,&rect);
	  xv_set(base_frame,WIN_SHOW,TRUE,0); /* always show this */
	}
      else if (!strncmp(keyword,"cmd_frame:",10)
	&& (fscanf(fp,"%d %d %d %d %d",&winopen,
	&rect.r_left,&rect.r_top,&rect.r_width,&rect.r_height)==5))
	{
	  frame_set_rect(cmd_frame,&rect);
	  xv_set(cmd_frame,WIN_SHOW,winopen,0);
	}
      else if (!strncmp(keyword,"msg_frame:",10)
	&& (fscanf(fp,"%d %d %d %d %d",&winopen,
	&rect.r_left,&rect.r_top,&rect.r_width,&rect.r_height)==5))
	{
	  frame_set_rect(msg_frame,&rect);
	  xv_set(msg_frame,WIN_SHOW,winopen,0);
	}
      else if (!strncmp(keyword,"script_frame:",13)
	&& (fscanf(fp,"%d %d %d %d %d",&winopen,
	&rect.r_left,&rect.r_top,&rect.r_width,&rect.r_height)==5))
	{
	  frame_set_rect(script_frame,&rect);
	  xv_set(script_frame,WIN_SHOW,(winopen || script_flag),0);
	}	
      else if (!strncmp(keyword,"setup_frame:",12)
	&& (fscanf(fp,"%d %d %d %d %d",&winopen,
	&rect.r_left,&rect.r_top,&rect.r_width,&rect.r_height)==5))
	{
	  frame_set_rect(setup_frame,&rect);
	  xv_set(setup_frame,WIN_SHOW,winopen,0);
	}
      else if (!strncmp(keyword,"history_frame:",14)
	&& (fscanf(fp,"%d %d %d %d %d",&winopen,
	&rect.r_left,&rect.r_top,&rect.r_width,&rect.r_height)==5))
	{
	  frame_set_rect(history_frame,&rect);
	  xv_set(history_frame,WIN_SHOW,winopen,0);
	}
      else if (!strncmp(keyword,"text_frame:",11)
        && (fscanf(fp,"%d %d %d %d %d",&winopen,
        &rect.r_left,&rect.r_top,&rect.r_width,&rect.r_height)==5))
	{
	  frame_set_rect(text_frame,&rect);
	  xv_set(text_frame,WIN_SHOW,winopen,0);
	}
      else if (!strncmp(keyword,"help_frame:",11)
        && (fscanf(fp,"%d %d %d %d %d",&winopen,
        &rect.r_left,&rect.r_top,&rect.r_width,&rect.r_height)==5))
	{
	  frame_set_rect(help_frame,&rect);
	  xv_set(help_frame,WIN_SHOW,winopen,0);
	}
      else for (i=0;i<NUM_PACKS;i++)
	{
	  sprintf(canvasname,"canvas_frame[%d]:",i);
	  if (!strncmp(keyword,canvasname,15)
            && (fscanf(fp,"%d %d %d %d %d",&winopen,
            &rect.r_left,&rect.r_top,&rect.r_width,&rect.r_height)==5))
	    {
	      frame_set_rect(canvas_frame[i],&rect);
	      xv_set(canvas_frame[i],WIN_SHOW,winopen,0);
	    }
	}
    } /* end of while */
  return 1;
} /* read_workspace */
		
int read_rgb_file()
/* read in color info from file given on cmd line */
{
  char keyword[81];
  int rd,gn,bl,indx,i;
  FILE *fp;

  if (!strlen(rgb_file) || (fp=fopen(rgb_file,"r"))==NULL)
    {
      fprintf(stderr,"Color file %s was not found.",rgb_file);
      return 0;
    }
  while (fscanf(fp,"%63s",keyword)!=EOF)
    {
      stripsp(keyword);
      if( *keyword!='#' && *keyword!='\n' && *keyword!='\0')
	{

/* ramp size given */
          if (!strncmp(keyword,"RAMP:",5) 
              && fscanf(fp,"%d",&color_ramp_size));

/* list of indices with red/green/blue levels */
          else if (!strncmp(keyword,"LIST_IND_R_G_B:",15))
	    {
	      bc_min=NUM_COLORS;
	      bc_max=0;
	      while( (fgets(keyword,80,fp)!=NULL)
                   && (strcmp(keyword,"END")!=0) )
		{
		  sscanf(keyword,"%d %d %d %d",&indx,&rd,&gn,&bl);
		  if (indx>=0 && indx<=NUM_COLORS 
		      && rd>=0 && rd<=NUM_COLORS
		      && gn>=0 && gn<=NUM_COLORS 
		      && bl>=0 && bl<=NUM_COLORS)
		    {
		      red[indx]=rd;
		      green[indx]=gn;
		      blue[indx]=bl;
		      if (indx<bc_min) bc_min=indx;
		      if (indx>bc_max) bc_max=indx;
		      indx=-1; /* re-initialize index */
		    }
		}
              /* mh: adjust bc_offset by 1 due to subtraction,
		 check that bc_min > -bc_offset */
	      if ((bc_offset=bc_min-bc_max-1)<0
		  && bc_offset > -NUM_COLORS/2
		  && bc_min > -bc_offset)
		{
		  for (i=bc_min;i<=bc_max;i++)
		    {
		      red[i+bc_offset]=
			(unsigned char)(.75*red[i]);
		      green[i+bc_offset]=
			(unsigned char)(.75*green[i]);
		      blue[i+bc_offset]=
			(unsigned char)(.75*blue[i]);
		    }
		}
	      else bc_offset=0;


	    }
	}
    } /* end of while */
  if (fp) fclose(fp);
  return 1;
} /* read_rgb_file */
		       

/* ============================ hex walking ======================== */

int hex_parallel_call(struct p_data *p,int v,int w,int n,complex *cmod,
		      int show)
/* Try to draw parallelogram (treat complex as though it were hex). 
show=1 means to draw faces and to compute complex modulus of 
parallelogram. */
{
	int hold,hold1,hold2,count,v1,v2,v3,v4;
	complex z0,z1,z2,zz;
	struct Vertlist *facelist,*ftrace;
	struct Vertlist *dlist,*curve1,*ctrace1,*curve2,*ctrace2;

	cmod->re=cmod->im=0.0;
	if (show) 
	 {
		dlist=try_hex_pg(p,v,w,n,&v1,&v2,&v3,&v4);
		layout_facelist(p,7,&dlist,1,0,SHOW); /* fill with bg */
			/* note: clears list*/
	 }
	ftrace=facelist=try_hex_pg(p,v,w,n,&v1,&v2,&v3,&v4);
	while (ftrace && ftrace->next) ftrace=ftrace->next;
	if (ftrace && ftrace->v!=facelist->v) /* didn't close up */
	 {vert_free(&facelist); return 0;}
/* create lists to get to corners */
	ftrace=facelist;
	if (ftrace!=NULL)
	 {
		curve1=(struct Vertlist *)
			calloc((size_t)1,sizeof(struct Vertlist));
		ctrace1=curve1;
		ctrace1->v=ftrace->v;
	 }
	else {vert_free(&facelist); return 0;}
	ftrace=ftrace->next;
	count=1;
	while (ftrace!=NULL && count < (2*n)-1)
	 {
		ctrace1=ctrace1->next=(struct Vertlist *)
			calloc((size_t)1,sizeof(struct Vertlist));
		ctrace1->v=ftrace->v;
		count++;
		ftrace=ftrace->next;
	 }
	if (!ftrace || !(ftrace->next)) 
	 {vert_free(&facelist);vert_free(&curve1);return 0;}
	hold1=ctrace1->v;  /* have to save this second-to-last face */
	ctrace1=ctrace1->next=(struct Vertlist *)
		calloc((size_t)1,sizeof(struct Vertlist));
	ctrace1->v=ftrace->v;
	ctrace1=ctrace1->next=(struct Vertlist *)
		calloc((size_t)1,sizeof(struct Vertlist));
	ctrace1->v=hold=curve1->v; /* close up */
	if (show)
	 {
		layout_facelist(p,7,&curve1,1,0,NOSHOW); /* clears curve1 */
		z0=p->packR_ptr[v1].center; /* first corner */
		z1=p->packR_ptr[v2].center; /* second corner */
	 }
/* second curve list */
	curve2=ctrace2=(struct Vertlist *)
	  calloc((size_t)1,sizeof(struct Vertlist));
 	ctrace2->next=(struct Vertlist *)
	  calloc((size_t)1,sizeof(struct Vertlist));
	ctrace2->v=hold; /* want start/end at same face as curve1 */
	ctrace2=ctrace2->next;
	ctrace2->v=hold2=ftrace->v;
	ftrace=ftrace->next;
	count=1;
	while (ftrace!=NULL && count < (2*n)-1)
	 {
		ctrace2=ctrace2->next=(struct Vertlist *)
		  calloc((size_t)1,sizeof(struct Vertlist));
		ctrace2->v=ftrace->v;
		count++;
		ftrace=ftrace->next;
	 }
	if (!ftrace) 
	 {vert_free(&facelist);vert_free(&curve1);vert_free(&curve2);return 0;}
	ctrace2=ctrace2->next=(struct Vertlist *)
	  calloc((size_t)1,sizeof(struct Vertlist));
	ctrace2->v=hold1;
	ctrace2=ctrace2->next=(struct Vertlist *)
	  calloc((size_t)1,sizeof(struct Vertlist));
	ctrace2->v=hold2;
	ctrace2=ctrace2->next=(struct Vertlist *)
	  calloc((size_t)1,sizeof(struct Vertlist));
	ctrace2->v=curve2->v; /* takes some doing to close up */
	if (show)
	 {
		layout_facelist(p,7,&curve2,1,0,NOSHOW); /* clears curve2*/
		z2=p->packR_ptr[v3].center; /* third corner */
	 }
/* later, may want to save curve1, curve2 as generators */
	vert_free(&curve1);vert_free(&curve2);
/* now to compute modulus */
	if (show)
	 {
		zz=csub(z1,z0);
		if (cAbs(zz)> .0000001)
			*cmod=cdiv(csub(z2,z1),zz);
		else {cmod->re=0.0;cmod->im=0.0;}
	 }
	return 1;
} /* hex_parallel_call */
	
struct Vertlist *try_hex_pg(struct p_data *p,int v,int w,int n,
			    int *v1,int *v2,int *v3,int *v4)
/* for hex_walk, called by hex_parallel_call and fix -h to draw 
parallelogram. Edge lengths n, start edge = v w, counterclockwise. 
Return corner verts; v2,v4 "sharp" corners. */
{
     int i,num;
     struct Edgelist *edgelist;
     struct Vertlist *vertlist,*vtrace,*facelist;
     struct K_data *pK_ptr;

     pK_ptr=p->packK_ptr;
     facelist=NULL;
     if (nghb(p,v,w)<0) return facelist; /* must start with edge */
/* record first two vertices */
     *v1=v;
     vertlist=(struct Vertlist *)calloc((size_t)1,sizeof(struct Vertlist));
     vertlist->v=v;
     vtrace=vertlist->next=(struct Vertlist *)
	calloc((size_t)1,sizeof(struct Vertlist));
     vtrace->v=w;
/* first edge */
     for (i=2;i<=n;i++) 
      {
          vtrace=vtrace->next=(struct Vertlist *)
	    calloc((size_t)1,sizeof(struct Vertlist));
          num=pK_ptr[w].num;
          vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-3) % num];
          v=w;
          w=vtrace->v;
      }
/* sharp left turn */
     *v2=w;
     vtrace=vtrace->next=(struct Vertlist *)
	calloc((size_t)1,sizeof(struct Vertlist));
     num=pK_ptr[w].num;
     vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-1) % num];
     v=w;
     w=vtrace->v;
/* second edge */
     for (i=2;i<=n;i++) 
      {
          vtrace=vtrace->next=(struct Vertlist *)
	    calloc((size_t)1,sizeof(struct Vertlist));
          num=pK_ptr[w].num;
          vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-3) % num];
          v=w;
          w=vtrace->v;
      }
/* left turn */
     *v3=w;
     vtrace=vtrace->next=(struct Vertlist *)
	calloc((size_t)1,sizeof(struct Vertlist));
     num=pK_ptr[w].num;
     vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-2) % num];
     v=w;
     w=vtrace->v;
/* third edge */
     for (i=2;i<=n;i++) 
      {
          vtrace=vtrace->next=(struct Vertlist *)
	    calloc((size_t)1,sizeof(struct Vertlist));
          num=pK_ptr[w].num;
          vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-3) % num];
          v=w;
          w=vtrace->v;
      }
/* sharp left turn */
     *v4=w;
     vtrace=vtrace->next=(struct Vertlist *)
       calloc((size_t)1,sizeof(struct Vertlist));
     num=pK_ptr[w].num;
     vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-1) % num];
     v=w;
     w=vtrace->v;
     for (i=2;i<=n;i++) /* last edge */
      {
          vtrace=vtrace->next=(struct Vertlist *)
	    calloc((size_t)1,sizeof(struct Vertlist));
          num=pK_ptr[w].num;
          vtrace->v=pK_ptr[w].flower[(nghb(p,w,v)+num-3) % num];
          v=w;
          w=vtrace->v;
      }
     edgelist=node_to_edge(p,vertlist);
     vert_free(&vertlist);
     facelist=path_follow(p,edgelist,1);
     edge_free(&edgelist);
     return facelist;
} /* try_hex_pg */

int j_ftn_call(struct p_data *p,char *datastr)
     /* call to build complex for classical j_function */
{
  int n0,n1,maxsize;

  if (sscanf(datastr,"%d %d %d",&n0,&n1,&maxsize)<3
      || n0<1 || n1<1 || maxsize< n0+n1+3)
    {
      sprintf(msgbuf,"Usage: j_ftn n0 n1 maxsize. n0, n1 are desired orders,"
	      " maxsize limits size");
      emsg();
      return 0;
    }
  return (build_j_function(p,n0,n1,maxsize));
} /* j_ftn_call */

int mend_call(struct p_data *p,char *datastr)
     /* call a routine to mend a localized flaw in a packing layout.
Two options: 
  - specify region with this call using "D v <n> (or c <x y>) r <r>"
    format.
  - Use vertices already marked in some earlier command. Just need to
    identify a reference vert inside the marked region for 'cookie' call.
    It can be specified with "v <n>", otherwise, we do a little
    search.
In either case, also have the 'l' option to specify the number of 
generations to be blended.
In sphere case we have to project to the plane; we use the identified
vert -- it its in the lower hemisphere, we move it and all the marked
verts. Thus, if marked region is extensive, this could cause an error.
     */
     /* fixup: -- make this automatic */
{
  int laps=1,vert=0,depth,max_depth,v,hits,dum,riffle_count=0;
  int *newold,*gen_list,mid_vert,i,j,done=3,qcount;
  double rad=.1;
  complex ctr;
  struct Edgelist *etrace,*eclobber;
  struct Vertlist *vertlist=NULL,*trace;
  char *nextptr,*ptr,*endptr,copydata[16];
  struct p_data *mendpack=NULL,*spack=NULL;
  Mobius mob;

  nextptr=datastr;
  if ((ptr=after_char(nextptr,"v")))
    /* read our reference vert */
    {
      if (!(vert=grab_one_vert(p,&ptr)))
	done = 16; /* some error */
      else 
	{
	  done -= 1;
	  ctr=p->packR_ptr[vert].center;
	}
    }
  if ((ptr=after_char(nextptr,"l"))) /* num. of gens to overlap. */
    {
      if (!sscanf(ptr,"%d",&laps) || laps<1)
	done=16; /* error */
    }
  /* Note: 1-bit of 'done' is for center (v OR c option), 2-bit is
     for radius (in 'D ..' option case), 16-bit reflects error. */

  if ((ptr=after_char(nextptr,"D"))) /* we're going to send string
					to Node_link_parse; have to
					read center first */
    {
      qcount=3;
      nextptr=datastr;
      while (qcount && done>0 && done<16)
	{
	  qcount--;
	  if ((done & 1) && (ptr=after_char(nextptr,"c"))) 
	    /* give x,y coords for center */
	    {
	      if (sscanf(ptr,"%lf %lf",&ctr.re,&ctr.im)!=2)
		done=16; /* some error */
	      else done -= 1;
	    }
	  if ((ptr=after_char(nextptr,"r"))) /* radius */
	    {
	      if (!sscanf(ptr,"%lf",&rad) || rad<m_toler
		  || (p->hes>0 && rad>M_PI_2))
		done=16; /* error */
	      else done -= 2;
	    }
	} /* end of while */
      if (done) /* didn't get all needed data or had error. */
	{
	  sprintf(msgbuf,
		  "usage mend: mend [D option] v <n> l <k>.");
	  emsg();
	  return 0;
	}

      /* now get the verts inside the specified disc */
      if (!(vertlist=node_link_parse(p,datastr,&endptr,&hits)))
	{
	  sprintf(msgbuf,"usage mend: mend -D v <n> (or c <x y>) r <x> -l <k>."
		  );
	  emsg();
	  return 0;
	}
    }
  else 
    {
      vertlist=node_link_parse(p,"m",&endptr,&hits);
      if (vertlist && (!vert || !p->packK_ptr[vert].mark)) 
	/* use already marked verts; need to find a seed interior 
	   to marked verts for use in the 'cookie' routine. */
	for (i=1;(!vert && i<=p->nodecount);i++)
	  if (!p->packK_ptr[i].bdry_flag)
	    {
	      j=0;
	      while (!vert && j<=p->packK_ptr[i].num
		     && p->packK_ptr[j].mark)
		j++;
	      if (j>p->packK_ptr[i].num) /* all neighbors are marked */
		vert=i;
	    }
      if (!vertlist || !vert) 
     /* didn't find seed or didn't find any marked vertices. */
	{
	  sprintf(msgbuf,"mend error: didn't find necessary marked"
		  " verts or seed vert.");
	  emsg();
	  vert_free(&vertlist);
	  return 0;
	}
    }

/* Summary: should have everything needed at this point. */	  
      
/* allocate pack for mend region fragment */
  if ((mendpack=(struct p_data *)
       calloc((size_t)1,sizeof(struct p_data)))==NULL
      || !copy_pack(p,mendpack))
    {
      sprintf(msgbuf,"mend: memory or copy problems for mend packing.");
      emsg();
      vert_free(&vertlist);
      if (mendpack) free_p_data(&mendpack);
      return 0;
    }
     
/* set up poison and cookie out mend fragment */

  for (i=1;i<=mendpack->nodecount;i++)
    mendpack->packK_ptr[i].util_flag=-1; /* set all to poison */
  trace=vertlist;
  while (trace)
    {
      mendpack->packK_ptr[trace->v].util_flag=0; /* non-poison */
      trace=trace->next;
    }
  vert_free(&vertlist);

  sprintf(copydata,"-v %d",vert);
  if (!cookie_cutter(mendpack,copydata,&newold) || !newold)
    {
      sprintf(msgbuf,"mend: error in cookie'ing out mend region.");
      emsg();
      if(newold) free(newold);
      free_p_data(&mendpack);
      return 0;
    }

/* set vertex_map of p */
  mendpack->vertex_map=etrace=eclobber=
    (struct Edgelist *)calloc((size_t)1,sizeof(struct Edgelist));
  for (i=1;i<=mendpack->nodecount;i++)
    {
      etrace->v=i;
      etrace->w=newold[i];
      etrace->next=
	(struct Edgelist *)calloc((size_t)1,sizeof(struct Edgelist));
      eclobber=etrace;
      etrace=etrace->next;
    }
  if (eclobber==etrace) /* none found; must be error */
    {
      free(mendpack->vertex_map);
      mendpack->vertex_map=NULL;
      sprintf(msgbuf,"mend: no vertex_map was generated.");
      emsg();
      free_p_data(&mendpack);
      return 0;
    }
  else
    {
      free(etrace);
      eclobber->next=NULL;
    }
  free(newold);

/* spherical case different: move center into northern hemisphere,
   project mend fragment to eucl plane, make copy to be repacked,
   blend that with original fragment, project back, move results into
   correct hemisphere, manually feed new centers/radii into p. */

  if (mendpack->hes>0) 
    {
      mob=ident_mob();
      if (ctr.im>M_PI_2) /* apply mobius for z --> 1/z to pack. */
	{
	  mob.a.re=mob.d.re=0.0;
	  mob.b.re=mob.c.re=1.0;
	  apply_Mobius(mendpack,"a",1,mob);
	}
      geom_to_e_pack_convert(mendpack);

      /* allocate spack for copy of mend fragment */
      if ((spack=(struct p_data *)
	   calloc((size_t)1,sizeof(struct p_data)))==NULL
	  || !copy_pack(mendpack,spack)
	  || !mendpack->vertex_map)
	{
	  sprintf(msgbuf,"mend: memory or copy problems for mend pack copy.");
	  emsg();
	  if (spack) free_p_data(&spack);
	  free_p_data(&mendpack);
	  return 0;
	}
      edge_free(&(mendpack->vertex_map)); /* keep link to p in spack's
					     vertex_map. Set mendpack's
					     vertex_map to identity. */
      mendpack->vertex_map=etrace=eclobber=
	(struct Edgelist *)calloc((size_t)1,sizeof(struct Edgelist));
      for (i=1;i<=mendpack->nodecount;i++)
	{
	  etrace->v=i;
	  etrace->w=i;
	  etrace->next=
	    (struct Edgelist *)calloc((size_t)1,sizeof(struct Edgelist));
	  eclobber=etrace;
	  etrace=etrace->next;
	}
      free(etrace);
      eclobber->next=NULL;
    }

/* now pack the mend fragment; it should have the bdry radii of p
   (or of spack in sph case). */

  if (mendpack->hes<=0)
    {
      riffle_count=riffle_call(mendpack,totalpasses,0,1);
      comp_pack_centers(mendpack,0,0,2,okerr);
    }

  /* find a bdry vert v, mark all bdry's util_flags to be used
     in generation counting. */
  v=0;
  for (i=1;i<=mendpack->nodecount;i++)
    {
      if (mendpack->packK_ptr[i].bdry_flag) 
	{
	  v=i;
	  mendpack->packK_ptr[i].util_flag=1;
	}
      else mendpack->packK_ptr[i].util_flag=0;
    }
  
  /* label generations of mendpack, find depth, vert deep inside, etc */

  if (!(gen_list=label_generations(mendpack,500,&mid_vert,&dum)))
    {
      sprintf(msgbuf,"mend: error in counting generations of mend region.");
      emsg();
      if (spack) free_p_data(&spack);
      free_p_data(&mendpack);
      return 0;
    } 

  mendpack->alpha=mid_vert; /* this should be deepest vertex */
  max_depth=gen_list[mid_vert];
  free(gen_list);gen_list=NULL;
  depth=(int)(max_depth/4);
  if (depth<1) depth=1;

/* Now call blend */

  if ((p->hes>0 && !blend(spack,mendpack,v,depth))
      || ((p->hes<=0 && !blend(p,mendpack,v,depth))))
    {
      sprintf(msgbuf,"mend: didn't blend correctly.");
      emsg();
      if (spack) free_p_data(&spack);
      free_p_data(&mendpack);
      return 0;
    }
  if (!spack) /* eucl or hyp cases */
    {
      sprintf(msgbuf,"mend successful:"
	      "mend patch size %d; blend depth %d; max_depth %d; "
	      "riffle count %d.",
	      mendpack->nodecount,depth,max_depth,riffle_count);
      msg();
      free_p_data(&mendpack);
      return 1;
    }

/* handle the sph case; spack has the new stuff in it. */

  geom_to_s_pack_convert(spack);
  apply_Mobius(spack,"a",1,mob); /* in case we had switched hemispheres */
  etrace=spack->vertex_map;
  while (etrace)
    {
      p->packR_ptr[etrace->w].rad=spack->packR_ptr[etrace->v].rad;
      p->packR_ptr[etrace->w].center=spack->packR_ptr[etrace->v].center;
      etrace=etrace->next;
    }
  sprintf(msgbuf,"mend successful:"
	  "mend patch size %d; blend depth %d; max_depth %d).",
	  mendpack->nodecount,depth,max_depth);
  msg();
  free_p_data(&spack);
  free_p_data(&mendpack);
  return 1;
} /* mend_call */

int h_dist_call(char *datastr)
     /* hyp distance in disc or uhp */
{
  int uhp_flag=0;
  double dist;
  complex z,w;

  stripsp(datastr);
  if (datastr[0]=='-')
    {
      if (datastr[1]!='u') return 0;
      datastr[0]=' ';
      datastr[1]=' ';
      uhp_flag=1;
    }
  sscanf(datastr,"%lf %lf %lf %lf",&z.re,&z.im,&w.re,&w.im);
  dist=h_dist(z,w,uhp_flag);
  if (fabs(dist)<m_toler)
    {
      sprintf(msgbuf,"h_dist: points essentially identical.");
      msg();
      return 0;
    }
  else if (dist<(-1)*m_toler)
    {
      sprintf(msgbuf,"h_dist: one or more points on ideal bdry");
      msg();
      return 0;
    }
  else
    {
      sprintf(msgbuf,"h_dist = %f ",dist);
      msg();
      return 1;
    }
} /* h_dist_call */

int focus_call(struct p_data *p,char *datastr)
     /* change screen view so specified vert, -v <v>, or (x,y) point, 
	-c <x y>, or active node (default) is at the center. 0 on error. */
{
  int got_pt=0,v=0,front;
  double rad;
  complex ctr;
  char *ptr;
  struct s_data *q=p->screen;

  /* use given x,y location. For sphere, this is actually the
     projected point, (y,z), on the view. */
  if ((ptr=after_char (datastr,"c")) 
      && sscanf(ptr,"%lf %lf",&ctr.re,&ctr.im)==2)
    got_pt=1;
  
  /* use center of specified vertex */
  else if (((ptr=after_char(datastr,"v")) && (v=grab_one_vert(p,&ptr)))
	   || (v=p->active_node))
    {
      if (p->hes<0) 
	h_to_e_data(p->packR_ptr[v].center,
		    p->packR_ptr[v].rad,&ctr,&rad);
      else if (p->hes==0) 
	ctr=p->packR_ptr[v].center;
      else 
	{
	  ctr=ss_view(p->screen,
		      p->packR_ptr[v].center,1,&front);
	  if (!front) return 1;
	  ctr=s_pt_to_visual_plane(ctr);
	}
    }
  else return 0;

  if (!screen_focus(q,ctr)) return 0;
  if (got_pt) sprintf(msgbuf,"Focused on %f %f",ctr.re,ctr.im);
  else sprintf(msgbuf,"Focused on %d.",v);
  msg();
  return 1;
} /* focus_call */
