// -*- mode: js; js-indent-level: 4; indent-tabs-mode: nil -*-

const Lang = imports.lang;
const Cinnamon = imports.gi.Cinnamon;
const St = imports.gi.St;

const Main = imports.ui.main;
const MessageTray = imports.ui.messageTray;
const AppletManager = imports.ui.appletManager;
const SignalManager = imports.misc.signalManager;

function WindowAttentionHandler() {
    this._init();
}

WindowAttentionHandler.prototype = {
    _init : function() {
        this._tracker = Cinnamon.WindowTracker.get_default();
        global.display.connect('window-demands-attention', Lang.bind(this, this._onWindowDemandsAttention));
        global.display.connect('window-marked-urgent', Lang.bind(this, this._onWindowDemandsAttention));
    },

    _onWindowDemandsAttention : function(display, window) {
        /* Ordinarily, new windows that don't specifically demand focus (like ones created
         * without user interaction) only get some indicator that the window wants the
         * user's attention (like blinking in the window list). Some windows aren't
         * directly tied to a user action, but are generated by the user's action anyhow -
         * these we make an exception for and focus them. */
        try {
            if (AppletManager.get_role_provider_exists(Main.AppletManager.Roles.WINDOW_ATTENTION_HANDLER)) {
                return;
            }

            if (!window || window.has_focus() || window.is_skip_taskbar() || !Main.isInteresting(window)) {
                return;
            }

            let app = this._tracker.get_window_app(window);

            let name = ""
            if (app !== null) {
                name = app.get_name();
            }

            let source = new WindowHandlerSource(app, window);
            Main.messageTray.add(source);

            let notification = new MessageTray.Notification(source, name, window.get_title());
            source.notify(notification);
        } catch (e) {
            global.logError('Error showing window demanding attention', e);
        }
    }
};

var WindowHandlerSource = class extends MessageTray.Source {
    constructor(app, window) {
        super(app.get_name());
        this.window = window;
        this.app = app;

        this._signals = new SignalManager.SignalManager();

        this._signals.connect(this.window, 'notify::demands-attention', () => this.windowStateChanged());
        this._signals.connect(this.window, 'notify::urgent', () => this.windowStateChanged());
        this._signals.connect(this.window, 'focus', () => this.destroy());
        this._signals.connect(this.window, 'unmanaged', () => this.destroy());
    }

    createNotificationIcon() {
        if (this.app !== null) {
            return this.app.create_icon_texture(this.ICON_SIZE);
        }

        return new St.Icon({ icon_name: 'dialog-information',
                             icon_type: St.IconType.SYMBOLIC,
                             icon_size: this.ICON_SIZE });
    }

    windowStateChanged() {
        if (!this.window.demands_attention && !this.window.urgent) {
            this.destroy();
        }
    }

    open() {
        this.window.activate(global.get_current_time());
    }

    destroy(params) {
        this._signals.disconnectAllSignals();
    }
};


