/*
 * Distributor.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  Distributor.hpp
   
   Utility for communication between several top windows or more general:
   between classes. Be `__distrib' a global pointer to a Distributor<int>
   instance. Classes [top windows] which want take part on the distribution have
   to login into `__distrib' with a STATIC class function and an user data
   (usualy their `this' pointer), which is deposited in `__distrib'. If now
   "anyone" in the application calls `__distrib->value(i)', then all registred
   functions are called with the address of `i' as first argument and their
   deposited user data (this-pointer) as second one. The classes can then do the
   needed local things. If `i' exists only temporarly compared to the registred
   classes -- e.g. a local variable of the caller, then the receiver should take
   copies of `i'.
   
   Later std::vector could be substituted by a slimmer utility, reduced to 
   the essential.
   
   Open question: Shall the Callback function be declared with constant or
   non-constant `pData' pointer...
      
      typedef void (Callback) (const void* pData, void* user)  or
      typedef void (Callback) (void* pData, void* user) ?
   
   Non-constant, if the pointed data shall to be changeable from inside the 
   callback functions. But I think, we don't need this. See also the discussion
   below at ValueDistributor: only constant pointer allow an `value(const T&)'.
   The constancy of the pointer `user' is a further question.

   TODO: 
    # Rename type "Callback" into "Br_Callback" or "Di_Callback".
*/
#ifndef Distributor_hpp
#define Distributor_hpp


#include <vector>

//----------------------------------------------------------------
// Type of our callback functions... 
/// @param pData: pointer to the info ("value") to distribute
/// @param user: user data deposites with login(), common caller's `this'
//----------------------------------------------------------------
typedef void (Callback) (const void* pData, void* user);
 

//=========================================
// DistributorBase...  (non-template stuff)
//=========================================
class DistributorBase
{
protected:
  
  struct Entry { 
    Callback* fp;  
    void*     user;   // read "user data" resp. "caller"; common the
                      // `this' pointer of the class `fp' belongs to.
    Entry (Callback* _fp, void* _u) : fp(_fp), user(_u) {} 
  };

  std::vector<Entry> array_;

public:
  void login  (Callback* fp, void* user);
  void logout (Callback* fp);   // guaranteed non-ambiguous  
  void logout (void* user);     // 'user' should be non-ambiguous in array_
  
  void report (const char* label) const;  // debug output
};


//===================================================================
// ValueDistributor... (template)
//
// Offen fuer mich nach wie vor, ob value(T), value(T&) oder value(const T&)?
//   Differenzen: (1) 'value(T)' schlechter fuer grosse T's. (2) Nur 'value(T)' 
//   und `value(const T&)' erlauben Dinge wie `value(1)' oder `value(fn())',
//   wo fn() eine Fkt `T fn()'. Wegen (2) sollte `value(T&)' vermieden werden.
//   Schliesslich wollen wir innerhalb value(..) den T-Wert ja nicht aendern, oder?
//   Alles klar, `value(const T&)' scheint die Wahl. Das verlangt dann aber 
//   auch obige Callback Funktion mit Konstantzeiger zu deklarieren! D.h.
//   'value(const T&)' impliziert, dass auch in den Callback-Funktionen das
//   dort per Zeiger referenzierte Value-Objekt niemals veraendert werden kann.
//====================================================================
template <typename T> 
class ValueDistributor : public DistributorBase
{
public:
  void value (const T& val);
  //void value (const T& val, void* caller);  // if caller would be needed
};

template <typename T>
void ValueDistributor<T>::value (const T& val)
{
  for (unsigned i=0; i < array_.size(); i++)
    array_[i].fp (&val, array_[i].user);
}


#endif // Distributor_hpp

// END OF FILE
