/*
 * Copyright (C) 2006, 2008 Apple Inc. All rights reserved.
 * Copyright (C) 2009 Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef ResourceResponse_h
#define ResourceResponse_h

#include "core/fileapi/File.h"
#include "core/platform/network/HTTPHeaderMap.h"
#include "core/platform/network/ResourceLoadInfo.h"
#include "core/platform/network/ResourceLoadTiming.h"
#include "weborigin/KURL.h"
#include "wtf/PassOwnPtr.h"
#include "wtf/RefPtr.h"
#include "wtf/text/CString.h"

#if OS(SOLARIS)
#include <sys/time.h> // For time_t structure.
#endif

namespace WebCore {

struct CrossThreadResourceResponseData;

class ResourceResponse {
    WTF_MAKE_FAST_ALLOCATED;
public:
    enum HTTPVersion { Unknown, HTTP_0_9, HTTP_1_0, HTTP_1_1 };

    class ExtraData : public RefCounted<ExtraData> {
    public:
        virtual ~ExtraData() { }
    };

    static PassOwnPtr<ResourceResponse> adopt(PassOwnPtr<CrossThreadResourceResponseData>);

    // Gets a copy of the data suitable for passing to another thread.
    PassOwnPtr<CrossThreadResourceResponseData> copyData() const;

    ResourceResponse();
    ResourceResponse(const KURL&, const AtomicString& mimeType, long long expectedLength, const AtomicString& textEncodingName, const String& filename);

    bool isNull() const { return m_isNull; }
    bool isHTTP() const;

    const KURL& url() const;
    void setURL(const KURL&);

    const AtomicString& mimeType() const;
    void setMimeType(const AtomicString&);

    long long expectedContentLength() const;
    void setExpectedContentLength(long long);

    const AtomicString& textEncodingName() const;
    void setTextEncodingName(const AtomicString&);

    // FIXME: Should compute this on the fly.
    // There should not be a setter exposed, as suggested file name is determined based on other headers in a manner that WebCore does not necessarily know about.
    const String& suggestedFilename() const;
    void setSuggestedFilename(const String&);

    int httpStatusCode() const;
    void setHTTPStatusCode(int);

    const AtomicString& httpStatusText() const;
    void setHTTPStatusText(const AtomicString&);

    String httpHeaderField(const AtomicString& name) const;
    String httpHeaderField(const char* name) const;
    void setHTTPHeaderField(const AtomicString& name, const String& value);
    void addHTTPHeaderField(const AtomicString& name, const String& value);
    const HTTPHeaderMap& httpHeaderFields() const;

    bool isMultipart() const { return mimeType() == "multipart/x-mixed-replace"; }

    bool isAttachment() const;

    // FIXME: These are used by PluginStream on some platforms. Calculations may differ from just returning plain Last-Modified header.
    // Leaving it for now but this should go away in favor of generic solution.
    void setLastModifiedDate(time_t);
    time_t lastModifiedDate() const;

    // These functions return parsed values of the corresponding response headers.
    // NaN means that the header was not present or had invalid value.
    bool cacheControlContainsNoCache() const;
    bool cacheControlContainsNoStore() const;
    bool cacheControlContainsMustRevalidate() const;
    bool hasCacheValidatorFields() const;
    double cacheControlMaxAge() const;
    double date() const;
    double age() const;
    double expires() const;
    double lastModified() const;

    unsigned connectionID() const;
    void setConnectionID(unsigned);

    bool connectionReused() const;
    void setConnectionReused(bool);

    bool wasCached() const;
    void setWasCached(bool);

    ResourceLoadTiming* resourceLoadTiming() const;
    void setResourceLoadTiming(PassRefPtr<ResourceLoadTiming>);

    PassRefPtr<ResourceLoadInfo> resourceLoadInfo() const;
    void setResourceLoadInfo(PassRefPtr<ResourceLoadInfo>);

    HTTPVersion httpVersion() const { return m_httpVersion; }
    void setHTTPVersion(HTTPVersion version) { m_httpVersion = version; }

    const CString& getSecurityInfo() const { return m_securityInfo; }
    void setSecurityInfo(const CString& securityInfo) { m_securityInfo = securityInfo; }

    long long appCacheID() const { return m_appCacheID; }
    void setAppCacheID(long long id) { m_appCacheID = id; }

    const KURL& appCacheManifestURL() const { return m_appCacheManifestURL; }
    void setAppCacheManifestURL(const KURL& url) { m_appCacheManifestURL = url; }

    bool wasFetchedViaSPDY() const { return m_wasFetchedViaSPDY; }
    void setWasFetchedViaSPDY(bool value) { m_wasFetchedViaSPDY = value; }

    bool wasNpnNegotiated() const { return m_wasNpnNegotiated; }
    void setWasNpnNegotiated(bool value) { m_wasNpnNegotiated = value; }

    bool wasAlternateProtocolAvailable() const
    {
      return m_wasAlternateProtocolAvailable;
    }
    void setWasAlternateProtocolAvailable(bool value)
    {
      m_wasAlternateProtocolAvailable = value;
    }

    bool wasFetchedViaProxy() const { return m_wasFetchedViaProxy; }
    void setWasFetchedViaProxy(bool value) { m_wasFetchedViaProxy = value; }

    bool isMultipartPayload() const { return m_isMultipartPayload; }
    void setIsMultipartPayload(bool value) { m_isMultipartPayload = value; }

    double responseTime() const { return m_responseTime; }
    void setResponseTime(double responseTime) { m_responseTime = responseTime; }

    const AtomicString& remoteIPAddress() const { return m_remoteIPAddress; }
    void setRemoteIPAddress(const AtomicString& value) { m_remoteIPAddress = value; }

    unsigned short remotePort() const { return m_remotePort; }
    void setRemotePort(unsigned short value) { m_remotePort = value; }

    const File* downloadedFile() const { return m_downloadedFile.get(); }
    void setDownloadedFile(PassRefPtr<File> downloadedFile) { m_downloadedFile = downloadedFile; }

    // Extra data associated with this response.
    ExtraData* extraData() const { return m_extraData.get(); }
    void setExtraData(PassRefPtr<ExtraData> extraData) { m_extraData = extraData; }

    // The ResourceResponse subclass may "shadow" this method to provide platform-specific memory usage information
    unsigned memoryUsage() const
    {
        // average size, mostly due to URL and Header Map strings
        return 1280;
    }

    static bool compare(const ResourceResponse&, const ResourceResponse&);

private:
    void parseCacheControlDirectives() const;
    void updateHeaderParsedState(const AtomicString& name);

    KURL m_url;
    AtomicString m_mimeType;
    long long m_expectedContentLength;
    AtomicString m_textEncodingName;
    String m_suggestedFilename;
    int m_httpStatusCode;
    AtomicString m_httpStatusText;
    HTTPHeaderMap m_httpHeaderFields;
    time_t m_lastModifiedDate;
    bool m_wasCached : 1;
    unsigned m_connectionID;
    bool m_connectionReused : 1;
    RefPtr<ResourceLoadTiming> m_resourceLoadTiming;
    RefPtr<ResourceLoadInfo> m_resourceLoadInfo;

    bool m_isNull : 1;

    mutable bool m_haveParsedCacheControlHeader : 1;
    mutable bool m_haveParsedAgeHeader : 1;
    mutable bool m_haveParsedDateHeader : 1;
    mutable bool m_haveParsedExpiresHeader : 1;
    mutable bool m_haveParsedLastModifiedHeader : 1;

    mutable bool m_cacheControlContainsNoCache : 1;
    mutable bool m_cacheControlContainsNoStore : 1;
    mutable bool m_cacheControlContainsMustRevalidate : 1;
    mutable double m_cacheControlMaxAge;

    mutable double m_age;
    mutable double m_date;
    mutable double m_expires;
    mutable double m_lastModified;

    // An opaque value that contains some information regarding the security of
    // the connection for this request, such as SSL connection info (empty
    // string if not over HTTPS).
    CString m_securityInfo;

    // HTTP version used in the response, if known.
    HTTPVersion m_httpVersion;

    // The id of the appcache this response was retrieved from, or zero if
    // the response was not retrieved from an appcache.
    long long m_appCacheID;

    // The manifest url of the appcache this response was retrieved from, if any.
    // Note: only valid for main resource responses.
    KURL m_appCacheManifestURL;

    // Set to true if this is part of a multipart response.
    bool m_isMultipartPayload;

    // Was the resource fetched over SPDY.  See http://dev.chromium.org/spdy
    bool m_wasFetchedViaSPDY;

    // Was the resource fetched over a channel which used TLS/Next-Protocol-Negotiation (also SPDY related).
    bool m_wasNpnNegotiated;

    // Was the resource fetched over a channel which specified "Alternate-Protocol"
    // (e.g.: Alternate-Protocol: 443:npn-spdy/1).
    bool m_wasAlternateProtocolAvailable;

    // Was the resource fetched over an explicit proxy (HTTP, SOCKS, etc).
    bool m_wasFetchedViaProxy;

    // The time at which the response headers were received.  For cached
    // responses, this time could be "far" in the past.
    double m_responseTime;

    // Remote IP address of the socket which fetched this resource.
    AtomicString m_remoteIPAddress;

    // Remote port number of the socket which fetched this resource.
    unsigned short m_remotePort;

    // The downloaded file if the load streamed to a file.
    RefPtr<File> m_downloadedFile;

    // ExtraData associated with the response.
    RefPtr<ExtraData> m_extraData;
};

inline bool operator==(const ResourceResponse& a, const ResourceResponse& b) { return ResourceResponse::compare(a, b); }
inline bool operator!=(const ResourceResponse& a, const ResourceResponse& b) { return !(a == b); }

struct CrossThreadResourceResponseData {
    WTF_MAKE_NONCOPYABLE(CrossThreadResourceResponseData); WTF_MAKE_FAST_ALLOCATED;
public:
    CrossThreadResourceResponseData() { }
    KURL m_url;
    String m_mimeType;
    long long m_expectedContentLength;
    String m_textEncodingName;
    String m_suggestedFilename;
    int m_httpStatusCode;
    String m_httpStatusText;
    OwnPtr<CrossThreadHTTPHeaderMapData> m_httpHeaders;
    time_t m_lastModifiedDate;
    RefPtr<ResourceLoadTiming> m_resourceLoadTiming;
    CString m_securityInfo;
    ResourceResponse::HTTPVersion m_httpVersion;
    long long m_appCacheID;
    KURL m_appCacheManifestURL;
    bool m_isMultipartPayload;
    bool m_wasFetchedViaSPDY;
    bool m_wasNpnNegotiated;
    bool m_wasAlternateProtocolAvailable;
    bool m_wasFetchedViaProxy;
    double m_responseTime;
    String m_remoteIPAddress;
    unsigned short m_remotePort;
    String m_downloadFilePath;
};

} // namespace WebCore

#endif // ResourceResponse_h
